#!/usr/bin/python
# -*- coding: utf-8 -*-

import os
import uuid

from cProfile import Profile

import mpfs.engine.process
from mpfs.config import settings


PLATFORM_PROFILER_DATA_FOLDER = settings.platform['profiler_data_folder']
PLATFORM_FILTER_PROFILE_HANDLERS = settings.platform['filter_profile_handlers']
PLATFORM_PROFILE_HANDLERS = settings.platform['profile_handlers']


class ProfilerMiddleware(object):
    """
    Класс для профилирования запросов к API. В него оборачивается WSGI app, например так:

    application = Flask(__name__)
    application.wsgi_app = ProfilerMiddleware(application.wsgi_app)

    За основу взят werkzeug.contrib.profiler.ProfilerMiddleware и допилен для наших нужд.
    """

    def __init__(self, app):
        self._app = app
        if not os.path.exists(PLATFORM_PROFILER_DATA_FOLDER):
            os.makedirs(PLATFORM_PROFILER_DATA_FOLDER)

    def __call__(self, environ, start_response):
        response_body = []

        def catching_start_response(status, headers, exc_info=None):
            start_response(status, headers, exc_info)
            return response_body.append

        def runapp():
            appiter = self._app(environ, catching_start_response)
            response_body.extend(appiter)
            if hasattr(appiter, 'close'):
                appiter.close()

        need_profiler = False
        uri = environ.get('PATH_INFO')
        if PLATFORM_FILTER_PROFILE_HANDLERS:
            for pattern in PLATFORM_PROFILE_HANDLERS:
                if uri.startswith(pattern):
                    need_profiler = True
                    break
        else:
            need_profiler = True

        if need_profiler:
            p = Profile()
            p.runcall(runapp)

            log = mpfs.engine.process.get_default_log()
            filepath = os.path.join(PLATFORM_PROFILER_DATA_FOLDER, uuid.uuid4().hex)
            try:
                p.dump_stats(filepath)
                log.info('Profile for %s: %s', uri, filepath)
            except IOError as e:
                log.exception('Unable to create profiler file (%s): %s', filepath, e)
        else:
            runapp()

        body = ''.join(response_body)
        return [body]
