# -*- coding: utf-8 -*-

from random import randint

from mpfs.common.util import filter_uid_by_percentage
from mpfs.common.util.user_agent_parser import UserAgentParser
from mpfs.config import settings
from mpfs.core.services.dataapi_rate_limiter_service import dataapi_rate_limiter
from mpfs.core.services.rate_limiter_service import rate_limiter
from mpfs.core.services.yarl_rate_limiter_service import yarl_rate_limiter, yarl_dataapi_rate_limiter
from mpfs.platform.common import GroupRateLimit
from mpfs.platform.exceptions import TooManyRequestsError, ServiceUnavailableError
from mpfs.platform.utils import parse_cookie, quote_string

PLATFORM_RATE_LIMITER_SPECIAL_LIMITS = settings.platform['rate_limiter']['special_limits']
PLATFORM_RATE_LIMITER_DEFAULT_USERS_COUNT = settings.platform['rate_limiter']['default_users_count']

if settings.feature_toggles['use_yarl_like_service_rate_limiter']:
    dataapi_rate_limiter_by_service = yarl_dataapi_rate_limiter
    rate_limiter_by_service = yarl_rate_limiter
else:
    dataapi_rate_limiter_by_service = dataapi_rate_limiter
    rate_limiter_by_service = dataapi_rate_limiter


class BaseRateLimiter(object):
    counter_key_special_limits = {sl['counter_key']: sl['group'] for sl in PLATFORM_RATE_LIMITER_SPECIAL_LIMITS if
                                  'counter_key' in sl}
    service = rate_limiter
    exceeded_limit_exception_cls = TooManyRequestsError
    limit_prefix = None

    def __init__(self, group):
        """
        Создаёт группу счётчиков.

        :param group: Идентификатор группы счётчиков.
        """
        self.group = group

    @property
    def enabled(self):
        return self.service.enabled

    def check(self, request, value=1):
        """
        Проверяет счётчик и в случае превышения лимита выбрасывает исключение.

        :raise TooManyRequestsError: Если клиентом превышен лимит запросов.
        :raise ServiceUnavailableError: Если превышен лимит неавторизованных запросов.

        :param request: экземпляр `mpfs.platform.common.PlatformRequest`.
        :param value: количество запросов, которые нужно посчитать при обработке
        :return: True или выбросит исключение.
        """
        if not request.check_rate_limit:
            return True

        counter_key = self.get_counter_key(request)

        group = self.get_special_limits_group(request, counter_key)

        if group and self.service.is_limit_exceeded(group, counter_key, value):
            self.handle_exceeded_limit(request)
        return True

    def get_special_limits_group(self, request, counter_key):
        limit = self.get_group_limit(request)

        if not limit:
            # Пытаемся получить спец. лимит для counter_key прописанный в `platform.rate_limiter.special_limits`.
            limit = GroupRateLimit(self.counter_key_special_limits.get(counter_key, self.get_group()))

        return limit.get_full_group_name()

    def get_counter_key(self, request):
        """
        Возвращает идентификатор счётчика.

        :rtype: str
        """
        raise NotImplementedError()

    def get_group_limit(self, request):
        if not self.limit_prefix:
            return None

        limits = request.client.limits if request.client and request.client.limits else []
        for limit in limits:
            if limit.name.startswith(self.limit_prefix):
                return limit

        return None

    def get_group(self):
        return self.group

    def handle_exceeded_limit(self, request):
        """
        Обрабатывает превышение лимита запросов.

        :param request: экземпляр `mpfs.platform.common.PlatformRequest`.
        """
        raise self.exceeded_limit_exception_cls()


class PerHandlerRateLimiter(BaseRateLimiter):
    """
    Ограничитель по количеству запросов ручки.

    Может применяться для ограничения количества ресурсов не требующих авторизации.
    """
    exceeded_limit_exception_cls = ServiceUnavailableError
    """
    Возвращаем 503 т.к. этот лимитер работает безотносительно клиента
    и поэтому не может быть причиной клиентской ошибки 4xx.
    """

    def get_counter_key(self, request):
        return request.path


class PerUserRateLimiter(BaseRateLimiter):
    """
    Ограничитель по количеству запросов пользователя.

    Может применяться для ограничения общего количества запросов пользователя в API.
    Если используется на ручке не требующей авторизации, то будет считать всех анонимусов одним пользоватлем
    и лимит будет для всех общий.
    """

    def get_counter_key(self, request):
        uid = 'anonymous'
        if request.user and request.user.uid:
            uid = request.user.uid
        return uid


class PerUserWithSeparateAnonymousRateLimiter(PerUserRateLimiter):
    """
    Ограничитель по количеству запросов пользователя.

    Полный аналог `mpfs.platform.rate_limiter.PerUserRateLimiter`, но имеет отдельную группу для анонимного пользователя.
    """

    def __init__(self, group, anonymous_group):
        super(PerUserWithSeparateAnonymousRateLimiter, self).__init__(group)
        self.anonymous_group = anonymous_group

    def get_counter_key(self, request):
        key = request.remote_addr
        if request.user and request.user.uid:
            key = request.user.uid
        return key

    def get_special_limits_group(self, request, counter_key):
        group = self.anonymous_group
        if request.user and request.user.uid:
            group = self.get_group()
        return group


class PerUserHandlerRateLimiter(BaseRateLimiter):
    """
    Ограничитель по количеству запросов пользователя к ручке.

    Может применяться для ограничения количества запросов пользователя к ручке.
    """

    def get_counter_key(self, request):
        if request.user and request.user.uid:
            uid = request.user.uid
        else:
            uid = 'anonymous'
        return quote_string('%s:%s' % (uid, request.path))


class PerPercentUserHandlerRateLimiter(PerUserHandlerRateLimiter):
    """
    Ограничитель по количеству запросов пользователя к ручке.

    Применяется на процент пользователей.

    Статистика в рейтлимитере считается для всех.
    """

    def __init__(self, group, user_percent, exceeded_limit_exception_cls=TooManyRequestsError):
        super(PerPercentUserHandlerRateLimiter, self).__init__(group)
        self.user_percent = user_percent
        self.exceeded_limit_exception_cls = exceeded_limit_exception_cls

    def handle_exceeded_limit(self, request):
        if request.user and request.user.uid:
            if not filter_uid_by_percentage(request.user.uid, self.user_percent):
                return
        raise self.exceeded_limit_exception_cls()


class PerHandlerRandomUserRateLimiter(PerUserHandlerRateLimiter):
    """
    Ограничитель по количеству запросов ручки.

    РПС нормируется на `users_count` юзеров для обхода ограничения
    по доступным лимитам на РПС и Burst.
    """
    exceeded_limit_exception_cls = ServiceUnavailableError
    """
    Возвращаем 503 т.к. этот лимитер работает безотносительно клиента
    и поэтому не может быть причиной клиентской ошибки 4xx.
    """

    def __init__(self, group, users_count=PLATFORM_RATE_LIMITER_DEFAULT_USERS_COUNT):
        super(PerHandlerRandomUserRateLimiter, self).__init__(group)
        self.users_count = users_count

    def get_counter_key(self, request):
        random_uid = randint(0, self.users_count)
        return quote_string('%s:%s' % (random_uid, request.path))


class PerYandexuidRateLimiter(BaseRateLimiter):
    """
    Ограничитель по количеству запросов по значению куки yandexuid. Значение yandexuid уникально для браузера.
    """

    def get_counter_key(self, request):
        if request.raw_headers and 'cookie' in request.raw_headers:
            cookies = parse_cookie(request.raw_headers['cookie'])
            if 'yandexuid' in cookies:
                return cookies['yandexuid']
        return 'anonymous'


class PerClientIdRateLimiter(BaseRateLimiter):
    """
    Ограничитель по идентификатору клиента.
    """

    def get_counter_key(self, request):
        return None if not request.client else request.client.id


class PerSomethingUniqueLimiter(BaseRateLimiter):
    """
    Ограничитель по какому-нибудь уникальному параметру запроса.

    Может быть использован:
        - uid
        - id из User-Agent мобильного приложения (device id для мобильных)
        - кука yandexuid, уникальна для браузера
    """

    def get_counter_key(self, request):
        if request.user and request.user.uid:
            return request.user.uid
        if request.raw_headers:
            ua = request.raw_headers.get('user-agent')
            unique_id = UserAgentParser.get_unique_id(ua)
            if unique_id:
                return unique_id

            cookie = request.raw_headers.get('cookie')
            if cookie:
                cookies = parse_cookie(request.raw_headers['cookie'])
                if 'yandexuid' in cookies:
                    return cookies['yandexuid']
        return 'anonymous'


class PerPublicDBRateLimiter(BaseRateLimiter):
    """Ограничитель по количеству запросов к публичной базе данных."""
    DEFAULT_GROUP_TMPL = 'cloud_api_datasync_public_db_%i'
    """Шаблон групп для RL.

    Используем несколько групп для RL из-за его ограничений.
    """

    def __init__(self, group=DEFAULT_GROUP_TMPL):
        super(PerPublicDBRateLimiter, self).__init__(group)

    def get_counter_key(self, request):
        return str(request.public_db_id)

    def get_group(self):
        return self.group % randint(1, 3)


class DataApiPerClientIdRateLimiter(PerClientIdRateLimiter):
    """Ограничитель по количеству запросов в DataApi по clientId."""
    DEFAULT_GROUP = settings.platform['rate_limiter']['default_groups']['dataapi']
    counter_key_special_limits = {}
    limit_prefix = 'dataapi'
    service = dataapi_rate_limiter_by_service

    def __init__(self, group=DEFAULT_GROUP):
        super(DataApiPerClientIdRateLimiter, self).__init__(group)


class DiskApiPerClientRateLimiter(PerClientIdRateLimiter):
    DEFAULT_GROUP = settings.platform['rate_limiter']['default_groups']['disk']
    limit_prefix = 'disk'
    service = rate_limiter_by_service

    def __init__(self, group=DEFAULT_GROUP):
        super(DiskApiPerClientRateLimiter, self).__init__(group)


class BatchRequestLimit(object):
    def __init__(self, limiter, processor, one_request=False):
        self.limiter = limiter
        """Инстанс рейтлимитера"""
        self.processor = processor
        """Хендлер для которого нужно вызывать этот рл"""
        self.one_request = one_request
        """Нужно ли делать один запрос в рл вместо нескольких"""


class BatchRateLimiter(object):
    """Проверяет ограничения для батчевых запросов. На примере первого запроса определяет нужный инстанс лимитера.
    Если для него включена опция one_request, то делается 1 запрос с указанием, что пришел n + 1 запрос.
    Если опция выключена то идет стандартный флоу - для каждого внутреннего запроса вызывает свой рл"""
    exceeded_limit_exception_cls = TooManyRequestsError

    def __init__(self, limits, default_limiter):
        self.limits = limits
        """Список объектов BatchRequestLimit"""
        self.default_limiter = default_limiter
        """Дефолтный обработчик. Вызывается если не подошел не один из кастомного списка лимитов"""

    @property
    def enabled(self):
        return True

    @staticmethod
    def get_request_count(requests, one_request):
        return len(requests) + 1 if one_request else 1

    def check(self, request):
        requests = request.body.get('items', [])
        if len(requests) == 0:
            return

        for limit in self.limits:
            if limit.limiter.enabled and limit.processor(request.dispatcher, request).check_request(requests[0]):
                limit.limiter.check(request, self.get_request_count(requests, limit.one_request))
                return

        self.default_limiter.check(request)


class CompositeRateLimiter(object):
    exceeded_limit_exception_cls = TooManyRequestsError

    def __init__(self, limiters):
        self.limiters = limiters

    @property
    def enabled(self):
        for limiter in self.limiters:
            if limiter.enabled:
                return True

        return False

    def check(self, request):
        for limiter in self.limiters:
            if limiter.enabled:
                limiter.check(request)
