# -*- coding: utf-8 -*-
from collections import OrderedDict
from inspect import isclass
from mpfs.platform.common import HTTP_VERBS, logger
from mpfs.platform.exceptions import MethodNotAllowedError
from mpfs.platform.handlers import BasePlatformHandler


class BaseResource(BasePlatformHandler):
    relations = {}
    """
    Маппинг операций поддерживаемых ресурсом на соответствующие классы хэндлеров

    Не HTTP-методы добавляются к ресурсу как AdHoc-методы.
    >>> class MyResourcesCollection(BaseResource):
    >>>     relations = {
    >>>         'GET': GetResourcesHandler,
    >>>         'download': adhoc('GET', GetResourcesDownloadHandler),
    >>>         '(?P<path>[^/]+)': MyResourcesResource,
    >>>     }
    """

    _router = None

    @property
    def router(self):
        return self._router or super(BaseResource, self).router

    def __init__(self, *args, **kwargs):
        if 'parent' not in kwargs and 'router' in kwargs:
            self._router = kwargs.get('router')
        super(BaseResource, self).__init__(*args, **kwargs)
        self._relations = self.relations
        self.initialize_relations(*args, **kwargs)

    def initialize_relations(self, *args, **kwargs):
        self.relations = OrderedDict()
        kwargs['parent'] = self
        for name, handler in self._relations.items():
            if isinstance(handler, (tuple, list)):
                handler = handler[0]
            if isclass(handler) or hasattr(handler, '__call__'):
                handler = handler(*args, **kwargs)
            self.relations[name] = handler

    def dispatch(self, request, *args, **kwargs):
        """
        Просто передаёт управление хэндлеру соответствующему HTTP-методу запроса
        или швыряет `PlatformNotImplemented`.
        """
        self.request = request
        self.args = args
        self.kwargs = kwargs

        # Обрабатываем метод OPTIONS
        if request.method == 'OPTIONS':
            allow_methods = ', '.join(self.get_allowed_methods(request))
            headers = {
                'Allow': allow_methods,
                'Access-Control-Allow-Methods': allow_methods,
                'Access-Control-Allow-Headers': ', '.join(self.get_allowed_headers(request)),
            }
            headers = {k: v for k, v in headers.iteritems() if v}
            result = (200, None, headers)
        else:
            handler = self.relations.get(request.method, None)
            if not handler:
                raise MethodNotAllowedError()
            result = handler.dispatch(request, *args, **kwargs)
        return self.make_response(result)

    def make_response(self, ret):
        resp = super(BaseResource, self).make_response(ret)
        # CORS
        resp.headers['Access-Control-Allow-Methods'] = ', '.join(self.get_allowed_methods(self.request))
        return resp

    def get_http_relations(self):
        return dict([(m, h) for m, h in self.relations.iteritems() if m in HTTP_VERBS])

    def get_allowed_headers(self, request, allow_headers=None):
        """
        Возвращает список заголовков для CORS заголовка Access-Control-Allow-Headers

        :param list|set|tuple allow_headers: Список заголовков, которые будут добавлены дополнительно к тем,
                                             что будут извлечены из хэндлеров.
        :rtype: list
        """
        headers = []
        for handler in self.get_http_relations().itervalues():
            assert isinstance(handler, BasePlatformHandler)
            headers.extend(handler.headers.get_fields().keys())
        if allow_headers:
            headers.extend(allow_headers)
        case_insensitive_uniq_headers = {k.lower(): k for k in headers}
        return case_insensitive_uniq_headers.values()

    def get_exposed_headers(self, request):
        """Возвращает список заголовков для CORS заголовка Access-Control-Expose-Headers"""
        headers = []
        for rel, handler in self.get_http_relations().iteritems():
            assert isinstance(handler, BasePlatformHandler)
            headers.extend(handler.get_exposed_headers(request))
        return list(set(headers))

    def get_allowed_methods(self, request):
        """
        Возвращает список доступных методов для ресурса соответствующего запросу.

        :param request: Пока не используется, но нужен чтобы была возможность даинамически вычислять список доступных
                        для конкретного объекта действий.
        :rtype: list
        """
        return self.get_http_relations().keys() + ['OPTIONS']

    @property
    def url_resolvers(self):
        from mpfs.platform.routers import url
        is_self_registered = False
        ret = []
        # Добавляем URL для подресурсов
        for name, resource in self.relations.iteritems():
            # достаём default_kwargs ресурса, указанные при регистрации в роутере
            res_args = self._relations[name][1:] if isinstance(self._relations[name], (tuple, list)) else []
            if name in HTTP_VERBS:
                if not is_self_registered:
                    ret.append(url(r'^/?$', self, *res_args))
                    is_self_registered = True
            elif isinstance(resource, AdHocResource):
                ret.append(url(r'^/%s/?$' % name, resource.url_resolvers, *res_args))
            elif isinstance(resource, BaseResource):
                ret.append(url(r'^/%s' % name, resource.url_resolvers, *res_args))
            else:
                logger.error_log.error('Unable to register object %s as relation for %s' % (resource, self))
        return ret

    def get_subresources(self, depth=None, hidden=False):
        """Возвращает все подресурсы на depth уровней вглубь"""
        ret = []
        if depth is not None:
            if depth > 0:
                depth -= 1
            else:
                return ret

        for rel, res in self.relations.iteritems():
            if isinstance(res, BaseResource):
                ret += res.get_subresources(depth=depth, hidden=hidden)
                if not hidden:
                    if not res.hidden:
                        ret.append(res)
                else:
                    ret.append(res)
        return ret


class NamespaceResource(BaseResource):
    """
    Ресурс не содержащий хэндлеров, служит для группировки других ресурсов.

    Т.к. предполагается что NamespaceResource не имеет собственных HTTP-методов и не является ресурсом
    в терминах HTTP, то наследники этого класса принудительно скрываются из полигона.
    """
    hidden = True

    def initialize_relations(self, *args, **kwargs):
        # Заготовочка для автолинковки ресурсов и всяких там self-discovery.
        # self._relations['GET'] = NamespaceHandler
        super(NamespaceResource, self).initialize_relations(*args, **kwargs)


def adhoc(method, handler_cls, hidden=False):
    """
    Возвращает новый класс ресурса созданный специально для AdHoc метода.

    :param method: HTTP-метод по которому будет доступен AdHoc-метод.
    :param handler_cls: Класс хэндлера AdHoc-метода.
    :param hidden: Флаг скрывает хендлер от схемы.
    :rtype: mpfs.platform.resources.BaseResource
    """
    resource = type(
        'AdHoc%sResource' % handler_cls.__name__,
        (AdHocResource,),
        {'relations': {method: handler_cls}, 'hidden': hidden}
    )
    return resource


class AdHocResource(BaseResource):
    pass


class InplaceResource(BaseResource):
    pass


def res(relations=None, **kwargs):
    """
    Создаёт ресурс с атрибутами kwargs

    Предназначен для использования там где нужно повесить на url ручку (хэндлер), а не полноценный ресурс.
    """
    attrs = {}
    attrs.update(kwargs)
    attrs['relations'] = relations or {}
    resource = type('InplaceResource', (InplaceResource,), attrs)
    return resource
