# -*- coding: utf-8 -*-
import types
import urllib
import urlparse
from inspect import isclass
import re

from mpfs.common.static import tags
from mpfs.common.util.ordereddict import OrderedDict
from mpfs.platform.async_mode import gevent_local_variables
from mpfs.platform.handlers import BasePlatformHandler
from mpfs.platform.resources import BaseResource
from mpfs.platform.utils import prepare_qs_val
from mpfs.platform.dispatchers import InternalDispatcher


@gevent_local_variables('request')
class BaseRouter(object):
    map = None
    reverse_map = None
    modules = None
    request = None
    """
    Запрос в рамках которого работает роутер.

    Если задан, то роутер будет получать base_url для генерации абсолютных URL из этого запроса,
    и если запрос внутренний, то роутер будет вставлять UID пользователя из запроса в генерируемые URL.
    """
    _base_url = None

    def __init__(self, base_url=None):
        """
        Создаёт новый экземпляр роутера.

        :param base_url: Если задан, то get_link будет возвращать абсолютные URL.
        :return:
        """
        self.base_url = base_url

    def build_reverse_map(self):
        reverse_map = OrderedDict()
        for link, handler in self.map.iteritems():
            reverse_map[handler] = link
        self.reverse_map = reverse_map

    def get_link(self, handler, query=None,  rfc6570=False, relative=False):
        """
        Возвращает кортеж из 3х элементов: Http-метода, URL ресурса и шаблонизирован URL или нет.

        If handler require any query arguments, tries to fill them using provided query dict.
        If any required by handler arguments is missing in query dict, return template url.
        TODO: If hyperschema if True then return URI template according to http://tools.ietf.org/html/rfc6570
        """
        raise NotImplementedError()

    def set_request(self, request):
        self.request = request
        request.router = self

    def set_base_url(self, value):
        self._base_url = value

    def get_base_url(self):
        if self._base_url:
            return self._base_url
        base_url = None
        if self.request:
            parsed_uri = urlparse.urlparse(self.request.url)
            base_url = '{uri.scheme}://{uri.netloc}'.format(uri=parsed_uri)
        return base_url

    base_url = property(get_base_url, set_base_url)

    def get_param_stub(self, rfc6570=False):
        return '%%(%s)s' if not rfc6570 else '{%s}'


class RegexRouter(BaseRouter):
    UID_PATTERN = r'(?P<uid>\d+)'

    _handler_cls_lookup_cache = None
    _handler_obj_lookup_cache = None

    def __init__(self, resource_cls):
        super(RegexRouter, self).__init__()
        self.patterns = None
        self.resolvers = None
        self.root_resource = None
        self.set_resolvers(resource_cls)

    def set_resolvers(self, resource_cls):
        self.root_resource = resource_cls(router=self)
        self.resolvers = self.root_resource.url_resolvers
        self.patterns = reduce(lambda ret, r: ret + r.get_patterns(), self.resolvers, [])
        hlc_cache = {}
        hlo_cache = {}
        for pattern in self.patterns:
            if isinstance(pattern.handler, BasePlatformHandler):
                handler_cls = type(pattern.handler)
                handler = pattern.handler
                if handler_cls not in hlc_cache:
                    hlc_cache[handler_cls] = {}
                if handler not in hlo_cache:
                    hlo_cache[handler] = {}
                path_params = reduce(lambda ret, p: ret + p.groupindex.keys(), pattern.patterns, [])
                required_qs_params = [k for k, f in pattern.handler.query.get_fields().iteritems() if f.required]
                params = tuple(path_params + required_qs_params)
                hlc_cache[handler_cls][params] = pattern
                hlo_cache[handler][params] = pattern
        self._handler_cls_lookup_cache = hlc_cache
        self._handler_obj_lookup_cache = hlo_cache

    def get_link(self, handler, params=None,  rfc6570=False, relative=False, force_request_mode=None):
        """
        Возвращает кортеж содержащий ссылку на ресурс.

        :param handler: Класс или экземпляр хэндлера. Осторожно, если один и тот же класс хэндлера используется
                        в нескольких версиях API, то роутер вернёт ссылку на тот, который был зарегистрирован раньше.
        :param dict params: Параметры для заполнения URL. Если целевой хэндлер принимает обязательные параметры, которых
                           нет в этом дикте, то возвращённый URL будет шаблонизирован, т.е. значения обазательных
                           параметров в URL будут заменены шаблонами.
        :param bool rfc6570: Использовать для генерации шаблонизированных URL RFC6570 или нет. Если нет,
                             то шаблнизированный URL будет содержать именованные шаблны python.
        :param bool relative: Заставляет метод вернуть относительный URL даже если `self.base_url` задан.
        :param str force_request_mode: Форсированный выбор типа запроса.
                                       Возможные значения `tags.platform.EXTERNAL`, `tags.platform.INTERNAL`
                                       или `None`(режим выбирается в зависимости от запроса).
        :return: Возвращает кортеж из 3х элементов: Http-метода, URL ресурса и флаг шаблонизирован URL или нет.
        :rtype: tuple
        """
        available_modes = (None, tags.platform.INTERNAL, tags.platform.EXTERNAL)
        if force_request_mode not in available_modes:
            raise TypeError('Wrong "force_request_mode". Got: "%s". Excepted: %s.' % (force_request_mode, available_modes))

        is_internal_mode = False
        if (force_request_mode is None and self.request.mode == tags.platform.INTERNAL or
                force_request_mode == tags.platform.INTERNAL):
            is_internal_mode = True

        params = params or {}
        # вытащили все паттерны связанные с хэндлером
        if not isclass(handler):
            handler_patterns = self._handler_obj_lookup_cache.get(handler, {})
        else:
            handler_patterns = self._handler_cls_lookup_cache.get(handler, {})

        if not handler_patterns:
            return None

        # теперь надо выбрать самый подходящий по параметрам паттерн
        # если параметры не заданы, то самый правильный линк -- это тот которому надо меньше всего параметров
        if not params:
            url_pattern = min(handler_patterns.iteritems(), key=lambda item: len(item[0]))[1]
        else:
            # если параметры заданы, то нужно искать линк с наибольшим пересечением по параметрам с переданными параметрами
            param_set = set(params.keys())
            url_pattern = max(handler_patterns.iteritems(), key=lambda item: len(set(item[0]) & param_set))[1]

        # Собираем полный паттерн URL
        url_chunks = url_pattern.chunks
        # Разэкранируем закэранированное
        url_chunks = map(lambda s: s.replace('\\', ''), url_chunks)
        # Добавляем UID во внутренний URL.
        if is_internal_mode:
            # костыль, чтобы поддержать ссылки для внутреннего api без uid'а, и с uid'ом
            _, uid, _, _ = InternalDispatcher.split_path(self.request._raw_path)
            if uid is not None:
                uid = self.request.user.uid if self.request.user and self.request.user.uid else self.UID_PATTERN
                url_chunks.insert(1, uid)
        # Добавляем id сервиса, если он пришел с идентификатором для авторизации по куке
        if self.request.cookie_auth_client_id:
            url_chunks.insert(0, self.request.cookie_auth_client_id)
        pattern = '/%s' % '/'.join(url_chunks)
        pattern = pattern.rstrip('/')

        # Заменяем именованные группы на значения параметров или шаблоны
        param_stub = self.get_param_stub(rfc6570)
        url, is_templated = URLPattern.expand_pattern(pattern, params, stub=param_stub)

        # Добавляем querystring
        if isinstance(handler, BasePlatformHandler) or issubclass(handler, BasePlatformHandler):
            q = OrderedDict()
            templates = {}
            for k, v in handler.query._fields.iteritems():
                is_specified = k in params
                if v.required or is_specified:
                    if is_specified:
                        v = params.get(k)
                        q[k] = prepare_qs_val(v)
                    else:
                        is_templated = True
                        templates[k] = param_stub % k
            # Добавляем client_id и clien_name во внутренние URL.
            if is_internal_mode:
                if self.request.args:
                    if 'client_id' in self.request.args:
                        q['client_id'] = prepare_qs_val(self.request.args['client_id'])
                    if 'client_name' in self.request.args:
                        q['client_name'] = prepare_qs_val(self.request.args['client_name'])
            # url-энкодим параметры со значениями
            qs = urllib.urlencode([(k, v) for k, v in q.iteritems()])
            # добавляем некодированные шаблонизированные параметры
            templated_qs = '&'.join(['%s=%s' % (k, t) for k, t in templates.iteritems()])
            qs = '&'.join(filter(None, [qs, templated_qs]))
            if qs:
                url = '%s?%s' % (url, qs)
            if self.base_url and not relative:
                url = '%s%s' % (self.base_url.rstrip('/'), url)
        return url_pattern.method, url, is_templated

    def resolve(self, request):
        """
        Резолвит запрос в URLResolverMatch

        :rtype: URLResolverMatch
        """
        for url in self.resolvers:
            match = url.resolve(request.path)
            if match:
                return match

    def get_allowed_methods(self, request):
        """
        Возвращает список поддерживаемых ресурсом HTTP-методов.
        """
        match = self.resolve(request)
        if match:
            get_allowed_methods = getattr(match.handler, 'get_allowed_methods', None)
            if get_allowed_methods and hasattr(get_allowed_methods, '__call__'):
                return get_allowed_methods(request)
        return []


def url(pattern, resource, kwargs=None):
    """
    Регистрирует ресурсы в роутере

    :param pattern: Регулярка соответствующая данному URL.
    :param resource: Ресурс или модуль содержащий атрибут url_patterns со списокм URL-паттернов или список URL-патернов.
    :param kwargs: Значения аргументов передаваемых ресурсу по-умолчанию при каждом запросе.
    :rtype: URLResolver
    """
    if isinstance(resource, (list, tuple, types.ModuleType)):
        return _URLModuleResolver(pattern, resource, kwargs)
    elif isclass(resource) and issubclass(resource, BaseResource):
        return _URLModuleResolver(pattern, resource().url_resolvers)
    elif isinstance(resource, BaseResource):
        return _URLPatternResolver(pattern, resource, kwargs)
    else:
        raise Exception('Unable to register object %s in router.' % resource)


class URLPattern(object):
    RE_PATTERN_BEGINNING = re.compile(r'^\^?/?')
    RE_PATTERN_ENDING = re.compile(r'/?\??\$?$')
    RE_PATTERN_GROUP = re.compile(r'\(\?P<([^\)]+)>[^\)]+\)')

    def __init__(self, method, patterns, handler):
        self.method = method
        self.patterns = patterns
        self.handler = handler

    @property
    def chunks(self):
        ret = []
        for p in self.patterns:
            chunk = self.RE_PATTERN_BEGINNING.sub('', getattr(p, 'pattern', p))
            chunk = self.RE_PATTERN_ENDING.sub('', chunk)
            ret.append(chunk)
        return ret

    @property
    def pattern(self):
        return '/%s' % '/'.join(self.chunks)

    @property
    def parameters(self):
        return self.get_parameters(self.pattern)

    @classmethod
    def get_parameters(cls, pattern):
        return cls.RE_PATTERN_GROUP.findall(pattern)

    def expand(self, params=None, stub=None):
        params = params or {}
        return self.expand_pattern(self.pattern, params, stub=stub)

    @classmethod
    def expand_pattern(cls, pattern, params=None, stub=None):
        params = params or {}
        stub = stub or '%%(%s)s'
        # если pattern -- скомпилированная регэкспа, то достаём из неё исходный паттерн
        pattern = getattr(pattern, 'pattern', pattern)
        is_templated = [False]
        def match_param(match):
            param_name = match.group(1)
            if param_name in params:
                v = params[param_name]
                return prepare_qs_val(v)
            else:
                is_templated[0] = True
                return stub % param_name
        return cls.RE_PATTERN_GROUP.sub(match_param, pattern), is_templated[0]


class URLResolverMatch(object):
    def __init__(self, handler, args, kwargs):
        self.handler = handler
        self.args = args
        self.kwargs = kwargs


class URLResolver(object):
    """Умеет резолвить path в URLResolverMatch"""
    def resolve(self, path):
        """Резолвит path в URLResolverMatch"""
        raise NotImplementedError()

    def get_patterns(self, patterns=None):
        """Возвращает список всех зарегистрированных"""
        raise NotImplementedError()


class _URLPatternResolver(URLResolver):
    def __init__(self, regex, resource, default_kwargs=None):
        self.regex = re.compile(regex, re.UNICODE | re.IGNORECASE)
        self.resource = resource
        self.default_kwargs = default_kwargs or {}

    def resolve(self, path):
        match = self.regex.search(path)
        if match:
            kwargs = match.groupdict() or {}
            if kwargs:
                args = ()
            else:
                args = match.groups()
            kwargs.update(self.default_kwargs)
            return URLResolverMatch(self.resource, args, kwargs)

    def get_patterns(self, patterns=None):
        patterns = patterns or []
        patterns = patterns + [self.regex]
        actions = self.resource.get_http_relations()
        return [URLPattern(m, patterns, h) for m, h in actions.iteritems()]


class _URLModuleResolver(URLResolver):
    def __init__(self, regex, resolvers, default_kwargs=None):
        self.regex = re.compile(regex, re.UNICODE | re.IGNORECASE)
        self._resolvers = resolvers
        self.default_kwargs = default_kwargs or {}

    @property
    def resolvers(self):
        patterns = getattr(self._resolvers, 'url_patterns', self._resolvers)
        return patterns

    @property
    def url_resolvers(self):
        return self.resolvers

    def resolve(self, path):
        match = self.regex.search(path)
        if match:
            new_path = path[match.end():]
            for resolver in self.resolvers:
                sub_match = resolver.resolve(new_path)
                if sub_match:
                    sub_match_kwargs = dict(match.groupdict(), **self.default_kwargs)
                    sub_match_kwargs.update(sub_match.kwargs)
                    return URLResolverMatch(sub_match.handler, sub_match.args, sub_match_kwargs)

    def get_patterns(self, patterns=None):
        patterns = patterns or []
        patterns = patterns + [self.regex]
        return reduce(lambda ret, r: ret + r.get_patterns(patterns), self.resolvers, [])
