# -*- coding: utf-8 -*-
"""
В данном контексте "сериализация" - это процесс формирования представления объекта.
В данном контексте "объект" - это любой набор данных, будь то экземпляр класса dict, list, object, string и т.д.

Сериализаторы отвечают за создание конечного представления объектов возвращаемых клиенту.

Сериализаторы это не форматтеры.
Представление характеризуется набором объектов, атрибутов и т.п., а формат относится больше к способу форматирования
представления. Например список и хэш-таблица - это представления данных, а JSON, XML, YAML - это способы форматирования.

Сериализаторы могут изменять структуру объектов и представление их атрибутов.
"""
import copy
from inspect import isclass
import sys
import mpfs

from mpfs.platform import get_api_mode
from mpfs.common.static import tags
from mpfs.platform.exceptions import ValidationError, FieldValidationError, NoDataProvidedError, InternalServerError
from mpfs.platform.fields import ParentMethodField, ParentAttrField, SerializerField, IntegerField, HalLinkField, \
    StringField
from mpfs.platform.common import logger


class SerializerMetaClass(type):
    """Метакласс обеспечивающий механизм наследования в атрибуте `fields` сериализаторов."""
    def __new__(mcs, name, parents, attrs):
        fields = {}
        # Закидываем родительские филды, если таковые имеются, чтоб поддержать наследование филдов в филдсетах.
        for parent in parents:
            if hasattr(parent, 'fields'):
                parent_fields = {k: copy.copy(f) for k, f in parent.fields.iteritems()}
                fields.update(parent_fields)

        # Обновляем собранные из родителей филды собственными филдами создаваемого класса.
        fields.update(attrs.get('fields', {}))
        # Фильтруем обнулённые филды
        fields = {k: v for k, v in fields.iteritems() if v is not None}

        attrs['fields'] = fields

        # Связываем филды с классом сериализатора.
        cls = super(SerializerMetaClass, mcs).__new__(mcs, name, parents, attrs)
        for name, field in cls.fields.iteritems():
            field.name = name
            field.parent = cls
            # если сериализатор является сериализатором патча объекта, то делаем все поля опциональными
            if cls.is_patch:
                field.required = False

        # Пороверяем, чтобы pbid не повторялись у филдов.
        pbids = {}
        for name, field in cls.fields.iteritems():
            if field.pbid is not None:
                if field.pbid in pbids:
                    msg = 'Protobuf field id %s in %s is already taken by "%s" field' % (
                        field.pbid, cls.__name__, pbids[field.pbid])
                    logger.error_log.error(msg)
                    raise ValueError(msg)
                else:
                    pbids[field.pbid] = name

        return cls

    def __str__(cls):
        return '%s.%s' % (cls.__module__, cls.__name__)


class FormatOptions(object):
    numeric_choices = False
    """Использовать номера опций вместо идентификаторов при сериализации/десериализации."""

    datetime_as_timestamp = False
    """Использовать timestamp вместо ISO8601 при сериализации/десериализации."""

    raw_binary = False
    """Не кодировать бинарные поля в base64."""

    filter_fields = True
    """Поддерживать вывод не всех атрибутов объектов, если в запросе передан параметр fields."""


class BaseSerializer(object):
    """
    Базовый класс сериализатора.

    Задаёт интерфейс всех сериализаторов, позволяет фильтровать набор возвращаемых аттрибутов объекта
    и обрабатывать аттрибуты объекта с помощью полей (:class:`mpfs.platform.fields.Field`).

    Например:

    >>> from mpfs.platform.v1.disk.fields import MpfsPathField
    >>> class MySerializer(BaseSerializer):
    ...     visible_fields = ['name', 'path']
    ...     path = MpfsPathField(source='id')
    ...
    >>> file = {'name': 'test.txt', 'id': '123456:/disk/test.txt', 'secret_field': '(O_o)'}
    >>> serializer = MySerializer(file)
    >>> serializer.data
    {'name': 'test.txt', 'path': 'disk:/test.txt'}
    >>> serializer = MySerializer(data=serializer.data)
    >>> serializer.object
    {'name': 'test.txt', 'id': '123456:/disk/test.txt'}
    """

    __metaclass__ = SerializerMetaClass

    visible_fields = []
    """Список имён атрибутов объекта, которые будет содержать представление.
    Если не задан, то представление будет содержать все атрибуты объекта.
    """

    excluded_fields = []
    """Список имён атрибутов объекта, которые будут исключены из представления."""

    native_client_fields = []
    """Список имен атрибутов объекта, которые будут исключены из представления для не нативных Дисковых клиентов"""

    internal_only_fields = []
    """Список имён атрибутов объекта, которые будут исключены из представления во внешнем API."""

    fields = {}
    """Филды сериализатора в формате {<имя атрибута>: <экземпляр филда>, ...}"""

    is_patch = False
    """
    Является сериализатор сериализатором патча объекта или нет.

    Сериализатор патча отличается от сериализатора объекта тем, что все поля в сериализаторе патча опциональны.
    """

    router = None
    """Роутер используемый для генерации HAL-контролов в представлении."""

    hal = False

    many = False

    _object = None

    initial_data = None

    initial_files = None

    format_options = FormatOptions

    def __init__(self, *args, **kwargs):
        """
        Создаёт новый экземпляр сериализатора.

        :param obj: Объект для сериализации.
        :param data: Данные для заполнения объекта.
        :param files: Файлы для заполнения объекта.
        :param visible_fields: `self.visible_fields` будет равно пересечению `visible_fields` с заданными в классе.
        :param excluded_fields: `self.excluded_fields` будет равно объединению с `excluded_fields` заданными в классе.
        :param router: Роутер используемый для генерации HAL контролов, если не задан,
                       то HAL будет отсутствовать в представлении.
        :param hal: Дополнять представление HAL контролами или нет.
        :param many: Пока не используется, но нужно пренести логику обработки списков объектов из
                     `:class:mpfs.platform.fields.SerializerField`.
        :param format_options: Параметры формата данных зависящие от используемого форматтера.
        :param fail_on_missing_required: Кидать ошибку если при десериализации не найдено обязательное значение.
        """
        self.fields = {k: copy.copy(f) for k, f in type(self).fields.iteritems()}
        self._fail_on_missing_required = False
        for name, field in self.fields.iteritems():
            field.parent = self
        self.initialize(*args, **kwargs)

    def initialize(self, obj=None, data=None, files=None, visible_fields=None, excluded_fields=None, router=None,
                   hal=True, many=False, format_options=FormatOptions,
                   fail_on_missing_required=False, *args, **kwargs):
        self._object = obj
        self.initial_data = data
        self.initial_files = files
        self.format_options = format_options
        self.excluded_fields = type(self).excluded_fields
        self.visible_fields = type(self).visible_fields
        self.router = router
        self.hal = hal
        self.many = many
        self._fail_on_missing_required = fail_on_missing_required

        if visible_fields and (not format_options or format_options.filter_fields):
            vfs = []
            for vf in visible_fields:
                if vf.split('.', 1)[0] in type(self).visible_fields:
                    vfs.append(vf)
            self.visible_fields = vfs

        if excluded_fields:
            self.excluded_fields = list(set(self.__class__.excluded_fields) | set(excluded_fields))

        if get_api_mode() == tags.platform.EXTERNAL:
            self.excluded_fields = list(set(self.excluded_fields + type(self).internal_only_fields))

    def __str__(self):
        return '%s.%s' % (self.__module__, self.__class__.__name__)

    @classmethod
    def get_subserializers(cls, serializer=None, memo=None):
        serializer = serializer or cls
        serializer = serializer if isclass(serializer) else type(serializer)
        memo = memo or []
        for field in serializer.fields.values():
            if isinstance(field, SerializerField):
                if field.serializer_cls not in memo:
                    memo.append(field.serializer_cls)
                    cls.get_subserializers(field.serializer_cls, memo=memo)
        return memo

    def get_links(self):
        """
        Возвращает dict содержащий HAL ссылки на доступные с сериализуемым объектом действия.

        Вынесен в качестве метода, а не в качестве подкласса т.к. возникает циклический импорт в ситуации,
        когда хэндлер содержит сериализатор, а сериализатор содержит атрибут/подкласс Links,
        содержащий экземпляры HalLinkField, которые ссылаются на хэндлеры.
        """
        return {}

    def restore_object(self, attrs, instance=None):
        """
        Десериализует dict атрибутов в экзепляр объекта (который тоже может быть dict'ом).

        Для правильной раскладки данных по аттрибутам объекта переопределите этот метод в дочерних сериализаторах.
        """
        self.validate(attrs)
        if instance is not None:
            if isinstance(instance, dict):
                for name, value in attrs.iteritems():
                    instance[name] = value
            else:
                for name, value in attrs.iteritems():
                    setattr(instance, name, value)
            return instance
        return attrs

    def validate(self, attrs):
        """
        Выполняет проверку корректсности данных подготовленных для заполнения объекта.
        Выбрасывает исключение ValidationError в случае некорректных данных.

        С помощью этого метода в дочерних классах можно проверять зависимости между несколькими атрибутами данных,
        например когда обязательно наличие одного из двух полей.

        :param dict attrs: Проверенные и преобразованные данные.
        """
        pass

    def get_fields(self):
        return self.fields

    def get_visible_fields(self):
        """Возвращает список имён атрибутов объекта, которые попадут в представление."""
        return self.visible_fields

    def get_excluded_fields(self):
        """Возвращает список имён атрибутов объекта, которые будут исключены из представления."""
        return self.excluded_fields

    def get_excluded_fields_of_request(self):
        """Возвращает список имен атрибутов объекта, которые надо исключить в рамках запроса"""
        excluded = list(self.get_excluded_fields())

        if (get_api_mode() == tags.platform.EXTERNAL
                and not (self.router and self.router.request and self.router.request.is_disk_native_client())):
            excluded += self.native_client_fields

        return excluded

    @property
    def data(self):
        """
        Возвращает представление объекта.

        :return: Представление объекта.
        :rtype: dict
        """
        return self._from_native(self._object)

    @property
    def object(self):
        """Возвращает объект инкапсулированный в сериализатор."""
        if self.initial_data is not None or self.initial_files is not None:
            return self._to_native(data=self.initial_data, files=self.initial_files)
        else:
            return self._object

    def prepare_object(self, obj):
        """
        Подготавливает объект для сериализации и возвращает dict используемый для формирования представления объекта.

        В дочерних классах в этом методе можно добавить необходимые сериализатору вычисляемые поля.
        """
        if isinstance(obj, dict):
            return obj
        else:
            return self.get_dict(obj)

    #
    # Internal methods.
    #

    @staticmethod
    def get_dict(obj):
        """Правильно делает `obj.__dict__`."""
        result = {}
        for attr in dir(obj):
            try:
                result[attr] = getattr(obj, attr)
            except Exception:
                # некоторые property могут вызывать ошибки при попытке получить их,
                # поэтому просто игнорируем эти атрибуты
                pass
        return result

    def _from_native(self, obj):
        """Выполняет преобразование внутреннего объекта во внешнее представление."""
        data = self.prepare_object(obj)
        ret = self._apply_fields(data)
        if self.router and self.hal:
            data.update(ret)
            links = self._get_hal_links(data)
            if links is not None:
                if '_links' not in ret:
                    ret['_links'] = {}
                ret['_links'].update(links)
        if self.get_excluded_fields():
            ret = self._exclude_fields(ret)
        return ret

    def _to_native(self, data, files=None):
        """Выполняет проверку и преобразование сырых данных во внутренне пердставление."""
        reverted_data = {}
        if data is not None or files is not None:
            if not isinstance(data, dict):
                raise ValidationError()

            for field_name, field in self.get_fields().iteritems():
                if not field.read_only:
                    field.initialize(parent=self, field_name=field_name)
                    source = field.source or field_name
                    if field_name in data or field.required:
                        raw_value = data.get(field_name)
                        try:
                            value = field.clean(raw_value)
                            # если поле вложенное, то создаём необходимые вложенные объекты
                            source_chunks = source.split('.')
                            target_obj = reverted_data
                            for chunk in source_chunks[:-1]:
                                if chunk not in target_obj:
                                    target_obj[chunk] = {}
                                target_obj = target_obj[chunk]
                            target_obj[source_chunks[-1]] = value
                        except ValidationError, e:
                            e = FieldValidationError(inner_exception=e, name=field_name, message=e.message,
                                                     description=e.description)
                            raise e, None, sys.exc_info()[2]
        else:
            raise NoDataProvidedError()
        return self.restore_object(reverted_data, instance=getattr(self, '_object', None))

    def _exclude_fields(self, data):
        """Возвращает представление из которого исключены атрибуты заданные в :attr:`.excluded_fields`"""
        ret = {}
        excluded_fields = self.get_excluded_fields()
        for k, v in data.iteritems():
            if k not in excluded_fields:
                ret[k] = v
        return ret

    def _apply_fields(self, data):
        """Возвращает представление, атрибуты которого были преобразованы с помощью полей."""
        visible_fields = self.get_visible_fields()
        # не применяем филды, которых нет в списке `self.visible_fields` или которые есть в `self.excluded_fields`
        if visible_fields:
            # Если в visible_fields передан податрибут (например, "some_field.subfield.subsubfield"),
            # то нам нужно включить в список visible_fields корневой атрибут этого податрибута ("some_field").
            visible_fields = [f.split('.', 1)[0] for f in visible_fields]
            fields = {k: v for k, v in self.get_fields().iteritems() if k in visible_fields}
        else:
            visible_fields = self.get_fields().keys()
            fields = {k: v for k, v in self.get_fields().iteritems()}

        excluded_fields_of_request = self.get_excluded_fields_of_request()
        if excluded_fields_of_request:
            excluded_fields = [f.split('.', 1)[-1] for f in excluded_fields_of_request]
            visible_fields = [f for f in visible_fields if f not in excluded_fields]
            fields = {k: v for k, v in fields.iteritems() if k not in excluded_fields}

        ret = {}
        for name in visible_fields:
            field = fields.get(name, None)
            source = getattr(field, 'source', None) or name
            field_required = False
            if field is not None:
                field.initialize(self)
                field_required = field.required
            # обрабатываем вложенные поля в source
            if '.' in source:
                source_chunks = source.split('.')
            else:
                source_chunks = [source]
            is_set = True
            value = data
            for chunk in source_chunks:
                if value is None:
                    is_set = False
                    break

                if chunk in value:
                    is_set = True
                    value = value[chunk]
                elif field_required:
                    is_set = True
                    value = None
                    if self._fail_on_missing_required:
                        is_set = False
                    break
                else:
                    is_set = False
                    value = None
                    break

            if is_set is False and value is None and field_required:
                if self._fail_on_missing_required:
                    raise InternalServerError()
            # Отдельно обрабатываем ParentMethodField и ParentAttrField т.к. они могут генерировать значение сами
            # без исходного значения в data.
            if is_set or isinstance(field, (ParentMethodField, ParentAttrField)):
                result = field.from_native(value) if field is not None else value
                ret[name] = result
        return ret

    def _get_hal_links(self, data):
        links = None
        if self.get_links():
            links = {}
            for name, field in self.get_links().iteritems():
                field.initialize(self)
                links[name] = field.from_native(data)
        return links


class ListSerializer(BaseSerializer):
    DEFAULT_OFFSET = 0
    DEFAULT_LIMIT = 20
    handler_cls = None
    fields = {
        'total': IntegerField(pbid=2, help_text=u'Общее количество элементов в списке'),
        'limit': IntegerField(default=DEFAULT_LIMIT, pbid=3, help_text=u'Количество элементов на странице'),
        'offset': IntegerField(default=DEFAULT_OFFSET, pbid=4, help_text=u'Смещение от начала списка'),
    }

    def get_links(self):
        offset = self._object.get('offset', self.DEFAULT_OFFSET)
        limit = self._object.get('limit', self.DEFAULT_LIMIT)
        total = self._object.get('total', -1)
        ret = super(ListSerializer, self).get_links()
        if -1 < total <= offset + limit:
            ret['next'] = HalLinkField(None)
        else:
            ret['next'] = HalLinkField(self.handler_cls, context={'offset': offset + limit})

        if offset - limit < 0:
            ret['previous'] = HalLinkField(None)
        else:
            ret['previous'] = HalLinkField(self.handler_cls, context={'offset': offset - limit})

        return ret


class ErrorSerializer(BaseSerializer):
    visible_fields = ['error', 'description', 'message']
    fields = {
        'error': StringField(required=True, source='code', pbid=1, help_text=u'Уникальный код ошибки'),
        'description': StringField(required=True, pbid=2, help_text=u'Техническое описание ошибки'),
        'message': StringField(required=True, pbid=3, help_text=u'Человекочитаемое описание ошибки'),
    }


class ErrorWithContextSerializer(ErrorSerializer):
    def prepare_object(self, exception_obj):
        result = super(ErrorWithContextSerializer, self).prepare_object(exception_obj)
        result.update(exception_obj.context)
        return result


class ApiInfoSerializer(BaseSerializer):
    visible_fields = ['api_version', 'build']
    api_version = 'v1'
    fields = {
        'api_version': ParentAttrField('api_version', field_type=StringField, pbid=1, help_text=u'Актуальная версия API'),
        'build': ParentMethodField('get_build', field_type=StringField, pbid=2, help_text=u'Версия сервера'),
    }

    def get_build(self, value):
        return mpfs.__version__


class AsIsSerializer(BaseSerializer):
    def _from_native(self, obj):
        return obj

    def _to_native(self, data, files=None):
        return data
