# -*- coding: utf-8 -*-
import re
import sys
import urllib
import urlparse
import StringIO
import collections

from flask import Request
from inspect import isclass

from mpfs.common.util.strings import smart_str


_first_cap_re = re.compile('(.)([A-Z][a-z]+)')
_all_cap_re = re.compile('([a-z0-9])([A-Z])')

_common_path_regexp = re.compile(r'^/v\d+/.*$')
_cookie_based_path_regexp = re.compile(r'^/(?P<client_id>[a-zA-Z0-9_]+)(?P<path>/v\d+/.*)$')

_uid_regex = re.compile(r'^(yateam-[0-9]+|yaid-[0-9]+|device-[0-9A-Za-z]+|share_production|system-production|[0-9]+)$')


def is_valid_uid(uid):
    # С вероятностью 0.9999 uid - это uid, если:
    #  а) похоже на число
    #  б) похоже на uid системных пользователей
    #  в) похоже на uid специального вида: yaid-N или device-N, где N - число (https://st.yandex-team.ru/CHEMODAN-30733)
    return _uid_regex.match(uid) is not None


def camelcase_to_underscore(word):
    """http://stackoverflow.com/a/1176023"""
    s1 = _first_cap_re.sub(r'\1_\2', word)
    return _all_cap_re.sub(r'\1_\2', s1).lower()


def model_name_for_serializer(serializer):
    """Возвращает имя модели по классу сериализатора представляющего эту модель"""
    if not isclass(serializer):
        serializer = type(serializer)
    return serializer.__name__[:-len('Serializer')]


def trim_docstring(docstring):
    """http://legacy.python.org/dev/peps/pep-0257/"""
    if not docstring:
        return ''
    # Convert tabs to spaces (following the normal Python rules)
    # and split into a list of lines:
    lines = docstring.expandtabs().splitlines()
    # Determine minimum indentation (first line doesn't count):
    indent = sys.maxint
    for line in lines[1:]:
        stripped = line.lstrip()
        if stripped:
            indent = min(indent, len(line) - len(stripped))
    # Remove indentation (first line is special):
    trimmed = [lines[0].strip()]
    if indent < sys.maxint:
        for line in lines[1:]:
            trimmed.append(line[indent:].rstrip())
    # Strip off trailing and leading blank lines:
    while trimmed and not trimmed[-1]:
        trimmed.pop()
    while trimmed and not trimmed[0]:
        trimmed.pop(0)
    # Return a single string:
    return '\n'.join(trimmed)


def split_docstring(docstring):
    """Split docstring to summary and description parts by empty line."""
    if isinstance(docstring, (str, unicode)):
        docstring = docstring.splitlines()
    empty_line_num = 0
    while len(docstring) > empty_line_num and docstring[empty_line_num]:
        empty_line_num += 1
    return ' '.join(docstring[:empty_line_num]), '\n'.join(docstring[empty_line_num+1:])


def prepare_qs_val(v):
    """
    Подготавливает значение к urlencode. Всегда возвращает str.
    :rtype: str
    """
    return smart_str(v) if isinstance(v, (str, unicode)) else str(v)


def quote_string(v):
    """
    Кодирует строки (bytes, str и unicode) в URL-encoded формат, остальные значения возвращает без изменений.
    """
    if isinstance(v, (bytes, str, unicode)):
        return quote(v)
    return v


def quote(v):
    """
    Кодирует значения в URL-encoded формат. Всегда возвращает str.
    :rtype: str
    """
    ret = repr(v)
    if isinstance(v, (bytes, str, unicode)):
        ret = smart_str(v)
    ret = urllib.quote(ret, safe='')
    return ret


def unquote(v):
    """
    Декодирует URL-encoded значение. Всегда возвращает unicode.
    :rtype: unicode
    """
    ret = v
    if isinstance(v, str):
        ret = ret.decode('utf-8')
    ret = ret.encode('utf-8')
    ret = urllib.unquote(ret)
    ret = ret.decode('utf-8')
    return ret


def build_flask_request(method, url, data=None, headers=None, rid=None, crid=None, ip=None):
    """
    Возвращает объект запроса, который можно передать диспатчеру платформы.

    :param method:
    :param url:
    :param data:
    :param headers:
    :return:
    """
    headers = headers or {}
    ip = ip or '127.0.0.1'
    headers = CaseInsensitiveDict(headers)
    data = data or ''
    uri_chunks = urlparse.urlparse(url)
    environ = {
        'REQUEST_METHOD': method,
        'REQUEST_URI': uri_chunks.path + '?' + uri_chunks.query if uri_chunks.query else uri_chunks.path,
        'HTTP_AUTHORIZATION': headers.pop('oauth', headers.pop('Authorization', '')),
        'SERVER_PROTOCOL': 'HTTP/1.1',
        'REMOTE_ADDR': ip,
        'HTTP_X_HTTP_METHOD': headers.get('X-HTTP-Method', ''),
        'PATH_INFO': uri_chunks.path,
        'QUERY_STRING': uri_chunks.query,
        'wsgi.url_scheme': uri_chunks.scheme,
        'SERVER_NAME': uri_chunks.hostname,
        'SERVER_PORT': str(uri_chunks.port) if uri_chunks.port else '80',
    }
    for k, v in headers.iteritems():
        environ['HTTP_%s' % str(k).upper().replace('-', '_')] = v

    data = data or ''
    stream = StringIO.StringIO(data)
    content_env = {
        'wsgi.input': stream,
        'CONTENT_TYPE': headers.get('Content-Type', ''),
        'CONTENT_LENGTH': stream.len,
    }
    environ.update(content_env)

    result = Request(environ)

    if rid:
        setattr(result, 'rid', rid)
    if crid:
        setattr(result, 'crid', crid)

    return result


def split_cookie_based_path(raw_path):
    """
    Функция для проверки, является ли raw_path путем с идентификатором сервиса, который хочет авторизовываться по
    cookie в raw_path, или это обычный raw_path.

    Если вначале raw_path нашли что-то отличное от v1, v2, ... vN, то это может быть похоже на идентификатор
    пользователя, который хочет авторизоваться по куке - отрезаем и сохраняем его отдельно,
    если нет, то идем по старому варианту.

    :param raw_path: path от url запроса пользователя
    raw_path пользователя с авторизацией по куке выглядит примерно так:
    /5a12ecda66154db8a76dc67750a632dc/v1/disk/resources
    а у обычного пользователя (для external OAuth) примерно так:
    /v1/disk/resources

    :return: tuple из двух строк - идентификатор клиента для авторизации по куке, raw_path, у которого отрезан этот
    идентификатор, если он там есть
    """
    client_id = None
    if not _common_path_regexp.match(raw_path):
        result = _cookie_based_path_regexp.match(raw_path)
        if result is not None:
            client_id = result.group('client_id')
            raw_path = result.group('path')
    return client_id, raw_path


def join_cookie_based_path(cookie_id, raw_path):
    if raw_path.startswith('/'):
        raw_path = raw_path[1:]
    return '/%s/%s' % (cookie_id, raw_path)


def parse_http_media_type_parameters(media_type):
    """Возвращает параметры Media-Type в соответствии с http://www.w3.org/Protocols/rfc2616/rfc2616-sec3.html#sec3.7"""
    ret = CaseInsensitiveDict()
    if media_type:
        ct_chunks = filter(None, [v.strip() for v in media_type.split(';')])
        if len(ct_chunks) > 1:
            params = ct_chunks[1:]
            for p in params:
                attribute, value = p.split('=', 1)
                ret[attribute] = value.strip('"')
    return ret


def flatten_dict(d, parent_key=None, separator='.'):
    """
    Делает dict плоским резолвя все ключи вложенных диктов в ключи корневого дикта с префиксом.

    >>> flatten_dict({'a': {'b': 1, 'c': 2}})
    {'a.b': 1, 'a.c': 2}
    """
    items = []
    for k, v in d.iteritems():
        new_key = '%s.%s' % (parent_key, k) if parent_key else k
        if isinstance(v, dict):
            items.extend(flatten_dict(v, parent_key=new_key, separator=separator).items())
        else:
            items.append((new_key, v))
    return dict(items)


# From https://github.com/kennethreitz/requests/blob/v1.2.3/requests/structures.py#L37
class CaseInsensitiveDict(collections.MutableMapping):
    """
    A case-insensitive ``dict``-like object.

    Implements all methods and operations of
    ``collections.MutableMapping`` as well as dict's ``copy``. Also
    provides ``lower_items``.

    All keys are expected to be strings. The structure remembers the
    case of the last key to be set, and ``iter(instance)``,
    ``keys()``, ``items()``, ``iterkeys()``, and ``iteritems()``
    will contain case-sensitive keys. However, querying and contains
    testing is case insensitive:

        cid = CaseInsensitiveDict()
        cid['Accept'] = 'application/json'
        cid['aCCEPT'] == 'application/json'  # True
        list(cid) == ['Accept']  # True

    For example, ``headers['content-encoding']`` will return the
    value of a ``'Content-Encoding'`` response header, regardless
    of how the header name was originally stored.

    If the constructor, ``.update``, or equality comparison
    operations are given keys that have equal ``.lower()``s, the
    behavior is undefined.

    """
    def __init__(self, data=None, **kwargs):
        self._store = dict()
        if data is None:
            data = {}
        self.update(data, **kwargs)

    def __setitem__(self, key, value):
        # Use the lowercased key for lookups, but store the actual
        # key alongside the value.
        self._store[key.lower()] = (key, value)

    def __getitem__(self, key):
        return self._store[key.lower()][1]

    def __delitem__(self, key):
        del self._store[key.lower()]

    def __iter__(self):
        return (casedkey for casedkey, mappedvalue in self._store.values())

    def __len__(self):
        return len(self._store)

    def lower_items(self):
        """Like iteritems(), but with all lowercase keys."""
        return (
            (lowerkey, keyval[1])
            for (lowerkey, keyval)
            in self._store.items()
        )

    def __eq__(self, other):
        if isinstance(other, collections.Mapping):
            other = CaseInsensitiveDict(other)
        else:
            return NotImplemented
        # Compare insensitively
        return dict(self.lower_items()) == dict(other.lower_items())

    # Copy is required
    def copy(self):
        return CaseInsensitiveDict(self._store.values())

    def __repr__(self):
        return '%s(%r)' % (self.__class__.__name__, dict(self.items()))


def parse_cookie(cookie):
    parts = cookie.split(';')
    result = {}
    for part in parts:
        key_value = part.split('=', 1)
        if len(key_value) == 2:
            result[key_value[0].strip()] = key_value[1]
    return result
