# -*- coding: utf-8 -*-

from mpfs.common.errors import DjfsError
from mpfs.common.static.codes import ALBUM_NOT_FOUND
from mpfs.core.services.djfs_albums import djfs_albums_legacy
from mpfs.platform import fields
from mpfs.platform.handlers import ServiceProxyHandler, ETagHandlerMixin
from mpfs.platform.v1.disk import exceptions
from mpfs.platform.v1.disk.permissions import WebDavPermission


ALBUM_TYPE_CHOICES = ['geo', 'favorites', 'personal', 'faces']


class DjfsAlbumsProxyHandler(ServiceProxyHandler):
    service = djfs_albums_legacy
    permissions = WebDavPermission()
    service_base_exception = DjfsError
    hidden = True

    error_map = {
        400: exceptions.UidNotFoundError,
        404: exceptions.DiskAlbumsNotFound,
    }

    def handle(self, request, *args, **kwargs):
        context = self.get_context()
        url = self.get_url(context)
        status_code, resp, _ = self.raw_request_service(url, method=self.service_method)
        return status_code, resp

    def get_service_error_code(self, exception):
        try:
            return exception.data['code']
        except (AttributeError, KeyError):
            return None


class GetSnapshotHandler(ETagHandlerMixin, DjfsAlbumsProxyHandler):
    service_method = 'GET'
    service_url = '/api/v1/albums/snapshot?uid=%(uid)s&type=%(type)s&collection_ids=%(collection_id)s'

    query = fields.QueryDict({
        'type': fields.MultipleChoicesField(choices=ALBUM_TYPE_CHOICES,
                                            help_text=u'Тип альбомов.'),
        'collection_id': fields.StringField(required=True, help_text=u'Список идентификатор альбомов.'),
    })

    def get_context(self, context=None):
        c = super(GetSnapshotHandler, self).get_context(context)
        c['type'] = ','.join(c['type'])
        return c


class GetDeltasHandler(DjfsAlbumsProxyHandler):
    error_map = {
        410: exceptions.DiskAlbumDeltaRevisionIsTooOld,
    }

    service_method = 'GET'
    service_url = '/api/v1/albums/deltas?uid=%(uid)s&base_revision=%(base_revision)s'

    query = fields.QueryDict({
        'base_revision': fields.IntegerField(required=True,
                                             help_text=u'Ревизия БД, для которой нужно получить изменения.'),
        'type': fields.MultipleChoicesField(choices=ALBUM_TYPE_CHOICES,
                                            help_text=u'Тип альбомов.'),
        'limit': fields.IntegerField(help_text=u'Количество изменений на странице.'),
    })

    def get_url(self, context=None, base_url=None):
        url = self.service_url
        limit = context.get('limit', None)
        if limit:
            url += '&limit=%d' % limit

        type = context.get('type')
        if type:
            url += '&type=%s' % ','.join(type)

        return self.build_url(url, context, base_url)


class MergePhotoAlbumsHandler(DjfsAlbumsProxyHandler):
    """Объединить два альбома. Изначаьно делается для альбомов лиц"""
    error_map = {
        404: exceptions.DiskAlbumsNotFound,
    }

    service_method = 'POST'
    service_url = '/api/v1/albums/faces/merge?uid=%(uid)s&src_album_id=%(src_album_id)s&dst_album_id=%(dst_album_id)s'

    query = fields.QueryDict({
        'src_album_id': fields.StringField(required=True, help_text=u'Альбом, из которого перемещаются фото.'),
        'dst_album_id': fields.StringField(required=True, help_text=u'Альбом, в который перемещаются фото.'),
    })


