# -*- coding: utf-8 -*-
import traceback
import urlparse
from copy import deepcopy

from mpfs.platform.common import logger
from mpfs.platform.utils import (build_flask_request, CaseInsensitiveDict,
                                 split_cookie_based_path, join_cookie_based_path)


class BaseBatchRequestProcessor(object):
    """Базовый класс обработчиков, задающий общий интерфейс."""

    preserve_headers = ['Authorization', 'Cookie']
    """Список заголовков, значения которых будут сохранены от основного запроса при вызове метода `build_raw_request`"""

    support_handlers = set()
    """Множество хэндлеров, которые умеет эмулировать обработчик."""

    def check_request(self, batch_request):
        """
        Проверяет подходит ли обработчик для запроса.

        По умолчанию проверяет есть ли хэндлер предназначенный для обработки запроса в списке `self.support_handlers`.

        :rtype: bool
        """
        raw_request = self.raw_requests.get(batch_request, None)
        if not raw_request:
            raw_request = self.build_raw_request(self.dispatcher.request, batch_request)
            self.raw_requests[batch_request] = raw_request

        native_request = self.native_requests.get(batch_request, None)
        if not native_request:
            native_request = self.build_native_request(raw_request)
            self.native_requests[batch_request] = native_request

        match = self.dispatcher.router.resolve(native_request)
        if match:
            # match.handler содержит экземпляр `BaseResource`
            # поэтому достаём из `BaseResource.relations` класс хэндлера
            resource_cls = type(match.handler)
            handler_cls = resource_cls.relations.get(native_request.method, None)
            return handler_cls in self.support_handlers
        return False

    def process(self, batch_requests):
        """
        Обрабатывает список запросов и возвращает маппинг запросов на ответы.

        Переопределяется в дочерних обработчиках.

        :param list batch_requests: Список объектов `mpfs.platform.v1.batch.common.BatchRequest`,
                                    отфильтрованный с помощью `self.check_request`.
        :return: Маппинг запросов на ответы.
        :rtype: dict
        """
        raise NotImplemented()

    #
    # Дальше описаны внутренние методы, переопределять и изменять которые не стоит.
    #

    def __init__(self, dispatcher, main_request=None):
        self.dispatcher = dispatcher
        """Диспатчер используемый для выполнения запроса."""

        self.main_request = main_request
        """Основной запрос. Запрос который содержит собственно batch-запрос."""

        self.raw_requests = {}
        """Маппинг batch-запросов на сырые запросы полученные от Flask'а."""

        self.native_requests = {}
        """Маппинг batch-запросов на платформенные запросы."""

    def __repr__(self):
        return self.__class__.__name__

    def dispatch(self, batch_requests):
        """
        Получает все запросы содержащиеся в пачке, выбирает те которые он может обработать,
        обрабатывает и возвращает результат.

        :param batch_requests: Нефильтрованный список объектов `mpfs.platform.v1.batch.common.BatchRequest`.
        :return: Маппинг обработанных запросов на ответы.
        :rtype: dict
        """
        # бэкапим основной запрос
        self.main_request = self.dispatcher.request
        try:
            batch_requests = batch_requests or []
            batch_requests = [r for r in batch_requests if self.check_request(r)]
            if batch_requests:
                return self.process(batch_requests)
        except Exception:
            # если произошла ошибка, то пишем её в лог, но не выбрасываем наружу
            logger.error_log.error(traceback.format_exc())
        finally:
            # т.к. при обработке пакетных запросов используются те же диспатчер и роутер,
            # что и при обработке основного запроса, то возвращаем правильный request в диспатчере на место
            self.dispatcher.request = self.main_request
        return {}

    @classmethod
    def build_raw_request(cls, main_request, batch_request, preserve_headers=None):
        """Собирает из batch-запроса нативный сырой запрос (который сейчас API получает от Flask'а)."""
        preserve_headers = preserve_headers or []
        preserve_headers.extend(cls.preserve_headers)

        client_id, raw_path = split_cookie_based_path(batch_request.url)
        if client_id is not None:
            # если клиент пришел с идентификатором для авторизации по куке и передал идентификатор
            # в теле batch запроса - для безопасности заменяем идентификатор из тела batch запроса на тот, с которым
            # он пришел
            raw_path = join_cookie_based_path(main_request.cookie_auth_client_id, raw_path)

        url = urlparse.urljoin(main_request.url, raw_path)
        headers = CaseInsensitiveDict([(k, v) for k, v in main_request.raw_headers.iteritems()])
        headers.update([(k, v) for k, v in batch_request.headers.iteritems() if k not in preserve_headers])
        raw_request = build_flask_request(batch_request.method, url, data=batch_request.body, headers=headers,
                                          rid=main_request.rid, crid=main_request.crid)
        return raw_request

    def build_native_request(self, raw_request):
        """
        Конвертирует сырой запрос (полученный от Flask'а) в нативный запрос платформы.

        Копирует значения атрибутов `request.client` и `request.user` из основного запроса.
        """
        request = self.dispatcher._build_request(raw_request)
        request.client = deepcopy(self.main_request.client)
        request.user = deepcopy(self.main_request.user)
        return request


class DefaultBatchRequestProcessor(BaseBatchRequestProcessor):
    """
    Обработчик пакетных запросов по умолчанию

    Принимает любые запросы из пакета и последовательно выполняет их.
    Этот обработчик способен переварить любые запросы, поэтому должен применяться после всех специфичных обработчиков.
    """

    def check_request(self, batch_request):
        return True

    def process(self, batch_requests):
        result = {}
        for req in batch_requests:
            raw_request = self.raw_requests.get(req) or self.build_raw_request(self.main_request, req)
            result[req] = self.dispatcher.dispatch(raw_request)
        return result
