# -*- coding: utf-8 -*-
import re
import traceback

from mpfs.common.static.tags.platform import EXTERNAL
from mpfs.config import settings
from mpfs.platform.auth import PassportCookieAuth
from mpfs.platform.common import logger
from mpfs.platform.exceptions import ServiceUnavailableError
from mpfs.platform.handlers import BasePlatformHandler
from mpfs.platform.permissions import AllowByClientIdPermission
from mpfs.platform.rate_limiters import BatchRateLimiter, BatchRequestLimit, DataApiPerClientIdRateLimiter, \
    PerUserRateLimiter
from mpfs.platform.v1.disk.batch_processors import GetResourcesBatchRequestProcessor
from mpfs.platform.v1.disk.permissions import WebDavPermission, DiskBatchPermission
from mpfs.platform.v1.batch.batch_processors import DefaultBatchRequestProcessor
from mpfs.platform.v1.batch.exceptions import BatchForbiddenNestedChainRequestError
from mpfs.platform.v1.batch.serializers import BatchResponseListSerializer, BatchRequestListSerializer
from mpfs.platform.v1.personality.batch_processors import ProfileBatchRequestProcessor


PLATFORM_BATCH_ALLOW_BY_YADISK_ALL = settings.platform['batch']['allow_by_yadisk_all']
PLATFORM_BATCH_ALLOWED_CLIENT_IDS = settings.platform['batch']['allowed_client_ids']


PERMISSIONS = AllowByClientIdPermission(PLATFORM_BATCH_ALLOWED_CLIENT_IDS) | DiskBatchPermission()
if PLATFORM_BATCH_ALLOW_BY_YADISK_ALL:
    PERMISSIONS = WebDavPermission() | PERMISSIONS


class BatchRequestHandler(BasePlatformHandler):
    """Выполнить пакетный запрос"""
    auth_methods = [PassportCookieAuth()]
    permissions = PERMISSIONS
    body_serializer_cls = BatchRequestListSerializer
    serializer_cls = BatchResponseListSerializer
    rate_limiter = BatchRateLimiter(
        [BatchRequestLimit(DataApiPerClientIdRateLimiter(), ProfileBatchRequestProcessor, True)],
        default_limiter=PerUserRateLimiter('cloud_api_user'))

    default_response_error = ServiceUnavailableError()
    """Ошибка по умолчанию возвращаемая если ни один из процессоров не вернул ответ на запрос"""

    default_processor_cls = DefaultBatchRequestProcessor
    """Процессор применяемый ко всем запросам оставшимся без ответа после применения процессоров из `self.processors`"""

    processors = [ProfileBatchRequestProcessor, GetResourcesBatchRequestProcessor]
    """Список классов процессоров последовательно применямых ко всем запросам"""

    chaining_request_internal_re = re.compile(r'^/v1(/[^/]+)?/chaining/request.*')
    chaining_request_external_re = re.compile(r'^/v1/chaining/request.*')

    def handle(self, request, *args, **kwargs):
        processors = [p(request.dispatcher) for p in self.processors + [self.default_processor_cls]]
        default_response = request.dispatcher.handle_error(self.default_response_error, log_trace=False)
        batch_requests = request.body.get('items', [])

        # Запрещаем в батчах выполнять чейн-запросы
        chaining_request_re = (self.chaining_request_external_re if request.mode == EXTERNAL
                               else self.chaining_request_internal_re)
        if any(chaining_request_re.match(r.url) for r in batch_requests):
            raise BatchForbiddenNestedChainRequestError()

        responses = {}
        # Обрабатываем запросы всеми процессорами кроме дефолтного. При этом героически замалчиваем ошибки,
        # чтоб вернуть корректный ответ на batch-запрос, даже если какие-то процессоры обвалятся в ошибки.
        for p in processors:
            try:
                unprocessed_requests = [req for req in batch_requests if req not in responses]
                responses.update(p.dispatch(unprocessed_requests))
            except Exception, e:
                logger.error_log.error(traceback.format_exc())

        # раскладываем ответы в том же порядке что и запросы
        result_responses = [responses.get(req, default_response) for req in batch_requests]

        return super(BatchRequestHandler, self).serialize({'items': result_responses})
