# -*- coding: utf-8 -*-
import re
import time

from dateutil import tz, relativedelta
from datetime import datetime, timedelta
import calendar

from mpfs.config import settings
from mpfs.common.errors import DataApiProfileError, XivaError, DjfsError, LentaLoaderNoResponse
from mpfs.config import web_server
from mpfs.common.static.codes import WH_USER_NEED_INIT, FORBIDDEN_PHOTOUNLIM
from mpfs.common.util.crypt import AesCbcCryptAgent, HmacCryptAgent
from mpfs.common.util.urls import urlencode
from mpfs.common.util.user_agent_parser import UserAgentParser
from mpfs.core.cache import cache_get, cache_keys
from mpfs.core.services.data_api_profile_service import data_api_profile
from mpfs.core.services.djfs_api_service import djfs_api_legacy
from mpfs.core.services.lenta_loader_service import lenta_loader
from mpfs.core.services.platform_search_service import platform_disk_search
from mpfs.core.services.xiva_service import xiva_service
from mpfs.platform import fields, validators
from mpfs.platform.auth import InternalTokenAuth
from mpfs.platform.common import ResponseObject
from mpfs.platform.exceptions import (
    BadRequestError,
    NotFoundError,
)
from mpfs.platform.handlers import BasePlatformHandler, ServiceProxyHandler
from mpfs.platform.permissions import (
    AllowByClientIdPermission,
    DenyAllPermission,
    AllowAllPermission
)
from mpfs.platform.rate_limiters import PerYandexuidRateLimiter
from mpfs.platform.serializers import BaseSerializer
from mpfs.platform.v1.case.exceptions import DocviewerExpiredUrlError, DocviewerInvalidFileUrlError, \
    DocviewerSignatureMismatchError, DocviewerYandexuidRequiredError, DiskCanDeleteItemSizeIsTooLarge
from mpfs.platform.v1.case.permissions import ImportYTTablePermissions, LentaCoolBlocksReadPermission, \
    NostalgiaPhotoBlocksReadPermission
from mpfs.platform.v1.data.handlers import _auto_initialize_user, DataApiProxyHandler
from mpfs.platform.v1.disk.exceptions import DiskUserNotFoundError, DiskPaymentRequired, DiskPhotoUnlimPaymentRequired
from mpfs.platform.v1.disk.handlers import MpfsProxyHandler
from mpfs.platform.v1.disk.handlers import _auto_initialize_user as _auto_initialize_user_in_mpfs
from mpfs.platform.v1.disk.billing.handlers import MpfsBillingUtilsHandlerMixin, MpfsProxyBillingHandler
from mpfs.platform.v1.personality.exceptions import YTObjectNotFoundError
from mpfs.platform.v1.personality.exceptions import PersonalityEventNotFoundError
from mpfs.platform.v1.personality.handlers import DataApiProfileBaseHandler, GenericObjectHandler
from mpfs.platform.v1.personality.serializers import FlightSerializer
from mpfs.platform.v1.case.serializers import (
    AutouploadLentaBlockUrlSerializer,
    DocviewerYabrowserFileDataSerializer,
    DocviewerYabrowserLinkSerializer,
    PromoStatusSerializer,
    UnlimitedAutouploadSetStatusRequestSerializer,
    WakeUpSessionIdSerializer,
    CanDeleteListBodySerializer, CanDeleteListResponseSerializer, UnlimitedAutouploadsSetStatusRequestSerializer,
    UnlimitedAutouploadsGetStatusRequestSerializer)
from mpfs.platform.v1.disk.permissions import WebDavPermission, DiskReadPermission
from fields import PushkinTokenField
from exceptions import PushkinTokenInvalidUUIDError, DiskUnsupportedTLDError

PLATFORM_TLDS = settings.platform['tlds']
PLATFORM_MOBILE_APPS_IDS = settings.platform['mobile_apps_ids']
PLATFORM_DISK_APPS_IDS = settings.platform['disk_apps_ids']
PLATFORM_DOCVIEWER_FILE_LINK_URL = settings.platform['docviewer']['file_link_url']
PLATFORM_DOCVIEWER_LINK_ALLOWED_CLIENT_IDS = settings.platform['docviewer']['link_allowed_client_ids']
PLATFORM_DOCVIEWER_FILE_DATA_ALLOWED_CLIENT_IDS = settings.platform['docviewer']['file_data_allowed_client_ids']
PLATFORM_DOCVIEWER_MDS_KEY_SECRET = settings.platform['docviewer']['mds_key_secret']
PLATFORM_DOCVIEWER_SIGN_SECRET = settings.platform['docviewer']['sign_secret']
PLATFORM_DJFS_LENTA_COOL_BLOCKS_SEND_INTERNAL_TOKEN = \
    settings.platform['djfs']['lenta_cool_blocks']['send_internal_token']
PLATFORM_DJFS_LENTA_COOL_BLOCKS_INTERNAL_TOKEN = settings.platform['djfs']['lenta_cool_blocks']['internal_token']
PROMO_AUTO_UPLOAD_32_GB = settings.promo['autoupload32gb']
PROMO_UNLIMITED_AUTOUPLOAD_START_TIMESTAMP = settings.promo['unlimited_autoupload']['start_timestamp']
WAKE_UP_INTERVAL = settings.wake_up['push_interval']


class CaseApiBaseHandler(ServiceProxyHandler):
    permissions = DenyAllPermission()


class ImportRegionsBatchHandler(CaseApiBaseHandler):
    service = data_api_profile
    service_method = 'PUT'
    service_base_exception = DataApiProfileError
    service_url = "/regions/import-regions-batch?url=%(url)s"

    query = fields.QueryDict({
        'url': fields.StringField(required=True, help_text=u'URL файла с данными для загрузки.'),
    })


class SaveActualFlightHandler(DataApiProfileBaseHandler):
    """Сохранить существующий или новый перелет"""
    service_method = 'PUT'
    service_url = '/events/flights/actual/%(event_id)s?__uid=%(uid)s'
    body_serializer_cls = FlightSerializer

    error_map = {
        'invalid-schema': BadRequestError
    }

    kwargs = fields.QueryDict({
        'event_id': fields.StringField(required=True, help_text=u'Идентификатор события.')
    })

    def request_service(self, url, *args, **kwargs):
        if 'departure' in self.request.body:
            self.preprocess_time(self.request.body['departure'])

        if 'arrival' in self.request.body:
            self.preprocess_time(self.request.body['arrival'])

        # only mail uses case-api
        self.request.body['data_source'] = 'mail'

        return super(SaveActualFlightHandler, self).request_service(url, *args, data=self.request.body,
                                                                   headers={"Content-Type": "application/json"})

    def preprocess_time(self, c):
        if 'time' in c and isinstance(c['time'], tuple):
            ts, tz_offset = c['time']
            c['time'] = ts
            c['tzoffset'] = tz_offset

    @_auto_initialize_user
    def handle(self, request, *args, **kwargs):
        return super(SaveActualFlightHandler, self).handle(request, *args, **kwargs)


class DeleteActualFlightHandler(DataApiProfileBaseHandler):
    """Удалить существующий перелет"""
    service_method = 'DELETE'
    service_url = '/events/flights/actual/%(event_id)s?__uid=%(uid)s'

    # invalid-delta нужно обрабатывать из-за внутреннего устройства data-api
    error_map = {
        'not-found': PersonalityEventNotFoundError,
        'invalid-delta': PersonalityEventNotFoundError
    }

    kwargs = fields.QueryDict({
        'event_id': fields.StringField(required=True, help_text=u'Идентификатор события.')
    })

    @_auto_initialize_user
    def handle(self, request, *args, **kwargs):
        return super(DeleteActualFlightHandler, self).handle(request, *args, **kwargs)


class BaseXivaHandler(CaseApiBaseHandler):
    permissions = WebDavPermission()
    service = xiva_service
    service_method = 'POST'
    service_base_exception = XivaError
    error_map = {
        'invalid-uuid': PushkinTokenInvalidUUIDError,
    }

    query = fields.QueryDict({
        'token': PushkinTokenField(required=True, help_text=u'Токен в формате Пушкина'),
    })

    invalid_uuid_re = re.compile('.*is not a \'uuid\'.*', re.DOTALL)

    def get_context(self, context=None):
        c = super(BaseXivaHandler, self).get_context(context)
        token = self.request.query['token']
        c.update({
            'uid': self.request.user.uid,
            'service': xiva_service.service,
            'token': xiva_service.service_token,
            'platform': token['platform'],
            'app_name': token['app_name'],
            'uuid': token['uuid'],
            'push_token': token['push_token'],
        })
        return c

    def get_service_error_code(self, exception):
        data = getattr(exception, 'data', {})
        body = data.get('text', None)
        if self.invalid_uuid_re.match(body):
            return 'invalid-uuid'
        else:
            return None


# https://wiki.yandex-team.ru/yxiva/api/v2/#podpisatmobilnoeprilozhenie
class XivaSubscribeMobileHandler(BaseXivaHandler):
    service_url = '/v2/subscribe/app?uid=%(uid)s&service=%(service)s&token=%(token)s' + \
                  '&uuid=%(uuid)s&push_token=%(push_token)s' + \
                  '&app_name=%(app_name)s&platform=%(platform)s'


# https://wiki.yandex-team.ru/yxiva/api/v2/#otpisatmobilnoeprilozhenie
class XivaUnsubscribeMobileHandler(BaseXivaHandler):
    service_url = '/v2/unsubscribe/app?uid=%(uid)s&service=%(service)s&token=%(token)s' + \
                  '&uuid=%(uuid)s&push_token=%(push_token)s'


class BaseAutouploadBonusHandler(MpfsProxyBillingHandler):
    """Вспомогательный хендлер для выдачи бонуса за автозагрузку"""


class StatusAutouploadBonusHandler(MpfsBillingUtilsHandlerMixin, BaseAutouploadBonusHandler):
    permissions = AllowByClientIdPermission(PLATFORM_MOBILE_APPS_IDS)
    serializer_cls = PromoStatusSerializer
    hidden = True
    test_login_re = re.compile(PROMO_AUTO_UPLOAD_32_GB['test_login_pattern'])

    error_map = {
        404: NotFoundError,
    }

    def handle(self, request, *args, **kwargs):
        if not self.is_test_user(request.user) and not self.check_if_date_is_valid():
            raise NotFoundError()
        result = {'show_notification': True}
        if self.check_if_service_exists('32_gb_autoupload'):
            result['show_notification'] = False
        return self.serialize(result)

    def is_test_user(self, user):
        """Для таких пользователей не проверяем временные границы действия акции"""
        if user and user.login and self.test_login_re.search(user.login):
            return True
        else:
            return False

    @classmethod
    def check_if_date_is_valid(cls):
        current_time = cls._get_current_time()
        if (current_time < PROMO_AUTO_UPLOAD_32_GB['start_timestamp'] or
                current_time >= PROMO_AUTO_UPLOAD_32_GB['end_timestamp']):
            return False
        return True

    @staticmethod
    def _get_current_time():
        return int(time.time())


class CanDeleteHandler(MpfsProxyHandler):
    service_method = 'POST'
    permissions = AllowByClientIdPermission(PLATFORM_MOBILE_APPS_IDS)
    error_map = {}
    hidden = True
    service_url = '/json/can_delete?uid=%(uid)s'
    body_serializer_cls = CanDeleteListBodySerializer
    serializer_cls = CanDeleteListResponseSerializer
    default_max_item_size = 200

    def handle(self, request, *args, **kwargs):
        context = self.get_context()
        if len(self.request.body['items']) > self.default_max_item_size:
            raise DiskCanDeleteItemSizeIsTooLarge()

        url = self.get_url(context)
        resp = self.request_service(url, method=self.service_method, data=self.request.body,
                                    headers={'Content-type': 'application/json'})
        return self.serialize(resp)

    def get_error_map(self, error_map=None):
        return {}


class UnlimitedAutouploadStatusHandler(BasePlatformHandler):
    permissions = AllowAllPermission()
    serializer_cls = BaseSerializer
    hidden = True
    auth_required = False

    error_map = {
        404: NotFoundError,
    }

    def handle(self, request, *args, **kwargs):
        current_time = int(time.time())
        if current_time < PROMO_UNLIMITED_AUTOUPLOAD_START_TIMESTAMP:
            raise NotFoundError()
        return None


class UnlimitedAutouploadSetStatusHandler(MpfsProxyHandler):
    permissions = AllowByClientIdPermission(PLATFORM_MOBILE_APPS_IDS
                                            + ['f8cab64f154b4c8e96f92dac8becfcaa',  # Android ПП
                                            ])
    hidden = True

    body_serializer_cls = UnlimitedAutouploadSetStatusRequestSerializer

    @_auto_initialize_user_in_mpfs
    def handle(self, request, *args, **kwargs):
        if request.body['activate']:
            url = '/json/enable_unlimited_autouploading?uid=%(uid)s'
        else:
            url = '/json/disable_unlimited_autouploading?uid=%(uid)s'
        url = self.build_url(url, context={'uid': request.user.uid})
        self.request_service(url)


class UnlimitedAutouploadsSetStatusHandler(MpfsProxyHandler):
    permissions = AllowByClientIdPermission(PLATFORM_MOBILE_APPS_IDS
                                            + ['f8cab64f154b4c8e96f92dac8becfcaa',  # Android ПП
                                            ])
    hidden = True
    body_serializer_cls = UnlimitedAutouploadsSetStatusRequestSerializer
    service_method = 'POST'
    service_url = '/json/set_unlimited_autouploading?uid=%(uid)s'
    serializer_cls = UnlimitedAutouploadsGetStatusRequestSerializer
    error_map = {
        402: DiskPaymentRequired,
        FORBIDDEN_PHOTOUNLIM: DiskPhotoUnlimPaymentRequired,
    }

    @_auto_initialize_user_in_mpfs
    def handle(self, request, *args, **kwargs):
        params = {}
        if 'unlimited_video_autoupload_enabled' in request.body:
            params['unlimited_video_autoupload_enabled'] = 1 if request.body['unlimited_video_autoupload_enabled'] else 0
            params['unlimited_video_autoupload_reason'] = 'by_user'
        if 'unlimited_photo_autoupload_enabled' in request.body:
            params['unlimited_photo_autoupload_enabled'] = 1 if request.body['unlimited_photo_autoupload_enabled'] else 0
        user_agent = UserAgentParser.parse(request.raw_headers.get('user-agent', ''))
        if user_agent.is_assessor:
            params['is_asessor'] = True
        url = self.build_url(self.service_url, context={'uid': request.user.uid})
        url = self.patch_url_params(url, params)

        return self.serialize(self.request_service(url))


class GetAutouploadLentaBlockUrlHandler(MpfsProxyHandler):
    permissions = WebDavPermission()
    serializer_cls = AutouploadLentaBlockUrlSerializer
    hidden = True
    lenta_url = (
        '/json/lenta_block_list?'
        'uid=%(uid)s&path=%(path)s&'
        'mtime_gte=%(mtime_gte)s&'
        'amount=0&media_type=image&'
        'meta=total_results_count'
    )
    query = fields.QueryDict({
        'begin_date': fields.DateTimeField(required=False, tz_required=True, help_text=u'Сформировать блок ленты от этой даты.'),
        'tld': fields.StringField(required=True, help_text=u'Домен верхнего уровня, на который будет сформирован урл.')
    })

    LENTA_BLOCK_URL = 'https://disk.yandex.%(tld)s/search-app'
    MAX_DELTA = relativedelta.relativedelta(months=1)
    DEFAULT_DELTA = relativedelta.relativedelta(days=1)
    DEFAULT_TLD = 'com'

    @classmethod
    def crop_dt(cls, begin_date):
        """Преобразовать дату согласно требованиям

        Если дата `None`, то одни сутки назад.
        Дата должна быть не более 1 месяца назад
        """
        now_dt = datetime.now(tz.tzlocal())
        if begin_date is None:
            return now_dt - cls.DEFAULT_DELTA

        min_begin_dt = now_dt - cls.MAX_DELTA
        if min_begin_dt > begin_date:
            return min_begin_dt
        return begin_date

    def _build_block_url(self, tld):
        return self.LENTA_BLOCK_URL % {'tld': tld}

    def handle(self, request, *args, **kwargs):
        context = self.get_context()
        begin_date = self.crop_dt(context['begin_date'])

        params = {
            'uid': context['uid'],
            'path': '/photostream',
            'mtime_gte': calendar.timegm(begin_date.utctimetuple()),
        }
        lenta_url = self.build_url(self.lenta_url, params)
        photostream_resource = self.request_service(lenta_url, self.service_method)[0]
        block_size = photostream_resource['meta']['total_results_count']
        if block_size < 1:
            raise NotFoundError()

        response_data = {
            'date': datetime.now(tz.tzlocal()),
            'total': block_size,
        }
        query_tld = request.query['tld']
        if query_tld in PLATFORM_TLDS:
            response_data['block_url'] = self._build_block_url(query_tld)
            return self.serialize(response_data)
        else:
            response_data['block_url'] = self._build_block_url(self.DEFAULT_TLD)
            response_data['unsupported_tld'] = query_tld
            raise DiskUnsupportedTLDError(**response_data)

    def handle_exception(self, exception):
        if isinstance(exception, self.service_base_exception):
            if self.get_mpfs_error_code(exception) == WH_USER_NEED_INIT:
                raise DiskUserNotFoundError()
        return super(GetAutouploadLentaBlockUrlHandler, self).handle_exception(exception)


class DocviewerYabrowserUrlService(object):
    aes_cbc = AesCbcCryptAgent(PLATFORM_DOCVIEWER_MDS_KEY_SECRET, urlsafe=True)
    hmac = HmacCryptAgent(PLATFORM_DOCVIEWER_SIGN_SECRET, urlsafe=True)
    url_regex = re.compile(r'^ya-browser://(.*)\?sign=(.*)$')

    def encrypt_mds_key_with_time(self, mds_key, time_=None):
        time_ = time_ or time.time()
        return self.aes_cbc.encrypt('%s|%s' % (mds_key, time_))

    def decrypt_mds_key_with_time(self, ciphertext):
        decrypted = self.aes_cbc.decrypt(ciphertext)
        (decrypted_mds_key, decrypted_time) = decrypted.split('|')
        return decrypted_mds_key, decrypted_time

    def sign_with_yandexuid(self, message, yandexuid):
        return self.hmac.sign(message + yandexuid)

    def format_yabrowser_url(self, encrypted, signature):
        return 'ya-browser://%s?sign=%s' % (encrypted, signature)

    def create_yabrowser_url(self, mds_key, yandexuid, time_=None):
        encrypted = self.encrypt_mds_key_with_time(mds_key, time_)
        signature = self.sign_with_yandexuid(encrypted, yandexuid)
        return self.format_yabrowser_url(encrypted, signature)

    def parse_yabrowser_url(self, url):
        match = self.url_regex.match(url)
        if not match:
            raise ValueError()
        return match.group(1), match.group(2)


class DocviewerYabrowserLinkHandler(BasePlatformHandler):
    permissions = AllowByClientIdPermission(PLATFORM_DOCVIEWER_LINK_ALLOWED_CLIENT_IDS)
    serializer_cls = DocviewerYabrowserLinkSerializer
    hidden = True
    auth_user_required = False

    query = fields.QueryDict({
        'mds_key': fields.StringField(required=True, help_text=u'Идентификатор файла в MDS.'),
        'file_name': fields.StringField(required=True, help_text=u'Имя файла.'),
    })

    headers = fields.QueryDict({
        'cookie': fields.CookieField(required=True),
    })

    rate_limiter = PerYandexuidRateLimiter('cloud_api_yandexuid')

    url_service = DocviewerYabrowserUrlService()

    def handle(self, request, *args, **kwargs):
        mds_key = request.query['mds_key']
        file_name = request.query['file_name']
        cookies = request.headers['cookie']

        if 'yandexuid' not in cookies:
            raise DocviewerYandexuidRequiredError()
        yandexuid = cookies['yandexuid']

        file_url = self.url_service.create_yabrowser_url(mds_key, yandexuid)

        return self.serialize({
            'href': PLATFORM_DOCVIEWER_FILE_LINK_URL % urlencode({'url': file_url, 'name': file_name}),
            'method': 'GET',
            'templated': False,
        })


class DocviewerYabrowserFileDataHandler(BasePlatformHandler):
    permissions = AllowByClientIdPermission(PLATFORM_DOCVIEWER_FILE_DATA_ALLOWED_CLIENT_IDS)
    serializer_cls = DocviewerYabrowserFileDataSerializer
    hidden = True

    query = fields.QueryDict({
        'file_url': fields.StringField(required=True, help_text=u'Зашифрованный url.'),
    })

    headers = fields.QueryDict({
        'cookie': fields.CookieField(required=True),
    })

    response_objects = [
        ResponseObject(DocviewerExpiredUrlError),
        ResponseObject(DocviewerInvalidFileUrlError),
        ResponseObject(DocviewerSignatureMismatchError),
    ]

    rate_limiter = PerYandexuidRateLimiter('cloud_api_yandexuid')

    url_service = DocviewerYabrowserUrlService()

    def handle(self, request, *args, **kwargs):
        time_ = time.time()
        file_url = request.query['file_url']
        cookies = request.headers['cookie']

        if 'yandexuid' not in cookies:
            raise DocviewerYandexuidRequiredError()
        yandexuid = cookies['yandexuid']

        try:
            encrypted, signature = self.url_service.parse_yabrowser_url(file_url)
        except ValueError:
            raise DocviewerInvalidFileUrlError()

        if not signature == self.url_service.sign_with_yandexuid(encrypted, yandexuid):
            raise DocviewerSignatureMismatchError()

        (decrypted_mds_key, decrypted_time) = self.url_service.decrypt_mds_key_with_time(encrypted)

        if timedelta(seconds=time_ - float(decrypted_time)) > timedelta(days=1):
            raise DocviewerExpiredUrlError()

        return self.serialize({'mds_key': decrypted_mds_key})


class BaseImportYTTableHandler(GenericObjectHandler):
    permissions = ImportYTTablePermissions()
    auth_methods = [InternalTokenAuth()]
    error_map = {
        'inaccessible-ypath': YTObjectNotFoundError
    }
    auth_user_required = False

    def handle(self, request, *args, **kwargs):
        context = self.get_context()
        url = self.get_url(context)
        resp = self.request_service(url, method=self.service_method, data=self.request.body)
        return resp


class ImportYTTableHandler(BaseImportYTTableHandler):
    service_url = '/api/generic_data/import/?typeName=%(resource_path)s&ypath=%(yt_path)s'
    service_method = 'POST'

    query = fields.QueryDict({
        'yt_path': fields.StringField(required=True, help_text=u'Путь до таблицы YT hahn'),
    })


class ImportYTTableWithIdHandler(BaseImportYTTableHandler):
    service_url = '/api/generic_data/import/?typeName=%(resource_path)s&ypath=%(yt_path)s&importId=%(import_id)s'
    service_method = 'PUT'

    kwargs = fields.QueryDict({
        'import_id': fields.StringField(required=True, help_text=u'Прегенерированный ID операции импорта таблицы')
    })

    query = fields.QueryDict({
        'yt_path': fields.StringField(required=True, help_text=u'Путь до таблицы YT hahn'),
    })


class GetImportOperationStatusHandler(BaseImportYTTableHandler):
    service_url = '/api/generic_data/import/%(import_id)s/counters?typeName=%(resource_path)s'
    service_method = 'GET'

    kwargs = fields.QueryDict({
        'import_id': fields.StringField(required=True, help_text=u'ID операции импорта таблицы')
    })


class DynamicSettingsChecksumHandler(BasePlatformHandler):
    auth_required = False
    hidden = True
    permissions = AllowAllPermission()

    query = fields.QueryDict({
        'key': fields.StringField(
            required=False,
            help_text=u'Секретный ключ для дополнительной проверки запроса.'
        ),
    })

    def handle(self, request, *args, **kwargs):
        # если вынести этот импорт на уровень модуля, то модуль нельзя будет импортнуть
        # в шеле, т.к. пакет добавляется динамически самим uWSGI во время запуска сервера
        # можно конечно его туда вынести с try/except, но я пока решил оставить так

        if (
            request.remote_addr != '::1' or
            request.query.get('key') != '33075d2fd4145f50a68652b9e1e985633fa6f58b93f4a172b069281f6574ea4a'
        ):
            # разрешаем делать запросы только с локалхоста и секретным ключом
            raise NotFoundError()

        cache_name = settings.platform['dynamic_settings']['cache_workers_states_name']
        keys = cache_keys(cache_name)
        state = {}
        for key in keys:
            state[key] = cache_get(key, cache_name)
        return {
            'state': state,
            'uwsgi': {'processes': web_server.options.get('processes')}
        }


class WakeUpPushStart(MpfsProxyHandler):
    permissions = AllowByClientIdPermission(PLATFORM_DISK_APPS_IDS)
    service_url = '/json/wake_up_push_start?uid=%(uid)s&device_id=%(id)s&interval=%(interval)d'
    headers = fields.QueryDict({'User-Agent': fields.StringField(required=False, help_text=u'User-Agent приложения')})
    hidden = True
    serializer_cls = WakeUpSessionIdSerializer
    error_map = {
        404: NotFoundError,
    }

    query = fields.QueryDict({
        'interval': fields.IntegerField(
            required=False,
            help_text=u'Интервал между wake_up запросами в секундах.'
        ),
    })

    query = fields.QueryDict({
        'interval': fields.IntegerField(
            required=False,
            default=WAKE_UP_INTERVAL,
            help_text=u'Интервал между wake_up запросами в секундах.'
        ),
    })

    def get_context(self, context=None):
        context = super(WakeUpPushStart, self).get_context(context)
        context['id'] = UserAgentParser.get_unique_id(context['User-Agent'])
        return context


class WakeUpPushStop(MpfsProxyHandler):
    permissions = AllowByClientIdPermission(PLATFORM_DISK_APPS_IDS)
    service_url = '/json/wake_up_push_stop?session_id=%(session_id)s'
    hidden = True
    resp_status_code = 204

    error_map = {
        400: BadRequestError,
        404: NotFoundError,
    }

    kwargs = fields.QueryDict({
        'session_id': fields.StringField(required=True, help_text=u'Идентификатор wake_up сессии.')
    })


class GetLentaResourcesPreviewDownloadLinks(CaseApiBaseHandler):
    service = djfs_api_legacy
    service_method = 'GET'
    service_url = '/api/lenta/block_previews?uid=%(uid)s&locale=%(locale)s&imageSize=%(image_size)s'
    permissions = DiskReadPermission() | NostalgiaPhotoBlocksReadPermission()
    service_base_exception = DjfsError
    hidden = True

    query = fields.QueryDict({
        'locale': fields.ChoiceField(required=True, choices=['ru', 'en', 'tr', 'uk', 'ua'],
                                     help_text=u'Локаль пользователя.'),
        'image_size': fields.StringField(required=True, validators=[validators.RegExpValidator(r'^\d+x\d+$')],
                                         help_text=u'Размер превью.')
    })

    def handle(self, request, *args, **kwargs):
        context = self.get_context()
        url = self.get_url(context)

        internal_token_auth_headers = None
        if PLATFORM_DJFS_LENTA_COOL_BLOCKS_SEND_INTERNAL_TOKEN:
            internal_token_auth_headers = {
                'Authorization': 'ClientToken token=%s;uid=%s' %
                                 (PLATFORM_DJFS_LENTA_COOL_BLOCKS_INTERNAL_TOKEN, context['uid']),
            }

        status_code, resp, _ = self.raw_request_service(url, headers=internal_token_auth_headers,
                                                        method=self.service_method)
        if status_code != 200:
            return 200, '{"empty": true}'
        return 200, resp

    def handle_exception(self, exception):
        """
        В случае любого исключения от DJFS возвращаем пустой ответ-заглушку, чтобы не аффектить клиентов
        """
        return 200, '{"empty": true}'


class GetLentaBlocks(CaseApiBaseHandler):
    service = djfs_api_legacy
    service_method = 'GET'
    service_url = '/api/lenta/block_from_pool?uid=%(uid)s&locale=%(locale)s&limit=%(limit)s&offset=%(offset)s'
    permissions = LentaCoolBlocksReadPermission()
    service_base_exception = DjfsError
    hidden = True

    error_map = {
        404: NotFoundError,
    }

    query = fields.QueryDict({
        'locale': fields.ChoiceField(required=True, choices=['ru', 'en', 'tr', 'uk', 'ua'],
                                     help_text=u'Локаль пользователя.'),
        'limit': fields.IntegerField(default=100, help_text=u'Количество выводимых блоков.'),
        'offset': fields.IntegerField(default=0, help_text=u'Смещение от начала списка блоков.'),
    })

    def handle(self, request, *args, **kwargs):
        url = self.get_url(self.get_context())
        status_code, resp, _ = self.raw_request_service(url, method=self.service_method)
        return status_code, resp

    def get_service_error_code(self, exception):
        return exception.data.get('code')


class GetLentaBlockById(CaseApiBaseHandler):
    service = djfs_api_legacy
    service_method = 'GET'
    service_url = '/api/lenta/block_info?uid=%(uid)s&locale=%(locale)s&blockId=%(block_id)s&fromPool=%(from_pool)s'
    permissions = LentaCoolBlocksReadPermission()
    service_base_exception = DjfsError
    hidden = True

    error_map = {
        404: NotFoundError,
    }

    kwargs = fields.QueryDict({
        'block_id': fields.StringField(required=True, help_text=u'Идентификатор блока.'),
    })

    query = fields.QueryDict({
        'locale': fields.ChoiceField(required=True, choices=['ru', 'en', 'tr', 'uk', 'ua'],
                                     help_text=u'Локаль пользователя.'),
        'from_pool': fields.BooleanField(default=False, help_text=u'Искать блок в пуле.'),
    })

    def handle(self, request, *args, **kwargs):
        url = self.get_url(self.get_context())
        status_code, resp, _ = self.raw_request_service(url, method=self.service_method)
        return status_code, resp

    def get_service_error_code(self, exception):
        return exception.data.get('code')


class GetMonitoringHandler(BasePlatformHandler):
    hidden = True
    resp_status_code = 200
    permissions = WebDavPermission() | AllowByClientIdPermission(PLATFORM_DISK_APPS_IDS)

    def handle(self, request, *args, **kwargs):
        return ''


class GetMobileMonitoringHandler(GetMonitoringHandler):
    pass


class GetDesktopMonitoringHandler(GetMonitoringHandler):
    pass


class SearchWarmupHandler(BasePlatformHandler):
    """
    Обработчик ручки прогрева поиска

    отнаследован от BasePlatformHandler, а не от CaseApiBaseHandler, потому что в сервисе поиска неподобающим образом
    переопределен метод open_url (перехватываются ошибки кидаемые open_url и вместо них кидаются более общие, а для этой
    ручки нужна кастомная логика обработки исключений)
    """
    permissions = AllowByClientIdPermission(PLATFORM_DISK_APPS_IDS)
    disk_search = platform_disk_search
    hidden = True

    def handle(self, request, *args, **kwargs):
        self.disk_search.warmup_cache(request.user.uid)
        return ''


class GetLentaSelectionBlockInfoHandler(CaseApiBaseHandler):
    service = lenta_loader
    service_method = 'GET'
    service_url = '/api/get-cool-lenta-block-info?uid=%(uid)s&blockId=%(block_id)s&lang=%(locale)s'
    service_base_exception = LentaLoaderNoResponse
    permissions = AllowByClientIdPermission(PLATFORM_DISK_APPS_IDS)
    hidden = True

    query = fields.QueryDict({
        'block_id': fields.StringField(required=True, help_text=u'Идентификатор блока.'),
        'locale': fields.ChoiceField(required=True, choices=['ru', 'en', 'tr', 'uk', 'ua'],
                                     help_text=u'Локаль пользователя.'),
    })
    error_map = {
        404: NotFoundError,
    }


class GetLentaReportBlockVisitHandler(CaseApiBaseHandler):
    service = lenta_loader
    service_method = 'POST'
    service_url = '/api/report-block-visit?uid=%(uid)s&blockId=%(block_id)s&os=%(os)s'
    service_base_exception = LentaLoaderNoResponse
    permissions = AllowByClientIdPermission(PLATFORM_DISK_APPS_IDS)
    hidden = True

    query = fields.QueryDict({
        'block_id': fields.StringField(required=True, help_text=u'Идентификатор блока.'),

    })
    error_map = {
        404: NotFoundError,
    }

    def get_context(self, context=None):
        ret = super(GetLentaReportBlockVisitHandler, self).get_context()
        ret['os'] = UserAgentParser.get_os(self.request.raw_headers.get('user-agent', ''))
        return ret
