# -*- coding: utf-8 -*-
from mpfs.common.errors.platform import MpfsProxyBadResponse
from mpfs.core.services.passport_service import passport
from mpfs.platform import fields
from mpfs.platform.permissions import AllowByClientIdPermission
from mpfs.platform.v1.case.qa.exceptions import DiskSharedFoldersConflictError, DiskSharedFoldersBadRightsError, \
    DiskSharedFoldersInviteNotFoundError, DiskSharedFoldersUsersLimitReachedError, DiskSharedFoldersLimitIsReachedError, \
    DiskSharedFoldersNotEnoughSpaceError, DiskSharedFoldersUserNotFound, DiskSharedFolderUserNotInternal
from mpfs.platform.v1.case.handlers import PLATFORM_MOBILE_APPS_IDS
from mpfs.platform.v1.case.permissions import YandexUserPermission
from mpfs.platform.v1.case.qa.fields import SharedFolderRightsField
from mpfs.platform.v1.case.qa.serializers import SharedFolderSerializer, SharedFolderIncomingInviteSerializer, \
    SharedFolderIncomingInvitesListSerializer
from mpfs.platform.v1.disk.fields import MpfsPathField
from mpfs.platform.v1.disk.handlers import MpfsProxyHandler


class DiskSharedFoldersBaseHandler(MpfsProxyHandler):
    hidden = True
    permissions = AllowByClientIdPermission(PLATFORM_MOBILE_APPS_IDS) & YandexUserPermission()
    auth_methods = []  # У MpfsProxyHandler добавлена Cookie-авторизация, отключаем её для QA ручек, ибо нефиг.


class DiskSharedFoldersCreateHandler(DiskSharedFoldersBaseHandler):
    """Создать общую папку"""
    error_map = {
        85: DiskSharedFoldersNotEnoughSpaceError,
        108: DiskSharedFoldersConflictError,
        124: DiskSharedFoldersLimitIsReachedError,
    }
    service_url = '/json/share_create_group?uid=%(uid)s&path=%(path)s'
    query = fields.QueryDict({
        'path': MpfsPathField(required=True, allowed_areas=('disk',),
                              help_text=u'Путь к папке, которую необходимо сделать общей.'),
    })
    serializer_cls = SharedFolderSerializer


class DiskSharedFoldersCreateInviteHandler(DiskSharedFoldersBaseHandler):
    """Пригласить в общую папку"""
    service_url = '/json/share_invite_user?' \
                  'uid=%(uid)s&gid=%(shared_folder_id)s&rights=%(rights)s' \
                  '&universe_login=%(email)s&universe_service=email' \
                  '&ip=%(client_ip)s'
    resp_status_code = 201
    kwargs = fields.QueryDict({
        'shared_folder_id': fields.StringField(required=True, help_text=u'Идентификатор общей папки.'),
    })
    query = fields.QueryDict({
        'email': fields.StringField(required=True, help_text=u'Email на который будет отправлено приглашение.'),
        'rights': SharedFolderRightsField(required=True, help_text=u'Права доступа, которые предоставляет приглашение.'),
    })

    def get_context(self, context=None):
        c = super(DiskSharedFoldersCreateInviteHandler, self).get_context(context)
        c['client_ip'] = self.request.client.ip
        if not passport.is_yandex_internal_qa_user(c['email']):
            raise DiskSharedFolderUserNotInternal()
        return c

    def serialize(self, obj, *args, **kwargs):
        # прячем всё что вернул бэкэнд
        return {}


class DiskSharedFoldersListIncomingInvitesHandler(DiskSharedFoldersBaseHandler):
    """Список не принятых приглашений в чужие общие папки"""
    service_url = '/json/share_list_not_approved_folders?uid=%(uid)s'
    serializer_cls = SharedFolderIncomingInvitesListSerializer
    query = fields.QueryDict({
        'limit': fields.IntegerField(default=100, help_text=u'Количество возвращаемых приглашений.'),
        'offset': fields.IntegerField(default=0, help_text=u'Смещение от начала списка приглашений.'),
    })

    def serialize(self, obj, *args, **kwargs):
        limit = self.request.query['limit']
        offset = self.request.query['offset']
        data = {
            'limit': limit,
            'offset': offset,
            'items': obj[offset:limit],
        }
        return super(DiskSharedFoldersListIncomingInvitesHandler, self).serialize(data, *args, **kwargs)


class DiskSharedFoldersAcceptInviteHandler(DiskSharedFoldersBaseHandler):
    """Принять приглашение в общую папку"""
    error_map = {
        108: DiskSharedFoldersInviteNotFoundError,
        123: DiskSharedFoldersUsersLimitReachedError,
        125: DiskSharedFoldersLimitIsReachedError,
        145: DiskSharedFoldersNotEnoughSpaceError,
    }
    service_url = "/json/share_activate_invite?uid=%(uid)s&hash=%(invite_key)s"
    kwargs = fields.QueryDict({
        'invite_key': fields.StringField(required=True, help_text=u'Ключ активации приглашения.'),
    })

    def serialize(self, obj, *args, **kwargs):
        # прячем всё что вернул бэкэнд
        return {}



class DiskSharedFoldersRejectInviteHandler(DiskSharedFoldersBaseHandler):
    """Отклонить приглашение в общую папку"""
    service_url = "/json/share_reject_invite?uid=%(uid)s&hash=%(invite_key)s"
    kwargs = fields.QueryDict({
        'invite_key': fields.StringField(required=True, help_text=u'Ключ активации приглашения.'),
    })

    def serialize(self, obj, *args, **kwargs):
        # прячем всё что вернул бэкэнд
        return {}


class DiskSharedFoldersSetRightsHandler(DiskSharedFoldersBaseHandler):
    """Изменить права гостя в общей папке"""
    service_url = "/json/share_change_rights?uid=%(uid)s&gid=%(shared_folder_id)s&user_uid=%(guest_uid)s&rights=%(rights)s"
    not_accepted_invite_url = "/json/share_change_rights?uid=%(uid)s" \
                              "&gid=%(shared_folder_id)s" \
                              "&universe_login=%(email)s" \
                              "&universe_service=email" \
                              "&rights=%(rights)s"
    kwargs = fields.QueryDict({
        'shared_folder_id': fields.StringField(required=True, help_text=u'Идентификатор общей папки.'),
    })
    query = fields.QueryDict({
        'email': fields.StringField(required=True, help_text=u'Email пользователя Яндекс.Диска члена общей папки, '
                                                             u'для которого необходимо установить права.'),
        'rights': SharedFolderRightsField(required=True, help_text=u'Права доступа.'),
    })

    @staticmethod
    def get_user_uid(email):
        user_info = passport.userinfo(login=email)
        if user_info['uid']:
            if not passport.is_yandex_internal_qa_user(uid=user_info['uid']):
                raise DiskSharedFolderUserNotInternal()
            return user_info['uid']
        raise DiskSharedFoldersUserNotFound()

    def get_context(self, context=None):
        c = super(DiskSharedFoldersSetRightsHandler, self).get_context(context)
        c['guest_uid'] = self.get_user_uid(email=self.request.query['email'])
        return c

    def serialize(self, obj, *args, **kwargs):
        # прячем всё что вернул бэкэнд
        return {}

    def handle(self, request, *args, **kwargs):
        context = self.get_context()
        url = self.get_url(context)
        try:
            status_code, resp, _ = self.raw_request_service(url, method=self.service_method)
        except MpfsProxyBadResponse as e:
            http_code = getattr(e, 'data', {}).get('code')
            if http_code == 404:
                context['universe_login'] = self.request.query['email']
                url = self.build_url(self.not_accepted_invite_url, context)
                resp = self.request_service(url, method=self.service_method)
            else:
                raise e

        return self.serialize(resp)


class DiskSharedFoldersRevokeAccessHandler(DiskSharedFoldersSetRightsHandler):
    """Закрыть гостю доступ к общей папке"""
    service_url = "/json/share_kick_from_group?uid=%(uid)s&gid=%(shared_folder_id)s&user_uid=%(guest_uid)s"
    query = fields.QueryDict({
        'rights': None,
    })

    def handle(self, request, *args, **kwargs):
        url = self.get_url(self.get_context())
        resp = self.request_service(url, method=self.service_method)
        return self.serialize(resp)


class DiskSharedFoldersRemoveHandler(DiskSharedFoldersBaseHandler):
    """
    Удалить или приватизировать общую папку.

    Содержимое папки и сама папка у владельца не удаляется.
    """
    service_url = "/json/share_unshare_folder?uid=%(uid)s&gid=%(shared_folder_id)s"
    kwargs = fields.QueryDict({
        'shared_folder_id': fields.StringField(required=True, help_text=u'Идентификатор общей папки.'),
    })
    resp_status_code = 204

    def handle(self, request, *args, **kwargs):
        super(DiskSharedFoldersRemoveHandler, self).handle(request, *args, **kwargs)
        return self.resp_status_code, None
