# -*- coding: utf-8 -*-
import re
import traceback

from mpfs.platform.common import logger

from mpfs.common.util import from_json
from mpfs.platform.utils import (
    CaseInsensitiveDict,
    quote_string,
)
from mpfs.platform.v1.chaining.exceptions import ChainingApplyTemplateError
from mpfs.platform.v1.batch.batch_processors import BaseBatchRequestProcessor


class ChainRequestProcessor(BaseBatchRequestProcessor):
    """
    Обработчик чейн-запросов

    Выполняет request и последовательно выполняет child_requests подставляя в них результат выполнения request.

    :param main_request: Главный запрос. Он содержит основной запрос и подзапросы.
    """
    template_pattern_re = re.compile("({[^\}]+})")

    def process(self, chain_request):
        raw_request = self.build_raw_request(self.main_request, chain_request)
        request_result = self.dispatcher.dispatch(raw_request)

        response_headers = request_result.headers
        if not isinstance(response_headers, CaseInsensitiveDict):
            response_headers = CaseInsensitiveDict(response_headers)
        response_content = from_json(request_result.content)

        sub_requests_results = []
        for subreq in chain_request.subrequests:
            try:
                subreq.url = self._apply_template(subreq.url, response_headers, response_content)
            except (KeyError, IndexError, TypeError, ChainingApplyTemplateError):
                logger.error_log.exception('Couldn\'t apply results %s to template %s' %
                                           (request_result.content, subreq.url))
                sub_requests_results.append(
                    self.main_request.dispatcher.handle_error(ChainingApplyTemplateError(), log_trace=True))
            else:
                raw_request = self.build_raw_request(self.main_request, subreq)
                subrequest_result = self.dispatcher.dispatch(raw_request)
                sub_requests_results.append(subrequest_result)
        return request_result, sub_requests_results

    def dispatch(self, chain_request):
        self.main_request = self.dispatcher.request
        try:
            return self.process(chain_request)
        except Exception:
            logger.error_log.error(traceback.format_exc())
            raise
        finally:
            self.dispatcher.request = self.main_request

    @classmethod
    def _apply_template(cls, template, response_headers, response_content):
        """
        Возвращает строку, после применения к ней шаблона. Все подставляемые значения подвергаются url-энкоду
        """
        result_string = template
        to_replace = cls.template_pattern_re.findall(template)
        for i in to_replace:
            message_part, key = i.strip('{}').split('.', 1)
            if message_part == 'body':
                resp_value = cls._get_value_from_json_doc(response_content, key.split('.'))
            elif message_part == 'headers':
                resp_value = cls._get_value_from_headers(response_headers, key)
            else:
                raise ChainingApplyTemplateError()
            result_string = result_string.replace(i, resp_value)
        return result_string

    @staticmethod
    def _get_value_from_headers(headers, key):
        return quote_string(str(headers[key]))

    @classmethod
    def _get_value_from_json_doc(cls, json_doc, key):
        return quote_string(str(cls._lookup_json_doc(json_doc, key)))

    @staticmethod
    def _lookup_json_doc(json_doc, keys):
        cur_obj = json_doc
        for k in keys:
            if isinstance(cur_obj, list):
                k = int(k)
                if k < 0:
                    raise ChainingApplyTemplateError()
            cur_obj = cur_obj[k]
        return cur_obj
