# -*- coding: utf-8 -*-
import re

from mpfs.common.static.tags.platform import EXTERNAL
from mpfs.config import settings
from mpfs.platform.handlers import BasePlatformHandler
from mpfs.platform.v1.chaining import exceptions
from mpfs.platform.v1.chaining.chaining_processors import ChainRequestProcessor
from mpfs.platform.v1.chaining.serializers import (
    ChainRequestSerializer,
    ChainResponseSerializer
)
from mpfs.platform.v1.disk.permissions import WebDavPermission


PLATFORM_CHAINING_MAX_SUBREQUESTS = settings.platform['chaining']['max_subrequests']


class ChainRequestHandler(BasePlatformHandler):
    """Выполнить чейн запрос"""
    permissions = WebDavPermission()
    body_serializer_cls = ChainRequestSerializer
    serializer_cls = ChainResponseSerializer
    container_request_internal_re = re.compile(r'^/v1(/[^/]+)?/(chaining/request|batch/request).*')
    container_request_external_re = re.compile(r'^/v1/(chaining/request|batch/request).*')
    hidden = True

    def handle(self, request, *args, **kwargs):
        processor = ChainRequestProcessor(request.dispatcher)
        chain_request = request.body

        # Запрещаем в чейн-запросах вызывать батч или чейн запросы
        container_request_re = (self.container_request_external_re if request.mode == EXTERNAL
                                else self.container_request_internal_re)
        urls = [chain_request.url] + [r.url for r in chain_request.subrequests]
        if any(container_request_re.match(u) for u in urls):
            raise exceptions.ChainingForbiddenNestedContainerRequestError()

        if len(chain_request.subrequests) > PLATFORM_CHAINING_MAX_SUBREQUESTS:
            raise exceptions.ChainingTooManySubrequestsError()

        first_response, subrequest_responses = processor.dispatch(chain_request)
        return self.serialize({'content': first_response.content, 'headers': first_response.headers,
                               'status_code': first_response.status_code, 'subrequests': subrequest_responses})
