# -*- coding: utf-8 -*-
import urllib

from mpfs.common.errors import CommentsBadResponse
from mpfs.common.util import from_json, to_json
from mpfs.common.util.ordereddict import OrderedDict
from mpfs.core.services.comments_service import comments
from mpfs.platform.auth import PassportCookieAuth
from mpfs.platform.formatters import UnicodeStringFormatter
from mpfs.platform.handlers import ServiceProxyHandler
from mpfs.platform.permissions import AllowAllPermission
from mpfs.platform.serializers import AsIsSerializer
from mpfs.platform.v1.comments.exceptions import (
    CommentsBadRequestError,
    CommentsBadRequestBodyError,
    CommentsNotAuthenticatedError,
    CommentsForbiddenError,
    CommentsNotFoundError,
    CommentsMethodNotAllowedError,
    CommentsReplyToReplyError,
    CommentsDisabledError,
    CommentsRateLimitExceededError,
    CommentsSpamDetectedError
)


class CommentsProxyHandler(ServiceProxyHandler):
    auth_required = False
    auth_user_required = False
    auth_methods = [PassportCookieAuth()]
    permissions = AllowAllPermission()

    content_types = OrderedDict({ServiceProxyHandler.CONTENT_TYPE_JSON: UnicodeStringFormatter()})
    body_serializer_cls = AsIsSerializer

    service = comments
    service_base_exception = CommentsBadResponse

    forward_headers = True
    do_retries = True

    error_map = {
        'validate': CommentsBadRequestError,
        'cant-parse': CommentsBadRequestBodyError,
        'reply-to-reply': CommentsReplyToReplyError,
        'not-authenticated': CommentsNotAuthenticatedError,
        'forbidden': CommentsForbiddenError,
        'comments-disabled': CommentsDisabledError,
        'not-found': CommentsNotFoundError,
        'method-not-allowed': CommentsMethodNotAllowedError,
        'comment-rate-limit-exceeded': CommentsRateLimitExceededError,
        'spam-detected': CommentsSpamDetectedError,
    }

    mocked_response = to_json({
        "users": [],
        "items": [],
        "entity": {
            "actions": [
                "view"
            ],
            "comments_count": 0,
            "dislikes_count": 0,
            "likes_count": 0
        }
    })

    def handle(self, request, *args, **kwargs):
        return 200, self.mocked_response

    def get_service_error_code(self, exception):
        data = getattr(exception, 'data', {})
        text = data.get('text', '{}')
        if text:
            try:
                resp = from_json(text)
            except (UnicodeDecodeError, ValueError):
                return None
            else:
                error = resp.get('error', {})
                return error.get('name', None)
