# -*- coding: utf-8 -*-

import base64

from mpfs.common.util import from_json, to_json
from mpfs.platform.fields import StringField
from mpfs.platform.v1.data import exceptions


class SubscriptionToken(object):
    REQUIRED_FIELDS = {'uid', 'service', 'uuid', 'push_token'}
    OPTIONAL_FIELDS = {'app_name', 'platform'}

    def __init__(self, uid, service, uuid, push_token, app_name=None, platform=None):
        # Xiva позволяет в поле service указывать фильтр по тегам: `<service_name>:<tags_filter_expr>`
        # Но тут теги нам не нужны
        assert ':' not in service, "Тэги не должны присутствовать в имени сервиса"
        self.uid = uid
        self.uuid = uuid
        self.push_token = push_token
        self.app_name = app_name
        self.platform = platform
        self.service = service

    @classmethod
    def parse(cls, value):
        """
        Токен нового вида выглядит как b64encoded заджейсоненная структура следующего вида:
        {
            "uid": 123456789,
            "service": "service_name:tag1+tag2+.+tag5*tag6..+tagN",
            "uuid": "RnufthegcHueUxV",
            "push_token": "KKYRKTlXPnxxBVwrtxVr",
            ["app_name": "Яндекс.Диск",]
            ["platform": "ios",]
        }, где
        uid - уникальный идентификатор пользователя, выдаваемый паспортом,
        service - сервис, на уведомления которого осуществляется подписка (состоит из имени сервиса [и набора тегов]),
        uuid - уникальный идентификатор инстанса установленного приложения,
        push_token - токен регистрации, выданный владельцем платформы приложения (Apple, Google, etc.),
        app_name - название приложения
        platform - платформа приложения (ios | gcm)
        """
        if not isinstance(value, basestring):
            raise TypeError('Expected string, got %s.' % type(value))

        try:
            value = base64.b64decode(value)
            value = from_json(value)
        except Exception as e:
            raise ValueError(e.message)

        if not isinstance(value, dict):
            raise TypeError('Expected encoded dict, got %s.' % type(value))

        for key in cls.REQUIRED_FIELDS:
            if key not in value:
                raise ValueError('Key %s not found.' % key)

        return cls(**value)

    def serialize(self):
        data = {}
        for key in self.REQUIRED_FIELDS | self.OPTIONAL_FIELDS:
            value = getattr(self, key)
            if not isinstance(value, basestring):
                if key in self.OPTIONAL_FIELDS:
                    continue
                else:
                    raise ValueError('Key %s has wrong type. Expected string, got %s.' % (key, type(value)))
            else:
                data[key] = value

        return base64.b64encode(to_json(data))

    def get_xiva_params(self):
        return {
            'uid': self.uid,
            'service': self.service,
            'uuid': self.uuid,
            'push_token': self.push_token,
            'app_name': self.app_name,
            'platform': self.platform
        }


class SubscriptionIdField(StringField):
    """Поле для хранения токена подписки.

    На вход принимает строку токена и преобразует ее в объект токена (~SubscriptionToken).
    На выход выдает представление объекта токена в виде строки.
    """

    def to_native(self, value):
        """Принимает на вход токен подписки в виде строки, возвращает объект токена (~SubscriptionToken)."""
        value = super(SubscriptionIdField, self).to_native(value)
        try:
            token = SubscriptionToken.parse(value)
        except (TypeError, ValueError):
            raise exceptions.DataAppSubscriptionIdValidationError()

        return token

    def from_native(self, value):
        """Принимает на вход объект токена (~SubscriptionToken), возвращает его представление в виде строки."""
        value = super(SubscriptionIdField, self).from_native(value)
        return value.serialize()
