# -*- coding: utf-8 -*-
import base64
import re
import uuid
import urlparse
import urllib
from collections import OrderedDict
from copy import copy

from mpfs.config import settings
from mpfs.common.errors import DataApiBadResponse, XivaError
from mpfs.common.errors.platform import MpfsProxyBadResponse
from mpfs.common.util import from_json
from mpfs.core.services.data_api_service import data_api
from mpfs.core.services.push_service import XivaSubscribeDataSyncService
from mpfs.platform import fields
from mpfs.platform.common import (
    ResponseObject,
    PlatformResponse,
    logger,
)
from mpfs.platform.exceptions import UnauthorizedError, ForbiddenError, ServiceUnavailableError
from mpfs.platform.formatters import ProtobufFormatter
from mpfs.platform.handlers import ServiceProxyHandler
from mpfs.platform.rate_limiters import PerClientIdRateLimiter, PerPublicDBRateLimiter, DataApiPerClientIdRateLimiter
from mpfs.platform.v1.data.backend_pb2 import Database, ListDatabases, Snapshot, ListOfDeltas
from mpfs.platform.v1.data import data_pb2
from mpfs.platform.v1.data.exceptions import (
    DataNotFoundError,
    DataInvalidDeltaError,
    DataTooOldDeltaError,
    DataDeltaIsNewerThanDatabaseRevisionError,
    DataDeltasForRevisionIsGoneError,
    DataInvalidIdError,
    DataReadOnlyError,
    DataDatabaseSizeLimitExceededError,
    DataDatabasesNumberLimitExceededError,
    DataBadRequestError,
    DataInvalidSchemaError,
    DataValidationError,
)
from mpfs.platform.v1.data.permissions import (
    DataApiAccessPermission,
    DataApiSubscriptionsAccessPermission,
    DataApiSetRevisionPermissions,
    DataApiXivaSubscribePermission)
from mpfs.platform.v1.data.serializers import (
    DatabaseSerializer,
    DatabaseListSerializer,
    DatabaseSnapshotSerializer,
    DeltaSerializer,
    DeltaListSerializer,
    DatabasePatchSerializer,
    UserListSerializer,
    SetRevisionSerializer,
    AppSubscriptionSerializer
)
from mpfs.platform.v1.data.fields import SubscriptionToken, SubscriptionIdField
from mpfs.platform.v1.disk.serializers import LinkSerializer
from mpfs.platform.v1.disk.permissions import WebDavPermission
from mpfs.platform.auth import PassportCookieAuth, YaTeamCookieAuth, ExternalTokenAuth
from mpfs.platform.permissions import AllowByClientIdPermission, DenyAllPermission, AllowAllPermission

PUBLIC_DB_ID_KEY = 'public_db_id'
PLATFORM_DATA_API_PUBLIC_DATABASES = settings.platform['data_api']['public_databases']
PLATFORM_DATA_API_MODE_DIRECT = settings.platform['data_api']['mode']['direct']
PLATFORM_NOTES_APP_ID = settings.platform['notes_app_id']
PLATFORM_DATA_SYNC_XIVA_PERMISSIONS = settings.platform['data_sync_xiva_permissions']

PLATFORM_TLDS = settings.platform['tlds']

FEATURE_TOGGLES_LOG_PROTOBUF_BODY_ON_DATA_API_ERRORS = settings.feature_toggles['log_protobuf_body_on_data_api_errors']


def _auto_initialize_user(func):
    def wrapper(handler, *args, **kwargs):
        try:
            return func(handler, *args, **kwargs)
        except handler.service_base_exception, e:
            if handler.get_service_error_code(e) == 'user-not-found':
                handler.init_user()
                return func(handler, *args, **kwargs)
            raise
    return wrapper


def _get_service_error_code_from_protobuf(raw_data_api_resp):
    try:
        error = data_pb2.Error.FromString(raw_data_api_resp)
        return error.error
    except Exception:
        logger.error_log.error('failed to parse error details from Data API response', exc_info=True)
        return None


def _get_service_error_code(exception):
    data = getattr(exception, 'data', {})
    raw_data_api_resp = data.get('text', '{}')
    if raw_data_api_resp:
        try:
            data_api_resp = from_json(raw_data_api_resp)
        except (UnicodeDecodeError, ValueError):
            # если нам в ответ приехал protobuf
            return _get_service_error_code_from_protobuf(raw_data_api_resp)
        else:
            error = data_api_resp.get('error', {})
            return error.get('name', None)

def extract_etag_from_direct_response_headers(headers):
    etag_header = {}
    if 'etag' in headers:
        etag_header['ETag'] = int(headers['etag'])
    return etag_header

def construct_direct_url(initial_url):
    return initial_url + '&direct=true'

def construct_backend_request_accept_header(response_content_type):
    if response_content_type == 'application/json':
        return {'Accept': 'application/json'}
    else:
        return {'Accept': 'application/x-protobuf, application/json'}


class PublicDbId(object):
    SEP = '@'

    def __init__(self, app_id, db_id):
        self.app_id = app_id
        self.db_id = db_id

    def __str__(self):
        return '%s%s%s' % (self.app_id, self.SEP, self.db_id)


class DataApiProxyHandler(ServiceProxyHandler):
    auth_methods = [PassportCookieAuth(), YaTeamCookieAuth(), ExternalTokenAuth()]
    service = data_api
    client_id_rate_limiter = DataApiPerClientIdRateLimiter()
    public_db_id_rate_limiter = None
    service_base_exception = DataApiBadResponse
    permissions = DataApiAccessPermission()
    service_pb2py_cls = None
    rate_limiter = None
    error_map = {
        'user-not-found': UnauthorizedError,
        'not-found': DataNotFoundError,
        'forbidden': ForbiddenError,
        'invalid-schema': DataInvalidSchemaError,
        'validate': DataValidationError,
    }
    user_init_method = 'PUT'
    user_init_url = '/api/register_user?__uid=%(uid)s&login=%(login)s'
    PUBLIC_DB_RE = re.compile(r'^\.pub\.(?P<app_id>[^@]+)@(?P<db_id>.+)$')

    kwargs = fields.QueryDict({
        'context': fields.ChoiceField(choices=['user', 'app'], default=False, required=True,
                                      help_text=u'''Контекст запроса.

*   ```app``` - приватный контекст приложения,
*   ```user``` - общий контекст пользователя, доступный всем приложениям.
'''),
    })

    @_auto_initialize_user
    def handle(self, request, *args, **kwargs):
        url = self.get_url(self.get_context())
        if PLATFORM_DATA_API_MODE_DIRECT:
            url = construct_direct_url(url)
            headers = construct_backend_request_accept_header(self.get_response_content_type())
            status_code, resp, headers = self.raw_request_service(url, method=self.service_method, headers=headers)
            return PlatformResponse(status_code, resp, extract_etag_from_direct_response_headers(headers))
        else:
            headers = {'Accept': 'application/x-protobuf, application/json'}
            status_code, resp, headers = self.raw_request_service(url, method=self.service_method, headers=headers)
            resp = self.serialize(resp)
            return status_code, resp

    def check_rate_limit(self, request):
        # Лимитируем запросы к публичным базкам
        public_db_id = getattr(request, PUBLIC_DB_ID_KEY, None)
        if (public_db_id is not None and
                self.public_db_id_rate_limiter is not None):
            return self.public_db_id_rate_limiter.check(request)

        self.client_id_rate_limiter.check(request)
        return super(DataApiProxyHandler, self).check_rate_limit(request)

    def raw_request_service(self, url, method=None, headers=None, data=None, service=None):
        if self.request.client and self.request.client.token:
            headers = headers or {}
            headers = headers.copy()
            headers.update({'Authorization': 'OAuth %s' % self.request.client.token})

        return super(DataApiProxyHandler, self).raw_request_service(
            url, method=method, headers=headers, data=data, service=service)

    def use_direct_datasync_invocation(self):
        return PLATFORM_DATA_API_MODE_DIRECT and self.get_response_content_type() == 'application/json'

    def parse_backend_response(self, resp):
        if self.service_pb2py_cls and isinstance(resp, (str, unicode, bytes)):
            return self.service_pb2py_cls.FromString(resp)
        return resp

    def serialize(self, obj, *args, **kwargs):
        obj = self.parse_backend_response(obj)
        return super(DataApiProxyHandler, self).serialize(obj, *args, **kwargs)

    def init_user(self):
        """Инициализирует пользователя Data API если тот ещё не проинициализирован"""
        user = self.request.user
        # Дёргаем Data API чтоб проинициализировал пользователя
        url = self.build_url(self.user_init_url, {'uid': user.uid, 'login': user.login or 'uid-%s' % user.uid})
        self.request_service(url, method=self.user_init_method)
        self.request.user_initialized = True

    def get_url(self, context=None):
        url = self.service_url
        if self.request.kwargs.get('context', None) == 'app':
            url = '%s&app=%%(app_id)s' % url
        return self.build_url(url, context=context)

    def get_context(self, context=None):
        c = super(DataApiProxyHandler, self).get_context(context)

        request = self.request
        public_db_id = getattr(request, PUBLIC_DB_ID_KEY, None)
        if public_db_id:
            c['app_id'] = public_db_id.app_id
            c['database_id'] = u'.pub.%s' % public_db_id.db_id
            c.update(PLATFORM_DATA_API_PUBLIC_DATABASES)
        else:
            c['app_id'] = request.client.id

        return c

    def get_service_error_code(self, exception):
        return _get_service_error_code(exception)

    def parse_public_db_id(self, db_id):
        """
        Вытаскивает из идентификатора публичной базки id-шник приложения и идентификатор базки.

        :return: PublicDbId(app_id, db_id) или None, если базка не публичная.
        """
        public_database_match = self.PUBLIC_DB_RE.match(db_id)
        if public_database_match:
            d = public_database_match.groupdict()
            app_id = d.get('app_id')
            db_id = d.get('db_id')
            if app_id and db_id:
                result = PublicDbId(app_id, db_id)
                return result

    def authorize(self, request, *args, **kwargs):
        db_id = kwargs.get('database_id') or ''
        public_db_id = self.parse_public_db_id(db_id)
        setattr(request, PUBLIC_DB_ID_KEY, public_db_id)

        try:
            authorized = super(DataApiProxyHandler, self).authorize(request, *args, **kwargs)
        except UnauthorizedError:
            authorized = False
        if authorized:
            return True

        # Если запрос к публичной базке, то считаем, что запрос авторизован, даже если это не так.
        if public_db_id:
            return True

        raise UnauthorizedError()

    def check_permissions(self, request):
        if getattr(request, PUBLIC_DB_ID_KEY, False):
            # Если пришел запрос на публичную базёнку, то разрешения не проверяем.
            pass
        else:
            super(DataApiProxyHandler, self).check_permissions(request)


class ListDatabasesHandler(DataApiProxyHandler):
    """Получить список баз данных

    #### OAuth-права

      1. Доступ для хранения данных приложения.
         Необходим для работы в контексте ```app```.
      2. Доступ к общим данным приложений пользователя.
         Необходим для работы в контексте ```user```.
    """
    service_url = '/api/databases?__uid=%(uid)s&limit=%(limit)s&offset=%(offset)s'
    serializer_cls = DatabaseListSerializer
    service_pb2py_cls = ListDatabases
    content_types = OrderedDict([
        ('application/protobuf', ProtobufFormatter(data_pb2.DatabaseList)),
    ])
    query = fields.QueryDict({
        'limit': fields.IntegerField(default=100, help_text=u'Количество выводимых ресурсов.'),
        'offset': fields.IntegerField(default=0, help_text=u'Смещение от начала списка ресурсов.'),
    })

    def serialize(self, obj, *args, **kwargs):
        obj = self.parse_backend_response(obj)
        obj = self.serializer_cls.get_dict(obj)
        limit = self.request.query['limit']
        offset = self.request.query['offset']
        databases = obj.get('databases', [])
        obj.update({
            'context': self.request.kwargs.get('context'),
            'limit': limit,
            'offset': offset,
            'total': len(databases),
            'databases': databases[offset:offset+limit]
        })
        return super(ListDatabasesHandler, self).serialize(obj, *args, **kwargs)


class DatabaseHandler(DataApiProxyHandler):
    serializer_cls = DatabaseSerializer
    service_url = '/api/databases/%(database_id)s?__uid=%(uid)s'
    service_pb2py_cls = Database
    kwargs = fields.QueryDict({
        'database_id': fields.StringField(required=True, help_text=u'Идентификатор базы данных.')
    })


class DataBaseWithETagHandler(DatabaseHandler):
    etag_attribute_name = 'rev'

    def parse_backend_response(self, resp):
        self.request.etag = None
        ret = super(DataBaseWithETagHandler, self).parse_backend_response(resp)
        if isinstance(ret, dict):
            self.request.etag = ret[self.etag_attribute_name]
        else:
            self.request.etag = getattr(ret, self.etag_attribute_name, None)
        return ret

    def handle(self, request, *args, **kwargs):
        if PLATFORM_DATA_API_MODE_DIRECT:
            return super(DataBaseWithETagHandler, self).handle(request, *args, **kwargs)
        else:
            status_code, resp = super(DataBaseWithETagHandler, self).handle(request, *args, **kwargs)
            if request.etag is None:
                raise KeyError('ETag is not defined.')
            return status_code, resp, {'ETag': request.etag}

    def get_exposed_headers(self, request):
        result = super(DataBaseWithETagHandler, self).get_exposed_headers(request)
        result.append('ETag')
        return result


class CollectionFilterHandlerMixin(ServiceProxyHandler):
    query = fields.QueryDict({
        'collection_id': fields.StringField(help_text=u'Фильтр по идентификатору коллекции.'),
    })

    def get_url(self, context=None):
        url = super(CollectionFilterHandlerMixin, self).get_url(context=context)
        collection_id = self.request.query.get('collection_id', None)
        if collection_id:
            url += '&collectionId=%(collection_id)s' % self.urlencode_context({'collection_id': collection_id})
        return url


class LogDetailsOnErrorMixin(ServiceProxyHandler):
    def handle_exception(self, exception):
        """При 500 логирует детали запроса."""
        if (FEATURE_TOGGLES_LOG_PROTOBUF_BODY_ON_DATA_API_ERRORS and
                isinstance(exception, self.service_base_exception)):
            data = getattr(exception, 'data', {})
            if (isinstance(data, dict) and
                        data.get('code') == 500 and
                        self.get_response_content_type() == 'application/protobuf' and
                        self.request.data is not None):
                logger.error_log.error('protobuf request failed for body: %s' % base64.b64encode(self.request.data))

        return super(LogDetailsOnErrorMixin, self).handle_exception(exception)


class GetDatabaseHandler(DataBaseWithETagHandler):
    """Получить метаданные базы данных

    Возвращает заголовок ```ETag```, содержащий текущую ревизию БД.

    #### OAuth-права

      1. Доступ для хранения данных приложения.
         Необходим для работы в контексте ```app```.
      2. Доступ к общим данным приложений пользователя.
         Необходим для работы в контексте ```user```.
    """
    content_types = OrderedDict([
        ('application/protobuf', ProtobufFormatter(data_pb2.Database)),
    ])


class GetOrCreateDatabaseHandler(LogDetailsOnErrorMixin, DataBaseWithETagHandler):
    """
    Создать или получить существующую базу данных

    Возвращает заголовок ```ETag```, содержащий текущую ревизию БД.

    Если база данных с указанным идентифкатором ещё не существует, то создаст её и вернёт ответ с кодом ```201```,
    иначе вернёт ответ с кодом ```200```.

    #### OAuth-права

      1. Доступ для хранения данных приложения.
         Необходим для работы в контексте ```app```.
      2. Доступ к общим данным приложений пользователя.
         Необходим для работы в контексте ```user```.
    """
    service_method = 'PUT'
    error_map = {
        'invalid-name': DataInvalidIdError,
        'read-only': DataReadOnlyError,
        'databases-count-limit-reached': DataDatabasesNumberLimitExceededError,
    }
    content_types = OrderedDict([
        ('application/protobuf', ProtobufFormatter(data_pb2.Database)),
    ])

    def get_response_objects(self):
        ret = super(GetOrCreateDatabaseHandler, self).get_response_objects()
        ret += [ResponseObject(self.serializer_cls, u'Создана новая база данных.', 201)]
        return ret


class UpdateDatabaseHandler(LogDetailsOnErrorMixin, DataBaseWithETagHandler):
    """
    Обновить атрибуты базы данных

    Возвращает заголовок ```ETag```, содержащий текущую ревизию БД. При изменении атрибутов БД ревизия не изменяется.

    #### OAuth-права

      1. Доступ для хранения данных приложения.
         Необходим для работы в контексте ```app```.
      2. Доступ к общим данным приложений пользователя.
         Необходим для работы в контексте ```user```.
    """
    service_method = 'PATCH'
    content_types = OrderedDict([
        ('application/protobuf', ProtobufFormatter(data_pb2.Database)),
    ])
    body_serializer_cls = DatabasePatchSerializer

    @_auto_initialize_user
    def handle(self, request, *args, **kwargs):
        url = self.get_url(self.get_context())
        if PLATFORM_DATA_API_MODE_DIRECT:
            url = construct_direct_url(url)

        headers = {'Content-Type': 'application/json', 'Accept': 'application/json'}
        assert isinstance(request.body, dict)

        data = copy(request.body)
        if 'description' in data and data['description'] is None:
            data.pop('description')
            data['removeDescription'] = True

        if PLATFORM_DATA_API_MODE_DIRECT:
            headers.update(construct_backend_request_accept_header(self.get_response_content_type()))
            status_code, resp, headers = self.raw_request_service(url, method=self.service_method, headers=headers, data=data)
            return PlatformResponse(status_code, resp, extract_etag_from_direct_response_headers(headers))
        else:
            resp = self.request_service(url, method=self.service_method, headers=headers, data=data)
            database = resp['result']
            ret = self.serialize(database)
            return ret


class DeleteDatabaseHandler(DatabaseHandler):
    """Удалить базу данных

    #### OAuth-права

      1. Доступ для хранения данных приложения.
         Необходим для работы в контексте ```app```.
      2. Доступ к общим данным приложений пользователя.
         Необходим для работы в контексте ```user```.
    """
    service_method = 'DELETE'
    serializer_cls = None
    service_pb2py_cls = None
    error_map = {
        'read-only': DataReadOnlyError,
    }
    response_objects = [
        ResponseObject(None, u'База данных удалена.', 204)
    ]
    content_types = OrderedDict([
        ('application/protobuf', ProtobufFormatter(data_pb2.Database)),
    ])

    def handle(self, request, *args, **kwargs):
        super(DeleteDatabaseHandler, self).handle(request, *args, **kwargs)
        return 204, None


class GetSnapshotHandler(CollectionFilterHandlerMixin, DataBaseWithETagHandler):
    """
    Получить снапшот базы данных

    Возвращает заголовок ```ETag```, содержащий текущую ревизию БД.

    #### OAuth-права

      1. Доступ для хранения данных приложения.
         Необходим для работы в контексте ```app```.
      2. Доступ к общим данным приложений пользователя.
         Необходим для работы в контексте ```user```.
    """
    service_url = '/api/databases/%(database_id)s/snapshot?__uid=%(uid)s'
    service_pb2py_cls = Snapshot
    serializer_cls = DatabaseSnapshotSerializer
    public_db_id_rate_limiter = PerPublicDBRateLimiter()
    content_types = OrderedDict([
        ('application/protobuf', ProtobufFormatter(data_pb2.DatabaseSnapshot)),
    ])

    def serialize(self, obj, *args, **kwargs):
        obj = self.parse_backend_response(obj)
        obj = self.serializer_cls.get_dict(obj)
        # Почему-то в снапшоте не отдаётся databaseId, исправляем это досадное недоразумение руками.
        obj['databaseId'] = self.request.kwargs['database_id']
        # немного перепаковываем данные, чтоб сериализаторам было удобнее
        obj['records'] = {'items': obj.pop('records', [])}

        resp = super(GetSnapshotHandler, self).serialize(obj, *args, **kwargs)

        return resp


class CreateDeltaHandler(LogDetailsOnErrorMixin, DatabaseHandler):
    """
    Применить изменение

    Возвращает заголовок ```ETag```, содержащий новую ревизию БД после применения изменения или текущую ревизию БД,
    если применить изменение не удалось.

    При создании изменения в объекте ```Value``` можно не указывать атрибут ```type```,
    достаточно указать значение в соответствующем типу значения атрибуте.
    Однако все объекты ```Value```, возвращаемые API, будут содержать атрибут ```type```.

    В случае успешного применения изменений вернёт ответ со статусом ```201```
    и ссылкой на список последних изменений БД.

    Если последняя ревизия БД немного старше ревизии указанной в заголовке ```If-Match```,
    то вернёт ответ со статусом ```409``` и объектом ```DeltaList``` со списком изменений,
    которые следует применить к локальной копии БД прежде чем вносить изменения в удалённую БД.

    Если последняя ревизия БД сильно старше ревизии указанной в заголовке ```If-Match```
    и обновить локальную копию базы данных при помощи списка изменений уже невозможно,
    то вернёт ошибку с кодом ```410```. При этом клиент должен обновить локальную копию БД получив свежий снапшот.

    #### OAuth-права

      1. Доступ для хранения данных приложения.
         Необходим для работы в контексте ```app```.
      2. Доступ к общим данным приложений пользователя.
         Необходим для работы в контексте ```user```.
    """
    service_method = 'PUT'
    resp_status_code = 201
    service_url = '/api/databases/%(database_id)s/diffs?rev=%(If-Match)s&__uid=%(uid)s'
    headers = fields.QueryDict({
        'If-Match': fields.IntegerField(required=True,
                                        help_text=u'Ревизия БД к которой должно быть применено изменение.'),
    })
    body_serializer_cls = DeltaSerializer
    serializer_cls = LinkSerializer
    error_map = {
        'invalid-delta': DataInvalidDeltaError,
        'too-new-change': DataDeltaIsNewerThanDatabaseRevisionError,
        'deltas-gone': DataDeltasForRevisionIsGoneError,
        'invalid-name': DataInvalidIdError,
        'read-only': DataReadOnlyError,
        'size-limit-reached': DataDatabaseSizeLimitExceededError,
    }
    content_types = OrderedDict([
        ('application/protobuf', ProtobufFormatter(in_cls=data_pb2.Delta, out_cls=data_pb2.Link)),
    ])

    response_objects = [
        ResponseObject(DeltaListSerializer, u'Ревизия БД на сервере старше текущей локальной ревизии БД.', status_code=409)
    ]

    @_auto_initialize_user
    def handle(self, request, *args, **kwargs):
        url = self.get_url(self.get_context())
        headers = {
            'Content-Type': 'application/x-protobuf',
            'Accept': 'application/x-protobuf, application/json',
        }

        if PLATFORM_DATA_API_MODE_DIRECT:
            url = construct_direct_url(url)
            headers.update(construct_backend_request_accept_header(self.get_response_content_type()))

        self.raw_request_service(url, method=self.service_method, headers=headers,
                                 data=request.body.SerializeToString())
        context = request.kwargs['context']
        revision = request.headers['If-Match']
        database_id = request.kwargs['database_id']
        link = self.router.get_link(
            ListDeltasHandler, params={'context': context, 'base_revision': revision + 1, 'database_id': database_id})
        resp = self.serialize(link)
        return 201, resp, {'ETag': revision + 1}

    def get_exposed_headers(self, request):
        result = super(CreateDeltaHandler, self).get_exposed_headers(request)
        result.append('ETag')
        return result

    def handle_exception(self, exception):
        # особо обрабатываем 409 ошибку, т.к. ответ на неё содержит дельты, которые надо отдать клиенту вместе с ошибкой
        if isinstance(exception, self.service_base_exception):
            data = getattr(exception, 'data', {})
            if data.get('code', None) == 409 and self.get_service_error_code(exception) is None:
                raw_deltas = data.get('text', None)
                if raw_deltas:
                    if PLATFORM_DATA_API_MODE_DIRECT:
                        if self.get_response_content_type() == 'application/json':
                            deltas = self.get_response_formatter().parse(raw_deltas)
                        else:
                            deltas = type(self.get_response_formatter())(data_pb2.DeltaList).parse(raw_deltas)
                    else:
                        pb_deltas = ListOfDeltas.FromString(raw_deltas)
                        deltas = DeltaListSerializer(obj=pb_deltas,
                                                     format_options=self.get_response_formatter().FormatOptions).data
                    # Ноги растут оттуда: https://wiki.yandex-team.ru/disk/data-api/backend-api#putdiff
                    # Ответ 409 - конфликт. В случае конфликта в теле ответа содержит список изменений в том же формате,
                    # что и ручка get diff. Если пришел пустой список, но при этом количество дельт указано большее 0,
                    # то это значит, что при запросе get diffs, ответ будет 410, т.е. нужно получать полный снапшот.
                    if 'items' not in deltas or len(deltas['items']) == 0:
                        raise DataTooOldDeltaError(headers={'ETag': deltas['revision']})
                    else:
                        if PLATFORM_DATA_API_MODE_DIRECT:
                            return PlatformResponse(409, raw_deltas, {'ETag': deltas['revision']})
                        else:
                            deltas['base_revision'] = deltas['items'][0]['base_revision']
                            return 409, deltas, {'ETag': deltas['revision']}

        return super(CreateDeltaHandler, self).handle_exception(exception)

    def make_response(self, ret):
        if isinstance(ret, tuple):
            formatter = self.get_response_formatter()
            if ret[0] == 409 and isinstance(formatter, ProtobufFormatter):
                return PlatformResponse(ret[0], type(formatter)(data_pb2.DeltaList).format(ret[1]), ret[2])
        return super(CreateDeltaHandler, self).make_response(ret)


class ListDeltasHandler(CollectionFilterHandlerMixin, DataBaseWithETagHandler):
    """
    Получить список изменений

    Возвращаемые объекты ```DeltaList``` и ```Delta``` содержат такие атрибуты как ```base_revision``` и ```revision```.
    Немного подробнее о значении этих атрибутов:
      * ```DeltaList.base_revision``` содержит номер ревизии БД к которой были применены изменения в списке.
      * ```Delta.base_revision``` содержит номер ревизии БД к которой было применено изменение.
      * ```DeltaList.revision``` содержит номер текущей актуальной ревизии БД.
      * ```Delta.revision``` содержит номер ревизии БД после применения изменения.

    Если запрошен список изменений для старой ревизии, то вернётся ответ со статусом ```410```. В этом случае вместо
    запроса изменений клиент должен запросить снапшот базы данных для получения актуального состояния БД.

    По умолчанию, на странице возвращается 100 последних изменений, т.е. параметр ```limit``` равен 100.
    Максимальное количество изменений на странице также равно 100, значения ```limit``` больше 100 игнорируются.

    #### OAuth-права

      1. Доступ для хранения данных приложения.
         Необходим для работы в контексте ```app```.
      2. Доступ к общим данным приложений пользователя.
         Необходим для работы в контексте ```user```.
    """
    service_method = 'GET'
    service_url = '/api/databases/%(database_id)s/diffs?rev=%(base_revision)s&limit=%(limit)s&__uid=%(uid)s'
    serializer_cls = DeltaListSerializer
    service_pb2py_cls = ListOfDeltas
    etag_attribute_name = 'currentDatabaseRev'
    error_map = {
        'deltas-gone': DataDeltasForRevisionIsGoneError,
    }
    public_db_id_rate_limiter = PerPublicDBRateLimiter()
    content_types = OrderedDict([
        ('application/protobuf', ProtobufFormatter(data_pb2.DeltaList)),
    ])

    query = fields.QueryDict({
        'base_revision': fields.IntegerField(required=True, help_text=u'Ревизия БД, для которой нужно получить изменения.'),
        'limit': fields.IntegerField(default=100, help_text=u'Количество изменений на странице.'),
    })

    def serialize(self, obj, *args, **kwargs):
        obj = self.parse_backend_response(obj)
        obj_dict = self.serializer_cls.get_dict(obj)
        obj_dict['base_revision'] = self.request.query['base_revision']
        obj_dict['limit'] = self.request.query['limit']
        return super(ListDeltasHandler, self).serialize(obj_dict, *args, **kwargs)

class ListAppUsersHandler(ServiceProxyHandler):
    """Получить список пользователей приложения"""
    auth_methods = [YaTeamCookieAuth(), ]
    auth_user_required = False
    permissions = DenyAllPermission()

    service = data_api
    service_method = 'GET'
    service_url = '/api/users?app=%(app_id)s'
    service_timeout = data_api.list_users_timeout

    serializer_cls = UserListSerializer
    service_base_exception = DataApiBadResponse

    query = fields.QueryDict({
        'database_id': fields.StringField(help_text=u'Фильтр по идентификатору базы'),
        'limit': fields.IntegerField(default=1000, help_text=u'Количество пользователей за запрос (не более 100к).'),
        'iteration_key': fields.StringField(help_text=u'Ключ для продолжения итерирования.'),
    })
    error_map = {
        'validate': DataBadRequestError,
    }

    def get_url(self, context=None):
        url = self.service_url

        for param in ['database_id', 'limit', 'iteration_key']:
            if context[param]:
                url += '&%s=%%(%s)s' % (param, param)
        return self.build_url(url, context=context)

    def get_context(self, context=None):
        c = super(ListAppUsersHandler, self).get_context(context)
        c['app_id'] = self.request.client.id
        return c

    def serialize(self, obj, *args, **kwargs):
        return super(ListAppUsersHandler, self).serialize(obj['result'], *args, **kwargs)

    def get_service_error_code(self, exception):
        return _get_service_error_code(exception)


class XivaSubscriptionUrlBaseHandler(ServiceProxyHandler):
    auth_methods = [PassportCookieAuth(), YaTeamCookieAuth(), ]
    permissions = DataApiSubscriptionsAccessPermission()
    service = XivaSubscribeDataSyncService()
    serializer_cls = LinkSerializer

    tld_regex = re.compile(r'(?<=yandex\.).*$')
    valid_tlds = set(PLATFORM_TLDS)

    def get_context(self, context=None):
        c = super(XivaSubscriptionUrlBaseHandler, self).get_context(context)
        c['client'] = self.request.client.id
        c['oauth_token'] = self.request.client.token
        if 'host' in self.request.raw_headers:
            host = self.request.raw_headers['host']
            matches = self.tld_regex.findall(host)
            if len(matches) > 0 and matches[0] in self.valid_tlds:
                c['tld'] = matches[0]
        return c

    def get_url_with_optional_params(self, context=None, optional_parameters=None, base_url=None):
        url = self.service_url
        scheme, netloc, path, query_string, fragment = urlparse.urlsplit(url)
        query = urlparse.parse_qsl(query_string)

        optional_parameters = optional_parameters or []
        c = context or {}
        for param in optional_parameters:
            if param in c and c[param]:
                query.append((param, '%(' + param + ')s'))

        updated_query_string = '&'.join(['%s=%s' % (k, v) for k, v in query])
        url = urlparse.urlunsplit((scheme, netloc, path, updated_query_string, fragment))

        return self.build_url(url, context=c, base_url=base_url)

    def serialize(self, obj, *args, **kwargs):
        link = ('GET', obj['url'], False,)
        return super(XivaSubscriptionUrlBaseHandler, self).serialize(link, *args, **kwargs)


class GetWebSubscriptionUrlHandler(XivaSubscriptionUrlBaseHandler):
    """
    Получить ссылку для подписки на уведомления об изменениях в базах данных

    Полученная ссылка подходит для осуществления подписки только при помощи WebSocket- или HTTP- соединения.

    Параметр ```session``` является необязательным и используется для того, чтобы различать подписки одного типа от
    конкретного пользователя. По умолчанию генерируется случайный.

    Формат параметра ```databases_ids```: список идентификаторов баз данных через запятую без пробелов.
    Формат идентификатора базы данных для ```databases_ids```: ```[{context}:]{database-id}```
    Если ```{context}:``` отсутствует, то считается, что он равен ```app```.
    """

    query = fields.QueryDict({
        'session': fields.StringField(
            required=False,
            help_text=u'Уникальный идентификатор пользовательской сессии или экземпляра приложения.'
        ),
        'databases_ids': fields.ListField(
            required=True,
            help_text=(u'Список идентификаторов баз данных, '
                       u'на уведомления об изменениях которых клиент хочет подписаться.')
        ),
    })

    service_url = '/v1/subscribe?uid=%(uid)s&client=%(client)s&service=%(service)s'
    resp_status_code = 200
    subscribe_to_service = 'datasync'
    forbidden_symbols_in_xiva_tags = re.compile('[^a-zA-Z0-9_.]')

    def get_service_with_tags(self):
        context = self.get_context()
        databases_ids = [re.sub(self.forbidden_symbols_in_xiva_tags, '_', d) for d in context['databases_ids']]
        raw_tags = '+'.join(databases_ids)
        return '%s:%s' % (self.subscribe_to_service, raw_tags)

    def handle(self, request, *args, **kwargs):
        context = self.get_context()
        url = self.service.base_url
        params = {
            'uid': context['uid'],
            'client': context['client'],
            'session': context.get('session', None) or uuid.uuid4().hex,
            'service': self.get_service_with_tags()
        }
        if 'oauth_token' in context:
            params['oauth_token'] = context['oauth_token']

        # Выполняется замена tld в url, если tld был передан в параметрах
        if 'tld' in context and context['tld'] != 'ru' and url.endswith('.ru'):
            url = url[:len(url)-3] + '.' + context['tld']

        return self.resp_status_code, self.serialize(
            {'url': self.get_url_with_optional_params(params, ['session', 'oauth_token'], url)}
        )


class GetXivaSubscriptionUrlBaseHandler(ServiceProxyHandler):
    auth_methods = [PassportCookieAuth(), YaTeamCookieAuth(), ]
    permissions = DataApiSubscriptionsAccessPermission()
    service = data_api
    service_base_exception = DataApiBadResponse
    serializer_cls = LinkSerializer

    tld_regex = re.compile(r'(?<=yandex\.).*$')
    valid_tlds = set(PLATFORM_TLDS)

    def get_context(self, context=None):
        c = super(GetXivaSubscriptionUrlBaseHandler, self).get_context(context)
        c['client'] = self.request.client.id
        c['oauth_token'] = self.request.client.token
        c['uid'] = self.request.user.uid
        c['databases_ids'] = ','.join(c['databases_ids'])

        # добавление TLD в запрос, https://st.yandex-team.ru/CHEMODAN-34942
        if 'host' in self.request.raw_headers:
            host = self.request.raw_headers['host']
            matches = self.tld_regex.findall(host)
            if len(matches) > 0 and matches[0] in self.valid_tlds:
                c['tld'] = matches[0]

        return c

    def get_url_with_optional_params(self, context=None, optional_parameters=None):
        url = self.service_url
        scheme, netloc, path, query_string, fragment = urlparse.urlsplit(url)
        query = urlparse.parse_qsl(query_string)

        optional_parameters = optional_parameters or []
        c = context or {}
        for param in optional_parameters:
            if param in c and c[param]:
                query.append((param, '%(' + param + ')s'))

        updated_query_string = '&'.join(['%s=%s' % (k, v) for k, v in query])
        url = urlparse.urlunsplit((scheme, netloc, path, updated_query_string, fragment))

        return self.build_url(url, context=c)

    def serialize(self, obj, *args, **kwargs):
        link = ('GET', obj['result']['url'], False,)
        return super(GetXivaSubscriptionUrlBaseHandler, self).serialize(link, *args, **kwargs)


class GetCallbackSubscriptionUrlHandler(GetXivaSubscriptionUrlBaseHandler):
    """
    Получить ссылку для подписки на уведомления об изменениях в базах данных

    Полученная ссылка подходит для осуществления подписки с посылкой уведомлений на callback-обработчик нотификаций
    на клиенте.

    Параметр ```session``` является необязательным и используется для того, чтобы различать подписки одного типа от
    конкретного пользователя. По умолчанию генерируется случайный.

    Формат параметра ```databases_ids```: список идентификаторов баз данных через запятую без пробелов.
    Формат идентификатора базы данных для ```databases_ids```: [{context}:]{database-id}
    Если {context}: отсутствует, то считается, что он равен ```app```.
    """

    service_url = '/api/databases/subscribe/url?__uid=%(uid)s&ctoken=%(client_token)s&callback=%(callback)s&database_ids=%(databases_ids)s'

    query = fields.QueryDict({
        'client_token': fields.StringField(required=True, help_text=u'Токен бекенда обработчика нотификаций на клиенте'),
        'callback': fields.StringField(required=True, help_text=u'URL обработчика нотификаций на клиенте.'),
        'session': fields.StringField(help_text=u'Уникальный идентификатор пользовательской сессии или экземпляра приложения.'),
        'databases_ids': fields.ListField(required=True, help_text=u'Список идентификаторов баз данных, на уведомления об изменениях которых клиент хочет подписаться.'),
    })

    def get_url(self, context=None):
        return self.get_url_with_optional_params(context, ['session', 'tld'])


class SetRevisionHandler(DatabaseHandler):
    """
    Уведомить об изменении БД
    """
    service_method = 'PUT'
    service_url = '/api/databases/%(database_id)s/on-change?rev=%(new_revision)s&__uid=%(uid)s'
    serializer_cls = None
    permissions = DataApiSetRevisionPermissions()
    service_pb2py_cls = None
    response_objects = [
        ResponseObject(None, u'Уведомление об изменении БД отправлено', 200)
    ]
    content_types = OrderedDict([
        ('application/protobuf', ProtobufFormatter(data_pb2.Database)),
    ])
    body_serializer_cls = SetRevisionSerializer

    def get_context(self, context=None):
        c = super(SetRevisionHandler, self).get_context(context)
        request_body = self.request.body
        assert isinstance(request_body, dict)
        c['new_revision'] = request_body['revision']
        return c

    def handle(self, request, *args, **kwargs):
        super(SetRevisionHandler, self).handle(request, *args, **kwargs)
        return 200, None


class XivaSubscribeBaseHandler(ServiceProxyHandler):
    auth_methods = [YaTeamCookieAuth(), ]
    service = XivaSubscribeDataSyncService()
    permissions = WebDavPermission() | AllowByClientIdPermission([PLATFORM_NOTES_APP_ID]) | DataApiXivaSubscribePermission(PLATFORM_DATA_SYNC_XIVA_PERMISSIONS)
    service_base_exception = XivaError
    error_map = {
        503: ServiceUnavailableError
    }

    def get_service_error_code(self, exception):
        return getattr(exception, 'status_code', None)


class CreateAppSubscriptionHandler(XivaSubscribeBaseHandler):
    """Подписать приложение на уведомления об изменении баз DataSync."""
    query = fields.QueryDict({
        'platform': fields.ChoiceField(
            choices=('apns', 'gcm', 'hms'),
            help_text=u'Платформа, в которой необходимо подписаться на уведомления.'
        ),
        'app_name': fields.StringField(
            required=True,
            help_text=u'Название приложения.'
        ),
        'registration_token': fields.StringField(
            required=True,
            help_text=u'Токен приложения в соответствующем сервисе.'
        ),
        'app_instance_id': fields.StringField(
            required=True,
            help_text=u'Уникальный идентификатор приложения в формате UUID.'
        ),
        'databases_ids': fields.ListField(
            required=True,
            help_text=(u'Список идентификаторов баз данных, '
                       u'на уведомления об изменениях которых клиент хочет подписаться.')
        ),
    })
    serializer_cls = AppSubscriptionSerializer
    resp_status_code = 201
    subscribe_to_service = 'datasync'
    forbidden_symbols_in_xiva_tags = re.compile('[^a-zA-Z0-9_.]')

    def get_xiva_filter(self):
        return None

    def get_device_id(self):
        return None

    def get_service_with_tags(self):
        context = self.get_context()
        databases_ids = [re.sub(self.forbidden_symbols_in_xiva_tags, '_', d) for d in context['databases_ids']]
        raw_tags = '+'.join(databases_ids)
        return '%s:%s' % (self.subscribe_to_service, raw_tags)

    def handle(self, request, *args, **kwargs):
        context = self.get_context()
        self.service.subscribe_app(
            self.get_service_with_tags(),
            context['app_instance_id'],
            context['registration_token'],
            context['app_name'],
            context['platform'],
            uid=context['uid'],
            filter_=self.get_xiva_filter(),
            device_id=self.get_device_id(),
        )
        subscription_token = SubscriptionToken(
            context['uid'],
            self.subscribe_to_service,
            context['app_instance_id'],
            context['registration_token'],
        )
        return 201, self.serialize({'subscription_token': subscription_token})


class DeleteAppSubscriptionHandler(XivaSubscribeBaseHandler):
    """Отписать приложение от уведомлений об изменениях баз DataSync."""
    auth_required = False
    auth_methods = tuple()
    permissions = AllowAllPermission()

    kwargs = fields.QueryDict({
        'subscription_id': SubscriptionIdField(required=True, help_text=u'Идентификатор подписки.')
    })

    response_objects = [
        ResponseObject(None, u'Подписка удалена.', 204)
    ]

    def handle(self, request, *args, **kwargs):
        subs_token = self.get_context()['subscription_id']
        self.service.unsubscribe_app(
            subs_token.service,
            subs_token.uuid,
            push_token=subs_token.push_token,
            uid=subs_token.uid,
        )
        return 204, None
