# -*- coding: utf-8 -*-
from collections import defaultdict

from mpfs.platform.permissions import ClientHasScopesPermission, ConditionalPermission, AllowByClientIdPermission


class DeprecatedDataAppDataPermission(ClientHasScopesPermission):
    """Доступ к данным приложения в Data API"""
    scopes = ('cloud_api:data.app_data',)


class DeprecatedDataUserDataPermission(ClientHasScopesPermission):
    """Доступ к данным пользователя в Data API"""
    scopes = ('cloud_api:data.user_data',)


class DataAppDataPermission(ClientHasScopesPermission):
    """Доступ к данным приложения в Data API"""
    scopes = ('cloud_api.data:app_data',)


class DataUserDataPermission(ClientHasScopesPermission):
    """Доступ к данным пользователя в Data API"""
    scopes = ('cloud_api.data:user_data',)


class DataApiAccessPermission(ConditionalPermission):
    conditions_permissions = {
        'app': (DataAppDataPermission() | DeprecatedDataAppDataPermission() |
                # Временное решение (смотри https://st.yandex-team.ru/CHEMODAN-72254)
                AllowByClientIdPermission(['7a54f58d4ebe431caaaa53895522bf2d'])),
        'user': (DataUserDataPermission() | DeprecatedDataUserDataPermission() |
                 # Временное решение (смотри https://st.yandex-team.ru/CHEMODAN-72254)
                 AllowByClientIdPermission(['7a54f58d4ebe431caaaa53895522bf2d'])),
    }

    def get_condition(self, request):
        return request.kwargs.get('context', None)


class DataApiSubscriptionsAccessPermission(ConditionalPermission):
    """Доступ к подпискам на данные пользователя или приложения в Data API в зависимости от используемых баз"""
    separator = ','

    conditions_permissions = {
        'app': DataAppDataPermission() | DeprecatedDataAppDataPermission(),
        'user': DataUserDataPermission() | DeprecatedDataUserDataPermission(),
        'all': (DataAppDataPermission() | DeprecatedDataAppDataPermission()) &
               (DataUserDataPermission() | DeprecatedDataUserDataPermission())
    }

    def get_condition(self, request):
        return self.extract_context(request.args.get('databases_ids', ''))

    def extract_context(self, databases_ids):
        has_app_databases = False
        has_user_databases = False
        for database_id in databases_ids.split(self.separator):
            if database_id.startswith('user:'):
                has_user_databases = True
            else:
                has_app_databases = True
        return 'all' if has_app_databases and has_user_databases else 'user' if has_user_databases else 'app'


class DataApiSetRevisionPermissions(ClientHasScopesPermission):
    """Доступ к ручке уведомления об изменении ревизии (БД) в Data API"""
    scopes = ('cloud_api.data:set_revision',)


class DataApiXivaSubscribePermission(AllowByClientIdPermission):
    """
    В конфиге прописываются какие client_id могут подписываться на какие базы
    """
    def __init__(self, client_ids_and_dbs):
        allowed_client_ids_dbs = defaultdict(set)
        for config in client_ids_and_dbs:
            client_id = config['client_id']
            db_ids = config['dbs']
            allowed_client_ids_dbs[client_id] |= set(db_ids)

        super(DataApiXivaSubscribePermission, self).__init__(allowed_client_ids_dbs.keys())
        self.allowed_client_ids_dbs = allowed_client_ids_dbs

    def has_permission(self, request):
        if not super(DataApiXivaSubscribePermission, self).has_permission(request):
            return False
        try:
            req_database_ids = set(request.query['databases_ids'])
        except:
            return False
        if not req_database_ids:
            return False
        client_id = request.client.id
        config_database_ids = self.allowed_client_ids_dbs.get(client_id)
        if not config_database_ids:
            return False
        # все запрошенные базы должны быть разрешены через конфиг
        return req_database_ids.issubset(config_database_ids)
