# -*- coding: utf-8 -*-
from collections import OrderedDict
import re
from google.protobuf.internal.containers import RepeatedCompositeFieldContainer
from google.protobuf.message import Message
from mpfs.platform import fields
from mpfs.platform.fields import ParentAttrField
from mpfs.platform.fields import ParentMethodField
from mpfs.platform.serializers import BaseSerializer, ListSerializer
from mpfs.platform.v1.data import exceptions
from mpfs.platform.v1.data.fields import SubscriptionIdField
from mpfs.platform.v1.data.backend_pb2 import RecordChange, FieldChange, DataFieldValue, Delta


class ProtobufSerializer(BaseSerializer):
    def restore_object(self, attrs, instance=None):
        self.validate(attrs)
        if instance is not None:
            if isinstance(instance, dict):
                for name, value in attrs.iteritems():
                    instance[name] = value
            else:
                for name, value in attrs.iteritems():
                    if isinstance(value, (list, tuple)):
                        getattr(instance, name).extend(value)
                    elif isinstance(getattr(instance, name, None), Message):
                        getattr(instance, name).CopyFrom(value)
                    else:
                        setattr(instance, name, value)
            return instance
        return attrs

    @staticmethod
    def get_dict(obj):
        if isinstance(obj, Message):
            obj_dict = {}
            for descriptor, v in obj.ListFields():
                obj_dict[descriptor.name] = v
            return obj_dict
        return BaseSerializer.get_dict(obj)


class DatabaseSerializer(ProtobufSerializer):
    visible_fields = ['revision', 'records_count', 'created', 'modified', 'size', 'database_id', 'handle', 'title', '_links']
    fields = {
        'revision': fields.IntegerField(required=True, source='rev', pbid=1, help_text=u'Ревизия БД'),
        'records_count': fields.IntegerField(required=True, source='count', pbid=2,
                                             help_text=u'Количество записей в БД'),
        'created': fields.DateTimeToTSField(required=True, milliseconds=True, pbid=3, help_text=u'Дата создания БД'),
        'modified': fields.DateTimeToTSField(required=True, milliseconds=True, pbid=4,
                                             help_text=u'Дата последнего изменения БД'),
        'size': fields.IntegerField(required=True, pbid=5, help_text=u'Размер БД в байтах'),
        'database_id': fields.StringField(required=True, source='databaseId', pbid=6, help_text=u'Идентификатор БД'),
        'title': fields.StringField(help_text=u'Название БД', pbid=7, source='description', default=None),
        'handle': fields.StringField(help_text=u'Уникальный идентификатор БД, меняется после удаление и создания заново', pbid=9, source='handle', default=None)
    }

    @property
    def handler_cls(self):
        from mpfs.platform.v1.data.handlers import DatabaseHandler
        return DatabaseHandler


class DatabasePatchSerializer(DatabaseSerializer):
    is_patch = True
    visible_fields = ['title']


class DatabaseListSerializer(ListSerializer):
    DEFAULT_LIMIT = 100
    visible_fields = ['limit', 'offset', 'total', 'items', '_links']
    fields = {
        'limit': fields.IntegerField(default=DEFAULT_LIMIT, pbid=3,
                                     help_text=u'Количество элементов на странице'),
        'items': fields.SerializerField(DatabaseSerializer, source='databases', many=True, pbid=1,
                                        help_text=u'Список БД'),
    }

    @property
    def handler_cls(self):
        from mpfs.platform.v1.data.handlers import ListDatabasesHandler
        return ListDatabasesHandler


class ValueSerializer(ProtobufSerializer):
    TYPE_CHOICES = ['double', 'integer', 'boolean', 'string',  'binary', 'null', 'inf', 'ninf', 'nan', 'datetime', 'list']
    TYPE_FLAGS_MAP = {
        'list': 'isList',
        'nan': 'isNan',
        'inf': 'isInfinity',
        'ninf': 'isNegativeInfinity',
        'null': 'isNull',
    }
    TYPE_FLAGS_REVERSE_LOOKUP = dict([(v, k) for k, v in TYPE_FLAGS_MAP.iteritems()])
    TYPE_RX = re.compile(r'^(?P<type>\w+)Values?|(?P<timestamp>timestamp)$')
    visible_fields = ['type', 'double', 'integer', 'boolean', 'string', 'binary', 'datetime', 'list', 'nan', 'inf',
                      'ninf', 'null']
    fields = {
        'type': fields.ChoiceField(choices=TYPE_CHOICES, pbid=13, num_choices_start=1, help_text=u'Тип значения'),
        'double': fields.FloatField(source='decimalValue', pbid=1, help_text=u'Значение типа double'),
        'integer': fields.IntegerField(source='integerValue', pbid=2, help_text=u'Значение типа integer'),
        'boolean': fields.BooleanField(source='boolValue', pbid=3, help_text=u'Значение типа boolean'),
        'string': fields.StringField(source='stringValue', pbid=4, help_text=u'Значение типа string'),
        'binary': fields.Base64ToBinaryField(source='bytesValue', pbid=5, help_text=u'Base64 encoded массив байт'),
        'datetime': fields.DateTimeToTSField(source='timestamp', pbid=10, milliseconds=True, help_text=u'Дата и время в UTC'),
        'list': fields.SerializerField(lambda: ValueSerializer, pbid=12, many=True, source='listValues',
                                       help_text=u'Список значений'),

        'nan': fields.BooleanField(source='isNan', pbid=9, help_text=u'Является ли значение NaN'),
        'inf': fields.BooleanField(source='isInfinity', pbid=7, help_text=u'Является ли значение неопределённостью'),
        'ninf': fields.BooleanField(source='isNegativeInfinity', pbid=8,
                                    help_text=u'Является ли значение отрицательной неопределённостью'),
        'null': fields.BooleanField(source='isNull', pbid=6, help_text=u'Является ли значение null'),
    }

    def __init__(self, *args, **kwargs):
        super(ValueSerializer, self).__init__(*args, **kwargs)
        self.TYPE_FIELDS = dict([(k, v.source) for k, v in self.get_fields().iteritems() if v.source])
        self.TYPE_FIELDS_REVERSE_LOOKUP = dict([(v, k) for k, v in self.TYPE_FIELDS.iteritems()])

    def validate(self, attrs):
        flag = None
        value_type = None
        for k, v in attrs.iteritems():
            match = self.TYPE_RX.match(k)
            if match:
                if not value_type:
                    # отмечаем, что значение уже установлено
                    gd = match.groupdict()
                    value_type = gd.get('type') or gd.get('timestamp')
                else:
                    # если значение установлено несколько раз, то швыряем исключение
                    raise exceptions.DataValueObjectProvideMultipleValuesError()
            elif k.startswith('is'):
                if not flag:
                    # отмечаем, что установлен флаг
                    flag = k
                else:
                    # если установлено несколько флагов, то швыряем исключение
                    raise exceptions.DataValueObjectProvideMultipleValuesError()
        if value_type and flag:
            if value_type != 'list' or flag != 'isList':
                # если одновременно установлено и значение и флаг, то швыряем исключние, если это не список
                raise exceptions.DataValueObjectProvideMultipleValuesError()

    def restore_object(self, attrs, instance=None):
        type_ = attrs.pop('type', None)
        # если type явно не задан то догадываемся о его значении
        if not type_:
            for k in attrs.keys():
                type_ = self.TYPE_FIELDS_REVERSE_LOOKUP.get(k, type_)
        elif type_ not in ('nan', 'inf', 'ninf', 'null', 'list') and self.fields[type_].source not in attrs:
            # если для указанного типа необходимо предоставить значение, а его нет, то швыряем исключение
            raise exceptions.DataTypeDoesntMatchValueError(type=type_)
        # расставляем флаги в соответствии с типом
        for t, f in self.TYPE_FLAGS_MAP.iteritems():
            if type_ == t:
                attrs[f] = True
        instance = instance or DataFieldValue()
        return super(ValueSerializer, self).restore_object(attrs, instance=instance)

    def prepare_object(self, obj):
        obj_dict = {}
        for k, v in super(ValueSerializer, self).prepare_object(obj).iteritems():
            if isinstance(v, RepeatedCompositeFieldContainer) and obj.isList:
                obj_dict[k] = v
            else:
                try:
                    if obj.HasField(k):
                        obj_dict[k] = v
                except:
                    pass
        # Определяем значение поля type
        for k, v in obj_dict.iteritems():
            if k in self.TYPE_FLAGS_REVERSE_LOOKUP and v:
                obj_dict['type'] = self.TYPE_FLAGS_REVERSE_LOOKUP[k]
                # добавляем аттрибуты nan, inf, ninf, null
                if k != 'isList':
                    obj_dict[k] = v
                break
            else:
                m = self.TYPE_RX.match(k)
                if m and v is not None:
                    obj_dict['type'] = self.TYPE_FIELDS_REVERSE_LOOKUP[k]
                    break
        # Адовая оптимизация, чтоб сериализатор не шерстил все возможные поля со значениями,
        # задаём жёстко, что объект будет содержать только одно поле соответствующее типу значения.
        self.visible_fields = list(set(type(self).visible_fields).intersection({'type', obj_dict['type']}))
        return obj_dict


class FieldSerializer(BaseSerializer):
    visible_fields = ['field_id', 'value']
    fields = {
        'field_id': fields.StringField(required=True, source='fieldId', pbid=1, help_text=u'Идентификатор поля'),
        'value': fields.SerializerField(ValueSerializer, required=True, pbid=2, help_text=u'Значение поля'),
    }


class RecordSerializer(BaseSerializer):
    visible_fields = ['revision', 'collection_id', 'record_id', 'fields']
    fields = {
        'revision': fields.IntegerField(required=True, source='rev', pbid=3, help_text=u'Ревизия БД'),
        'collection_id': fields.StringField(required=True, source='collectionId', pbid=1,
                                            help_text=u'Идентификатор коллекции'),
        'record_id': fields.StringField(required=True, source='recordId', pbid=2, help_text=u'Идентификатор записи'),
        'fields': fields.SerializerField(FieldSerializer, many=True, pbid=4, help_text=u'Список полей записи'),
    }


class RecordsListSerializer(ListSerializer):
    visible_fields = ['items']
    fields = {
        'items': fields.SerializerField(RecordSerializer, many=True, pbid=1, help_text=u'Список записей')
    }


class DatabaseSnapshotSerializer(DatabaseSerializer):
    visible_fields = ['revision', 'records_count', 'created', 'modified', 'size', 'database_id', 'handle', 'title', 'records']
    fields = {
        'title': fields.StringField(help_text=u'Название БД', pbid=6),  # переопределяем, чтоб нумерация совпала с нумерацией бэкэнда
        'database_id': fields.StringField(required=True, source='databaseId', pbid=8, help_text=u'Идентификатор БД'),  # переопределяем, чтоб нумерация совпала с нумерацией бэкэнда
        'records': fields.SerializerField(RecordsListSerializer, required=True, pbid=7, help_text=u'Записи'),
    }


class FieldChangeSerializer(ProtobufSerializer):
    CHANGE_TYPE_CHOICES = OrderedDict([
        (u'delete', 1),
        (u'set', 2),
        (u'list_item_set', 3),
        (u'list_item_delete', 4),
        (u'list_item_insert', 5),
        (u'list_item_move', 6),
    ])
    LIST_ITEM_CHANGE_TYPE_CHOICES = dict([(k, v) for k, v in CHANGE_TYPE_CHOICES.items() if k.startswith('list_item_')])
    visible_fields = ['change_type', 'field_id', 'value', 'list_item', 'list_item_dest']
    fields = {
        'change_type': fields.ChoiceField(required=True, source='type', choices=CHANGE_TYPE_CHOICES, pbid=1,
                                          num_choices_start=1, help_text=u'Тип изменения'),
        'field_id': fields.StringField(required=True, source='fieldId', pbid=2, help_text=u'Идентификатор поля'),
        'value': fields.SerializerField(ValueSerializer, source='putFieldValue', pbid=3,
                                        help_text=u'Значение поля. Необходимо при типах изменений ```set``` '
                                                  u', ```list_item_insert```, или ```list_item_set```.'),
        'list_item': fields.IntegerField(source='listIndex', pbid=4,
                                         help_text=u'Индекс элемента списка, к которому применяется изменение. '
                                                   u'Необходим для любого изменения начинающегося с "list_item".'),
        'list_item_dest': fields.IntegerField(source='listMoveDestIndex', pbid=5,
                                             help_text=u'Новый индекс элемента списка. '
                                                       u'Необходим при типе изменения ```list_item_move```.'),
    }

    def restore_object(self, attrs, instance=None):
        instance = instance or FieldChange()
        return super(FieldChangeSerializer, self).restore_object(attrs, instance=instance)

    def validate(self, attrs):
        ch_type = attrs['type']
        if ch_type == self.CHANGE_TYPE_CHOICES[u'set']:
            if attrs.get('putFieldValue') is None:
                raise exceptions.DataFieldSetWithoutValueError()
        else:
            if ch_type in self.LIST_ITEM_CHANGE_TYPE_CHOICES.values():
                if attrs.get('listIndex') is None:
                    raise exceptions.DataFieldListItemWithoutIndexError()
                elif ch_type == self.LIST_ITEM_CHANGE_TYPE_CHOICES[u'list_item_set'] and attrs.get('putFieldValue') is None:
                    raise exceptions.DataFieldSetWithoutValueError()
                elif ch_type == self.LIST_ITEM_CHANGE_TYPE_CHOICES[u'list_item_move'] and attrs.get('listMoveDestIndex') is None:
                    raise exceptions.DataFieldListItemMoveWithoutDestError()


class RecordChangeSerializer(ProtobufSerializer):
    CHANGE_TYPE_CHOICES = OrderedDict([
        (u'insert', 1),
        (u'delete', 2),
        (u'update', 3),
        (u'set', 4)
    ])
    visible_fields = ['change_type', 'collection_id', 'record_id', 'changes']
    fields = {
        'change_type': fields.ChoiceField(
            required=True, source='type', choices=CHANGE_TYPE_CHOICES, pbid=1, num_choices_start=1,
            help_text=u'Тип изменения. ```delete``` - удалить запись. ```insert``` - создать новую запись. '
                      u'```update``` - обновить существующую запись. ```set``` - сохранить запись. '
                      u'Отличие ```set``` от ```update``` в том, что ```set``` может применяться как к существующим, '
                      u'так и к не существующим записям и принимает не список изменений, а полный список полей записи, '
                      u'которые будут сохранены.'),
        'collection_id': fields.StringField(required=True, source='collectionId', pbid=2,
                                            help_text=u'Идентификатор коллекции'),
        'record_id': fields.StringField(required=True, source='recordId', pbid=3, help_text=u'Идентификатор записи'),
        'changes': fields.SerializerField(FieldChangeSerializer, many=True, source='fieldChanges', pbid=4,
                                          help_text=u'Список изменений полей в записи'),
    }

    def restore_object(self, attrs, instance=None):
        instance = instance or RecordChange()
        return super(RecordChangeSerializer, self).restore_object(attrs, instance=instance)


class DeltaSerializer(ProtobufSerializer):
    visible_fields = ['base_revision', 'revision', 'delta_id', 'changes']
    fields = {
        'revision': fields.IntegerField(source='newRev', pbid=4, help_text=u'Ревизия БД после применения дельты'),
        'base_revision': fields.IntegerField(source='rev', pbid=3, help_text=u'Ревизия БД к которой была применена дельта'),
        'delta_id': fields.StringField(source='id', pbid=2, help_text=u'Уникальный идентифкатор дельты'),
        'changes': fields.SerializerField(RecordChangeSerializer, many=True, required=True, pbid=1,
                                          help_text=u'Список изменений записей'),
    }

    def restore_object(self, attrs, instance=None):
        instance = instance or Delta()
        return super(DeltaSerializer, self).restore_object(attrs, instance=instance)


class DeltaListSerializer(BaseSerializer):
    DEFAULT_LIMIT = 100
    visible_fields = ['base_revision', 'revision', 'total', 'limit', 'items']
    fields = {
        'total': fields.IntegerField(required=True, source='count', pbid=1, help_text=u'Общее количество изменений'),
        'base_revision': fields.IntegerField(pbid=4, help_text=u'Ревизия для которой выбраны изменения'),
        'revision': fields.IntegerField(pbid=5, source='currentDatabaseRev', help_text=u'Текущая актуальная ревизия БД'),
        'limit': fields.IntegerField(default=DEFAULT_LIMIT, pbid=3,
                                     help_text=u'Количество изменений на странице'),
        'items': fields.SerializerField(DeltaSerializer, required=True, source='deltas', many=True, pbid=2,
                                        help_text=u'Список измненией'),
    }

    @property
    def handler_cls(self):
        from mpfs.platform.v1.data.handlers import ListDeltasHandler
        return ListDeltasHandler

    def get_links(self):
        rev = self._object['base_revision']
        limit = self._object.get('limit', self.DEFAULT_LIMIT)
        total = self._object.get('total', -1)
        ret = super(DeltaListSerializer, self).get_links()
        if -1 < rev + total <= rev + limit:
            ret['next'] = fields.HalLinkField(None)
        else:
            ret['next'] = fields.HalLinkField(self.handler_cls, context={'base_revision': rev + limit})
        return ret


class UserSerializer(BaseSerializer):
    visible_fields = ['uid']
    fields = {
        'uid': fields.StringField(help_text=u'Идентификатор пользователя')
    }


class UserListSerializer(BaseSerializer):
    visible_fields = ['limit', 'iteration_key', 'items']
    fields = {
        'limit': fields.IntegerField(help_text=u'Максимальное количество пользователей в ответе'),
        'iteration_key': fields.StringField(help_text=u'Ключ для продолжения итерирования. Отсутствует когда пользователи закончились'),
        'items': fields.SerializerField(UserSerializer, source='items', many=True, help_text=u'Список пользователей'),
    }


class SetRevisionSerializer(BaseSerializer):
    visible_fields = ['revision']
    fields = {
        'revision': fields.StringField(help_text=u'Новая ревизия измененной БД')
    }


class AppSubscriptionSerializer(BaseSerializer):
    visible_fields = ['subscription_id']
    fields = {
        'subscription_id': SubscriptionIdField(required=True, source='subscription_token',
                                               help_text=u'Идентификатор подписки.')
    }
