# -*- coding: utf-8 -*-
from mpfs.common.static import codes
from mpfs.common.util import to_json
from mpfs.platform.v1.batch.batch_processors import BaseBatchRequestProcessor
from mpfs.platform.v1.disk.fields import MpfsPathField
from mpfs.platform.v1.disk.handlers import GetResourceHandler
from mpfs.core.services.mpfsproxy_service import mpfsproxy


class GetResourcesBatchRequestProcessor(BaseBatchRequestProcessor):
    GET_RESOURCES_MPFS_URL_PATTERN = r'.*/json/list\?(?=.*&?uid=%s)(?=.*&?path=%s)'
    support_handlers = {GetResourceHandler, }

    def process(self, batch_requests):
        paths = [self._platform_path_to_mpfs_path(self.native_requests.get(x).args['path']) for x in batch_requests]
        uids = {y for y in [self.native_requests.get(x).user.uid for x in batch_requests]
                + [self.main_request.user.uid]}

        if len(uids) > 1:
            raise NotImplementedError('tried to make batch get resources request for several users at the same time')
        uid = self.main_request.user.uid

        res = mpfsproxy.bulk_info(uid, paths)

        requests_with_mpfs_responses = self._zip_requests_with_mpfs_responses(batch_requests, res)
        request_to_dispatched_response_map = self._dispatch_responses(requests_with_mpfs_responses, uid)

        return request_to_dispatched_response_map

    def _zip_requests_with_mpfs_responses(self, ordered_requests, ordered_responses):
        result = []

        requests_i = 0
        responses_i = 0

        while requests_i < len(ordered_requests) and responses_i < len(ordered_responses):
            native_request = self.native_requests.get(ordered_requests[requests_i])
            if self._platform_path_to_mpfs_path(native_request.args['path']) != ordered_responses[responses_i]['path']:
                result.append((ordered_requests[requests_i], {}))
                requests_i += 1
                continue
            result.append((ordered_requests[requests_i], ordered_responses[responses_i]))
            requests_i += 1
            responses_i += 1
        for requests_i in xrange(requests_i, len(ordered_requests)):
            result.append((ordered_requests[requests_i], {}))

        return result

    def _dispatch_responses(self, requests_with_mpfs_responses, uid):
        result_map = {}
        for request, mpfs_response in requests_with_mpfs_responses:
            data = mpfs_response
            status_code = 200
            if not mpfs_response:
                status_code = 404
                data = {'code': codes.LIST_NOT_FOUND}
            elif mpfs_response['type'] == 'dir':
                # В настоящий момент батч ручку не используют для листинга папок (только файлов). Батчевый листинг папок
                # делать немого сложнее, поэтому пока что не делали по ненадобности и будем просто возвращать 500.
                status_code = 500

            mpfsproxy.clean_response_patches()
            mpfsproxy.set_response_patch(
                self.native_requests.get(request).method,
                self.GET_RESOURCES_MPFS_URL_PATTERN % (
                    uid, self._platform_path_to_mpfs_path(self.native_requests.get(request).args['path'])),
                status_code,
                data=to_json(data)
            )
            result_map[request] = self.dispatcher.dispatch(self.raw_requests.get(request))
        return result_map

    @staticmethod
    def _platform_path_to_mpfs_path(platform_path):
        return MpfsPathField().to_native(platform_path)
