# -*- coding: utf-8 -*-
"""API работы с Дисковым биллингом"""
import re
import time

from datetime import datetime
from dateutil.relativedelta import relativedelta
from functools import wraps
from httplib import CREATED

from mpfs.common.static import tags
from mpfs.common.util.iptools import IPRangeList
from mpfs.config import settings
from mpfs.core.services.passport_service import passport
from mpfs.platform.common import PlatformUser
from mpfs.platform.exceptions import (
    ForbiddenError,
    ServiceUnavailableError,
    UnauthorizedError,
)
from mpfs.common.util import from_json
from mpfs.common.errors.billing import BillingClientNotBindedToMarket
from mpfs.platform import fields
from mpfs.platform.permissions import AllowByClientIdPermission
from mpfs.platform.v1.disk import exceptions
from mpfs.platform.v1.disk.handlers import MpfsProxyHandler, _auto_initialize_user
from mpfs.platform.v1.disk.permissions import DiskPartnerPermission, DiskBillingReadPermission, WebDavPermission
from mpfs.platform.v1.disk.billing import exceptions as billing_exceptions
from mpfs.platform.v1.disk.billing.serializers import (
    BillingPartnerServiceSerializer,
    BillingPartnerServicesSerializer,
    BillingSubscriptionsListSerializer,
    BillingProductsListSerializer,
)

PLATFORM_MOBILE_APPS_IDS = settings.platform['mobile_apps_ids']
PLATFORM_PS_APPS_IDS = settings.platform['ps_apps_ids']
BILLING_PARTNERS_STUB_SERVICE_CREATION = settings.billing['partners']['stub_service_creation']


def auto_bind_market(func):
    @wraps(func)
    def wrapper(handler, *args, **kwargs):
        try:
            return func(handler, *args, **kwargs)
        except handler.service_base_exception, e:
            if handler.get_service_error_code(e) == BillingClientNotBindedToMarket.code:
                handler.bind_market()
                return func(handler, *args, **kwargs)
            raise

    return wrapper


class MpfsProxyBillingHandler(MpfsProxyHandler):
    """Настройки для работы с биллингом"""

    def get_context(self, context=None):
        c = super(MpfsProxyBillingHandler, self).get_context(context)
        c['ip'] = self.request.get_real_remote_addr()
        return c

    def get_service_error_code(self, exception):
        response_text = exception.data['text']
        parsed_response_text = from_json(response_text)
        code = int(parsed_response_text['code'])
        if code in self.error_map:
            return code
        else:
            return super(MpfsProxyBillingHandler, self).get_service_error_code(exception)


class MpfsBillingUtilsHandlerMixin(MpfsProxyHandler):
    service_list_url = '/billing/service_list?uid=%(uid)s&ip=%(ip)s'

    def get_service_info(self, product_id):
        uid = self.get_context()['uid']
        if not hasattr(self.request, 'users_services'):
            self.request.users_services = dict()
        if uid not in self.request.users_services:
            url = self.build_url(self.service_list_url, self.get_context())
            self.request.users_services[uid] = self.request_service(url)

        for service in self.request.users_services[uid]:
            if service['name'] == product_id:
                return service['sid'], service['expires']
        raise billing_exceptions.DiskServiceNotFoundError()

    def get_services_info(self, product_id):
        uid = self.get_context()['uid']
        if not hasattr(self.request, 'users_services'):
            self.request.users_services = dict()
        if uid not in self.request.users_services:
            url = self.build_url(self.service_list_url, self.get_context())
            self.request.users_services[uid] = self.request_service(url)

        services = []
        for service in self.request.users_services[uid]:
            if service['name'] == product_id:
                services.append((service['sid'], service['expires']))

        return services

    def check_if_service_exists(self, product_id):
        try:
            self.get_service_info(product_id)
        except billing_exceptions.DiskServiceNotFoundError:
            return False
        return True

    def get_service_by_service_id(self, service_id):
        url = self.build_url(self.service_list_url, self.get_context())
        services = self.request_service(url)
        for service in services:
            if service['sid'] == service_id:
                return service
        raise billing_exceptions.DiskServiceNotFoundError()


class ListProductsHandler(MpfsProxyBillingHandler):
    """Получить список актуальных продуктов (включая скидки)"""
    need_auto_initialize_user = False

    permissions = DiskBillingReadPermission() | WebDavPermission()
    serializer_cls = BillingProductsListSerializer
    hidden = True
    service_url = '/billing/verstka_products?uid=%(uid)s&locale=%(locale)s&ip=%(ip)s'
    client_bind_to_market = '/billing/client_bind_market?ip=%(ip)s&uid=%(uid)s'
    query = fields.QueryDict({
        'locale': fields.ChoiceField(choices=['ru', 'en', 'tr', 'uk', 'ua'],
                                     help_text=u'Локаль пользователя.', default='ru'),
    })

    error_map = {
        131: billing_exceptions.DiskClientNotBoundToMarketError,
    }

    @auto_bind_market
    def handle(self, request, *args, **kwargs):
        return super(ListProductsHandler, self).handle(request, *args, **kwargs)

    def bind_market(self):
        """Делает запрос на привязывание к market'у"""
        user = self.request.user
        context = self.get_context()

        url = self.build_url(self.client_bind_to_market, {'uid': user.uid, 'ip': context['ip']})
        self.request_service(url, method='GET')


class ListSubscriptionsHandler(MpfsProxyBillingHandler):
    """Получить список активных подписок пользователя"""
    need_auto_initialize_user = False

    permissions = DiskBillingReadPermission() | AllowByClientIdPermission(
        PLATFORM_MOBILE_APPS_IDS) | AllowByClientIdPermission(PLATFORM_PS_APPS_IDS)
    serializer_cls = BillingSubscriptionsListSerializer
    hidden = True
    service_url = '/billing/service_list?uid=%(uid)s&ip=%(ip)s'
    query = fields.QueryDict({
        'limit': fields.IntegerField(default=100, help_text=u'Количество выводимых подписок.'),
        'offset': fields.IntegerField(default=0, help_text=u'Смещение от начала списка подписок.'),
        'payment_methods': fields.ListField(help_text=u'Список методов оплаты.'),
    })

    def serialize(self, obj, *args, **kwargs):
        filtering_payment_methods = set(self.request.query['payment_methods'])
        services = [x for x in obj if (not filtering_payment_methods
                                       or ('payment_method' in x and x['payment_method'] in filtering_payment_methods))]
        limit = self.request.query['limit']
        offset = self.request.query['offset']
        data = {
            'total': len(services),
            'items': services[offset:offset + limit],
            'limit': limit,
            'offset': offset,
        }
        return super(ListSubscriptionsHandler, self).serialize(data, *args, **kwargs)


class BasePartnerServiceHandler(MpfsProxyBillingHandler):
    """Настройки для работы с партнерскими продуктами"""
    permissions = DiskPartnerPermission()
    partner_re = re.compile(r'^cloud_api:disk\.partners\.(\w+)\.manage_services$')
    product_list_url = '/billing/product_list?line=%(line)s&market=%(market)s'
    scope_template = 'cloud_api:disk.partners.%s.manage_services'
    default_product_line = 'partner'
    unavailable_product_exception = billing_exceptions.DiskProductNotFoundError

    query = fields.QueryDict({
        'product_id': fields.StringField(required=True, help_text=u'Имя продукта.'),
        'email': fields.StringField(required=False, help_text=u'Email пользователя.'),
    })
    kwargs = fields.QueryDict({
        'partner': fields.StringField(required=True, help_text=u'Имя партнера.'),
    })
    auth_user_required = False

    @classmethod
    def email2user(cls, email):
        user = cls._build_user_by_bb_user_info(passport.userinfo(login=email))
        if user is None:
            raise exceptions.EmailNotFoundError()
        return user

    @classmethod
    def uid2user(cls, uid):
        user = cls._build_user_by_bb_user_info(passport.userinfo(uid=uid))
        if user is None:
            raise exceptions.UidNotFoundError()
        return user

    def get_product_id(self, **kwargs):
        context = self.get_context()
        if 'product_id' in context:
            product_id = context['product_id']
        else:
            product_id = kwargs['product_id']

        return product_id

    def load_products(self):
        """
        Загружает все партнерские продукты и кладет в request.products
        """
        params = {'line': self.default_product_line, 'market': 'RU'}
        url = self.build_url(self.product_list_url, params)
        response = self.request_service(url)
        if isinstance(response, list):
            self.request.products = response
        else:
            raise exceptions.DiskServiceUnavailableError()

    def get_partner_products(self, partner):
        """
        Получить список всех продуктов партнера
        """
        if not hasattr(self.request, 'products'):
            self.load_products()

        return [p for p in self.request.products if p.get('partner') == partner]

    def get_partner_product(self, partner, product_id):
        """
        Получить конкретный партнерский продукт
        """
        if not hasattr(self.request, 'products'):
            self.load_products()

        for product in self.request.products:
            if ('partner' in product and
                    product['partner'] == partner and
                    product['id'] == product_id):
                return product
        else:
            raise self.unavailable_product_exception()

    def check_permissions_for_product(self, partner, product):
        """
        Проверяет есть ли доступ партнера к продукту
        """
        if self.request.mode == tags.platform.EXTERNAL:
            # проверка ip
            context = self.get_context()
            request_ip = context['ip']
            if ('partner_ips' in product and
                    type(product['partner_ips']) is list and
                    request_ip not in IPRangeList(*product['partner_ips'])):
                raise ForbiddenError()
            # проверка скоупа
            required_scope = self.scope_template % partner
            if required_scope not in self.request.client.scopes:
                raise ForbiddenError()

    def load_product_check_permissions(self, *args, **kwargs):
        partner = kwargs['partner']
        product_id = self.get_product_id(**kwargs)

        self.request.partner_product = self.get_partner_product(partner, product_id)
        self.check_permissions_for_product(partner, self.request.partner_product)

    @_auto_initialize_user
    def handle(self, request, *args, **kwargs):
        self.load_product_check_permissions(*args, **kwargs)
        return super(BasePartnerServiceHandler, self).handle(request, *args, **kwargs)

    def check_permissions(self, request):
        super(BasePartnerServiceHandler, self).check_permissions(request)
        email = self.request.query.get('email')
        uid = self.request.query.get('uid')
        if request.user:
            return
        elif email:
            request.user = self.email2user(email)
        elif uid:
            request.user = self.uid2user(uid)
        else:
            raise UnauthorizedError()

    @staticmethod
    def _build_user_by_bb_user_info(user_info):
        uid = user_info.get('uid')
        login = user_info.get('login')
        if not (uid and login):
            return
        user = PlatformUser()
        user.uid = uid
        user.login = login
        return user


class GetServiceInfoByProductHandler(MpfsBillingUtilsHandlerMixin, BasePartnerServiceHandler):
    """Получить информацию об услуге по указанному продукту"""
    query = fields.QueryDict({
        'product_id': fields.StringField(required=False, help_text=u'Имя продукта.'),
    })
    resp_status_code = 200
    serializer_cls = BillingPartnerServicesSerializer

    def handle(self, request, *args, **kwargs):
        partner = kwargs['partner']
        partner_products = self.get_partner_products(partner)
        user_services = []
        for product in partner_products:
            try:
                self.check_permissions_for_product(partner, product)
                try:
                    services = self.get_services_info(product['id'])
                    for sid, expires in services:
                        service = {'product_id': product['id'], 'sid': sid}
                        if expires:
                            service['btime'] = float(expires)
                        user_services.append(service)
                except self.service_base_exception, e:
                    # если пользователь не инициализирован, то возвращаем пустой список
                    # https://st.yandex-team.ru/CHEMODAN-25467
                    if self.get_service_error_code(e) == 44:
                        pass
                except billing_exceptions.DiskServiceNotFoundError:
                    pass
            except ForbiddenError:
                pass
        obj = {'partner': partner, 'items': user_services}
        return super(GetServiceInfoByProductHandler, self).serialize(obj)


class CreateServiceByProductHandler(BasePartnerServiceHandler):
    """Создать услугу по указанному продукту"""
    service_url = '/billing/service_create?uid=%(uid)s&line=partner&pid=%(product_id)s&ip=%(ip)s'
    resp_status_code = 201

    error_map = {
        135: billing_exceptions.DiskServiceMultiProvidingError,
        686: billing_exceptions.DiskProductNotFoundError,
    }
    serializer_cls = BillingPartnerServiceSerializer

    def serialize(self, obj, *args, **kwargs):
        obj = obj or dict()
        obj['product_id'] = self.request.query['product_id']
        if 'btime' in obj and obj['btime'] is None:
            del obj['btime']
        return super(CreateServiceByProductHandler, self).serialize(obj, *args, **kwargs)

    def handle(self, request, *args, **kwargs):
        if BILLING_PARTNERS_STUB_SERVICE_CREATION['enabled']:
            partner = kwargs['partner']
            product_id = self.get_product_id(**kwargs)

            if (partner in BILLING_PARTNERS_STUB_SERVICE_CREATION['enable_for'] and
                    product_id in BILLING_PARTNERS_STUB_SERVICE_CREATION['enable_for'][partner]):
                return CREATED, self.serialize({'product_id': product_id})

        return super(CreateServiceByProductHandler, self).handle(request, *args, **kwargs)


class ProlongateServiceByProductHandler(MpfsBillingUtilsHandlerMixin, BasePartnerServiceHandler):
    """Продлить услугу"""
    service_set_attribute_url = ('/billing/service_set_attribute?uid=%(uid)s&'
                                 'ip=%(ip)s&'
                                 'sid=%(sid)s&'
                                 'key=service.btime&'
                                 'value=%(btime)s')
    serializer_cls = BillingPartnerServiceSerializer

    @staticmethod
    def get_new_billing_ts(service_expires_ts, product_prolongation_delta, product_expiration_delta=None):
        service_expires_dt = datetime.fromtimestamp(int(service_expires_ts))
        current_dt = datetime.now()

        if service_expires_dt > current_dt + (product_expiration_delta or product_prolongation_delta):
            raise billing_exceptions.DiskServiceAlreadyProlongatedError()

        new_billing_dt = service_expires_dt + product_prolongation_delta
        return int(time.mktime(new_billing_dt.timetuple()))

    @staticmethod
    def get_product_prolongation_period(product):
        try:
            [(duration, amount)] = product['period'].items()
            return relativedelta(**{duration: +amount})
        except Exception:
            raise billing_exceptions.DiskServiceNotProlongableError()

    def set_new_billing_time(self, sid, new_billing_ts):
        context = self.get_context({'sid': sid, 'btime': new_billing_ts})
        url = self.build_url(self.service_set_attribute_url, context)
        response = self.request_service(url)
        return response

    @_auto_initialize_user
    def handle(self, request, *args, **kwargs):
        self.load_product_check_permissions(*args, **kwargs)
        product = request.partner_product

        if product['id'] == 'rostelecom_2014_100gb':  # костыль https://st.yandex-team.ru/CHEMODAN-22375
            prolongation_delta = relativedelta(months=1)
        else:
            prolongation_delta = self.get_product_prolongation_period(product)

        sid, service_expires_ts = self.get_service_info(product['id'])
        if product['id'] == 'rostelecom_2014_100gb':  # костыль https://st.yandex-team.ru/CHEMODAN-22375
            try:
                new_billing_ts = self.get_new_billing_ts(service_expires_ts, prolongation_delta, relativedelta(months=4))
            except billing_exceptions.DiskServiceAlreadyProlongatedError, e:
                new_billing_ts = service_expires_ts
        else:
            new_billing_ts = self.get_new_billing_ts(service_expires_ts, prolongation_delta)

        # продлеваем, если дата окончания услуги изменилась
        if new_billing_ts > service_expires_ts:
            self.set_new_billing_time(sid, new_billing_ts)

        obj = {'product_id': product['id'], 'btime': new_billing_ts}
        return super(ProlongateServiceByProductHandler, self).serialize(obj)


class DeleteServiceByProductHandler(MpfsBillingUtilsHandlerMixin, BasePartnerServiceHandler):
    """Удалить услугу"""
    service_url = '/billing/service_delete?uid=%(uid)s&sid=%(sid)s&ip=%(ip)s'
    resp_status_code = 204

    error_map = {
        133: billing_exceptions.DiskServiceNotFoundError,
    }

    def delete_service(self, product_id):
        sid, _ = self.get_service_info(product_id)
        context = self.get_context({'sid': sid})
        url = self.build_url(self.service_url, context)
        status_code, resp, headers = self.raw_request_service(url)
        if status_code == 200:
            return
        raise ServiceUnavailableError()

    @_auto_initialize_user
    def handle(self, request, *args, **kwargs):
        self.load_product_check_permissions(*args, **kwargs)
        self.delete_service(request.partner_product['id'])
        return None


class DeleteServiceByServiceIdHandler(MpfsBillingUtilsHandlerMixin, BasePartnerServiceHandler):
    """Удалить услугу"""
    url = '/billing/service_delete?uid=%(uid)s&sid=%(service_id)s&ip=%(ip)s'
    resp_status_code = 204

    error_map = {
        133: billing_exceptions.DiskServiceNotFoundError,
    }

    query = fields.QueryDict({
        'product_id': None,
    })
    kwargs = fields.QueryDict({
        'service_id': fields.StringField(required=True, help_text=u'Идентификатор услуги.'),
    })

    def delete_service(self):
        url = self.build_url(self.url, self.get_context())
        status_code, resp, headers = self.raw_request_service(url)
        if status_code == 200:
            return
        raise ServiceUnavailableError()

    @_auto_initialize_user
    def handle(self, request, *args, **kwargs):
        service_id = kwargs['service_id']
        service = self.get_service_by_service_id(service_id)
        kwargs['product_id'] = service['name']
        self.load_product_check_permissions(*args, **kwargs)
        self.delete_service()
