# -*- coding: utf-8 -*-
from mpfs.common.static.codes import BILLING_IN_APP_INVALID_VERIFICATION, BILLING_IN_APP_UID_MISMATCH
from mpfs.common.static.tags.billing import APPLE_APPSTORE, GOOGLE_PLAY
from mpfs.platform import fields
from mpfs.platform.permissions import AllowByClientIdPermission
from mpfs.platform.v1.case.handlers import PLATFORM_MOBILE_APPS_IDS
from mpfs.platform.v1.disk.billing.inapp.exceptions import DiskBillingInappReceiptIsInvalidError, \
    DiskBillingInappUidMismatchError
from mpfs.platform.v1.disk.billing.inapp.serializers import ProcessInappReceiptBodySerializer, \
    BillingInappProductsListSerializer, BillingInAppTariffsSerializer
from mpfs.platform.v1.disk.handlers import MpfsProxyHandler


class ProcessInappReceiptHandler(MpfsProxyHandler):
    """Сообщить MPFS об in-app оплате"""
    service_url = '/billing/process_receipt?uid=%(uid)s&store_id=%(store_id)s&store_product_id=%(store_product_id)s' \
                  '&package_name=%(package_name)s&currency=%(currency)s'
    permissions = AllowByClientIdPermission(PLATFORM_MOBILE_APPS_IDS)
    hidden = True
    body_serializer_cls = ProcessInappReceiptBodySerializer
    query = fields.QueryDict({
        'store_id': fields.StringField(required=True, help_text=u'идентификатор стора.'),
        'store_product_id': fields.StringField(required=False, help_text=u'Идентификатор продукта в сторе.'),
        'package_name': fields.StringField(required=True, help_text=u'Имя бандла.'),
        'currency': fields.StringField(required=False, help_text=u'Валюта, в которой совершалась покупка.'),
    })
    error_map = {
        BILLING_IN_APP_INVALID_VERIFICATION: DiskBillingInappReceiptIsInvalidError,
        BILLING_IN_APP_UID_MISMATCH: DiskBillingInappUidMismatchError,
    }

    def handle(self, request, *args, **kwargs):
        context = self.get_context()
        url = self.get_url(context)
        data = self.request_service(url,
                                    method=self.service_method,
                                    data=self.request.body,
                                    headers={'Content-type': 'application/json'})
        return self.serialize(data)


class ListInAppTariffsHandler(MpfsProxyHandler):
    """Получить in-app тарифы"""
    service_url = '/billing/inapp_product_card?uid=%(uid)s&store_id=%(store_id)s'
    permissions = AllowByClientIdPermission(PLATFORM_MOBILE_APPS_IDS)
    hidden = True
    serializer = BillingInAppTariffsSerializer
    query = fields.QueryDict({
        'store_id': fields.ChoiceField(required=True, choices=[APPLE_APPSTORE, GOOGLE_PLAY], help_text=u'ID платформы для фильтрации.')
    })


class ListInappProductsHandler(MpfsProxyHandler):
    """Получить списоке in-app продуктов

    Новые киленты используют ListInappTariffsHandler
    """
    service_url = '/billing/inapp_products?uid=%(uid)s&store_id=%(store_id)s&locale=%(locale)s'
    permissions = AllowByClientIdPermission(PLATFORM_MOBILE_APPS_IDS)
    hidden = True
    body_serializer_cls = BillingInappProductsListSerializer

    query = fields.QueryDict({
        'locale': fields.ChoiceField(choices=['ru', 'en', 'tr', 'uk', 'ua'], help_text=u'Локаль пользователя.', default='ru'),
        'store_id': fields.ChoiceField(required=True, choices=[APPLE_APPSTORE, GOOGLE_PLAY], help_text=u'ID платформы для фильтрации.')
    })
