# -*- coding: utf-8 -*-
from mpfs.core.user.constants import DEFAULT_LOCALE
from mpfs.platform.fields import (
    BooleanField,
    DateTimeToTSField,
    IntegerField,
    SerializerField,
    StringField,
    FloatField,
)
from mpfs.platform.serializers import BaseSerializer, ListSerializer


class BillingPartnerServiceSerializer(BaseSerializer):
    visible_fields = ['product_id', 'expires', 'service_id']
    fields = {
        'product_id': StringField(required=True, pbid=1, help_text=u'Идентификатор продукта'),
        'expires': DateTimeToTSField(required=False, pbid=2, source='btime', help_text=u'Дата истечения предоставленной услуги.'),
        'service_id': StringField(required=False, pbid=3, source='sid', help_text=u'Идентификатор услуги'),
    }


class BillingPartnerServicesSerializer(BaseSerializer):
    visible_fields = ['partner', 'items']
    fields = {
        'partner': StringField(required=True, pbid=1, help_text=u'Идентификатор партнёра'),
        'items': SerializerField(BillingPartnerServiceSerializer, many=True,
                                 help_text=u'Список предоставленных пользователю партнерских услуг')
    }


class BillingServiceSerializer(BaseSerializer):
    visible_fields = ['name', 'key', 'space']
    fields = {
        'name': StringField(required=True, pbid=1, help_text=u'Локализованное название услуги.'),
        'key': StringField(required=True, pbid=2, help_text=u'Идентификатор услуги.'),
        'space': IntegerField(required=False, help_text=u'Объем предоставляемого места в байтах.'),
    }


class BillingOrderSerializer(BaseSerializer):
    visible_fields = ['payment_method']
    fields = {
        'payment_method': StringField(required=False, pbid=1, help_text=u'Способ оплаты.'),
    }


class BillingSubscriptionSerializer(BaseSerializer):
    lang = DEFAULT_LOCALE
    """Переменная определяет в какой локализации будет выведено имя услуги."""

    visible_fields = ['subscription_id', 'active_until', 'next_charge', 'auto_renewable', 'service', 'order']
    fields = {
        'subscription_id': StringField(required=True, pbid=3, source='sid', help_text=u'Идентификатор подписки.'),
        'active_until': DateTimeToTSField(required=False, pbid=4, help_text=u'Дата, до которой подписка активна.'),
        'next_charge': DateTimeToTSField(required=False, pbid=5, help_text=u'Дата следующего автоматического списания.'),
        'auto_renewable': BooleanField(required=True, pbid=6, source='subscription', help_text=u'Является ли подписка автопродляемой.'),
        'service': SerializerField(BillingServiceSerializer, required=True, help_text=u'Информация об услуге.'),
        'order': SerializerField(BillingOrderSerializer, required=False, help_text=u'Информация о заказе.'),
    }

    def prepare_object(self, obj):
        default_subscription_name = obj['names'][DEFAULT_LOCALE]
        obj['service'] = {
            'name': obj['names'].get(self.lang, default_subscription_name),
            'key': obj['name'],
        }
        if obj.get('size') is not None:
            obj['service']['space'] = obj['size']
        if obj.get('expires') is not None:
            obj['active_until'] = obj['expires']
        if obj['subscription']:
            obj['next_charge'] = obj['expires']
        if obj.get('payment_method') is not None:
            obj['order'] = {
                'payment_method': obj['payment_method']
            }
        return super(BillingSubscriptionSerializer, self).prepare_object(obj)

    def initialize(self, obj=None, lang=None, *args, **kwargs):
        self.lang = lang
        super(BillingSubscriptionSerializer, self).initialize(obj=obj, *args, **kwargs)


class BillingSubscriptionsListSerializer(ListSerializer):
    lang = DEFAULT_LOCALE  # необходима для того, чтобы проставить нужное значение lang в BillingSubscriptionSerializer

    visible_fields = ['items', 'limit', 'offset', 'total']
    fields = {
        'items': SerializerField(BillingSubscriptionSerializer, many=True, help_text=u'Список услуг')
    }

    def initialize(self, obj=None, lang=None, *args, **kwargs):
        self.lang = lang or DEFAULT_LOCALE
        super(BillingSubscriptionsListSerializer, self).initialize(obj=obj, *args, **kwargs)

    def get_fields(self):
        for f in self.fields.values():
            if isinstance(f, SerializerField):
                f.init.update({'lang': self.lang})
        return super(BillingSubscriptionsListSerializer, self).get_fields()


class BillingDiscountSerializer(BaseSerializer):
    visible_fields = ['percentage', 'active_until_ts']
    fields = {
        'percentage': IntegerField(required=True, help_text=u'Процент скидки от цены.'),
        'active_until_ts': IntegerField(required=False, help_text=u'Timestamp до какого времени скидка активна.'),
    }


class BillingPeriodSerializer(BaseSerializer):
    visible_fields = ['product_id', 'price']
    fields = {
        'product_id': StringField(required=True, help_text=u'ID продукта за период.'),
        'price': FloatField(required=True, help_text=u'Цена за продукт.'),
    }


class BillingPeriodsSerializer(BaseSerializer):
    visible_fields = ['year', 'month']
    fields = {
        'year': SerializerField(BillingPeriodSerializer, required=False, help_text=u'Годовой период для продукта.'),
        'month': SerializerField(BillingPeriodSerializer, required=False, help_text=u'Месячный период для продукта.'),
    }


class BillingProductSerializer(BaseSerializer):
    visible_fields = ['display_space', 'display_space_units', 'periods', 'space', 'currency', 'discount']
    fields = {
        'display_space': IntegerField(required=True, help_text=u'Объем предоставляемого места для отображения.'),
        'display_space_units': StringField(required=True, help_text=u'Единицы измерения для поля display_space.'),
        'periods': SerializerField(BillingPeriodsSerializer, required=True,
                                   help_text=u'Временные периоды выдачи продуктов.'),
        'space': IntegerField(required=True, help_text=u'Объем предоставляемого места в байтах.'),
        'currency': StringField(required=True, help_text=u'Валюта.'),
        'discount': SerializerField(BillingDiscountSerializer, required=False, help_text=u'Скидка.')
    }


class BillingProductsListSerializer(ListSerializer):
    visible_fields = ['items', 'total']
    fields = {
        'items': SerializerField(BillingProductSerializer, many=True, help_text=u'Список продуктов')
    }
