# -*- coding: utf-8 -*-

import time
import hashlib
import re

from collections import defaultdict, OrderedDict

from mpfs.common.errors import AddressError
from mpfs.common.util.user_agent_parser import UserAgentParser
from mpfs.core.address import ResourceId
from mpfs.core.services.event_history_search_service import event_history_search_service
from mpfs.core.services.mpfsproxy_service import mpfsproxy
from mpfs.core.services.passport_service import passport
from mpfs.platform import fields
from mpfs.platform.auth import PassportCookieAuth
from mpfs.platform.handlers import ServiceProxyHandler, ETagHandlerMixin, PlatformHandlerMetaClass
from mpfs.platform.v1.disk.permissions import WebDavPermission
from mpfs.common.errors import EventHistoryBadResponse

from metadata import EVENT_TYPES, ApiPlatform, EventSchema
from serializers import ClusterizeActionSerializer, GroupActionSerializer, SearchActionSerializer


def _list_to_lucene_expression(values):
    return '(%s)' % ' or '.join(values)


class PlatformFilter(object):
    def __init__(self, values):
        self.values = set(values or [])

    def __nonzero__(self):
        return bool(self.values) and self.values != set(ApiPlatform.all_platforms)

    def to_lucene_expression(self):
        if ApiPlatform.undisclosed_platform_substitute in self.values:
            return '(* !%s)' % _list_to_lucene_expression(set(ApiPlatform.disclosed_platforms) - self.values)
        else:
            return _list_to_lucene_expression(self.values)


class BaseSearchProxyHandlerMetaClass(PlatformHandlerMetaClass):
    """Метакласс обеспечивающий механизм наследования атрибута `optional_query_params`."""
    def __new__(mcs, name, parents, attrs):
        attrs = mcs.__inherit_list(parents, attrs, 'optional_query_params')
        return super(BaseSearchProxyHandlerMetaClass, mcs).__new__(mcs, name, parents, attrs)

    @staticmethod
    def __inherit_list(parents, attrs, attr_name):
        attrs[attr_name] = BaseSearchProxyHandlerMetaClass.__collect_lists(parents, attrs, attr_name)
        return attrs

    @staticmethod
    def __collect_lists(parents, attrs, attr_name):
        values = []
        for parent in parents:
            values.extend(getattr(parent, attr_name, []))
        values.extend(attrs.get(attr_name, []))
        return values


class BaseSearchProxyHandler(ServiceProxyHandler):
    permissions = WebDavPermission()
    service = event_history_search_service
    service_base_exception = EventHistoryBadResponse
    auth_methods = [PassportCookieAuth()]
    __metaclass__ = BaseSearchProxyHandlerMetaClass

    query = fields.QueryDict({
        # service parameters
        'page_load_date': fields.DateTimeToTSWithTimezoneField(milliseconds=True,
                                                               help_text=u'Дата захода пользователя на страницу'),

        # filtering
        'start_date': fields.DateTimeToTSField(
                milliseconds=True,
                help_text=u'Дата возникновения первого события, которое требуется получить в листинге'),
        'end_date': fields.DateTimeToTSField(
                milliseconds=True,
                help_text=u'Дата возникновения последнего события, которое требуется получить в листинге'),
        'event_type': fields.MultipleChoicesField(choices=EVENT_TYPES,
                                                  help_text=u'Тип событий, которые следует оставить в выдаче'),
        'platform': fields.MultipleChoicesField(
                choices=ApiPlatform.all_platforms,
                help_text=u'Платформа, на которой возникли события, которые следует оставить в выдаче'),
        'parent_path': fields.StringField(help_text=u'Фильтр по папке, в которой произошло событие'),
        'exclude_resource_type': fields.ChoiceField(choices=OrderedDict([('file', 'file'), ('dir', 'directory')]),
                                                    help_text=u'Фильтр по типам ресурсов.'),

        # pagination
        'offset': fields.IntegerField(
                default=0,
                help_text=u'Для постраничной выдачи. Смещение, начиная с которого выдавать элементы.'),
        'limit': fields.IntegerField(
                default=20,
                help_text=u'Для постраничной выдачи. Количество элементов, которое нужно выдать.'),

        # preview parameters
        'preview_size': fields.StringField(help_text=u'Размер превью'),
        'preview_crop': fields.IntegerField(default=0, help_text=u'Надо ли делать кроп'),
    })

    optional_query_params = ['exclude_resource_type']
    query_parameter_mapping = {
        'start_date': 'start_timestamp',
        'end_date': 'end_timestamp',
        'limit': 'max_amount',
        'limit_per_group': 'max_events_per_group',
        'parent_path': 'parent_folder',
        'event_group_key': 'group_key',
        'event_type_filter': 'event_type',
        'platform_filter': 'platform',
    }

    def handle(self, request, *args, **kwargs):
        if request.query['offset'] != 0 and not request.query['page_load_date']:
            raise self.query.get_fields()['page_load_date'].errors['required']()

        return super(BaseSearchProxyHandler, self).handle(request, *args, **kwargs)

    def get_context(self, context=None):
        c = super(BaseSearchProxyHandler, self).get_context(context=context)
        if not c['page_load_date']:
            c['page_load_date'] = (int(round(time.time() * 1000)), 0)

        c['page_load_timestamp'], _ = c['page_load_date']
        return c

    def get_url(self, context=None):
        if context is None:
            context = self.get_context()

        url = self.service_url

        self._add_filter(context, 'event_type_filter')
        self._add_platform_filter(context)

        for param in self.optional_query_params:
            if param in context and context[param]:
                search_param = self.query_parameter_mapping.get(param, param)
                url = '%s&%s=%%(%s)s' % (url, search_param, param)

        return self.build_url(url, context=context)

    @staticmethod
    def _add_platform_filter(context):
        platform_filter = PlatformFilter(context.get('platform', None))
        if platform_filter:
            context['platform_filter'] = platform_filter.to_lucene_expression()

    def _add_filter(self, context, target_key):
        source_key = self.query_parameter_mapping[target_key]
        if target_key in self.optional_query_params and source_key in context and context[source_key]:
            context[target_key] = _list_to_lucene_expression(context[source_key])

    def get_service_error_code(self, exception):
        return None


class BaseEventSearchProxyHandler(BaseSearchProxyHandler):
    # порядок значений внутри списка важен:
    #   * если у события есть путь к целевому ресурсу (target_path), то метаданные отдаем для целевого ресурса
    #   * если у события есть только путь к исходному ресурсу (source_path), то метаданные отдаем для исходного ресурса
    resource_path_keys = ['target_path', 'source_path']
    resource_events_keys = resource_path_keys + ['entity_id']
    uid_keys = ['owner_uid', 'user_uid']

    def __init__(self, parent=None, *args, **kwargs):
        super(BaseEventSearchProxyHandler, self).__init__(parent=None, *args, **kwargs)

    def get_context(self, context=None):
        c = super(BaseEventSearchProxyHandler, self).get_context(context=context)
        c['group_interval'] = self.service.group_interval
        return c

    def serialize(self, resp, *args, **kwargs):
        resp = self._add_page_load_date(resp)
        resp = self._add_meta_to_resource_events(resp)
        resp = self._add_meta_to_album_events(resp, 'album_id', 'album')
        resp = self._add_meta_to_album_events(resp, 'child_album_id', 'child_album')
        resp = self._add_user_info_to_events(resp)
        resp = self._add_lenta_block_to_events(resp)

        self.request.raw_data = resp
        return super(BaseEventSearchProxyHandler, self).serialize(resp, *args, **kwargs)

    def extract_events(self, resp):
        raise NotImplementedError()

    def extract_resource_events(self, resp):
        events = self.extract_events(resp)
        return [event for event in events if any(event.get(path_key, None) for path_key in self.resource_events_keys)]

    def extract_album_events(self, resp, album_id_key):
        events = self.extract_events(resp)
        return [event for event in events if event.get(album_id_key, None) is not None]

    def _add_page_load_date(self, resp):
        resp['page_load_date'] = self.get_context()['page_load_date']
        return resp

    def _add_user_info_to_events(self, resp):
        uids = {
            uid for e in self.extract_events(resp)
            for uid in [e.get(key, None) for key in self.uid_keys]
            if uid and not EventSchema.is_comment_event(e)
        }
        uids.add(self.request.user.uid)
        resp['users'] = [u for u in [passport.userinfo(uid=uid, personal=True) for uid in uids] if u.get('uid', None)]
        return resp

    def _add_meta_to_resource_events(self, resp):
        events_with_resource_id = defaultdict(list)
        events_with_path_only = defaultdict(list)
        for event in self.extract_resource_events(resp):
            try:
                resource_id = ResourceId(event.get('owner_uid', None), event.get('resource_file_id', None))
            except AddressError:
                try:
                    resource_id = ResourceId.parse(event.get('entity_id', ''))
                except ValueError:
                    resource_id = None

            paths = [event[path_key] for path_key in self.resource_path_keys if event.get(path_key, None) is not None]

            if resource_id:
                events_with_resource_id[str(resource_id)].append(event)
            elif paths:
                events_with_path_only[self._unify_path(paths[0])].append(event)

        c = self.get_context()
        preview_size = c.get('preview_size', None)
        preview_crop = c.get('preview_crop', False)
        preview_quality = c.get('preview_quality', None)
        preview_allow_big_size = c.get('preview_allow_big_size', False)

        if events_with_resource_id:
            resource_infos = mpfsproxy.bulk_info_by_resource_ids(self.request.user.uid,
                                                                 list(events_with_resource_id.keys()),
                                                                 preview_size=preview_size,
                                                                 preview_crop=preview_crop,
                                                                 preview_quality=preview_quality,
                                                                 preview_allow_big_size=preview_allow_big_size)

            def get_events_key(info):
                return info.get('meta', {}).get('resource_id', None)

            self._add_meta_to_resource_events_basic(events_with_resource_id, resource_infos, get_events_key)

        if events_with_path_only:
            resource_infos = mpfsproxy.bulk_info(self.request.user.uid,
                                                 list(events_with_path_only.keys()),
                                                 preview_size=preview_size,
                                                 preview_crop=preview_crop,
                                                 preview_quality=preview_quality,
                                                 preview_allow_big_size=preview_allow_big_size)

            def get_events_key(info):
                return self._unify_path(info['path'])

            self._add_meta_to_resource_events_basic(events_with_path_only, resource_infos, get_events_key)

        return resp

    @staticmethod
    def _add_meta_to_resource_events_basic(events, resource_infos, get_events_key):
        for info in resource_infos:
            key = get_events_key(info)
            if not key:
                continue

            for event in events.get(key, []):
                event['resource'] = info

    @staticmethod
    def _unify_path(path):
        if not isinstance(path, basestring):
            return path

        return path.rstrip('/')

    def _add_meta_to_album_events(self, resp, album_id_key, album_meta_key):
        events = defaultdict(list)
        for event in self.extract_album_events(resp, album_id_key):
            album_id = event.get(album_id_key, None)
            if album_id:
                events[album_id].append(event)

        if events:
            c = self.get_context()
            for album_id, albums_events in events.iteritems():
                album_info = mpfsproxy.get_album(self.request.user.uid, album_id, amount=0)
                if album_info:
                    for event in albums_events:
                        event[album_meta_key] = album_info

        return resp

    def _add_lenta_block_to_events(self, resp):
        events = self.extract_events(resp)
        comment_events = [event for event in events if event['event_type'].startswith('comment-')]
        for event in comment_events:
            if event.get('entity_type', '') != 'private_resource':
                continue

            event['lenta_block'] = {
                'uid': self.request.user.uid,
                'modify_uid': event['user_uid'],
                'media_type': event.get('resource', {}).get('meta', {}).get('media_type', None),
                'mtime': event['event_timestamp'],
                'type': event['entity_type'],
                'resource_file_id': event['entity_id']
            }

        return resp


class ClusterizeSearchProxyHandler(ETagHandlerMixin, BaseEventSearchProxyHandler):
    service_url = '/clusterize?count_distinct=resource_type,resource_media_type&interval=%(group_interval)s&get=*' + \
                  '&uid=%(uid)s&tz_offset=%(tz_offset)s&page_load_timestamp=%(page_load_timestamp)s' + \
                  '&offset=%(offset)s&max_amount=%(limit)s'

    query = fields.QueryDict({
        'tz_offset': fields.IntegerField(help_text=u'Временная зона пользователя в мс относительно UTC+0'),
        'limit_per_group': fields.IntegerField(help_text=u'Максимальное кол-во событий в группе.'),
    })

    optional_query_params = ['start_date', 'end_date', 'parent_path', 'limit_per_group',
                             'event_type_filter', 'platform_filter']

    serializer_cls = ClusterizeActionSerializer

    _time_ms_regexp = re.compile('\.\d+')

    def handle(self, request, *args, **kwargs):
        if request.query['offset'] == 0 and \
                        request.query['page_load_date'] is None and \
                        request.query['tz_offset'] is None:
            raise self.query.get_fields()['tz_offset'].errors['required']()

        result = super(ClusterizeSearchProxyHandler, self).handle(request, *args, **kwargs)

        # не все клиенты умеют обрабатывать ответ с миллисекундами, для мобильных возвращаем без них
        # https://st.yandex-team.ru/CHEMODAN-34097
        ua = request.raw_headers.get('user-agent')
        if UserAgentParser.is_yandex_disk_mobile(ua):
            for group in result['groups']:
                for event in group['events']:
                    if 'event_date' in event:
                        event['event_date'] = self._time_ms_regexp.sub('', event['event_date'])
                if 'min_date' in group['group']:
                    group['group']['min_date'] = self._time_ms_regexp.sub('', group['group']['min_date'])
                if 'max_date' in group['group']:
                    group['group']['max_date'] = self._time_ms_regexp.sub('', group['group']['max_date'])
            if 'page_load_date' in result:
                result['page_load_date'] = self._time_ms_regexp.sub('', result['page_load_date'])

        return result

    def get_context(self, context=None):
        c = super(ClusterizeSearchProxyHandler, self).get_context(context=context)
        if c.get('tz_offset', None) is None:
            _, page_load_timezone = c['page_load_date']
            c['tz_offset'] = page_load_timezone * 1000
        else:
            rounded_timezone = int(round(c['tz_offset'] / 1000 / 60)) * 60
            c['page_load_date'] = (c['page_load_date'][0], rounded_timezone)
        return c

    def extract_events(self, resp):
        return [event for group in resp['hitsArray'] for event in group['merged_docs']]

    # https://st.yandex-team.ru/CHEMODAN-28497
    def _calc_weak_etag(self, response):
        group_ids = [self.__build_group_id(group) for group in self.__get_groups()]
        content = {'fields': self.request.query['fields'], 'group_ids': group_ids}
        return hashlib.md5(str(content)).hexdigest()

    def __get_groups(self):
        return self.request.raw_data.get('hitsArray', [])

    @staticmethod
    def __build_group_id(group):
        resources = [doc.get('resource', {}) for doc in group['merged_docs']]
        return group['merged_docs'][0]['group_key'], group['size'], resources


class GroupSearchProxyHandler(BaseEventSearchProxyHandler):
    service_url = '/group?interval=%(group_interval)s&get=*' + \
                  '&uid=%(uid)s&page_load_timestamp=%(page_load_timestamp)s' + \
                  '&group_key=%(event_group_key)s' +\
                  '&start_timestamp=%(start_date)s&end_timestamp=%(end_date)s' + \
                  '&offset=%(offset)s&max_amount=%(limit)s'

    query = fields.QueryDict({
        'event_group_key': fields.StringField(required=True, help_text=u'Общий ключ группы событий'),
        'start_date': fields.DateTimeToTSField(required=True, milliseconds=True,
                                               help_text=u'Дата возникновения первого события в группе'),
        'end_date': fields.DateTimeToTSField(required=True, milliseconds=True,
                                             help_text=u'Дата возникновения последнего события в группы'),
    })

    serializer_cls = GroupActionSerializer

    def extract_events(self, resp):
        return resp['hitsArray']


class SearchProxyHandler(BaseEventSearchProxyHandler):
    service_url = '/search?uid=%(uid)s&get=*' + \
                  '&page_load_timestamp=%(page_load_timestamp)s&text=%(text)s' + \
                  '&offset=%(offset)s&max_amount=%(limit)s'

    query = fields.QueryDict({
        'text': fields.StringField(required=True, help_text=u'Текст поискового запроса.'),
    })

    optional_query_params = ['start_date', 'end_date', 'parent_path', 'event_type_filter', 'platform_filter']

    serializer_cls = SearchActionSerializer

    def extract_events(self, resp):
        return resp['hitsArray']


class CountersProxyHandler(BaseSearchProxyHandler):
    service_url = '/counters?uid=%(uid)s&page_load_timestamp=%(page_load_timestamp)s&field=%(field)s'

    optional_query_params = ['start_date', 'end_date', 'parent_path', 'event_type_filter', 'platform_filter']

    query = fields.QueryDict({
        'field': fields.StringField(required=True,
                                    help_text=u'Название поля, встречаемость которого необходимо получить'),
    })

    def serialize(self, resp, *args, **kwargs):
        return super(CountersProxyHandler, self).serialize(self._alter_resp(resp), *args, **kwargs)

    def _alter_resp(self, resp):
        if self.get_context().get('field') == 'platform':
            new_resp = defaultdict(int)
            for platform, count in resp.iteritems():
                new_resp[str(ApiPlatform(platform))] += count
            return new_resp
        else:
            return resp
