# -*- coding: utf-8 -*-


class EventSchema(object):
    persistent_common = ('version', 'id', 'group_key', 'event_type', 'event_timestamp', 'platform')

    def __init__(self, attrs):
        self.persistent_specific = tuple(attr for attr in attrs if attr not in self.persistent_common)

    def __add__(self, other):
        return EventSchema(self.persistent_specific + other.persistent_specific)

    def __contains__(self, item):
        return item in self.complete

    @property
    def persistent(self):
        return self.persistent_common + self.persistent_specific

    @property
    def mixed_in(self):
        persistent_schema = self.persistent
        return tuple(mixed_in_attr for mixed_in_attr, trigger_attrs in _MIXED_IN_TO_TRIGGER_ATTRS.iteritems()
                     if any(trigger_attr in persistent_schema for trigger_attr in trigger_attrs))

    @property
    def complete(self):
        return self.persistent + self.mixed_in

    @property
    def public(self):
        return tuple(attr for attr in self.complete if attr not in _HIDDEN_ATTRS)

    def __str__(self):
        return '%s=%s' % ('EventAttrs', str(self.persistent_specific))

    @staticmethod
    def get_predefined(event_type):
        return _EVENT_SCHEMAS.get(event_type, None)

    @classmethod
    def get_predefined_or_create_new(cls, event_data):
        return cls.get_predefined(event_data['event_type']) or cls(event_data.keys())

    @staticmethod
    def is_comment_event(data):
        return data.get('event_type', '').startswith('comment-')

_REF_SCHEMA_PREFIX = 'r:'


def _build_schema_dict(src_attrs=None, ref_schemas=None, ref_prefix=_REF_SCHEMA_PREFIX):
    if src_attrs is None:
        return {}

    if ref_schemas is None:
        ref_schemas = {}

    src_attrs = src_attrs.copy()
    result = {}

    def is_ref_schema(name):
        return name.startswith(ref_prefix)

    def add_if_missing_and_get(schema_name):
        if is_ref_schema(schema_name):
            schema_name = schema_name[len(ref_prefix):]

        if schema_name in ref_schemas:
            return ref_schemas[schema_name]

        if schema_name not in result:
            names = src_attrs.pop(schema_name)

            schema_from_attrs = EventSchema([name for name in names if not is_ref_schema(name)])
            schemas_from_refs = tuple(add_if_missing_and_get(name) for name in names if name not in schema_from_attrs)

            result[schema_name] = reduce((lambda s1, s2: s1 + s2), schemas_from_refs + (schema_from_attrs,))

        return result[schema_name]

    for src_schema_name in src_attrs.keys():
        add_if_missing_and_get(src_schema_name)

    return result


_MIXED_IN_TO_TRIGGER_ATTRS = {
    'resource':         ('target_path', 'source_path', 'entity_id'),
    'album':            ('album_id',),
    'child_album':      ('child_album_id',),
    'product_names':    ('product_name_ru', 'product_name_en', 'product_name_ua', 'product_name_tr'),
}

_REF_SCHEMAS = _build_schema_dict({
    'product':          ('product_id', 'product_period',
                         'product_name_ru', 'product_name_ua', 'product_name_en', 'product_name_tr'),
    'common-invite':    ('invite_login', 'invite_service'),
    'resource-types':   ('resource_type', 'resource_media_type'),

    'fs-target':        ('target_path', 'user_uid', 'r:resource-types'),
    'fs-source-target': ('source_path', 'r:fs-target'),
    'sharing-basic':    ('target_path', 'owner_uid'),
    'sharing-user':     ('r:sharing-basic', 'user_uid', 'rights'),
    'album-basic':      ('album_id', 'album_title'),
    'album-item':       ('r:album-basic', 'item_id', 'child_album_id', 'child_album_title',
                         'source_path', 'r:resource-types'),
    'billing':          ('r:product', 'currency', 'price'),
    'space':            ('r:product', 'old_limit', 'new_limit'),
    'comment-entity':   ('resource_name', 'resource_type', 'resource_short_url', 'lenta_block'),
})

_HIDDEN_ATTRS = _MIXED_IN_TO_TRIGGER_ATTRS['product_names']

_EVENT_SCHEMAS = _build_schema_dict(ref_schemas=_REF_SCHEMAS, src_attrs={
    'fs-mkdir':             ('target_path', 'user_uid'),
    'fs-mksysdir':          ('type',),
    'fs-copy':              ('r:fs-source-target', 'resource_overwritten'),
    'fs-move':              ('r:fs-source-target', 'resource_overwritten'),
    'fs-rename':            ('r:fs-source-target',),
    'fs-rm':                ('r:fs-target',),
    'fs-store':             ('r:fs-target',),
    'fs-store-download':    ('r:fs-target',),
    'fs-store-office':      ('r:fs-target',),
    'fs-store-photostream': ('r:fs-target',),
    'fs-store-photounlim':  ('r:fs-target',),
    'fs-store-update':      ('r:fs-target',),
    'fs-trash-append':      ('r:fs-source-target',),
    'fs-trash-drop-all':    ('owner_uid',),
    'fs-trash-drop':        ('r:fs-target',),
    'fs-trash-restore':     ('r:fs-source-target',),
    'fs-set-public':        ('r:fs-target', 'resource_short_url'),
    'fs-set-private':       ('r:fs-target',),
    'fs-aviary-render':     ('r:fs-target',),

    'social-import': ('r:fs-target', 'invite_service'),
    'social-export': ('r:fs-target', 'invite_service'),

    'album-create':                 ('r:album-item',),
    'album-remove':                 ('r:album-basic',),
    'album-change-cover':           ('r:album-basic',),
    'album-change-cover-offset':    ('r:album-basic',),
    'album-change-publicity':       ('r:album-basic', 'album_is_public'),
    'album-change-title':           ('r:album-basic', 'album_prev_title'),
    'album-create-item':            ('r:album-item',),
    'album-items-append':           ('r:album-item',),
    'album-items-remove':           ('r:album-item',),
    'album-post-to-social':         ('r:album-basic', 'invite_service'),

    'billing-order-new':        ('r:billing',),
    'billing-buy-new':          ('r:billing',),
    'billing-prolong':          ('r:billing',),
    'billing-prolong-error':    ('r:billing', 'reason'),
    'billing-subscribe':        ('r:billing',),
    'billing-unsubscribe':      ('r:billing',),
    'billing-delete':           ('r:billing',),

    'share-create-group':           ('r:sharing-basic',),
    'share-unshare-folder':         ('r:sharing-basic',),
    'share-unshare-folder-user':    ('r:sharing-user', 'user_type'),
    'share-invite-user':            ('r:sharing-user', 'r:common-invite'),
    'share-remove-invite':          ('r:sharing-user', 'r:common-invite'),
    'share-change-invite-rights':   ('r:sharing-user', 'r:common-invite', 'prev_rights'),
    'share-change-rights':          ('r:sharing-user', 'prev_rights'),
    'share-change-group-owner':     ('r:sharing-user',),
    'share-kick-from-group':        ('r:sharing-user',),
    'share-activate-invite':        ('r:sharing-user',),
    'share-reject-invite':          ('r:sharing-user',),
    'share-leave-group':            ('r:sharing-user',),

    'invite-sent':          ('r:common-invite',),
    'invite-activation':    ('owner_uid', 'user_uid'),

    'space-product-reduce': ('r:space',),
    'space-promo-enlarge':  ('r:space',),
    'space-promo-reduce':   ('r:space',),

    'comment-add': ('r:comment-entity',),
    'comment-delete': ('r:comment-entity',),
    'comment-restore': ('r:comment-entity',),
    'comment-like-add': ('r:comment-entity',),
    'comment-like-delete': ('r:comment-entity',),
    'comment-dislike-add': ('r:comment-entity',),
    'comment-dislike-delete': ('r:comment-entity',),
})

EVENT_TYPES = _EVENT_SCHEMAS.keys()


class ApiPlatform(object):
    disclosed_platforms = (
        # mobile
        'ios', 'andr', 'wp',

        # desktop
        'win', 'mac', 'lnx',

        # web
        'web',
    )

    # https://st.yandex-team.ru/CHEMODAN-28857
    # known substitutees: rest, mpfs, dav, davadm, sdk, sw
    # change to 'other' after support on frontend
    undisclosed_platform_substitute = 'rest'

    all_platforms = disclosed_platforms + (undisclosed_platform_substitute,)

    def __init__(self, value):
        self.value = value

    def __str__(self):
        if self.value in self.disclosed_platforms:
            return self.value
        else:
            return self.undisclosed_platform_substitute
