# -*- coding: utf-8 -*-

from mpfs.platform import fields

from metadata import EventSchema
from fields import ApiPlatformField
from mpfs.platform.serializers import BaseSerializer
from mpfs.platform.v1.disk.serializers import ResourceSerializer, PhotoAlbumSerializer, InternalUserSerializer
from mpfs.platform.v1.disk.fields import PathField, RightsField


class BaseHistorySerializer(BaseSerializer):
    pass


class BaseHistoryActionSerializer(BaseHistorySerializer):
    fields = {
        'users': fields.SerializerField(InternalUserSerializer, many=True,
                                        help_text=u'Список пользователей, присутствующих в выдаче'),
        'page_load_date': fields.DateTimeToTSWithTimezoneField(
                required=True, milliseconds=True, help_text=u'Дата захода пользователя на первую страницу'),
    }


class _Event(object):
    __product_name_locales = {
        'ru': 'ru',
        'en': 'en',
        'tr': 'tr',
        'uk': 'ua',
        'ua': 'ua',  # deprecated: https://st.yandex-team.ru/CHEMODAN-24941
    }

    def __init__(self, data):
        self.schema = EventSchema.get_predefined_or_create_new(data)
        self.data = data

    def get(self, key, default=None):
        return self.data.get(key, default)

    def has_attr(self, attr):
        return attr in self.schema

    def prepare_data(self):
        if self.has_attr('product_names'):
            self._add_product_names()

        if self.has_attr('resource_short_url'):
            self._hide_public_url_if_inactive()

        self._add_missing_attrs()

        if EventSchema.is_comment_event(self.data):
            self.data['resource_short_url'] = self.get('resource', {}).get('meta', {}).get('short_url', None)

        return self.data

    def _add_product_names(self):
        self.data['product_names'] = {locale: self.__get_product_name(suffix)
                                      for locale, suffix in self.__product_name_locales.iteritems()}

    def __get_product_name(self, suffix):
        return self.get('product_name_' + suffix, None)

    def _hide_public_url_if_inactive(self):
        history_public_key = self.get('resource_public_key', None)
        current_public_key = self.get('resource', {}).get('meta', {}).get('public_hash', None)

        if not (history_public_key and history_public_key == current_public_key):
            self.data['resource_short_url'] = None

    def _add_missing_attrs(self):
        for attr in self.schema.persistent:
            if attr not in self.data:
                self.data[attr] = None


class EventSerializer(BaseHistorySerializer):
    fields = {
        'id': fields.StringField(required=True, help_text=u'Идентификатор события'),
        'version': fields.IntegerField(required=True, help_text=u'Версия'),
        'event_type': fields.StringField(required=True, help_text=u'Тип события'),
        'event_date': fields.DateTimeToTSField(source='event_timestamp', required=True, milliseconds=True,
                                               help_text=u'Дата возникновения события'),
        'platform': ApiPlatformField(required=True, help_text=u'Платформа, на которой произошло событие'),

        'from': PathField(source='source_path', help_text=u'Исходный путь к ресурсу'),
        'path': PathField(source='target_path', help_text=u'Результирующий путь к ресурсу'),
        'resource_type': fields.ChoiceField(choices=['file', 'directory'], help_text=u'Тип ресурса: папка или файл'),
        'resource_name': fields.StringField(help_text=u'Название ресурса'),
        'resource_media_type': fields.StringField(help_text=u'Медиа-тип ресурса'),
        'resource_overwritten': fields.BooleanField(help_text=u'Флаг, сигналазирующий,что ресурс был перезаписан',
                                                    boolify=True),
        'resource_public_url': fields.StringField(source='resource_short_url', help_text=u'Публичный URL ресурса'),

        'resource': fields.SerializerField(ResourceSerializer, help_text=u'Мета-информация ресурса'),

        'album_id': fields.StringField(help_text=u'Идентификатор альбома'),
        'album_title': fields.StringField(help_text=u'Название альбома'),
        'album_prev_title': fields.StringField(help_text=u'Старое название альбома'),
        'album_is_public': fields.BooleanField(help_text=u'Публичность альбома', boolify=True),
        'child_album_id': fields.StringField(help_text=u'Идентификатор дочернего альбома'),
        'child_album_title': fields.StringField(help_text=u'Название дочернего альбома'),
        'item_id': fields.StringField(help_text=u'Идентификатор элемента'),
        'item_album_id': fields.StringField(help_text=u'Идентификатор дочернего альбома'),
        'item_album_title': fields.StringField(help_text=u'Название дочернего альбома'),

        'album': fields.SerializerField(PhotoAlbumSerializer, help_text=u'Мета-информация альбома'),
        'child_album': fields.SerializerField(PhotoAlbumSerializer, help_text=u'Мета-информация дочернего альбома'),

        'product_id': fields.StringField(help_text=u'Идентификатор услуги'),
        'product_period': fields.StringField(help_text=u'Период действия услуги'),
        'product_names': fields.JsonObjectField(help_text=u'Названия услуги на разных языках'),
        'currency': fields.StringField(help_text=u'Валюта, в которой произведен платеж в формате ISO 4217 (RUB, USD)'),
        'payment_amount': fields.StringField(source='price', help_text=u'Сумма платежа'),

        'user_id': fields.StringField(source='user_uid', help_text=u'UID пользователя'),
        'owner_id': fields.StringField(source='owner_uid', help_text=u'UID владельца'),

        'provider_login': fields.StringField(source='invite_login',
                                             help_text=u'Адрес электронной почты или логин в соц. сети/сервисе'),
        'provider_id': fields.StringField(source='invite_service',
                                          help_text=u'ID сервиса, через который приглашен пользователь'),

        'rights': RightsField(help_text=u'Права доступа'),
        'previous_rights': RightsField(source='prev_rights', help_text=u'Старые права доступа'),
        'old_limit': fields.IntegerField(help_text=u'Предыдущее значение лимита дискового пространства'),
        'new_limit': fields.IntegerField(help_text=u'Новое значение лимита дискового пространства'),
        'reason': fields.StringField(help_text=u'Причина неудачи платежа'),

        'entity_type': fields.StringField(help_text=u'Тип комментируемой сущности'),
        'entity_id': fields.StringField(help_text=u'Идентификатор комментируемой сущности'),

        'lenta_block': fields.JsonObjectField(help_text=u'Блок Ленты', max_nesting_level=0)
    }
    visible_fields = fields.keys()
    source_to_field_names = {field.source or name: name for name, field in fields.iteritems()}

    def prepare_object(self, obj):
        obj = super(EventSerializer, self).prepare_object(obj)
        obj['event_timestamp'] = int(obj['event_timestamp'])
        return _Event(obj).prepare_data()

    def get_visible_fields(self):
        visible_fields = super(EventSerializer, self).get_visible_fields()
        if visible_fields:
            schema_fields = self.__get_schema_fields()
            return [fld for fld in visible_fields if fld.partition('.')[0] in schema_fields]
        else:
            return self.__get_schema_fields()

    def __get_schema_fields(self):
        public_schema = EventSchema.get_predefined_or_create_new(self._object).public
        return [field for src_attr, field in self.source_to_field_names.iteritems() if src_attr in public_schema]


class ClusterizeGroupDataSerializer(BaseHistorySerializer):
    fields = {
        'event_group_key': fields.StringField(required=True, help_text=u'Общий ключ группы событий'),
        'min_date': fields.DateTimeToTSField(required=True, milliseconds=True,
                                             help_text=u'Дата возникновения самого старого события в группе'),
        'max_date': fields.DateTimeToTSField(required=True, milliseconds=True,
                                             help_text=u'Дата возникновения самого свежего события в группе'),
    }
    visible_fields = ['event_group_key', 'min_date', 'max_date']


# https://st.yandex-team.ru/CHEMODAN-27308
# отдельный сериализатор добавлен только из-за документации
class ClusterizeCountersSerializer(BaseHistorySerializer):
    fields = {
        'resource_type': fields.JsonObjectField(
                help_text=u'Список типов ресурсов группы (файлов и папок) с указанием кол-ва каждого из них'),
        'resource_media_type': fields.JsonObjectField(
                help_text=u'Список медиа-типов группы, с указанием кол-ва каждого из них'),
    }
    visible_fields = ['resource_type', 'resource_media_type']


class ClusterizeGroupSerializer(BaseHistorySerializer):
    fields = {
        'total_count': fields.IntegerField(source='size', required=True,
                                           help_text=u'Общее количество событий, входящих в группу'),
        'group': fields.SerializerField(ClusterizeGroupDataSerializer, required=True,
                                        help_text=u'Данные группы событий'),
        'events': fields.SerializerField(EventSerializer, source='merged_docs', required=True, many=True,
                                         help_text=u'Список событий, входящих в группу'),
        'counters': fields.SerializerField(ClusterizeCountersSerializer,
                                           help_text=u'Количество уникальных значений полей ' + \
                                                     u'resource_type и resource_media_type')
    }
    visible_fields = ['total_count', 'group', 'events', 'counters']

    def prepare_object(self, obj):
        obj = super(ClusterizeGroupSerializer, self).prepare_object(obj)
        obj['group'] = {
            'event_group_key': obj['merged_docs'][0]['group_key'],
            'min_date': obj['min'],
            'max_date': obj['max'],
        }
        return obj


class ClusterizeActionSerializer(BaseHistoryActionSerializer):
    fields = {
        'total_count': fields.IntegerField(source='hitsCount', required=True, help_text=u'Общее кол-во групп'),
        'groups': fields.SerializerField(ClusterizeGroupSerializer, source='hitsArray', required=True, many=True,
                                         help_text=u'Список групп'),
    }
    visible_fields = ['total_count', 'groups', 'users', 'page_load_date']


class BaseHistoryEventListActionSerializer(BaseHistoryActionSerializer):
    fields = {
        'total_count': fields.IntegerField(source='hitsCount', required=True, help_text=u'Общее кол-во событий'),
        'events': fields.SerializerField(EventSerializer, source='hitsArray', required=True, many=True,
                                         help_text=u'Список событий'),
    }
    visible_fields = ['total_count', 'events', 'users', 'page_load_date']


class GroupActionSerializer(BaseHistoryEventListActionSerializer):
    pass


class SearchActionSerializer(BaseHistoryEventListActionSerializer):
    pass
