# -*- coding: utf-8 -*-
from mpfs.common import errors
from mpfs.common.util import public_urls
from mpfs.core.address import ResourceId
from mpfs.platform import exceptions
from mpfs.platform.exceptions import ForbiddenError, NotFoundError
from mpfs.platform.fields import StringField, SortField, ListField, ChoiceField, BaseField, BooleanField
from mpfs.platform.v1.disk.exceptions import (
    DiskAreaDoesNotExistsError, DiskPathFormatError, DiskMailAttachServiceFileIDFormatError,
    DiskPublicKeyFormatError,
    IncorrectResourceIdFormatError
)
from mpfs.platform.v1.disk.permissions import WebDavPermission
from mpfs.platform.v1.disk.validators import ForbiddenPathSymbolsValidator


class QuotedListField(ListField):
    def to_native(self, value):
        result = super(QuotedListField, self).to_native(value)
        return [i.strip('"') for i in result]


class PathField(StringField):
    area_separator = ':'
    default_errors = {
        'area_does_not_exist': DiskAreaDoesNotExistsError,
        'path_is_incorrect': DiskPathFormatError,
    }

    def __init__(self, default_area='disk', allowed_areas=None, *args, **kwargs):
        super(PathField, self).__init__(*args, **kwargs)
        self.default_area = default_area
        self.allowed_areas = allowed_areas or ('disk', 'app')

    def to_native(self, value):
        value = super(PathField, self).to_native(value)
        if value not in self.empty_values:
            path_chunks = value.split(self.area_separator, 1)
            if len(path_chunks) == 1:
                area = self.default_area
                path = path_chunks[0]
            else:
                area = path_chunks[0]
                path = path_chunks[1]
                if not path.startswith('/'):
                    raise self.errors['path_is_incorrect'](path=path)
            if not path.startswith('/'):
                path = '/%s' % path
            if area not in self.allowed_areas:
                raise self.errors['area_does_not_exist'](area=area)
            return self.area_separator.join((area, path,))
        return value


class MpfsPathField(PathField):
    """
    Gets platform path and converts it to MPFS-style path.

    Использует `request` для определения прав доступа к ограниченым разделам (`only_official_clients_areas`).
    """
    def __init__(self, only_official_clients_areas=(), *args, **kwargs):
        """
        :param only_official_clients_areas: Список разделов доступных только официальным клиентам.
        """
        path_validator = ForbiddenPathSymbolsValidator()
        kwargs['validators'] = kwargs.get('validators', [])
        kwargs['validators'].append(path_validator)

        super(MpfsPathField, self).__init__(*args, **kwargs)
        self.only_official_clients_areas = only_official_clients_areas

    def to_native(self, value):
        platform_path = super(MpfsPathField, self).to_native(value)
        if platform_path not in self.empty_values:
            # Ограничиваем некоторые AREAs для не официальных клиентов
            if self.only_official_clients_areas:
                area = platform_path.split(self.area_separator, 1)[0]
                if area in self.only_official_clients_areas:
                    try:
                        # 1ый parent - словарь `query`, где лежат все поля
                        # 2ой parent - объект-handler, где доступен `request`
                        WebDavPermission().has_permission(self.parent.parent.request)
                    except ForbiddenError:
                        raise self.errors['area_does_not_exist'](area=area)
            return '/%s%s' % tuple(platform_path.split(':', 1))
        return value

    def from_native(self, value):
        mpfs_path = super(MpfsPathField, self).from_native(value)
        if value not in self.empty_values:
            if not mpfs_path.startswith('/'):
                raw_address_id = mpfs_path
                mpfs_path = raw_address_id.split(':', 1)[1]
            area, path = self._split_path(mpfs_path)
            return '%s:/%s' % (area, path)
        return value

    @staticmethod
    def _split_path(path):
        chunks = path.lstrip('/').split('/', 1)
        area = chunks[0]
        path = ''
        if len(chunks) > 1:
            path = chunks[1]
        return area, path


class OrganizationPathField(MpfsPathField):
    ORGANIZATION_FOLDER = '/disk/.organization'

    def to_native(self, value):
        result = super(OrganizationPathField, self).to_native(value)
        return result.replace('/disk', self.ORGANIZATION_FOLDER)

    def from_native(self, value):
        return super(OrganizationPathField, self).from_native(value.replace(self.ORGANIZATION_FOLDER, '/disk'))


class PublicPathField(StringField):
    """
    Отрывает публичный ключ от пути подресурса при from_native
    """
    public_key_separator = ':'

    def to_native(self, value):
        path = super(PublicPathField, self).to_native(value)
        if not path.startswith('/'):
            path = '/%s' % path
        return path

    def from_native(self, value):
        mpfs_path = super(PublicPathField, self).from_native(value)
        path_chunks = mpfs_path.split(self.public_key_separator, 1)
        return path_chunks[-1] or '/'


class PublicKeyField(StringField):
    """
    Принимает снаружи public_key или public_url, внутрь отдаёт только public_key.
    Если надо, то вытаскивает public_key из public_url.
    """
    read_only = True
    unquote = True

    error_map = {
        404: NotFoundError,
        400: DiskPublicKeyFormatError,
    }

    def to_native(self, value):
        value = super(PublicKeyField, self).to_native(value)
        try:
            value = public_urls.any_url_to_private_hash(value)
        except errors.ClckNoResponse, e:
            code = e.data.get('code')
            exc = self.error_map.get(code)
            if exc is None:
                raise
            else:
                raise exc(data=e.data)
        return value


class MpfsSortField(SortField):
    def __init__(self, serializer_cls=None, parse_order=True, asc=1, desc=0, fields_mapping=None, *args, **kwargs):
        """
        :param serializer_cls: Сериализатор ресурсов которые сортируются,
                               например для файлов и папок -- это `ResourceSerializer`.
        :param bool parse_order: Доставить из поля направление сортировки или нет.
                                 Если True, то значение будет содержать dict вида {'field': 'key', 'order': 1},
                                 иначе -- строку вида `key`.
        :param asc: Значение возвращаемое в случае сортировки по возрастанию.
        :param desc: Значение возвращаемое в случае сортировки по убыванию (префикс `-`).
        :param dict fields_mapping: Маппинг входных значений на возвращаемые. Имеет бОльший приоритет чем сериализатор.
        """
        super(MpfsSortField, self).__init__(parse_order=parse_order, asc=asc, desc=desc, *args, **kwargs)
        self.serializer_cls = serializer_cls
        self.fields_mapping = fields_mapping

    def to_native(self, value):
        result = super(MpfsSortField, self).to_native(value)
        if isinstance(result, dict):
            raw_sort_by = result['field']
        else:
            raw_sort_by = result

        sort_by = None

        if self.fields_mapping:
            v = self.fields_mapping.get(raw_sort_by)
            if v:
                sort_by = v

        if not sort_by:
            field = None
            if self.serializer_cls:
                field = self.serializer_cls().get_fields().get(raw_sort_by)
            if field:
                source = field.source
                # Отрубаем префикс meta, если он встречается в source, т.к. MPFS сначала пытается найти поле сортировки
                # в корне ресурса, а если не находит, то на автомате ищет это поле в мете.
                if source:
                    source = source.split('.')[-1]
                    sort_by = source

        if not sort_by:
            sort_by = raw_sort_by

        if isinstance(result, dict):
            result['field'] = sort_by
            return result
        else:
            return sort_by

    def from_native(self, value):
        if isinstance(value, dict):
            sort_by = value['field']
            prefix = '' if value['order'] == 1 else '-'
        else:
            sort_by = value
            prefix = ''
        if self.serializer_cls:
            key, source = None, None
            for key, field in self.serializer_cls().get_fields().iteritems():
                source = field.source
                if source:
                    source = source.split('.')[-1]
                if sort_by == source:
                    return prefix + key
        return sort_by


class MpfsSortChoiseField(MpfsSortField, ChoiceField):
    """
    Для MpfsSortField добавляет ограничения от ChoiceField
    """
    def __init__(self, choices=None, *args, **kwargs):
        if choices:
            kwargs['choices'] = choices + ['-' + c for c in choices]
        ChoiceField.__init__(self, *args, **kwargs)
        MpfsSortField.__init__(self, *args, **kwargs)

    def to_native(self, value):
        return MpfsSortField.to_native(self, value)

    def from_native(self, value):
        return MpfsSortField.from_native(self, value)

    def validate(self, value):
        value = self.from_native(value)
        return ChoiceField.validate(self, value)


class RightsField(StringField):
    default = None
    unquote = False

    def from_native(self, value):
        value = super(RightsField, self).from_native(value)
        if value in self.empty_values:
            return self.default

        if value in ('660', 660):
            return 'rw'
        elif value in ('640', 640):
            return 'r'
        else:
            raise ValueError('Unknown permissions: %s' % value)


class MailAttachServiceFileIDField(StringField):
    service_file_id_sep = '/'
    default_errors = {
        'service_file_id_is_incorrect': DiskMailAttachServiceFileIDFormatError
    }

    def to_native(self, value):
        value = super(MailAttachServiceFileIDField, self).to_native(value)
        if value not in self.empty_values:
            chunks = value.split(self.service_file_id_sep)
            if len(chunks) != 2:  # mail_mid, mail_hid
                raise self.errors['service_file_id_is_incorrect'](service_file_id=value)
        return value


class ResourceIdField(StringField):

    def to_native(self, value):
        value = super(ResourceIdField, self).to_native(value)
        try:
            value = ResourceId.parse(value)
        except (ValueError, errors.AddressError):
            raise IncorrectResourceIdFormatError(resource_id=value)
        return value


class AntiVirusStatusField(BaseField):
    read_only = True

    mapping = {
        0: 'not-scanned',  # к примеру если поле отсутствует в базе у документа
        1: 'clean',
        2: 'infected',
        4: 'not-scanned',
    }

    default_errors = {
        'unsupported_value': exceptions.FieldWrongChoiceError,
    }

    def validate(self, value):
        if value not in self.mapping:
            raise self.errors['unsupported_value'](choices=self.mapping.keys())

    def from_native(self, value):
        value = super(AntiVirusStatusField, self).from_native(value)
        return self.mapping[value]


class AutouploadFlagField(BooleanField):
    read_only = True
