# -*- coding: utf-8 -*-
"""API Яндекс Диска"""
import sys
import urllib
import urlparse

from datetime import datetime
from dateutil.relativedelta import relativedelta

from mpfs.common.errors import DjfsApiProxyError
from mpfs.common.errors.platform import MpfsProxyBadResponse
from mpfs.common.util import from_json, filter_value_by_percentage
from mpfs.common.util.video_unlim import filter_users_experiment, RKUB, is_enabled_users_experiment
from mpfs.config import settings
from mpfs.config.constants import DISK_ANDROID_VIDEOUNLIM_ALERT_FILE_PATH
from mpfs.common import errors
from mpfs.common.static import tags
from mpfs.common.static.codes import (
    WH_USER_NEED_INIT,
    PASSPORT_PASSWORD_NEEDED,
    USER_BLOCKED,
    OWNER_HAS_NO_FREE_SPACE,
    FOLDER_DELETION_BY_RESOURCE_ID_FORBIDDEN,
    PRECONDITIONS_FAILED,
    MD5_CHECK_NOT_SUPPORTED,
    CLIENT_BAD_REQUEST,
    REQUESTS_LIMIT_EXCEEDED_429,
    API_RESTRICTED_FOR_OVERDRAFT_USER,
    ALBUM_NOT_FOUND,
    ALBUMS_UNABLE_TO_APPEND_ITEM,
    ALBUM_ALREADY_EXISTS,
    ALBUMS_UNABLE_TO_DELETE,
    ALBUMS_UNABLE_TO_PUBLISH_UNSAVED,
    OFFICE_IS_NOT_ALLOWED,
    MOVE_WRONG_DESTINATION,
    FOLDER_TOO_DEEP,
    GROUP_NOT_PERMIT,
    VIDEO_STREAMING_UNPROCESSABLE_ENTITY,
    UPLOAD_TRAFFIC_LIMIT_EXCEEDED,
    UPLOAD_TRAFFIC_LIMIT_WITHOUT_DETAILS,
    UPLOAD_FILE_SIZE_LIMIT_EXCEEDED,
)
from mpfs.core.office.util import make_resource_id
from mpfs.core.services.djfs_api_service import djfs_api
from mpfs.core.services.hbf_service import HbfService
from mpfs.core.services.passport_service import passport
from mpfs.core.services.video_service import video
from mpfs.core.services.uaas_service import new_uaas
from mpfs.core.user.constants import (
    DEFAULT_FOLDERS,
    PHOTOUNLIM_AREA,
    PUBLIC_UID,
    DISK_AREA_PATH,
    ATTACH_AREA,
    PHOTOUNLIM_AREA_PATH,
    NOTES_AREA_PATH,
    CLIENT_AREA_PATH,
    DISK_AREA,
)
from mpfs.engine.process import get_error_log, get_default_log
from mpfs.platform.async_mode import gevent_local_variables
from mpfs.platform.common import ResponseObject, PlatformUser
from mpfs.platform.exceptions import (
    NotFoundError, UnauthorizedError, ForbiddenError,
    ServiceUnavailableError, FieldRequiredError, UnavailableForLegalReasons, UserNotFoundInPassportError,
    TooManyRequestsError, VersionNotFoundError, BadRequestError, FieldValidationError, UnsupportedMediaTypeError,
    FieldsAllOrNoneValidationError)
from mpfs.platform.handlers import (
    ServiceProxyHandler,
    ETagHandlerMixin,
    CheckUserKarmaHandlerMixin,
    GetResouceInfoByIDMixin,
    BasePlatformHandler
)
from mpfs.core.services.mpfsproxy_service import mpfsproxy
from mpfs.core.services.notifier_service import notifier
from mpfs.core.services.smartcache_service import smartcache
from mpfs.common.static import messages
from mpfs.common.util import from_json, to_json, hashed
from mpfs.common.util.filetypes import builtin_extensions
from mpfs.common.util.mobile_client import MobileClientVersion
from mpfs.common.util.urls import update_qs_params
from mpfs.common.util.user_agent_parser import UserAgent, UserAgentParser
from mpfs.common.util.ycrid_parser import YcridParser, YcridPlatformPrefix
from mpfs.platform import fields
from mpfs.platform import validators
from mpfs.platform.auth import OriginAuth, PassportCookieAuth, UserAgentAuth, TVM2Auth
from mpfs.platform.permissions import DenyAllPermission, AllowAllPermission, AllowByClientIdPermission
from mpfs.platform.rate_limiters import (
    PerUserRateLimiter, PerHandlerRandomUserRateLimiter, PerSomethingUniqueLimiter, PerPercentUserHandlerRateLimiter
)
from mpfs.platform.v1.disk import exceptions
from mpfs.platform.v1.disk.fields import (
    MpfsPathField, PublicKeyField, PathField, MpfsSortField, QuotedListField,
    OrganizationPathField, MpfsSortChoiseField, MailAttachServiceFileIDField,
    ResourceIdField
)
from mpfs.platform.v1.disk.permissions import (
    DiskReadPermission,
    DiskWritePermission,
    DiskAppFolderPermission,
    DiskPartnerPermission,
    DiskVideoPublicPermission,
    DiskInfoPermission,
    WebDavPermission,
    DiskSearchPermission,
    DiskAttachWritePermission,
    MobileMailPermission
)
from mpfs.platform.v1.notes.permissions import NotesReadPermission, NotesWritePermission
from mpfs.platform.v1.disk.rate_limiters import PerUploadExternalHostRateLimiter
from mpfs.platform.v1.disk.serializers import (
    ResourceSerializer, LinkSerializer, OperationStatusSerializer,
    OperationListSerializer, ShareProvidersListSerializer,
    PublicResourceSerializer, DiskSerializer,
    LastUploadedResourceListSerializer, FilesResourceListSerializer,
    VideoStreamListSerializer, ResourcePatchSerializer,
    PublicResourcesListSerializer, PhotosliceSnapshotLinkSerializer,
    PhotosliceSnapshotSerializer, PhotosliceDeltaListSerializer,
    PhotoAlbumsListSerializer, PhotoAlbumSerializer,
    PhotoAlbumNewSerializer, PhotoAlbumItemNewSerializer,
    PhotoAlbumItemSerializer, PhotoAlbumPatchSerializer,
    PhotoAlbumItemsListSerializer, OrganizationsListSerializer,
    OrganizationSerializer, OrganizationResourceSerializer,
    UploadResourceSerializer, TrashResourceSerializer,
    ResourceDimensionsSerializer, SnapshotSerializer, SnapshotIterationKeySerializer,
    DeltasSerializer, DeltaResourceSerializer, ResourceUploadLinkSerializer,
    ResourceListSerializer, PhotounlimLastModifiedResourceListSerializer,
    ClientsConfigSerializer, ClientsInstallerSerializer, ExperimentsSerializer,
    UserFeatureTogglesSerializer, SearchResourceListSerializer,
    ClientBodySerializer, UserActivityInfoSerializer, ExclusionsFromGeneratedAlbumsItemsSerializer,
    SearchResourceSerializer, ResourceVersionsSerializer, ResourceVersionSerializer, PhotoAlbumWithItemsSerializer,
    PhotoAlbumItemsNewSerializer, OfficeOnlineEditorURLSerializer, OnlyOfficeCallbackBodySerializer,
    ClientsInstallerWithAutologonSerializer, OfficeFileFiltersSerializer, OfficeFileURLsSerializer,
    SetPublicSettingsSerializer
)
from mpfs.platform.v1.disk.exceptions import (
    DiskUserBlockedError,
    DiskNotFoundError,
    DiskOfficeUnsupportedExtensionError,
    DiskOfficeIsDisabledError,
    DiskNoWritePermissionForSharedFolderError,
)

USER_DEFAULT_TLD = settings.user['default_tld']

PLATFORM_DISK_APPS_IDS = settings.platform['disk_apps_ids']
PLATFORM_PS_APPS_IDS = settings.platform['ps_apps_ids']
PLATFORM_MTS_TV_APP_ID = settings.platform['mts_tv_app_id']
PLATFORM_SKIP_CHECK_SPACE_ALLOWED_CLIENT_IDS = settings.platform['skip_check_space']['allowed_client_ids']
PLATFORM_VIDEO_NEW_PERCENTAGE = settings.platform['video']['new_percentage']
PLATFORM_VIDEO_NEW_VIDEO_ENABLED_FOR_UIDS = settings.platform['video']['new_video_enabled_for_uids']

SNAPSHOT_PER_HANDLER_RATE_LIMITER_ENABLED = settings.snapshot['per_handler_rate_limiter']['enabled']
SNAPSHOT_PER_HANDLER_RATE_LIMITER_USERS_COUNT = settings.snapshot['per_handler_rate_limiter']['users_count']

DELTAS_PER_HANDLER_RATE_LIMITER_ENABLED = settings.deltas['per_handler_rate_limiter']['enabled']
DELTAS_PER_HANDLER_RATE_LIMITER_USERS_COUNT = settings.deltas['per_handler_rate_limiter']['users_count']
DELTAS_PER_USER_HANDLER_LIMITER_PERCENT = settings.deltas['per_user_handler_limiter_percent']

SOFTWARE_INSTALLER_PATH = settings.platform['software_installer']['path']
SOFTWARE_INSTALLER_WITH_AUTOLOGON_PATH = settings.platform['software_installer']['with_autologon']['path']
SOFTWARE_INSTALLER_TEST_BASE_URL = settings.platform['software_installer']['test_builds_base_url']
SOFTWARE_INSTALLER_ALLOWED_NETWORK_MACROS = settings.platform['software_installer']['allowed_network_macros']
PLATFORM_EXCLUDE_EXPS = [{'testids': exp_settings['testids'],
                          'for_uids': set(exp_settings['for_uids'])}
                         for exp_settings in settings.platform['exclude_exps']]
JAVA_DJFS_API_PROXY_PLATFORM_BULK_INFO_ENABLED = settings.java_djfs_api['proxy_platform']['bulk_info']['enabled']


JAVA_DJFS_API_PROXY_PLATFORM_INFO_ENABLED = settings.java_djfs_api['proxy_platform']['info']['enabled']

PLATFORM_TELEMOST_ENABLED_EXPERIMENTS_TEST_IDS = settings.platform['telemost']['experiments']['testids']

FEATURE_TOGGLES_BLOCK_PUBLIC_RESOURCE_DOWNLOAD = settings.feature_toggles['block_public_resource_download']

error_log = get_error_log()
default_log = get_default_log()


def _auto_initialize_user(func):
    def wrapper(handler, *args, **kwargs):
        try:
            return func(handler, *args, **kwargs)
        except handler.service_base_exception, e:
            if handler.get_mpfs_error_code(e) == WH_USER_NEED_INIT:
                if not handler.need_auto_initialize_user:
                    raise exceptions.DiskUserNotFoundError()
                try:
                    handler.init_user(handler.request.user.uid)
                    return func(handler, *args, **kwargs)
                except errors.PassportBadResult, e:
                    raise ServiceUnavailableError()
                except (errors.PassportTokenExpired, errors.PassportCookieInvalid), e:
                    raise UnauthorizedError()
                except handler.service_base_exception, e:
                    if handler.get_mpfs_error_code(e) == PASSPORT_PASSWORD_NEEDED:
                        raise exceptions.DiskUnsupportedUserAccountTypeError()
                    raise
            elif handler.get_mpfs_error_code(e) == USER_BLOCKED:
                # Случай, если пользователь заблокирован
                raise DiskUserBlockedError()
            elif handler.get_mpfs_error_code(e) == API_RESTRICTED_FOR_OVERDRAFT_USER:
                raise exceptions.DiskAPIDisabledForOverdraftUserError()
            elif handler.get_mpfs_error_code(e) == PASSPORT_PASSWORD_NEEDED:
                raise exceptions.DiskUnsupportedUserAccountTypeError()
            raise e.__class__, e, sys.exc_info()[2]
    return wrapper


class MpfsProxyHandler(CheckUserKarmaHandlerMixin, ServiceProxyHandler):
    service = mpfsproxy
    service_base_exception = errors.platform.MpfsProxyBadResponse
    service_url = None
    """
    MPFS handler URL pattern.
    You can use all Args and Body parameters as named mapping keys with any available conversion types.
    If you need more complicated logic to create URL, then override get_url method.
    """
    error_map = {
        109: exceptions.DiskNoWritePermissionForSharedFolderError,
        API_RESTRICTED_FOR_OVERDRAFT_USER: exceptions.DiskAPIDisabledForOverdraftUserError,
        404: exceptions.DiskNotFoundError,
        503: exceptions.DiskServiceUnavailableError,
        UPLOAD_TRAFFIC_LIMIT_EXCEEDED: exceptions.DiskUploadTrafficLimitExceeded,
        UPLOAD_TRAFFIC_LIMIT_WITHOUT_DETAILS: exceptions.DiskUserReadOnly,
        UPLOAD_FILE_SIZE_LIMIT_EXCEEDED: exceptions.DiskUploadFileSizeLimitExceeded,
    }
    user_init_url = '/json/user_init?uid=%(uid)s&locale=%(locale)s&source=%(source)s'
    user_init_default_locale = 'ru'
    auth_methods = [PassportCookieAuth()]

    service_client_id_to_name_map = {c.get('oauth_client_id'): c['name'] for c in settings.platform['auth']}
    need_auto_initialize_user = True

    """Пытаться ли проинициализировать юзера в диске, если он отсутствует"""

    def init_user(self, uid):
        """Инициализирует пользователя Диска если тот ещё не проинициализирован"""
        # Идём в паспорт за локалью пользователя
        user_info = passport.userinfo(uid)

        # Если пользователя такого нет, то швыряем 401
        if not user_info.get('uid'):
            if self.request.mode == tags.platform.EXTERNAL:
                raise UnauthorizedError()
            else:
                raise UserNotFoundInPassportError()

        locale = user_info.get('language', None) or self.user_init_default_locale

        # Дёргаем MPFS чтоб проинициализировал пользователя
        context = {'uid': uid, 'locale': locale, 'source': self._get_source()}
        urlencoded_context = self.urlencode_context(context)
        url = '%s%s' % (mpfsproxy.base_url, self.user_init_url % urlencoded_context)

        self.request_service(url)
        self.request.user_initialized = True

    def handle_exception(self, exception):
        if isinstance(exception, DjfsApiProxyError):
            platform_exception = self.get_platform_exception(exception)
            if platform_exception:
                # re-raise exception preserving original stack trace and value
                raise platform_exception, None, sys.exc_info()[2]
        return super(MpfsProxyHandler, self).handle_exception(exception)

    def get_mpfs_error_code(self, exception):
        ex_data = getattr(exception, 'data', None)
        if isinstance(ex_data, dict):
            raw_mpfs_resp = ex_data.get('text', '{}')
            mpfs_resp = from_json(raw_mpfs_resp)
            return mpfs_resp.get('code', None)
        return None

    @_auto_initialize_user
    def handle(self, request, *args, **kwargs):
        return super(MpfsProxyHandler, self).handle(request, *args, **kwargs)

    def raw_request_service(self, url, method=None, headers=None, data=None, service=None):
        user_ip = self.request.get_real_remote_addr()

        if not headers:
            headers = {}
        headers.update({'X-Real-Ip': user_ip})

        if self.request.mode == tags.platform.INTERNAL:
            url = update_qs_params(url, {'skip_overdraft_check': '1'})

        return super(MpfsProxyHandler, self).raw_request_service(url, method, headers, data, service)

    def build_url(self, url, context=None, base_url=None):
        url = super(MpfsProxyHandler, self).build_url(url, context=context, base_url=base_url)
        if self.serializer_cls and issubclass(self.serializer_cls, ResourceSerializer):
            return self._get_service_url_for_resource_serializer(url, context)
        return url

    def _get_parsed_response_from_exc(self, exception):
        data = getattr(exception, 'data', None)
        if data and isinstance(data, dict):
            response_text = data.get('text', None)

            if response_text is None:
                return None

            try:
                parsed_response = from_json(response_text)
            except Exception:
                return None

            return parsed_response
        return None

    def get_json_response_error_code(self, exception):
        code = None

        data = getattr(exception, 'data', None)
        if data and isinstance(data, dict):
            response_text = data.get('text', None)

            if response_text is None:
                return None

            try:
                parsed_response_text = from_json(response_text)
            except Exception:
                return None

            code = parsed_response_text.get('code', None)
            if code and isinstance(code, (str, unicode)) and code.isdigit():
                code = int(code)

        return code

    def get_http_error_code(self, exception):
        code = None
        data = getattr(exception, 'data', None)
        if data and isinstance(data, dict):
            code = data.get('code', None)
            if code and isinstance(code, (str, unicode)) and code.isdigit():
                code = int(code)
        return code

    def get_service_error_code(self, exception):
        code = self.get_json_response_error_code(exception)
        if code is not None and code in self.get_error_map():
            return code

        code = self.get_http_error_code(exception)
        return code

    def get_exception_context(self, context=None, exception=None):
        exc_context = super(MpfsProxyHandler, self).get_exception_context(context)

        if self.get_mpfs_error_code(exception) in (UPLOAD_TRAFFIC_LIMIT_EXCEEDED,
                                                   UPLOAD_FILE_SIZE_LIMIT_EXCEEDED):
            parsed = self._get_parsed_response_from_exc(exception)
            if parsed and parsed.get('title'):
                try:
                    title = from_json(parsed['title'])
                except Exception:
                    title = None
                if title:
                    for k, v in title.iteritems():
                        exc_context[k] = v

        return exc_context

    def _get_source(self):
        """Получить источник регистрации."""
        client_id = self.request.client.id
        if client_id in self.service_client_id_to_name_map:
            source = 'rest_api_%s' % self.service_client_id_to_name_map[client_id]
        else:
            source = 'rest_api_other'
        return source

    def _get_fields_to_filter(self):
        fields = super(MpfsProxyHandler, self)._get_fields_to_filter()
        fields_from_request = []
        extra_default_visible_fields = []
        for field in fields:
            if field[0] == '+':
                extra_default_visible_fields.append(field[1:])
                continue
            fields_from_request.append(field)

        # Если нет строгой фильтрации по филдам, то должны вернуть все видимые филды, поэтому None
        visible_fields = (fields_from_request + extra_default_visible_fields if fields_from_request
                          else None)

        return visible_fields

    def _get_service_url_for_resource_serializer(self, url, context):
        meta_to_request_explicitly = set()
        for field in context.get('fields', []):
            if field[0] == '+':
                field = field[1:]
            if self.serializer_cls and field in self.serializer_cls.explicitly_requested_meta_fields:
                meta_to_request_explicitly.add(field)

        if not meta_to_request_explicitly:
            return url
        parsed_url = urlparse.urlsplit(url)
        query = urlparse.parse_qs(parsed_url.query)
        query['meta'] = ','.join([query['meta'][0],
                                  ','.join(meta_to_request_explicitly)])
        url = urlparse.urlunsplit(parsed_url._replace(query=urllib.urlencode(query, doseq=True)))
        return url


class GetDiskHandler(MpfsProxyHandler):
    """Получить метаинформацию о диске пользователя"""
    permissions = DiskInfoPermission() | DiskReadPermission() | WebDavPermission()
    user_info_url = '/json/user_info?uid=%(uid)s'
    serializer_cls = DiskSerializer

    error_map = {
        403: exceptions.DiskUserBlockedError,
        API_RESTRICTED_FOR_OVERDRAFT_USER: exceptions.DiskAPIDisabledForOverdraftUserError,
    }

    @_auto_initialize_user
    def handle(self, request, *args, **kwargs):
        context = self.get_context()
        result = {}
        # user_info
        user_info_url = self.build_url(self.user_info_url, context)
        result['user_info'] = self.request_service(user_info_url, self.service_method)
        # user
        user_info = passport.userinfo(self.request.user.uid)
        result['user'] = {
            'login': request.user.login,
            'uid': request.user.uid,
        }
        for field in ('country', 'display_name', 'avatar', 'has_staff'):
            if field in user_info and user_info[field] is not None:
                result['user'][field] = user_info[field]
        return self.serialize(result)


class GetClientsConfigHandler(MpfsProxyHandler):
    """Получить настройки клиента ПО"""
    permissions = WebDavPermission()
    serializer_cls = ClientsConfigSerializer

    error_map = {
        403: exceptions.DiskUserBlockedError,
        API_RESTRICTED_FOR_OVERDRAFT_USER: exceptions.DiskAPIDisabledForOverdraftUserError,
    }

    def handle(self, request, *args, **kwargs):
        user_info = passport.userinfo(request.user.uid)

        config_db_prefix = ''
        if user_info.get('has_staff'):
            config_db_prefix = 'staff'

        result = {
            'config_db_prefix': config_db_prefix,
        }

        return self.serialize(result)


@gevent_local_variables('_app_folder_path', '_app_folder_exists')
class MpfsProxyWithAppFolderHandler(MpfsProxyHandler):
    """
    Работает всё это так:
      1. В самом начале обработки запроса в методе initialize перед вызовом родительского диспатча мы получаем путь к
         папке приложения и добавляем в request новый атрибут app_folder_path, который необходим .
      2. Перед обработкой querystring в методе process_query мы резолвим все пути начинающиеся с app:/ относительно
         папки приложения. В итоге из "app:/app_file.ext" получаются пути вида "disk:/Приложения/my_app/app_file.ext",
         которые потом благополучно резолвятся MpfsPathField'ами в нормальные MPFS'ные пути.
      3. Дальше штатно отрабатывает проверка разрешений.
      4. В начало обработки запроса (метод handle) с помощью декортатора добавляется автосоздание папки приложения,
         если она фигурирует в querystring.
    """
    app_area = 'app'
    app_folder_name_separator = '_'
    auto_create_app_folder = True
    auto_resolve_app_folder = True
    _app_folder_handle_method_wrapped = False
    _app_folder_path = None
    _app_folder_exists = False

    def get_auto_create_app_folder(self):
        if self.request.client is not None and self.request.client.id in PLATFORM_DISK_APPS_IDS:
            return False
        return self.auto_create_app_folder

    def get_auto_resolve_app_folder(self):
        if self.request.client is not None and self.request.client.id in PLATFORM_DISK_APPS_IDS:
            return False
        return self.auto_resolve_app_folder

    # Переопределение стандартных методов хэндлера

    def initialize(self, request, *args, **kwargs):
        self._app_folder_path = None
        self._app_folder_exists = False
        super(MpfsProxyWithAppFolderHandler, self).initialize(request, *args, **kwargs)
        if self.get_auto_create_app_folder():
            self.request.app_folder_path = self.get_app_folder_path()

    def clean_query(self, raw_query):
        processed_q = {}
        for name, field in self.query.get_fields().iteritems():
            source = getattr(field, 'source', None) or name
            if source in raw_query:
                val = raw_query.get(source)
                if (isinstance(field, PathField) and
                        isinstance(val, (str, unicode)) and
                        self.get_auto_resolve_app_folder()):
                    val = self.resolve_path(val)
                processed_q[source] = val
        return super(MpfsProxyWithAppFolderHandler, self).clean_query(processed_q)

    def __init__(self, *args, **kwargs):
        super(MpfsProxyWithAppFolderHandler, self).__init__(*args, **kwargs)
        if not self._app_folder_handle_method_wrapped:
            # К моменту вызова `self.handle()` проверки прав доступа уже пройдены и соответственно у приложения есть
            # права на папку приложения, если она фигурирует в запросе. Поэтому можно и нужно её создать если её нет.
            self.handle = self.auto_create_app_folder_on_method(self.handle)
            self._app_folder_handle_method_wrapped = True

    # Собственные методы класса

    def auto_create_app_folder_on_method(self, func):
        def wrapper(*args, **kwargs):
            if not self._app_folder_exists and self.get_auto_create_app_folder():
                for name, field in self.query.get_fields().iteritems():
                    val = self.request.query.get(name)
                    if isinstance(field, PathField) and val:
                        if val.startswith(self.get_app_folder_path()):
                            self.create_app_folder(self.get_app_folder_path())
                            self._app_folder_exists = True
                            # Достаточно создать папку один раз
                            break
            return func(*args, **kwargs)
        return wrapper

    def create_apps_folder(self):
        url = self.build_url(
            '/json/mksysdir?uid=%(uid)s&type=applications',
            {'uid': self.request.user.uid}
        )
        self.request_service(url)

    def bind_folder_to_app(self, uid, path, oauth_app_id):
        """Устанавливает мета-атрибут папки закрепляющий её за приложением"""
        url = self.build_url(
            '/json/setprop?uid=%(uid)s&path=%(path)s&oauth_app_id=%(oauth_app_id)s',
            {'uid': uid, 'path': path, 'oauth_app_id': oauth_app_id}
        )
        self.request_service(url)

    def create_app_folder(self, path):
        """Создаёт папку и закрепляет её за приложением."""
        uid = self.request.user.uid
        oauth_app_id = self.request.client.id
        self.create_apps_folder()
        url = self.build_url(
            '/json/mkdir?uid=%(uid)s&path=%(path)s',
            {'uid': uid, 'path': path}
        )
        self.request_service(url)
        self.bind_folder_to_app(uid, path, oauth_app_id)

    def build_app_folder_name(self, long=False):
        """
        Формирует имя папки приложения

        Если имя приложения пустое, то и короткое и длинное имя приложения будут равны его OAuth App Id.

        :param long: Если True, то имя папки будет <OAuth имя приложения>-<OAuth App Id приложения>.
        :rtype: str
        """
        if long:
            long_name_parts = (self.request.client.name, self.request.client.id)
            return self.app_folder_name_separator.join(long_name_parts)
        else:
            return self.request.client.name

    def build_app_folder_path(self, apps_folder_path, long=False):
        return '%s/%s' % (apps_folder_path.rstrip('/'), self.build_app_folder_name(long=long))

    @_auto_initialize_user
    def get_app_folder_path(self, cut_prefix=False):
        # если папка приложения ещё не определена и пользователь авторизован
        if not self._app_folder_path and self.request.user and self.request.user.uid:
            default_folders = self.service.default_folders(self.request.user.uid)
            apps_folder_path = default_folders.get('applications')

            # Пытаемся найти папку приложения по-хорошему, с помощью фильтра по oauth_app_id.
            try:
                url = self.build_url(
                    '/json/list?uid=%(uid)s&path=%(path)s&meta=oauth_app_id&oauth_app_id=%(oauth_app_id)s',
                    {'uid': self.request.user.uid, 'path': apps_folder_path, 'oauth_app_id': self.request.client.id}
                )
                # Первой в результате list всегда идёт сама папка по которой делали list. Сразу отбрасываем её.
                app_folders = self.request_service(url)[1:]
            except self.service_base_exception, e:
                if int(e.data['code']) == 404:
                    # Если у пользователя нет папки Приложения.
                    self._app_folder_path = self.build_app_folder_path(apps_folder_path)
                else:
                    # Иначе пробрасываем исключение дальше, пусть наверху разбираются.
                    raise e
            else:
                if app_folders:
                    # Если папка приложения нашлась без проблем.
                    self._app_folder_path = app_folders[0]['path']
                    self._app_folder_exists = True
                else:
                    # Иначе, если по-хорошему папка приложения находиться не захотела.
                    # Ищем по-плохому и делаем так чтоб в следующий раз она находилась по-хорошему.
                    # Выгребаем с помощью list все папки в папке приложений пользователя.
                    # Не используем tree, т.к. эта ручка не отдаёт мета-атрибут oauth_app_id.
                    url = self.build_url(
                        '/json/list?uid=%(uid)s&path=%(path)s&meta=oauth_app_id',
                        {'uid': self.request.user.uid, 'path': apps_folder_path, 'oauth_app_id': self.request.client.id}
                    )
                    # Первой в результате list всегда идёт сама папка по которой делали list. Сразу отбрасываем её.
                    apps_folder_resources = self.request_service(url)[1:]
                    app_folder_short_name_occupied = False
                    # Сначала ищем папки с именем приложения потом ищем папки оканчивающиеся на id приложения.
                    is_short_app_name = lambda name: name == self.build_app_folder_name()
                    is_long_app_name = lambda name: \
                        len(name.rsplit(self.app_folder_name_separator, 1)) > 1 \
                            and name.rsplit(self.app_folder_name_separator, 1) == self.request.client.id
                    for is_acceptable in (is_short_app_name, is_long_app_name):
                        for r in apps_folder_resources:
                            name = r['name']
                            if is_acceptable(name):
                                # Нашли подходящую папку.
                                if is_short_app_name(name):
                                    app_folder_short_name_occupied = True
                                # Проверяем не занята ли она другим приложением.
                                if 'oauth_app_id' not in r.get('meta', {}):
                                    # Если не занята, то помечаем её как папку нашего приложения,
                                    # чтоб в следующий раз она находилась сразу по-хорошему.
                                    self.bind_folder_to_app(self.request.user.uid, r['path'], self.request.client.id)
                                    self._app_folder_path = r['path']
                                    self._app_folder_exists = True
                                    break
                        if self._app_folder_path:
                            break

                    if not self._app_folder_path:
                        # Если так ничего и не нашли, то возвращаем какое у папки должно быть имя
                        self._app_folder_exists = False
                        if not app_folder_short_name_occupied:
                            self._app_folder_path = self.build_app_folder_path(apps_folder_path)
                        else:
                            self._app_folder_path = self.build_app_folder_path(apps_folder_path, long=True)

        if not cut_prefix:
            if self._app_folder_path:
                return self._app_folder_path.rstrip('/')
            else:
                # Если каким-то чудом всё таки оказалось None
                return self._app_folder_path
        else:
            return self._app_folder_path[len('/disk'):].rstrip('/')

    def resolve_path(self, path):
        """Resolve "app:/*" paths to "disk:/Applications/<app folder name>/*" paths"""
        app_area_prefix = '%s:/' % self.app_area
        app_path = self.get_app_folder_path(cut_prefix=True)
        platform_app_path = u'disk:%s/' % app_path
        if path.startswith(app_area_prefix):
            return u'%s%s' % (platform_app_path, path[len(app_area_prefix):])
        return path

    def unresolve_path(self, path):
        """Unresolve "disk:/Applications/<app folder name>/*" paths to "app:/*" paths"""
        app_area_prefix = '%s:/' % self.app_area
        app_path = self.get_app_folder_path(cut_prefix=True)
        platform_app_path = u'disk:%s/' % app_path
        if path.startswith(platform_app_path):
            return u'%s%s' % (app_area_prefix, path[len(platform_app_path):])
        return path


class ListVideoStreamsHandler(MpfsProxyWithAppFolderHandler):
    """
    Получить список потоков для стримминга
    """
    auth_user_required = True
    permissions = AllowByClientIdPermission(PLATFORM_DISK_APPS_IDS + [PLATFORM_MTS_TV_APP_ID])

    service_url = '/json/video_url?uid=%(uid)s&path=%(path)s'
    video_streaming_url = '/json/video_streams?uid=%(uid)s&path=%(path)s&use_http=%(use_http)s&user_ip=%(user_ip)s&client_id=%(client_id)s'
    album_video_streaming_url = '/json/album_video_streams?uid=%(uid)s&album_id=%(album_id)s&item_id=%(item_id)s&user_ip=%(user_ip)s&use_http=%(use_http)s&client_id=%(client_id)s'
    query = fields.QueryDict({
        'path': MpfsPathField(required=False, allowed_areas=('disk', 'app', ATTACH_AREA, PHOTOUNLIM_AREA), help_text=u'Путь к ресурсу.'),
        'album_id': fields.StringField(required=False, help_text=u'ID альбома'),
        'item_id': fields.StringField(required=False, help_text=u'ID видео в альбоме'),
        'no-redirects': fields.BooleanField(default=False, required=False, help_text=u'true, если клиент не поддерживает редиректы.'),
        'http-only': fields.BooleanField(default=False, required=False, help_text=u'true, если клиент не поддерживает https.'),
    })
    serializer_cls = VideoStreamListSerializer

    ya_video_formats_map = {
        'mp4-standard-quality': ('mp4', '360p', 'sq.mp4', 'H.264', 'AAC'),
        'mpeg4-low': ('mp4', '360p', 'm450x334.mp4', 'H.264', 'AAC'),
        'mpeg4-med': ('mp4', '480p', 'medium.mp4', 'H.264', 'AAC'),
        'mpeg4-high': ('mp4', '720p', 'm1280x720.mp4', 'H.264', 'AAC'),
        'mp4-480p': ('mp4', '480p', '480p.mp4', 'H.264', 'AAC'),
        'mp4-720p': ('mp4', '720p', '720p.mp4', 'H.264', 'AAC'),
        'hls-360p': ('hls', '360p', '360p.m3u8', 'H.264', 'AAC'),
        'hls-480p': ('hls', '480p', '480p.m3u8', 'H.264', 'AAC'),
        'hls-720p': ('hls', '720p', '720p.m3u8', 'H.264', 'AAC'),
    }

    hls_const_params = {
        'container': 'hls',
        'video_codec': 'H.264',
        'audio_codec': 'AAC'
    }

    error_map = {
        VIDEO_STREAMING_UNPROCESSABLE_ENTITY: exceptions.VideoStreamingUnprocessableEntity
    }

    @_auto_initialize_user
    def handle(self, request, *args, **kwargs):
        return self._handle(request, *args, **kwargs)

    def _handle(self, request, *args, **kwargs):
        if self._use_old_video_streaming():
            resp = self._old_video_streaming()
        else:
            resp = self._new_video_streaming()
        return self.serialize(resp)

    def _use_old_video_streaming(self):
        if PLATFORM_VIDEO_NEW_PERCENTAGE <= 0:
            return True
        if PLATFORM_VIDEO_NEW_PERCENTAGE >= 100:
            return False
        group_field = self._get_group_field()
        if group_field in PLATFORM_VIDEO_NEW_VIDEO_ENABLED_FOR_UIDS:
            return False
        group_num = int(hashed(group_field), 16) % 100
        if group_num < PLATFORM_VIDEO_NEW_PERCENTAGE:
            return False
        return True

    def _get_group_field(self):
        return self.request.user.uid

    def _new_video_streaming(self):
        context = self.get_context()
        context['use_http'] = int(context['http-only'])
        context['user_ip'] = self.request.get_real_remote_addr(prefer_x_forwarded_for=False)
        context['client_id'] = self.request.client.id
        # это нужно для публичной ручки
        if 'uid' not in context or not context['uid']:
            context['uid'] = ''

        if 'album_id' in context and context['album_id'] and 'item_id' in context and context['item_id']:
            url = self.build_url(self.album_video_streaming_url, context=context)
        elif ('path' in context and context['path']) or ('public_key' in context and context['public_key']):
            url = self.build_url(self.video_streaming_url, context=context)
        else:
            raise FieldRequiredError()
        return self.request_service(url)

    def _old_video_streaming(self):
        # TODO после полного перехода на новый видеостриминг - удалить
        url = self.get_url(self.get_context())
        mpfs_resp = self.request_service(url, method=self.service_method)

        host = mpfs_resp['host']
        stream = mpfs_resp['stream']
        stream_short = mpfs_resp.get('stream_short', None)

        url = host.replace('https', 'http') + '/video-info?stream=' + stream
        resp = self.request_service(url, method=self.service_method, service=video)
        resp['streamId'] = stream_short or stream
        resp['host'] = host
        resp['no-redirects'] = self.request.query['no-redirects']
        resp['http-only'] = self.request.query['http-only']

        items = {'hls': {}, 'mp4': {}}

        stream_id = resp['streamId']
        streaming_host = resp['host']
        transcoder_direct_url = resp.get('forceTranscoderHlsHost', None)
        no_redirects = resp['no-redirects']

        dimensions = resp.get('dimensions', {})

        http_only = resp['http-only']
        protocol = 'http' if http_only else 'https'

        if http_only:
            # https://st.yandex-team.ru/CHEMODAN-23338
            # Лучше бы чтобы ссылки на http возвращали сами бэкэнды
            # Но сейчас надо сделать быстро
            # И мы рассчитываем на то, что SmartTV все-таки научится играть по https и этот костыль мы уберем
            transcoder_direct_url = transcoder_direct_url.replace("https://", "http://")
            transcoder_direct_url = transcoder_direct_url.replace(":443", "")

            streaming_host = streaming_host.replace("https://", "http://")

        for dimension in dimensions.keys():
            if no_redirects and transcoder_direct_url:
                link = transcoder_direct_url \
                       + "/txc/stream-hls/" + stream_id \
                       + "/HLS/" + dimension \
                       + "/0/playlist.m3u8?force-init=true"
            else:
                link = streaming_host \
                       + "/hls-playlist/" + stream_id \
                       + "/" + "playlist-redirect.m3u8?dimension=" + dimension

            items['hls'][dimension] = {
                'links': {
                    protocol: link
                },
                'resolution': dimension,
                'width': dimensions[dimension]['width'],
                'height': dimensions[dimension]['height'],
            }
            items['hls'][dimension].update(self.hls_const_params)

        if 'yandexVideo' in resp:
            ya_video = resp.get('yandexVideo')
            token = ya_video.get('videodiskToken')
            storage_dir = ya_video.get('storageDir')
            login = ya_video.get('login')

            ya_video_host = ya_video['host']

            for format in ya_video.get('formats'):
                if format in self.ya_video_formats_map:
                    container, dimension, filename, video_codec, audio_codec = self.ya_video_formats_map[format]
                    if dimension in dimensions:
                        # Возвращаем только те размеры, что есть в стримминге
                        # Потому что про остальные мы не знаем размеры
                        # Это костыль, но пусть будет так потому что мы скоро полностью все переделаем
                        # и не будет платформы вообще
                        items[container][dimension] = {
                            'container': container,
                            'links': {
                                protocol: protocol + "://" + ya_video_host
                                         + "/get-film/"
                                         + login + "/"
                                         + storage_dir + "/"
                                         + filename + "?"
                                         + "stream-id=" + stream_id + ";" + token
                                         + ("&noredirect=true" if no_redirects else "")
                            },
                            'resolution': dimension,
                            'width': dimensions[dimension]['width'],
                            'height': dimensions[dimension]['height'],
                            'video_codec': video_codec,
                            'audio_codec': audio_codec
                        }

        streams = items['hls'].values() + items['mp4'].values()
        adaptive_item = {
            'resolution': 'adaptive',
            'links': {
                protocol: "%s/hls-playlist/%s/master-playlist.m3u8" % (streaming_host, stream_id)
            }
        }
        adaptive_item.update(self.hls_const_params)
        streams.append(adaptive_item)
        return {
            'duration': resp['duration'],
            'streamId': stream_id,
            'items': streams,
            'total': len(streams),
        }


class ListPublicVideoStreamsHandler(ListVideoStreamsHandler):
    """Получить список потоков для стриминга для публичного файла"""
    auth_methods = [UserAgentAuth(), OriginAuth()]
    permissions = ListVideoStreamsHandler.permissions | DiskVideoPublicPermission()
    rate_limiter = PerSomethingUniqueLimiter('cloud_api_user')
    hidden = True
    auth_user_required = False
    auto_create_app_folder = False
    query = fields.QueryDict({
        'path': None,
        'public_key': PublicKeyField(required=True, help_text=u'Ключ или публичный URL ресурса.')
    })
    error_map = {
        409: UnavailableForLegalReasons,
        404: NotFoundError,
        API_RESTRICTED_FOR_OVERDRAFT_USER: exceptions.DiskAPIDisabledForOverdraftUserError,
    }
    service_url = '/json/public_video_url?private_hash=%(public_key)s&uid=%(uid)s'
    video_streaming_url = '/json/public_video_streams?private_hash=%(public_key)s&use_http=%(use_http)s&user_ip=%(user_ip)s&uid=%(uid)s&client_id=%(client_id)s'

    def handle(self, request, *args, **kwargs):
        return super(ListPublicVideoStreamsHandler, self)._handle(request, *args, **kwargs)

    def _get_group_field(self):
        return self.request.query['public_key']

    def get_context(self, context=None):
        result_context = super(ListPublicVideoStreamsHandler, self).get_context(context=context)
        # Если uid'а не было в запросе, то используем публичный uid
        if 'uid' not in result_context:
            result_context['uid'] = PUBLIC_UID
        return result_context


class CustomPreviewSizeHandlerMixin(MpfsProxyHandler):
    """
    Добавляет в хэндлер возможность выбора размера превьюшек.

    Да простит меня Михаил, но при существующем дереве классов:

                                                               |-- GetResourcesHandler
                         |-- MpfsProxyWithAppFolderHandler <|--|
                         |                                     |-- ... (ещё штук 10 классов)
    MpfsProxyHandler <|--|
                         |
                         |-- GetLastUploadedResourceListHandler
                         |
                         |-- ... (ещё штук 15 классов)

    Если сделать это без миксина /* три раза поплевал через плечо, постучал по дереву и перекрестился */, то получится,
    что все наследники MpfsProxyHandler унаследуют параметр preview_size и его придётся каими-то
    костылями удалят из query в большинстве дочерних классов.
    """
    query = fields.QueryDict({
        'preview_size': fields.StringField(default=u'S',
                                           validators=[validators.RegExpValidator(r'^(S|M|L|XL|XXL|XXXL|(\d+x?\d*)|(x\d+))$')],
                                           help_text=u'Размер превью.'),
        'preview_crop': fields.BooleanField(default=False, help_text=u'Разрешить обрезку превью.'),
        'preview_quality': fields.IntegerRangeField(bottom_border=0, upper_border=100, hidden=True, permissions=WebDavPermission(), help_text=u'Качество превью.'),
        'preview_allow_big_size': fields.BooleanField(default=False, hidden=True, permissions=WebDavPermission(), help_text=u'Разрешить превью больших размеров.'),
    })

    def get_url(self, context=None):
        """Добавит в query string сгенерированого хэндлером url параметр preview_size"""
        url = super(CustomPreviewSizeHandlerMixin, self).get_url(context=context)
        parsed_url = urlparse.urlsplit(url)
        query = urlparse.parse_qs(parsed_url.query)
        query['preview_size'] = self.request.query['preview_size']
        query['preview_crop'] = int(self.request.query['preview_crop'])

        if self.request.query.get('preview_quality'):
            query['preview_quality'] = self.request.query['preview_quality']
        if self.request.query.get('preview_allow_big_size'):
            query['preview_allow_big_size'] = int(self.request.query['preview_allow_big_size'])

        url = urlparse.urlunsplit(parsed_url._replace(query=urllib.urlencode(query, doseq=True)))
        return url


class GetResourceHandler(CustomPreviewSizeHandlerMixin, MpfsProxyWithAppFolderHandler):
    """
    Получить метаинформацию о файле или каталоге

    Если путь указывает на каталог, в ответе также описываются ресурсы этого каталога.
    """
    permissions = DiskReadPermission() | DiskAppFolderPermission('path') | WebDavPermission()
    service_url = ('/json/list?uid=%(uid)s&path=%(path)s&amount=%(limit)s&offset=%(offset)s&sort=%(sort.field)s&order=%(sort.order)s&'
                   'meta=' + ResourceSerializer.get_raw_mpfs_meta(extra_meta=['numchildren']))
    serializer_cls = ResourceSerializer

    query = fields.QueryDict({
        'path': MpfsPathField(required=True, allowed_areas=('disk', 'app', PHOTOUNLIM_AREA), help_text=u'Путь к ресурсу.',
                              only_official_clients_areas=(PHOTOUNLIM_AREA,)),
        'sort': MpfsSortField(serializer_cls=ResourceSerializer, help_text=u'Поле для сортировки вложенных ресурсов.',
                              fields_mapping={'exif.date_time': 'etime'},
                              ),
        'limit': fields.IntegerField(default=20, help_text=u'Количество выводимых вложенных ресурсов.'),
        'offset': fields.IntegerField(default=0, help_text=u'Смещение от начала списка вложенных ресурсов.'),
    })

    def serialize(self, obj, *args, **kwargs):
        if isinstance(obj, list):
            res = obj[0]
            if '_embedded' not in res:
                res['_embedded'] = {}
            # По хорошему всё это должен делать сериализатор, но MPFS не отдаёт нам достаточно данных для того,
            # чтобы делать нормальную пагинацию на стороне сериализатора не прибегая к ручному добиванию
            # исходных данных для сериализатора.
            # Поэтому приходится тут ручками добивать недостающие данные.

            native_sort = self.request.query['sort']
            external_sort = self.query.get_fields()['sort'].from_native(native_sort)
            embedded = {
                'path': res['path'],
                'items': obj[1:],
                'sort': external_sort,
                'limit': self.request.query['limit'],
                'offset': self.request.query['offset'],
                'total': res.get('meta', {}).get('numchildren', 0)
            }
            res['_embedded'].update(embedded)
        else:
            res = obj
        return super(GetResourceHandler, self).serialize(res, *args, **kwargs)


class GetLastUploadedFilesListHandler(CustomPreviewSizeHandlerMixin, MpfsProxyHandler):
    """Получить список файлов упорядоченный по дате загрузки"""
    MEDIA_TYPE_CHOICES = sorted(builtin_extensions.keys())

    permissions = DiskReadPermission()
    service_url = ('/json/timeline?uid=%(uid)s&amount=%(limit)s&sort=utime&order=0&media_type=%(media_type)s&'
                   'meta=' + ResourceSerializer.get_raw_mpfs_meta())
    serializer_cls = LastUploadedResourceListSerializer
    query = fields.QueryDict({
        'limit': fields.IntegerField(default=20, help_text=u'Количество выводимых вложенных ресурсов.'),
        'media_type': fields.MultipleChoicesField(choices=MEDIA_TYPE_CHOICES, help_text=u'Фильтр по медиа типу.'),
    })

    def serialize(self, obj, *args, **kwargs):
        if isinstance(obj, list):
            res = {
                'limit': self.request.query['limit'],
                'items': obj[1:],
                'media_type': self.request.query['media_type']
            }
        else:
            res = obj
        return super(GetLastUploadedFilesListHandler, self).serialize(res, *args, **kwargs)

    def get_url(self, context=None):
        context = context or {}
        media_type = context.get('media_type', [])
        context['media_type'] = ','.join(media_type)
        return super(GetLastUploadedFilesListHandler, self).get_url(context=context)


class GetOnlineEditorURLHandler(MpfsProxyHandler):
    """Получить ссылку на онлайн-редактор для файла."""
    hidden = True
    permissions = DiskReadPermission() | AllowByClientIdPermission(PLATFORM_DISK_APPS_IDS)
    service_url = '/json/office_generate_online_editor_url?uid=%(uid)s&tld=%(tld)s&path=%(path)s'
    serializer_cls = OfficeOnlineEditorURLSerializer
    query = fields.QueryDict({
        'tld': fields.StringField(
            default=USER_DEFAULT_TLD,
            help_text=u'Домен верхнего уровня для генерируемой ссылки.'
        ),
        'path': MpfsPathField(
            required=True,
            allowed_areas=(DISK_AREA,),
            help_text=u'Путь к ресурсу.'
        ),
    })

    error_map = {
        415: DiskOfficeUnsupportedExtensionError,
        403: DiskUserBlockedError,
        OFFICE_IS_NOT_ALLOWED: DiskOfficeIsDisabledError,
        API_RESTRICTED_FOR_OVERDRAFT_USER: exceptions.DiskAPIDisabledForOverdraftUserError,
    }


class AddGeneratedAlbumExclusionHandler(MpfsProxyHandler):
    """Добавить файлу исключение из автосгенерированных альбомов."""

    hidden = True
    permissions = WebDavPermission()
    service_url = '/json/albums_exclude_from_generated?uid=%(uid)s&album_type=%(album_type)s&path=%(path)s'
    serializer_cls = ExclusionsFromGeneratedAlbumsItemsSerializer
    query = fields.QueryDict({
        'album_type': fields.StringField(
            required=True,
            help_text=u'Тип автосгенерированного альбома, из которого нужно исключить ресурс.'
        ),
        'path': MpfsPathField(
            required=True,
            allowed_areas=('disk', PHOTOUNLIM_AREA),
            help_text=u'Путь к ресурсу.'
        ),
    })


class GetPhotounlimLastModifiedFilesListHandler(MpfsProxyHandler):
    """Получить список файлов загруженных в безлимитный раздел для фото."""
    hidden = True
    permissions = WebDavPermission()
    service_url = ('/json/photounlim_last_modified?uid=%(uid)s&iteration_key=%(iteration_key)s&'
                   'start_date=%(start_date)s&meta=' + ResourceSerializer.get_raw_mpfs_meta())
    serializer_cls = PhotounlimLastModifiedResourceListSerializer
    query = fields.QueryDict({
        'iteration_key': fields.StringField(help_text=u'Ключ для получения следующей страницы списка. '
                                                      u'Если задан, то start_date игнорируется.',
                                            default=''),
        'start_date': fields.DateTimeToTSField(help_text=u'Дата, с которой начинается список файлов. '
                                                         u'Передаётся только при получении первой страницы списка.',
                                               default=''),
    })


class GetFlatFilesListHandler(GetLastUploadedFilesListHandler):
    """Получить список файлов упорядоченный по имени"""
    service_url = ('/json/timeline?uid=%(uid)s&amount=%(limit)s&offset=%(offset)s&sort=%(sort.field)s&order=%(sort.order)s&media_type=%(media_type)s&'
                   'meta=' + ResourceSerializer.get_raw_mpfs_meta())
    serializer_cls = FilesResourceListSerializer
    query = fields.QueryDict({
        'offset': fields.IntegerField(default=0, help_text=u'Смещение от начала списка вложенных ресурсов.'),
        'sort': MpfsSortField(serializer_cls=ResourceSerializer, help_text=u'Поле для сортировки ресурсов.'),
    })

    SORT_KEY_MAP = {
        'path': 'key',
    }

    def get_context(self, context=None):
        c = super(GetFlatFilesListHandler, self).get_context(context=context)
        sort_field = c['sort']['field']
        c['sort']['field'] = self.SORT_KEY_MAP.get(sort_field, sort_field)
        return c

    def serialize(self, obj, *args, **kwargs):
        res = super(GetFlatFilesListHandler, self).serialize(obj, *args, **kwargs)
        res['offset'] = self.request.query['offset']
        return res


class ListPublicResourcesHandler(CustomPreviewSizeHandlerMixin, MpfsProxyHandler):
    """Получить список опубликованных ресурсов"""
    RESOURCE_TYPE_CHOICES = ['file', 'dir']
    permissions = DiskReadPermission() | WebDavPermission()
    serializer_cls = PublicResourcesListSerializer
    query = fields.QueryDict({
        'limit': fields.IntegerField(default=20, help_text=u'Количество выводимых ресурсов.'),
        'offset': fields.IntegerField(default=0, help_text=u'Смещение от начала списка ресурсов.'),
        'type': fields.ChoiceField(choices=RESOURCE_TYPE_CHOICES, help_text=u'Фильтр по типам ресурсов.')
    })
    service_url = ('/json/list_public?uid=%(uid)s&amount=%(limit)s&offset=%(offset)s&'
                   'meta=' + ResourceSerializer.get_raw_mpfs_meta())

    def get_url(self, context=None):
        url = super(ListPublicResourcesHandler, self).get_url(context=context)
        res_type = self.request.query.get('type')
        if res_type:
            parsed_url = urlparse.urlsplit(url)
            query = urlparse.parse_qs(parsed_url.query)
            query['type'] = res_type
            url = urlparse.urlunsplit(parsed_url._replace(query=urllib.urlencode(query, doseq=True)))
        return url

    def serialize(self, obj, *args, **kwargs):
        data = {
            'items': obj,
            'limit': self.request.query['limit'],
            'offset': self.request.query['offset'],
        }
        res_type = self.request.query['type']
        if res_type:
            data['type'] = res_type
        return super(ListPublicResourcesHandler, self).serialize(data, *args, **kwargs)


class AsyncOperationHandler(MpfsProxyWithAppFolderHandler):
    """
    Base class for MPFS asynchronous operations.

    Work as follows:
      1. Get result of `list_url`,
      2. Check if resource is empty dir or file then see 3 else see 4,
      3. Call `operation_url` and return 200 OK.
      4. Call `async_operation_url` and return link to async operation status.
    So in common it's enough to override `enabled`, `operation_url` and `async_operation_url` to create new handler.
    """
    resp_status_code = 201
    async_resp_status_code = 202
    serializer_cls = LinkSerializer
    async_serializer_cls = LinkSerializer
    list_url = None
    operation_url = None
    async_operation_url = None
    error_map = {
        405: exceptions.DiskPathPointsToExistentDirectoryError,
        409: exceptions.DiskPathDoesntExistsError,
        412: exceptions.DiskResourceAlreadyExistsError,
        423: exceptions.DiskResourceLockedError,
        API_RESTRICTED_FOR_OVERDRAFT_USER: exceptions.DiskAPIDisabledForOverdraftUserError,
    }
    query = fields.QueryDict({
        'force_async': fields.BooleanField(required=False, help_text=u'Выполнить асинхронно.')
    })

    def get_operation_url(self):
        return self.operation_url

    def get_async_operation_url(self):
        return self.async_operation_url

    def get_resource_link(self, obj):
        path = self.query.get_fields()['path'].from_native(self.request.query['path'])
        link = self.router.get_link(GetResourceHandler, {'path': path})
        return link

    def handle(self, request, *args, **kwargs):
        context = self.urlencode_context(self.get_context())

        url = self.list_url % context
        resp = self.service.open_url('%s%s' % (self.service.base_url, url), retry=False)
        data = from_json(resp)
        is_file = isinstance(data, dict)
        is_empty_dir = (isinstance(data, list) and len(data) == 1)
        is_empty_resource = is_file or is_empty_dir

        if request.query.get('force_async') or not is_empty_resource:
            # если path указывает ни на файл ни на пустой каталог или принудительная асинхронщина
            url = self.get_async_operation_url() % context
            resp = self.service.open_url('%s%s' % (self.service.base_url, url), retry=False)
            data = from_json(resp)
            oid = data['oid']
            link = self.router.get_link(GetOperationStatusHandler, {'operation_id': oid})
            return self.async_resp_status_code, self.async_serialize(link)

        if is_empty_resource:
            # если вернулся JSON Object - значит path указывает на файл
            # если вернулся JSON List единичной длины - значит path указывает на пустую папку
            url = self.get_operation_url() % context
            obj = self.request_service('%s%s' % (self.service.base_url, url))
            link = self.get_resource_link(obj)
            return self.resp_status_code, self.serialize(link)

        raise exceptions.DiskRemoveNotEmptyFolderNonRecursivelyError()

    def async_serialize(self, obj, *args, **kwargs):
        """Сериализует объект сериализатором хэндлера. Умеет включать/выключать HAL контролы в сериализаторе."""
        if self.async_serializer_cls:
            serializer = self.async_serializer_cls(
                obj, router=self.router, hal=self.is_hal_requested(), lang=self.request.language, *args, **kwargs)
            obj = serializer.data
        return obj

    def get_response_objects(self):
        ret = super(AsyncOperationHandler, self).get_response_objects()
        ret += [ResponseObject(self.async_serializer_cls, u'Операция выполняется асинхронно.', self.async_resp_status_code)]
        return ret


class DeleteResourceHandler(AsyncOperationHandler):
    """Удалить файл или папку

    По умолчанию удалит ресурс в Корзину. Чтобы удалить ресурс не помещая в корзину, следует указать параметр
    ```permanently=true```.

    Если удаление происходит асинхронно, то вернёт ответ со статусом ```202``` и ссылкой на асинхронную операцию.
    Иначе вернёт ответ со статусом ```204``` и пустым телом.
    """
    list_url = '/json/list?uid=%(uid)s&path=%(path)s&amount=1'
    permissions = DiskWritePermission() | DiskAppFolderPermission('path') | WebDavPermission()
    resp_status_code = 204
    serializer_cls = None

    error_map = {
        PRECONDITIONS_FAILED: exceptions.MD5DifferError,
        MD5_CHECK_NOT_SUPPORTED: exceptions.MD5CheckNotSupportedError,
        API_RESTRICTED_FOR_OVERDRAFT_USER: exceptions.DiskAPIDisabledForOverdraftUserError,
    }

    query = fields.QueryDict({
        'path': MpfsPathField(required=True, allowed_areas=('disk', 'app', PHOTOUNLIM_AREA), help_text=u'Путь к файлу или папке.',
                              only_official_clients_areas=(PHOTOUNLIM_AREA,)),
        'permanently': fields.BooleanField(help_text=u'Удалить ресурс не помещая в Корзину.'),
        'md5': fields.StringField(help_text=u'md5 удаляемого файла.'),
    })

    def get_operation_url(self):
        if self.request.query.get('permanently'):
            url = '/json/rm?uid=%(uid)s&path=%(path)s'
        else:
            url = '/json/trash_append?uid=%(uid)s&path=%(path)s'
        if self.request.query.get('md5'):
            url += '&md5=%(md5)s'
        return url

    def get_async_operation_url(self):
        if self.request.query.get('permanently'):
            url = '/json/async_rm?uid=%(uid)s&path=%(path)s'
        else:
            url = '/json/async_trash_append?uid=%(uid)s&path=%(path)s'
        if self.request.query.get('md5'):
            url += '&md5=%(md5)s'
        return url

    @_auto_initialize_user
    def handle(self, request, *args, **kwargs):
        result = super(DeleteResourceHandler, self).handle(request, *args, **kwargs)
        if result and isinstance(result, tuple) and len(result) >= 1 and result[0] == self.resp_status_code:
            return None
        return result


class CopyResourceHandler(AsyncOperationHandler):
    """Создать копию файла или папки

    Если копирование происходит асинхронно, то вернёт ответ с кодом ```202``` и ссылкой на асинхронную операцию.
    Иначе вернёт ответ с кодом ```201``` и ссылкой на созданный ресурс.
    """
    list_url = '/json/list?uid=%(uid)s&path=%(from)s&amount=1'
    operation_url = '/json/copy?uid=%(uid)s&src=%(from)s&dst=%(path)s&force=%(overwrite)d'
    async_operation_url = '/json/async_copy?uid=%(uid)s&src=%(from)s&dst=%(path)s&force=%(overwrite)d'
    permissions = (DiskReadPermission() | DiskAppFolderPermission('from')) \
        & (DiskWritePermission() | DiskAppFolderPermission('path')) | WebDavPermission()

    query = fields.QueryDict({
        'path': MpfsPathField(required=True, help_text=u'Путь к создаваемому ресурсу.'),
        'from': MpfsPathField(required=True, allowed_areas=('disk', 'app', PHOTOUNLIM_AREA), help_text=u'Путь к копируемому ресурсу.'),
        'overwrite': fields.BooleanField(help_text=u'Перезаписать существующий ресурс.'),
    })

    error_map = {
        507: exceptions.DiskStorageQuotaExhaustedError,
        OWNER_HAS_NO_FREE_SPACE: exceptions.DiskOwnerStorageQuotaExhaustedError,
        API_RESTRICTED_FOR_OVERDRAFT_USER: exceptions.DiskAPIDisabledForOverdraftUserError,
    }


class MoveResourceHandler(AsyncOperationHandler):
    """Переместить файл или папку

    Если перемещение происходит асинхронно, то вернёт ответ с кодом ```202``` и ссылкой на асинхронную операцию.
    Иначе вернёт ответ с кодом ```201``` и ссылкой на созданный ресурс.
    """
    list_url = '/json/list?uid=%(uid)s&path=%(from)s&amount=1'
    operation_url = '/json/move?uid=%(uid)s&src=%(from)s&dst=%(path)s&force=%(overwrite)d'
    async_operation_url = '/json/async_move?uid=%(uid)s&src=%(from)s&dst=%(path)s&force=%(overwrite)d'
    permissions = (DiskReadPermission() & DiskWritePermission()) \
        | (DiskAppFolderPermission('from') & DiskAppFolderPermission('path')) \
        | (DiskAppFolderPermission('from') & DiskWritePermission()) \
        | WebDavPermission()

    query = fields.QueryDict({
        'path': MpfsPathField(required=True, allowed_areas=('disk', PHOTOUNLIM_AREA),
                              only_official_clients_areas=(PHOTOUNLIM_AREA,),
                              help_text=u'Путь к создаваемому ресурсу.'),
        'from': MpfsPathField(required=True, allowed_areas=('disk', PHOTOUNLIM_AREA),
                              only_official_clients_areas=(PHOTOUNLIM_AREA,),
                              help_text=u'Путь к перемещаемому ресурсу.'),
        'overwrite': fields.BooleanField(help_text=u'Перезаписать существующий ресурс.'),
    })

    error_map = {
        507: exceptions.DiskStorageQuotaExhaustedError,
        OWNER_HAS_NO_FREE_SPACE: exceptions.DiskOwnerStorageQuotaExhaustedError,
        API_RESTRICTED_FOR_OVERDRAFT_USER: exceptions.DiskAPIDisabledForOverdraftUserError,
        MOVE_WRONG_DESTINATION: exceptions.DiskMoveWrongDestinationError,
    }


class OldGetOperationStatusHandler(MpfsProxyHandler):
    """Получить статус асинхронной операции.

    ..warning:: Устаревший обработчик, см. :class:`GetOperationStatusHandler`
    """
    serializer_cls = OperationStatusSerializer
    service_url = '/json/status?uid=%(uid)s&oid=%(id)s'
    query = fields.QueryDict({
        'id': fields.StringField(required=True, help_text=u'Идентификатор операции.'),
    })
    permissions = AllowAllPermission()
    error_map = {
        404: exceptions.DiskOperationNotFoundError,
        409: exceptions.DiskNotFoundError,
        API_RESTRICTED_FOR_OVERDRAFT_USER: exceptions.DiskAPIDisabledForOverdraftUserError,
    }


class GetOperationStatusHandler(MpfsProxyHandler):
    """Получить статус асинхронной операции"""
    serializer_cls = OperationStatusSerializer
    service_url = '/json/status?uid=%(uid)s&oid=%(operation_id)s&meta=short_url'
    kwargs = fields.QueryDict({
        'operation_id': fields.StringField(required=True, help_text=u'Идентификатор операции.'),
    })
    permissions = AllowAllPermission()
    error_map = {
        404: exceptions.DiskOperationNotFoundError,
        409: exceptions.DiskNotFoundError,
        API_RESTRICTED_FOR_OVERDRAFT_USER: exceptions.DiskAPIDisabledForOverdraftUserError,
    }

    def serialize(self, obj, *args, **kwargs):
        result = super(GetOperationStatusHandler, self).serialize(obj, *args, **kwargs)
        # https://st.yandex-team.ru/CHEMODAN-37726
        self.add_public_url_to_result_just_for_store_attach_operation(obj, result)
        return result

    @staticmethod
    def add_public_url_to_result_just_for_store_attach_operation(obj, result):
        operation_type = obj['type']
        resource = obj.get('resource')

        if operation_type == 'store' and resource:
            if resource['id'].startswith('/attach/'):
                if 'meta' in resource and 'short_url' in resource['meta']:
                    result.setdefault('data', {'public_url': resource['meta']['short_url']})


class ListActiveOperationsHandler(MpfsProxyHandler):
    """Получить список активных асинхронных операций"""
    serializer_cls = OperationListSerializer
    service_url = '/json/active_operations?uid=%(uid)s'
    permissions = DiskReadPermission() | WebDavPermission()

    # Порядок сортировки по-убыванию ctime.
    query = fields.QueryDict({
        'limit': fields.IntegerField(default=20, help_text=u'Количество выводимых операций.'),
        'offset': fields.IntegerField(default=0, help_text=u'Смещение от начала списка операций.'),
    })

    @_auto_initialize_user
    def handle(self, request, *args, **kwargs):
        context = self.get_context()
        offset = context['offset']
        limit = context['limit']
        url = self.get_url(context)
        resp = self.request_service(url, method=self.service_method)
        # Делаем преобразование ответа ручки active_operations для сериализатора статусов
        for active_task in resp:
            active_task['status'] = messages.true_operation_titles[active_task['state']]
        resp.sort(key=lambda x: x['ctime'])
        prepared_resp = {
            'items': resp[offset: offset + limit],
            'total': len(resp),
            'limit': limit,
            'offset': offset,
        }
        return self.serialize(prepared_resp)


class GetResourceDownloadLinkHandler(MpfsProxyWithAppFolderHandler):
    """Получить ссылку на скачивание файла"""
    permissions = DiskReadPermission() | DiskAppFolderPermission('path') | WebDavPermission()
    service_url = '/json/url?uid=%(uid)s&path=%(path)s'
    serializer_cls = LinkSerializer
    query = fields.QueryDict({
        'path': MpfsPathField(required=True, allowed_areas=('disk', 'app', PHOTOUNLIM_AREA), help_text=u'Путь к ресурсу.',
                              only_official_clients_areas=(PHOTOUNLIM_AREA,)),
    })

    def serialize(self, obj, *args, **kwargs):
        url = obj.get('file', obj.get('folder', None))
        link = ('GET', url, False)
        return super(GetResourceDownloadLinkHandler, self).serialize(link, *args, **kwargs)


class GetPhotounlimResourceDownloadLinkHandler(GetResourceDownloadLinkHandler):
    """Получить ссылку на скачивание файла Безлимитного раздела"""
    hidden = True
    permissions = WebDavPermission()
    query = fields.QueryDict({
        'path': MpfsPathField(required=True, allowed_areas=(PHOTOUNLIM_AREA,),
                              help_text=u'Путь к ресурсу в Безлимитном разделе.'),
    })


class GetResourceDownloadLinkByResourceIDHandler(GetResouceInfoByIDMixin, GetResourceDownloadLinkHandler):
    """Получить ссылку на скачивание файла по идентификатору ресурса."""
    hidden = True
    permissions = WebDavPermission()
    kwargs = fields.QueryDict({
        'resource_id': ResourceIdField(required=True),
    })
    query = fields.QueryDict({
        'path': None
    })

    def handle(self, request, *args, **kwargs):
        raw_resource_id = request.kwargs['resource_id'].serialize()
        resource_info = self.get_resource_info(request.user.uid, raw_resource_id)
        request.query['path'] =  resource_info['path']
        return super(GetResourceDownloadLinkByResourceIDHandler, self).handle(request, *args, **kwargs)


class UploadResourceHandler(MpfsProxyWithAppFolderHandler):
    """Создать операцию по загрузке файла."""

    hidden = True

    permissions = WebDavPermission()
    serializer_cls = UploadResourceSerializer
    service_url = '/json/store?uid=%(uid)s&path=%(path)s&force=%(overwrite)d&' \
                  'md5=%(md5)s&sha256=%(sha256)s&size=%(size)d'
    query = fields.QueryDict({
        'path': MpfsPathField(required=True, help_text=u'Путь к загружаемому файлу на Диске.'),
        'overwrite': fields.BooleanField(help_text=u'Перезаписать существующий файл.'),
        'md5': fields.StringField(default='', help_text=u'md5 загружаемого файла.', hidden=True),
        'sha256': fields.StringField(default='', help_text=u'sha256 загружаемого файла.', hidden=True),
        'size': fields.IntegerField(default=0, help_text=u'Размер загружаемого файла.', hidden=True),
    })

    error_map = {
        405: exceptions.DiskPathPointsToExistentDirectoryError,
        409: exceptions.DiskPathDoesntExistsError,
        412: exceptions.DiskResourceAlreadyExistsError,
        423: exceptions.DiskResourceLockedError,
        507: exceptions.DiskStorageQuotaExhaustedError,
        OWNER_HAS_NO_FREE_SPACE: exceptions.DiskOwnerStorageQuotaExhaustedError,
        API_RESTRICTED_FOR_OVERDRAFT_USER: exceptions.DiskAPIDisabledForOverdraftUserError,
    }

    def get_resource_link(self):
        path = self.query.get_fields()['path'].from_native(self.request.query['path'])
        link = self.router.get_link(GetResourceHandler, {'path': path})
        return link

    def get_context(self, context=None):
        return super(UploadResourceHandler, self).get_context()

    @_auto_initialize_user
    def handle(self, request, *args, **kwargs):
        context = self.get_context()
        service_url = self.build_url(self.service_url, context=context)

        if request.client.id in PLATFORM_SKIP_CHECK_SPACE_ALLOWED_CLIENT_IDS:
            service_url = update_qs_params(service_url, {'skip_check_space': '1'})

        device_collections = self.request.raw_headers.get('X-Yandex-Device-Collections')
        if device_collections is not None:
            service_url = update_qs_params(service_url, {'device_collections': device_collections})

        if self.request.mode == tags.platform.INTERNAL:
            service_url = update_qs_params(service_url, {'skip_speed_limit': '1'})

        store_response = self.request_service(service_url, data=request.data)
        if store_response.get('status') == 'hardlinked':
            resource_link = self.get_resource_link()
            return 201, LinkSerializer(resource_link).data
        upload_link = ('PUT', store_response.get('upload_url'), False,)
        operation_link = self.router.get_link(GetOperationStatusHandler, {'operation_id': store_response.get('oid')})
        return self.serialize({'upload_link': upload_link, 'operation_link': operation_link})


class GetResourceUploadLinkHandler(MpfsProxyWithAppFolderHandler):
    """Получить ссылку для загрузки файла"""
    permissions = DiskWritePermission() | DiskAppFolderPermission('path') | WebDavPermission()
    serializer_cls = ResourceUploadLinkSerializer
    service_url = '/json/store?uid=%(uid)s&path=%(path)s&force=%(overwrite)d'
    query = fields.QueryDict({
        'path': MpfsPathField(required=True, help_text=u'Путь к загружаемому файлу на Диске.'),
        'overwrite': fields.BooleanField(help_text=u'Перезаписать существующий файл.'),
    })

    error_map = {
        405: exceptions.DiskPathPointsToExistentDirectoryError,
        409: exceptions.DiskPathDoesntExistsError,
        412: exceptions.DiskResourceAlreadyExistsError,
        423: exceptions.DiskResourceLockedError,
        507: exceptions.DiskStorageQuotaExhaustedError,
        OWNER_HAS_NO_FREE_SPACE: exceptions.DiskOwnerStorageQuotaExhaustedError,
        API_RESTRICTED_FOR_OVERDRAFT_USER: exceptions.DiskAPIDisabledForOverdraftUserError,
    }

    def serialize(self, obj, *args, **kwargs):
        response = ('PUT', obj.get('upload_url'), False, obj.get('oid'))
        return super(GetResourceUploadLinkHandler, self).serialize(response, *args, **kwargs)

    @_auto_initialize_user
    def handle(self, request, *args, **kwargs):
        context = self.get_context()
        service_url = self.build_url(self.service_url, context=context)

        if request.client.id in PLATFORM_SKIP_CHECK_SPACE_ALLOWED_CLIENT_IDS:
            service_url = update_qs_params(service_url, {'skip_check_space': '1'})

        if self.request.mode == tags.platform.INTERNAL:
            service_url = update_qs_params(service_url, {'skip_speed_limit': '1'})

        device_collections = self.request.raw_headers.get('X-Yandex-Device-Collections')
        if device_collections is not None:
            service_url = update_qs_params(service_url, {'device_collections': device_collections})

        resp = self.request_service(service_url, data=self.request.body)
        resp = self.serialize(resp)
        return resp


class ResourceLinkHandler(MpfsProxyWithAppFolderHandler):
    """Хэндлер отвечающий ссылкой на ресурс над которым была выполнена операция"""
    permissions = DiskWritePermission() | DiskAppFolderPermission('path') | WebDavPermission()
    serializer_cls = LinkSerializer
    error_map = {
        423: exceptions.DiskResourceLockedError,
        API_RESTRICTED_FOR_OVERDRAFT_USER: exceptions.DiskAPIDisabledForOverdraftUserError,
    }

    def handle(self, request, *args, **kwargs):
        super(ResourceLinkHandler, self).handle(request, *args, **kwargs)
        path = self.query.get_fields()['path'].from_native(self.request.query['path'])
        link = self.router.get_link(GetResourceHandler, {'path': path})
        return self.serialize(link)


class CreateResourceHandler(ResourceLinkHandler):
    """Создать папку"""
    resp_status_code = 201
    mkdir_url = '/json/mkdir?uid=%(uid)s&path=%(path)s'
    mksysdir_url = '/json/mksysdir?uid=%(uid)s&type=%(folder_type)s'
    default_folders_url = '/json/default_folders?uid=%(uid)s'
    query = fields.QueryDict({
        'path': MpfsPathField(required=True, help_text=u'Путь к создаваемой папке.'),
        # 'overwrite': fields.BooleanField(help_text=u'Перезаписать или нет, если файл уже существует.'),
        # 'type': fields.ChoiceField(['folder', 'file'], default='folder', help_text=u'Тип создаваемого ресурса.'),
    })
    forced_mkdir_folder_types = {'yateamnda'}

    error_map = {
        403: exceptions.DiskUserBlockedError,
        405: exceptions.DiskPathPointsToExistentDirectoryError,
        409: exceptions.DiskPathDoesntExistsError,
        412: exceptions.DiskResourceAlreadyExistsError,
        507: exceptions.DiskStorageQuotaExhaustedError,
        OWNER_HAS_NO_FREE_SPACE: exceptions.DiskOwnerStorageQuotaExhaustedError,
        API_RESTRICTED_FOR_OVERDRAFT_USER: exceptions.DiskAPIDisabledForOverdraftUserError,
        FOLDER_TOO_DEEP: exceptions.DiskFolderTooDeepError,
    }

    @staticmethod
    def normalize_path(path):
        return path.rstrip('/')

    @property
    def default_folders_paths(self):
        if hasattr(self, '_default_folders_paths'):
            return self._default_folders_paths
        else:
            self._default_folders_paths = set()
            for items in DEFAULT_FOLDERS.itervalues():
                for default_folder in items.itervalues():
                    default_folder = self.normalize_path(default_folder)
                    self._default_folders_paths.add(default_folder)
            return self._default_folders_paths

    def get_default_folder_type(self, path):
        path = self.normalize_path(path)
        if path not in self.default_folders_paths:
            return None
        url = self.build_url(self.default_folders_url, self.get_context())
        response = self.request_service(url)
        for folder_type, folder_name in response.iteritems():
            folder_name = self.normalize_path(folder_name)
            if folder_name == path:
                return folder_type
        return None

    def handle(self, request, *args, **kwargs):
        path = request.query['path']
        request.folder_type = self.get_default_folder_type(path)
        return super(CreateResourceHandler, self).handle(request, *args, **kwargs)

    def get_url(self, context=None):
        if self.request.folder_type is None or self.request.folder_type in self.forced_mkdir_folder_types:
            return self.build_url(self.mkdir_url, context)
        else:
            c = context or {}
            c['folder_type'] = self.request.folder_type
            return self.build_url(self.mksysdir_url, context)


class UpdateResourceHandler(MpfsProxyWithAppFolderHandler):
    """Обновить пользовательские данные ресурса"""
    permissions = DiskWritePermission() | DiskAppFolderPermission('path') | WebDavPermission()
    resource_url = ('/json/info?uid=%(uid)s&path=%(path)s&'
                    'meta=' + ResourceSerializer.get_raw_mpfs_meta())
    service_url = '/json/setprop?uid=%(uid)s&path=%(path)s&custom_properties=%(custom_properties)s'
    query = fields.QueryDict({
        'path': MpfsPathField(required=True, help_text=u'Путь к обновляемому ресурсу.'),
    })
    body_serializer_cls = ResourcePatchSerializer
    serializer_cls = ResourceSerializer

    @_auto_initialize_user
    def handle(self, request, *args, **kwargs):
        context = super(UpdateResourceHandler, self).get_context()

        # достаём оригинальный ресурс
        resource_url = self.build_url(self.resource_url, context=context)
        resource = self.request_service(resource_url, method=self.service_method)

        # мержим существующие custom_properties ресурса с новыми
        custom_properties = resource.get('meta', {}).get('custom_properties', {})
        custom_properties.update(self.request.body.get('meta', {}).get('custom_properties', {}))

        # выкидываем обнулённые поля
        custom_properties = dict([(k, v) for k, v in custom_properties.iteritems() if v is not None])

        custom_properties = to_json(custom_properties)
        # проверяем размер
        max_size = ResourceSerializer.fields['custom_properties'].max_size
        if max_size is not None and len(custom_properties) > max_size:
            raise exceptions.exceptions.FieldObjectMaxSizeExceededError(max_size=max_size)

        context['custom_properties'] = custom_properties
        if 'meta' not in resource:
            resource['meta'] = {}
        resource['meta']['custom_properties'] = from_json(custom_properties)

        # сохраняем обновлённые custom_properties
        url = self.get_url(context)
        self.request_service(url, method=self.service_method)

        # возвращаем обновлённый ресурс
        return self.serialize(resource)


class PublishResourceHandler(ResourceLinkHandler):
    """Опубликовать ресурс"""

    error_map = {
        109: exceptions.DiskReadOnlyError,
        507: exceptions.DiskStorageQuotaExhaustedError,
        API_RESTRICTED_FOR_OVERDRAFT_USER: exceptions.DiskAPIDisabledForOverdraftUserError,
    }

    service_url = '/json/set_public?uid=%(uid)s&path=%(path)s'
    query = fields.QueryDict({
        'path': MpfsPathField(required=True, allowed_areas=('disk', 'app', PHOTOUNLIM_AREA), help_text=u'Путь к публикуемому ресурсу.',
                              only_official_clients_areas=(PHOTOUNLIM_AREA,)),
    })


class UnpublishResourceHandler(ResourceLinkHandler):
    """Отменить публикацию ресурса"""
    service_url = '/json/set_private?uid=%(uid)s&path=%(path)s'
    query = fields.QueryDict({
        'path': MpfsPathField(required=True, allowed_areas=('disk', 'app', PHOTOUNLIM_AREA), help_text=u'Путь к ресурсу.',
                              only_official_clients_areas=(PHOTOUNLIM_AREA,)),
    })


class GetPublicResourceHandler(GetResourceHandler):
    """Получить метаинформацию о публичном файле или каталоге"""
    auto_create_app_folder = False
    auto_resolve_app_folder = False
    auth_required = False
    permissions = AllowAllPermission()
    service_url = ('/json/public_list?private_hash=%(public_key)s:%(path)s&amount=%(limit)s&offset=%(offset)s&'
                   'sort=%(sort.field)s&order=%(sort.order)s&'
                   'meta=' + PublicResourceSerializer.get_raw_mpfs_meta(extra_meta=['numchildren', 'user', 'blockings']))
    serializer_cls = PublicResourceSerializer
    query = fields.QueryDict({
        'public_key': PublicKeyField(required=True, help_text=u'Ключ или публичный URL ресурса.'),
        'path': fields.StringField(help_text=u'Путь к ресурсу в публичной папке.'),
    })
    error_map = {
        109: exceptions.DiskNoWritePermissionForSharedFolderError,
        API_RESTRICTED_FOR_OVERDRAFT_USER: exceptions.DiskAPIDisabledForOverdraftUserError,
        404: exceptions.DiskNotFoundError,
        409: exceptions.DiskNotFoundError,
        503: exceptions.DiskServiceUnavailableError,
    }

    def serialize(self, obj, *args, **kwargs):
        if isinstance(obj, list):
            res = obj[0]
            if '_embedded' not in res:
                res['_embedded'] = {}
            res['_embedded']['public_key'] = self.request.query['public_key']
        return super(GetPublicResourceHandler, self).serialize(obj, *args, **kwargs)

    @_auto_initialize_user
    def handle(self, request, *args, **kwargs):
        url = self.get_url(self.get_context())
        public_key = request.args['public_key']
        resp = self.request_service(url)
        ycrid = request.crid
        if isinstance(resp, dict) \
                and YcridParser.get_platform(ycrid) == YcridPlatformPrefix.REST \
                and resp['meta'].get('blockings') \
                and filter_value_by_percentage(public_key, FEATURE_TOGGLES_BLOCK_PUBLIC_RESOURCE_DOWNLOAD):
            raise exceptions.DiskResourceDownloadLimitExceededError()
        return self.serialize(resp)


class GetPublicResourceDownloadLinkHandler(GetResourceDownloadLinkHandler):
    """Получить ссылку на скачивание публичного ресурса"""
    auto_create_app_folder = False
    auto_resolve_app_folder = False
    auth_required = False
    permissions = AllowAllPermission()
    rate_limiter = PerSomethingUniqueLimiter('cloud_api_user')
    service_url = '/json/public_url?private_hash=%(public_key)s:%(path)s&check_blockings=%(check_blockings)d'
    query = fields.QueryDict({
        'public_key': PublicKeyField(required=True, help_text=u'Ключ или публичный URL ресурса.'),
        'path': fields.StringField(help_text=u'Путь к ресурсу в публичной папке.'),
    })

    def get_context(self, context=None):
        result = super(GetPublicResourceDownloadLinkHandler, self).get_context(context)
        is_yandex_disk_mobile = UserAgentParser.is_yandex_disk_mobile(self.request.raw_headers.get('user-agent'))
        result['check_blockings'] = 0 if is_yandex_disk_mobile else 1
        return result


class SaveToDiskPublicResourceHandler(AsyncOperationHandler):
    """
    Сохранить публичный ресурс в папку Загрузки

    Если сохранение происходит асинхронно, то вернёт ответ с кодом ```202``` и ссылкой на асинхронную операцию.
    Иначе вернёт ответ с кодом ```201``` и ссылкой на созданный ресурс.
    """
    auto_create_app_folder = False
    auto_resolve_app_folder = False
    permissions = DiskWritePermission() | WebDavPermission()
    list_url = '/json/public_list?private_hash=%(public_key)s:%(path)s&amount=1'
    operation_url = '/json/public_copy?uid=%(uid)s&private_hash=%(public_key)s:%(path)s&name=%(name)s'
    async_operation_url = '/json/async_public_copy?uid=%(uid)s&private_hash=%(public_key)s:%(path)s&name=%(name)s'
    query = fields.QueryDict({
        'public_key': PublicKeyField(required=True, help_text=u'Ключ или публичный URL ресурса.'),
        'path': fields.StringField(help_text=u'Путь к копируемому ресурсу в публичной папке.'),
        'save_path': MpfsPathField(help_text=u'Путь к папке, в которую будет сохранен ресурс. По умолчанию «Загрузки».'),
        'name': fields.StringField(help_text=u'Имя, под которым ресурс будет сохранён в папке.'),
    })
    error_map = {
        507: exceptions.DiskStorageQuotaExhaustedError,
        OWNER_HAS_NO_FREE_SPACE: exceptions.DiskOwnerStorageQuotaExhaustedError,
        API_RESTRICTED_FOR_OVERDRAFT_USER: exceptions.DiskAPIDisabledForOverdraftUserError,
    }

    def __init__(self, *args, **kwargs):
        super(SaveToDiskPublicResourceHandler, self).__init__(*args, **kwargs)
        self.path_field = MpfsPathField()

    def _add_save_path(self, base_url):
        save_path = self.request.query.get('save_path')
        if save_path:
            return "%s&save_path=%s" % (base_url, save_path)
        return base_url

    def get_operation_url(self):
        url = super(SaveToDiskPublicResourceHandler, self).get_operation_url()
        return self._add_save_path(url)

    def get_async_operation_url(self):
        url = super(SaveToDiskPublicResourceHandler, self).get_async_operation_url()
        return self._add_save_path(url)

    def get_resource_link(self, obj):
        path = self.path_field.from_native(obj.get('path'))
        link = self.router.get_link(GetResourceHandler, {'path': path})
        return link


class GetTrashResourceHandler(GetResourceHandler):
    """Получить содержимое Корзины"""
    auto_create_app_folder = False
    auto_resolve_app_folder = False
    permissions = DiskReadPermission() | WebDavPermission()
    serializer_cls = TrashResourceSerializer
    service_url = ('/json/list?uid=%(uid)s&path=%(path)s&amount=%(limit)s&offset=%(offset)s&sort=%(sort.field)s&order=%(sort.order)s&'
                   'meta=' + TrashResourceSerializer.get_raw_mpfs_meta(extra_meta=['numchildren']))
    query = fields.QueryDict({
        'path': MpfsPathField(required=True, default='trash:/', allowed_areas=('trash',), default_area='trash',
                              help_text=u'Путь к ресурсу в Корзине.'),
        'sort': MpfsSortChoiseField(choices=['deleted', 'created'], serializer_cls=TrashResourceSerializer,
                                    help_text=u'Поле для сортировки вложенных ресурсов.'),
        'limit': fields.IntegerField(default=20, help_text=u'Количество выводимых вложенных ресурсов.'),
        'offset': fields.IntegerField(default=0, help_text=u'Смещение от начала списка вложенных ресурсов.'),
    })


class RestoreFromTrashHandler(AsyncOperationHandler):
    """
    Восстановить ресурс из Корзины

    Если восстановление происходит асинхронно, то вернёт ответ с кодом ```202``` и ссылкой на асинхронную операцию.
    Иначе вернёт ответ с кодом ```201``` и ссылкой на созданный ресурс.
    """
    auto_create_app_folder = False
    auto_resolve_app_folder = False
    permissions = DiskReadPermission() & DiskWritePermission() | WebDavPermission()
    list_url = '/json/list?uid=%(uid)s&path=%(path)s&amount=1'
    operation_url = '/json/trash_restore?uid=%(uid)s&path=%(path)s&force=%(overwrite)d&name=%(name)s'
    async_operation_url = '/json/async_trash_restore?uid=%(uid)s&path=%(path)s&force=%(overwrite)d&name=%(name)s'
    query = fields.QueryDict({
        'path': MpfsPathField(required=True, allowed_areas=('trash',), default_area='trash',
                              help_text=u'Путь к ресурсу в Корзине.'),
        'name': fields.StringField(help_text=u'Имя, под которым будет восстановлен ресурс.'),
        'overwrite': fields.BooleanField(help_text=u'Перезаписать существующий ресурс восстанавливаемым.'),
    })
    error_map = {
        507: exceptions.DiskStorageQuotaExhaustedError,
        OWNER_HAS_NO_FREE_SPACE: exceptions.DiskOwnerStorageQuotaExhaustedError,
        API_RESTRICTED_FOR_OVERDRAFT_USER: exceptions.DiskAPIDisabledForOverdraftUserError,
    }

    def get_resource_link(self, obj):
        path = self.query.get_fields()['path'].from_native(obj)
        link = self.router.get_link(GetResourceHandler, {'path': path})
        return link


class ClearTrashHandler(AsyncOperationHandler):
    """
    Очистить Корзину

    Если удаление происходит асинхронно, то вернёт ответ со статусом ```202``` и ссылкой на асинхронную операцию.
    Иначе вернёт ответ со статусом ```204``` и пустым телом.

    Если параметр ```path``` не задан или указывает на корень Корзины, то корзина будет полностью очищена,
    иначе из Корзины будет удалён только тот ресурс, на который указывает ```path```.
    """
    auto_create_app_folder = False
    auto_resolve_app_folder = False
    resp_status_code = 204
    permissions = DiskWritePermission() | WebDavPermission()
    list_url = '/json/list?uid=%(uid)s&path=%(path)s&amount=1'

    query = fields.QueryDict({
        'path': MpfsPathField(default='trash:/', allowed_areas=('trash',), default_area='trash',
                              help_text=u'Путь к ресурсу в Корзине.'),
    })

    def get_operation_url(self):
        if self.request.query.get('path') == '/trash/':
            return '/json/trash_drop_all?uid=%(uid)s'
        else:
            return '/json/trash_drop?uid=%(uid)s&path=%(path)s'

    def get_async_operation_url(self):
        if self.request.query.get('path') == '/trash/':
            return '/json/async_trash_drop_all?uid=%(uid)s'
        else:
            return '/json/async_trash_drop?uid=%(uid)s&path=%(path)s'

    @_auto_initialize_user
    def handle(self, request, *args, **kwargs):
        result = super(ClearTrashHandler, self).handle(request, *args, **kwargs)
        if result and isinstance(result, tuple) and len(result) >= 1 and result[0] == 204:
            return None
        return result


class UploadExternalResourceHandler(MpfsProxyWithAppFolderHandler):
    """
    Загрузить файл в Диск по URL

    Загрузка происходит асинхронно. Поэтому в ответ на запрос возвращается ссылка на асинхронную операцию.
    """
    permissions = DiskWritePermission() | DiskAppFolderPermission('path')
    service_url = (
        '/json/async_store_external?'
        'uid=%(uid)s&path=%(path)s&external_url=%(url)s&'
        'disable_redirects=%(disable_redirects)d&disable_retries=1'
    )
    resp_status_code = 202
    serializer_cls = LinkSerializer
    rate_limiter = PerUserRateLimiter('cloud_api_external_upload_user')
    rate_limiter_per_host = PerUploadExternalHostRateLimiter('cloud_api_external_upload_host')
    query = fields.QueryDict({
        'path': MpfsPathField(required=True, help_text=u'Путь, куда будет помещён ресурс.'),
        'url': fields.UrlField(required=True, help_text=u'URL внешнего ресурса, который следует загрузить.'),
        'disable_redirects': fields.BooleanField(
            required=False, default=False, help_text=u'Запретить делать редиректы.'
        ),
    })

    error_map = {
        404: exceptions.DiskPathDoesntExistsError,
        409: exceptions.DiskResourceAlreadyExistsError,
        507: exceptions.DiskStorageQuotaExhaustedError,
        OWNER_HAS_NO_FREE_SPACE: exceptions.DiskOwnerStorageQuotaExhaustedError,
        API_RESTRICTED_FOR_OVERDRAFT_USER: exceptions.DiskAPIDisabledForOverdraftUserError,
    }

    def serialize(self, obj, *args, **kwargs):
        link = self.router.get_link(GetOperationStatusHandler, {'operation_id': obj['oid']})
        return super(UploadExternalResourceHandler, self).serialize(link, *args, **kwargs)

    def check_rate_limit(self, request):
        super(UploadExternalResourceHandler, self).check_rate_limit(request)
        self.rate_limiter_per_host.check(request)


class SmartcacheProxyHandler(CheckUserKarmaHandlerMixin, ServiceProxyHandler):
    """
    Базовый класс для хэндлеров-проксей smartcache
    """
    service = smartcache
    service_base_exception = errors.DataApiBadResponse
    permissions = WebDavPermission() | DiskReadPermission()
    auth_methods = [PassportCookieAuth()]
    error_map = {
        'user-not-found': UnauthorizedError,
        'not-found': exceptions.DiskNotFoundError,
        'forbidden': ForbiddenError,
        404: exceptions.DiskNotFoundError,
    }

    def get_service_error_code(self, exception):
        code = None
        data = getattr(exception, 'data', None)
        if data and isinstance(data, dict):
            code = data.get('code', None)
            if code:
                if isinstance(code, (str, unicode)) and code.isdigit():
                    code = int(code)
        return code


class PhotosliceInitSnapshotHandler(SmartcacheProxyHandler):
    """
    Получить метаданные снапшота (если снапшота не было, то он будет создан)
    """
    service_method = 'GET'
    service_url = '/smartcache/photoslice-init-snapshot?__uid=%(uid)s'
    error_map = {
        'read-only': exceptions.DiskReadOnlyError,
        'databases-count-limit-reached': exceptions.DiskPhotosliceNumberLimitExceededError,
    }
    serializer_cls = PhotosliceSnapshotLinkSerializer

    def get_resource_link(self, photoslice_id, revision):
        params = {
            'photoslice_id': photoslice_id,
            'revision': revision
            }
        link = self.router.get_link(PhotosliceGetSnapshotHandler, params=params)
        return link

    def handle(self, request, *args, **kwargs):
        url = self.get_url(self.get_context())
        resp = self.request_service(url, method=self.service_method)
        result_node = resp['result']
        resp['link'] = self.get_resource_link(result_node['photoslice_id'], result_node['revision'])
        resp = self.serialize(resp)
        return resp


class PhotosliceGetSnapshotHandler(SmartcacheProxyHandler):
    """
    Получить снапшот
    """
    service_method = 'GET'
    service_url = '/smartcache/smartcache-snapshot?__uid=%(uid)s&photoslice-id=%(photoslice_id)s'
    serializer_cls = PhotosliceSnapshotSerializer
    kwargs = fields.QueryDict({
        'photoslice_id': fields.StringField(required=True, help_text=u'Идентификатор фотосреза.'),
    })
    query = fields.QueryDict({
        'revision': fields.IntegerField(required=False, help_text=u'Ревизия снапшота'),
        'cluster_ids': fields.StringField(default=None, required=False, help_text=u'Идентификаторы кластеров, которые надо вернуть'),
        'limit': fields.IntegerField(required=False, help_text=u'Максимальное количество кластеров в ответе'),
        'offset': fields.IntegerField(required=False, help_text=u'Номер первого кластера в ответе'),
    })

    def get_url(self, context=None):
        url = self.service_url
        if 'cluster_ids' in context and context['cluster_ids']:
            url = '%s&cluster_ids=%%(cluster_ids)s' % url

        if context['fields']:
            url = '%s&fields=%s' % (url, ','.join(context['fields']))

        if context['limit']:
            url = '%s&limit=%s' % (url, context['limit'])

        if context['offset']:
            url = '%s&offset=%s' % (url, context['offset'])

        if context['revision']:
            url = '%s&rev=%s' % (url, context['revision'])

        return self.build_url(url, context=context)


class PhotosliceGetDeltaListHandler(SmartcacheProxyHandler):
    """
    Получить список дельт
    """
    service_method = 'GET'
    service_url = '/smartcache/smartcache-deltas-list?__uid=%(uid)s&photoslice-id=%(photoslice_id)s&rev=%(base_revision)s&limit=%(limit)s'
    serializer_cls = PhotosliceDeltaListSerializer
    kwargs = fields.QueryDict({
        'photoslice_id': fields.StringField(required=True, help_text=u'Идентификатор фотосреза.'),
    })
    query = fields.QueryDict({
        'base_revision': fields.IntegerField(required=True, help_text=u'Ревизия снапшота, которая есть у клиента'),
        'limit': fields.IntegerField(required=False, default=100, help_text=u'Максимальное количество дельт, которое требуется вернуть. По-умолчанию 100'),
    })


class ListPhotoAlbumsHandler(ETagHandlerMixin, CustomPreviewSizeHandlerMixin, MpfsProxyHandler):
    """
    Получить список фотоальбомов
    """
    service_method = 'GET'
    permissions = DiskReadPermission() | WebDavPermission()
    service_url = ('/json/albums_list?uid=%(uid)s&'
                   'meta=' + ResourceSerializer.get_raw_mpfs_meta())
    serializer_cls = PhotoAlbumsListSerializer

    query = fields.QueryDict({
        'limit': fields.IntegerField(default=20, help_text=u'Количество выводимых фотоальбомов.'),
        'offset': fields.IntegerField(default=0, help_text=u'Смещение от начала списка фотоальбомов.'),
    })

    def handle(self, request, *args, **kwargs):
        context = self.get_context()
        offset = context['offset']
        limit = context['limit']
        url = self.get_url(context)
        resp = self.request_service(url, method=self.service_method)
        for album in resp:
            if 'cover' in album and album['cover'] is None:
                album.pop('cover')
        prepared_resp = {
            'items': resp[offset:offset + limit],
            'total': len(resp),
            'limit': limit,
            'offset': offset,
        }
        return self.serialize(prepared_resp)


class SharePhotoAlbumHandler(MpfsProxyHandler):
    """
    Опубликовать фотоальбом в социальной сети
    """
    permissions = WebDavPermission()
    service_url = '/json/async_public_album_social_wall_post?uid=%(uid)s&provider=%(provider_id)s&album_id=%(album_id)s'
    resp_status_code = 202
    serializer_cls = LinkSerializer

    kwargs = fields.QueryDict({
        'album_id': fields.StringField(required=True, help_text=u'Идентификатор альбома.'),
        'provider_id': fields.StringField(required=True, help_text=u'Идентификатор социальной сети, в которую публикуется альбом.'),
    })

    error_map = {
        159: exceptions.DiskPhotoAlbumNotPublicError,
        176: exceptions.DiskPhotoAlbumIncorrectProviderError,
        ALBUM_NOT_FOUND: exceptions.DiskPhotoAlbumNotFoundError,
        API_RESTRICTED_FOR_OVERDRAFT_USER: exceptions.DiskAPIDisabledForOverdraftUserError,
    }

    def serialize(self, obj, *args, **kwargs):
        link = self.router.get_link(GetOperationStatusHandler, {'operation_id': obj['oid']})
        return super(SharePhotoAlbumHandler, self).serialize(link, *args, **kwargs)


class ListPhotoAlbumShareProvidersHandler(MpfsProxyHandler):
    """
    Получить список социальных сетей, в которых пользователь может публиковать альбомы
    """
    service_method = 'GET'
    permissions = WebDavPermission()
    service_url = '/json/social_rights?uid=%(uid)s&scenario=wall_post'

    serializer_cls = ShareProvidersListSerializer

    def serialize(self, obj, *args, **kwargs):
        # obj format:
        #  {"vkontakte":{"link":"","auth":true}}
        #  {"vkontakte":{"link":"<auth_link>","auth":false}}

        socials = {
            'items': [
                {'provider_id': provider_id} for provider_id in obj.keys() if obj[provider_id]["auth"]
            ]
        }

        return super(ListPhotoAlbumShareProvidersHandler, self).serialize(socials)


class CreatePhotoAlbumHandler(CustomPreviewSizeHandlerMixin, MpfsProxyHandler):
    """
    Создать новый фотоальбом
    """
    service_method = 'POST'
    permissions = DiskWritePermission() | WebDavPermission()
    service_url = ('/json/albums_create_with_items?uid=%(uid)s&album_type=%(album_type)s&is_public=%(is_public)s&'
                   'meta=' + ResourceSerializer.get_raw_mpfs_meta())
    serializer_cls = PhotoAlbumSerializer
    body_serializer_cls = PhotoAlbumNewSerializer
    error_map = {
        155: exceptions.DiskPhotoAlbumCoverIndexOutOfRangeError,
        API_RESTRICTED_FOR_OVERDRAFT_USER: exceptions.DiskAPIDisabledForOverdraftUserError,
        ALBUM_ALREADY_EXISTS: exceptions.DiskAlbumAlreadyExists,
    }
    query = fields.QueryDict({
        'album_type': fields.StringField(required=False, default='personal', help_text=u'Тип альбома.'),
    })

    def get_service_error_code(self, exception):
        return self.get_mpfs_error_code(exception)

    def get_context(self, context=None):
        c = super(CreatePhotoAlbumHandler, self).get_context()
        c['is_public'] = int(self.request.body.get('is_public', True))
        return c

    def handle(self, request, *args, **kwargs):
        context = self.get_context()
        url = self.get_url(context)
        try:
            resp = self.request_service(url, method=self.service_method, data=self.request.body,
                                        headers={'Content-type': 'application/json'})
        except MpfsProxyBadResponse as e:
            res = from_json(e.data['text'])
            if 'data' in res and 'album_id' in res['data']:
                raise exceptions.DiskAlbumAlreadyExists(album_id=res['data']['album_id'])
            else:
                raise
        return self.serialize(resp)


class BulkCreatePhotoAlbumItemHandler(CustomPreviewSizeHandlerMixin, MpfsProxyHandler):
    """
    Добавить список элементов в фотоальбом
    """
    service_method = 'GET'
    permissions = DiskWritePermission() | WebDavPermission()
    service_url = ('/json/album_append_items?uid=%(uid)s&album_id=%(album_id)s&if_not_exists=1&'
                   'meta=' + ResourceSerializer.get_raw_mpfs_meta())
    serializer_cls = PhotoAlbumWithItemsSerializer
    body_serializer_cls = PhotoAlbumItemsNewSerializer
    kwargs = fields.QueryDict({
        'album_id': fields.StringField(required=True, help_text=u'Идентификатор альбома.'),
    })
    error_map = {
        ALBUM_NOT_FOUND: exceptions.DiskPhotoAlbumNotFoundError,
    }

    def handle(self, request, *args, **kwargs):
        context = self.get_context()
        req_body = self.request.body
        if len(req_body['resources']) > 100:
            raise FieldValidationError(name='resources',
                                       message=u'лимит 100 ресурсов за запрос',
                                       description='limit 100 resources per request')
        mpfs_body = {
            'items': [{"type": "resource", "path": item["path"]}
                      for item in req_body['resources']]
        }
        url = self.get_url(context)
        resp = self.request_service(url, method=self.service_method, data=mpfs_body)
        return self.serialize(resp)


class CreatePhotoAlbumItemHandler(CustomPreviewSizeHandlerMixin, MpfsProxyHandler):
    """
    Добавить элемент в фотоальбом
    """
    service_method = 'POST'
    permissions = DiskWritePermission() | WebDavPermission()
    service_url = ('/json/album_append_item?uid=%(uid)s&album_id=%(album_id)s&type=resource&path=%(path)s&'
                   'if_not_exists=1&meta=' + ResourceSerializer.get_raw_mpfs_meta())
    serializer_cls = PhotoAlbumItemSerializer
    body_serializer_cls = PhotoAlbumItemNewSerializer
    kwargs = fields.QueryDict({
        'album_id': fields.StringField(required=True, help_text=u'Идентификатор альбома.'),
    })
    error_map = {
        ALBUM_NOT_FOUND: exceptions.DiskPhotoAlbumNotFoundError,
        ALBUMS_UNABLE_TO_APPEND_ITEM: exceptions.DiskAlbumsUnableToAppendItem
    }

    def handle(self, request, *args, **kwargs):
        context = self.get_context()
        context['path'] = self.request.body['path']
        url = self.get_url(context)
        resp = self.request_service(url, method=self.service_method, data=self.request.body)
        return self.serialize(resp)


class DeletePhotoAlbumItemHandler(CustomPreviewSizeHandlerMixin, MpfsProxyHandler):
    """Удалить элемент из фотоальбома"""

    permissions = DiskWritePermission() | WebDavPermission()
    resp_status_code = 204
    service_url = '/json/album_item_remove?uid=%(uid)s&album_id=%(album_id)s&item_id=%(item_id)s'
    kwargs = fields.QueryDict({
        'album_id': fields.StringField(required=True, help_text=u'Идентификатор альбома.'),
        'item_id': fields.StringField(required=True, help_text=u'Идентификатор элемента альбома.'),
    })
    error_map = {
        ALBUM_NOT_FOUND: exceptions.DiskPhotoAlbumNotFoundError,
    }


class ListPhotoAlbumItemsHandler(ETagHandlerMixin, CustomPreviewSizeHandlerMixin, MpfsProxyHandler):
    """
    Получить список элементов фотоальбома
    """
    service_method = 'GET'
    permissions = DiskReadPermission() | WebDavPermission()
    service_url = ('/json/album_get?uid=%(uid)s&album_id=%(album_id)s&amount=%(limit)s&last_item_id=%(last_item_id)s&'
                   'meta=' + ResourceSerializer.get_raw_mpfs_meta())
    serializer_cls = PhotoAlbumItemsListSerializer

    query = fields.QueryDict({
        'limit': fields.IntegerField(default=20, help_text=u'Количество элементов на странице.'),
        'last_item_id': fields.StringField(default=u"", help_text=u'Идентификатор последнего элемента полученного на предыдущей странице.'),
    })
    kwargs = fields.QueryDict({
        'album_id': fields.StringField(required=True, help_text=u'Идентификатор альбома.'),
    })
    error_map = {
        ALBUM_NOT_FOUND: exceptions.DiskPhotoAlbumNotFoundError,
    }

    def serialize(self, obj, *args, **kwargs):
        context = self.get_context()
        obj = {
            'items': obj['items'],
            'limit': context['limit'],
            'last_item_id': context['last_item_id'],
        }
        return super(ListPhotoAlbumItemsHandler, self).serialize(obj, *args, **kwargs)


class GetPhotoAlbumItemHandler(ETagHandlerMixin, CustomPreviewSizeHandlerMixin, MpfsProxyHandler):
    """
    Получить информацию об элементе фотоальбома
    """
    service_method = 'GET'
    permissions = DiskReadPermission() | WebDavPermission()
    service_url = ('/json/album_item_info?uid=%(uid)s&album_id=%(album_id)s&item_id=%(item_id)s&'
                   'meta=' + ResourceSerializer.get_raw_mpfs_meta())
    serializer_cls = PhotoAlbumItemSerializer
    kwargs = fields.QueryDict({
        'album_id': fields.StringField(required=True, help_text=u'Идентификатор альбома.'),
        'item_id': fields.StringField(required=True, help_text=u'Идентификатор элемента.'),
    })
    error_map = {
        ALBUM_NOT_FOUND: exceptions.DiskPhotoAlbumNotFoundError,
    }


class GetPhotoAlbumHandler(ETagHandlerMixin, CustomPreviewSizeHandlerMixin, MpfsProxyHandler):
    """Получить фотоальбом"""

    service_method = 'GET'
    permissions = DiskReadPermission() | WebDavPermission()
    service_url = ('/json/album_get?uid=%(uid)s&album_id=%(album_id)s&limit=0'
                   'meta=' + ResourceSerializer.get_raw_mpfs_meta())
    serializer_cls = PhotoAlbumSerializer
    kwargs = fields.QueryDict({
        'album_id': fields.StringField(required=True, help_text=u'Идентификатор альбома.'),
    })
    error_map = {
        ALBUM_NOT_FOUND: exceptions.DiskPhotoAlbumNotFoundError,
    }


class ChangePhotoAlbumHandler(CustomPreviewSizeHandlerMixin, MpfsProxyHandler):
    """Изменить атрибуты фотоальбома"""

    permissions = DiskWritePermission() | WebDavPermission()

    _meta_fields = '&meta=' + ResourceSerializer.get_raw_mpfs_meta()
    # no_items=1 везде, т.к. PhotoAlbumSerializer не использует поле items, а его получение -- медленная операция
    service_url = '/json/album_set_attr?uid=%(uid)s&album_id=%(album_id)s&no_items=1' + _meta_fields
    unpublish_url = '/json/album_unpublish?uid=%(uid)s&album_id=%(album_id)s&no_items=1' + _meta_fields
    publish_url = '/json/album_publish?uid=%(uid)s&album_id=%(album_id)s&no_items=1' + _meta_fields

    serializer_cls = PhotoAlbumSerializer
    body_serializer_cls = PhotoAlbumPatchSerializer
    kwargs = fields.QueryDict({
        'album_id': fields.StringField(required=True, help_text=u'Идентификатор альбома.'),
    })
    error_map = {
        ALBUM_NOT_FOUND: exceptions.DiskPhotoAlbumNotFoundError,
        ALBUMS_UNABLE_TO_PUBLISH_UNSAVED: exceptions.DiskAlbumUnableToPublish,
    }

    def get_url(self, context=None):
        """Добавляем в query string аргументы из тела запроса"""
        url = super(ChangePhotoAlbumHandler, self).get_url(context=context)
        parsed_url = urlparse.urlsplit(url)

        query = urlparse.parse_qs(parsed_url.query)
        query.update(self.request.body)

        for k, v in query.iteritems():
            if isinstance(v, unicode):
                query[k] = v.encode('utf-8')

        url_parts = list(parsed_url)
        url_parts[-2] = urllib.urlencode(query, doseq=True)
        return urlparse.urlunsplit(url_parts)

    def handle(self, request, *args, **kwargs):
        context = self.get_context()

        attributes = self.request.body.copy()

        is_public = attributes.pop('is_public', None)
        if is_public is not None:

            url = self.unpublish_url
            if is_public is True:
                url = self.publish_url

            url = self.build_url(url, context=context)
            resp = self.request_service(url, method='GET')
            if not attributes:
                return self.serialize(resp)

        context.update(attributes)
        url = self.get_url(context)
        resp = self.request_service(url, method=self.service_method)
        return self.serialize(resp)


class DeletePhotoAlbumHandler(CustomPreviewSizeHandlerMixin, MpfsProxyHandler):
    """Удалить фотоальбом"""

    permissions = DiskWritePermission() | WebDavPermission()
    service_url = '/json/album_remove?uid=%(uid)s&album_id=%(album_id)s'
    resp_status_code = 204
    kwargs = fields.QueryDict({
        'album_id': fields.StringField(required=True, help_text=u'Идентификатор альбома.'),
    })
    error_map = {
        ALBUMS_UNABLE_TO_DELETE: exceptions.DiskAlbumUnableToDelete,
        ALBUM_NOT_FOUND: exceptions.DiskPhotoAlbumNotFoundError,
    }


class OrganizationBaseHander(MpfsProxyHandler):
    YAMB_CLIENT_IDS = settings.platform['organizations']['allowed_client_ids']
    permissions = AllowByClientIdPermission(YAMB_CLIENT_IDS)
    kwargs = fields.QueryDict({
        'organization_id': fields.StringField(required=True)
    })

    @property
    @_auto_initialize_user
    def organization_group(self):
        attr_name = '_organization_group_cache'
        if not hasattr(self.request, attr_name):
            url = '/json/info?uid=%(organization_id)s&path=%(organization_base_path)s&meta=group'
            url = self.build_url(url, {
                'organization_base_path': '/disk/.organization',
                'organization_id': self.request.kwargs['organization_id']
            })
            group = self.request_service(url, method='GET')
            setattr(self.request, attr_name, group)
        return getattr(self.request, attr_name, None)

    @property
    @_auto_initialize_user
    def user_links(self):
        attr_name = '_user_links_cache'
        if not hasattr(self.request, attr_name):
            url = '/json/share_list_all_folders?uid=%(uid)s&meta=group'
            url = self.build_url(url, {'uid': self.request.user.uid})
            links = self.request_service(url, method='GET')
            setattr(self.request, attr_name, {l['meta']['group']['gid']: l for l in links})
        return getattr(self.request, attr_name, None)

    @property
    def user_organization_group(self):
        """Папка организации к которй у пользователя есть доступ."""
        gid = self.organization_group['meta']['group']['gid']
        if gid in self.user_links:
            return self.organization_group
        else:
            return None

    def has_access_to_organization(self):
        return bool(self.user_organization_group)

    def has_access_to_organization_path(self, mpfs_path):
        if mpfs_path.startswith(self.user_organization_group['path']):
            return True
        return False

    def check_permissions(self, request):
        super(OrganizationBaseHander, self).check_permissions(request)

        # у пользователя нет доступа к организации
        if request.kwargs.get('organization_id') and not self.has_access_to_organization():
            self.permission_denied(request)

        # в запросе указан путь, но у пользователя нет доступа к этому пути
        path = request.query.get('path')
        if path and not self.has_access_to_organization_path(path):
            self.permission_denied(request)


class ListOrganizationsHandler(OrganizationBaseHander):
    """Список организаций доступных пользователю"""
    serializer_cls = OrganizationsListSerializer
    kwargs = fields.QueryDict({
        'organization_id': None,  # отключаем унаследованный параметр
    })

    @_auto_initialize_user
    def handle(self, request, *args, **kwargs):
        organizations = []

        # проверяем наличие организаций в линках пользователя
        links = self.user_links.viewvalues()
        url_tpl = '/json/share_folder_info?uid=%(owner_uid)s&gid=%(gid)s'
        for link in links:
            organization_id = link['meta']['group']['owner']['uid']
            url = self.build_url(url_tpl, self.get_context({
                'owner_uid': organization_id,
                'gid': link['meta']['group']['gid']
            }))
            resp = self.request_service(url)
            path = resp.get('id')
            if path and path.startswith('/disk/.organization'):
                organizations.append({'organization_id': organization_id})

        # # проверяем не является ли пользователь сам организацией
        # url = '/json/info?uid=%(uid)s&path=%(path)s'
        # url = self.build_url(url, self.get_context({'path': '/disk/.organization'}))
        # try:
        #     status_code, _, _ = self.raw_request_service(url)
        #     if status_code == 200:
        #         organizations.append({'organization_id': request.user.uid})
        # except self.service_base_exception:
        #     pass

        return self.serialize({'items': organizations})


class CreateOrganizationHandler(MpfsProxyHandler):
    """
    Создать организацию

    Пока создаёт организацию из пользователя сделавшего запрос.
    """
    permissions = OrganizationBaseHander.permissions
    resp_status_code = 201
    serializer_cls = LinkSerializer

    @_auto_initialize_user
    def handle(self, request, *args, **kwargs):
        # создаём корневую папку организации
        url = '/json/mkdir?uid=%(uid)s&path=%(path)s'
        url = self.build_url(url, self.get_context({'path': '/disk/.organization'}))
        self.request_service(url)

        # шарим папку
        url = '/json/share_create_group?uid=%(uid)s&path=%(path)s'
        url = self.build_url(url, self.get_context({'path': '/disk/.organization'}))
        self.request_service(url)

        link = self.router.get_link(GetOrganizationHandler, {'organization_id': request.user.uid})
        return self.serialize(link)


class GetOrganizationHandler(OrganizationBaseHander):
    """Метаданные организации"""
    serializer_cls = OrganizationSerializer

    def handle(self, request, *args, **kwargs):
        return self.serialize(request.kwargs)


class CreateOrganizationFolderHandler(OrganizationBaseHander):
    """Создать папку"""
    resp_status_code = 201
    query = fields.QueryDict({
        'path': OrganizationPathField(required=True, allowed_areas=['disk'], help_text=u'Путь к создаваемой папке.')
    })
    serializer_cls = LinkSerializer

    def handle(self, request, *args, **kwargs):
        # создаём корневую папку организации
        url = '/json/mkdir?uid=%(organization_id)s&path=%(path)s'
        url = self.build_url(url, self.get_context())
        self.request_service(url)

        path = self.query.get_fields()['path'].from_native(request.query['path'])
        link = self.router.get_link(GetOrganizationResourceHandler,
                                    {'organization_id': request.kwargs['organization_id'], 'path': path})
        return self.serialize(link)


class GetOrganizationResourceHandler(OrganizationBaseHander, GetResourceHandler):
    """Получить ресурс организации"""
    query = fields.QueryDict({
        'path': OrganizationPathField(required=True, allowed_areas=['disk'], help_text=u'Путь к ресурсу.')
    })
    serializer_cls = OrganizationResourceSerializer
    service_url = ('/json/list?uid=%(organization_id)s&path=%(path)s&amount=%(limit)s&offset=%(offset)s&sort=%(sort.field)s&order=%(sort.order)s&'
                   'meta=' + ResourceSerializer.get_raw_mpfs_meta(extra_meta=['numchildren']))


class GetOrganizationUploadLinkHandler(OrganizationBaseHander):
    """Получить ссылку на загрузку файла"""
    serializer_cls = LinkSerializer
    service_url = '/json/store?uid=%(uid)s&path=%(path)s&force=%(overwrite)d'
    query = fields.QueryDict({
        'path': OrganizationPathField(required=True, help_text=u'Путь к загружаемому файлу на Диске.'),
        'overwrite': fields.BooleanField(help_text=u'Перезаписать существующий файл.'),
    })

    error_map = {
        405: exceptions.DiskPathPointsToExistentDirectoryError,
        409: exceptions.DiskPathDoesntExistsError,
        412: exceptions.DiskResourceAlreadyExistsError,
        507: exceptions.DiskStorageQuotaExhaustedError,
        OWNER_HAS_NO_FREE_SPACE: exceptions.DiskOwnerStorageQuotaExhaustedError,
        API_RESTRICTED_FOR_OVERDRAFT_USER: exceptions.DiskAPIDisabledForOverdraftUserError,
    }

    def get_context(self, context=None):
        c = super(GetOrganizationUploadLinkHandler, self).get_context(context=context)
        # подменяем путь в диске организации на путь в диске пользователя иначе ничерта не загрузится по ссылке
        org_path = self.request.query['path']
        user_path = None
        if org_path.startswith(self.organization_group['path']):
            link = self.user_links.get(self.organization_group['meta']['group']['gid'])
            if link:
                user_path = org_path.replace(self.organization_group['path'].rstrip('/'), link['path'].rstrip('/'))
        if not user_path:
            raise exceptions.DiskPathDoesntExistsError()
        else:
            c['path'] = user_path
        return c

    def serialize(self, obj, *args, **kwargs):
        link = ('PUT', obj.get('upload_url'), False,)
        return super(OrganizationBaseHander, self).serialize(link, *args, **kwargs)


class GetOrganizationDownloadLinkHandler(OrganizationBaseHander):
    """Получить ссылку на скачивание ресурса"""
    service_url = '/json/url?uid=%(uid)s&path=%(path)s'
    serializer_cls = LinkSerializer
    query = fields.QueryDict({
        'path': OrganizationPathField(required=True, help_text=u'Путь к ресурсу.'),
    })

    def get_context(self, context=None):
        c = super(GetOrganizationDownloadLinkHandler, self).get_context(context=context)
        # подменяем путь в диске организации на путь в диске пользователя иначе ничерта не загрузится по ссылке
        org_path = self.request.query['path']
        user_path = None
        if org_path.startswith(self.organization_group['path']):
            link = self.user_links.get(self.organization_group['meta']['group']['gid'])
            if link:
                user_path = org_path.replace(self.organization_group['path'].rstrip('/'), link['path'].rstrip('/'))
        if not user_path:
            raise exceptions.DiskPathDoesntExistsError()
        else:
            c['path'] = user_path
        return c

    def serialize(self, obj, *args, **kwargs):
        url = obj.get('file', obj.get('folder', None))
        link = ('GET', url, False)
        return super(GetOrganizationDownloadLinkHandler, self).serialize(link, *args, **kwargs)


class GrantAccessToOrganizationFolderHandler(OrganizationBaseHander):
    """Предоставить пользователю доступ в папку организации"""
    query = fields.QueryDict({
        'path': OrganizationPathField(required=True, help_text=u'Путь к ресурсу.'),
        'email': fields.StringField(help_text=u'E-mail пользователя, которого нужно добавить в папку.'),
        'uid': fields.StringField(help_text=u'UID пользователя, которого нужно добавить в папку.'),
    })

    def handle(self, request, *args, **kwargs):
        if request.query['email']:
            user_info = passport.userinfo(login=request.query['email'])
        elif request.query['uid']:
            user_info = passport.userinfo(uid=request.query['uid'])
        else:
            raise FieldRequiredError()

        invited_email = user_info['email']
        invited_uid = user_info['uid']

        url = '/json/share_invite_user?uid=%(organization_id)s&gid=%(gid)s&rights=660&universe_login=%(invited_email)s&universe_service=email'
        url = self.build_url(url, self.get_context({'gid': self.organization_group['meta']['group']['gid'], 'invited_email': invited_email}))
        invite = self.request_service(url)

        url = '/json/share_activate_invite?uid=%(invited_uid)s&hash=%(hash)s&meta=group,uid'
        url = self.build_url(url, self.get_context({'invited_uid': invited_uid, 'hash': invite['hash']}))
        self.request_service(url)

        return '{}'  # не знал что вернуть, поэтому пока ручки не отдаём наружу, пусть будет так


class WOPIURLResourcesHandler(MpfsProxyHandler):
    """Получить ссылку для работы по протоколу WOPI с данным файлом
    """
    permissions = DenyAllPermission()
    service_url = '/json/info?uid=%(uid)s&path=%(path)s&unzip_file_id=1&meta=file_id,group'
    query = fields.QueryDict({
        'path': fields.StringField(required=True, help_text=u'Путь к ресурсу.')
    })
    serializer_cls = LinkSerializer

    @_auto_initialize_user
    def handle(self, request, *args, **kwargs):
        context = self.get_context()
        url = self.get_url(context)
        info = self.request_service(url)

        if info['type'] != u'file':
            raise errors.ResourceNotFound()

        owner_uid = request.user.uid
        if 'group' in info['meta']:
            owner_uid = info['meta']['group']['owner']['uid']

        file_id = info['meta']['file_id']

        resource_id = make_resource_id(owner_uid, file_id)
        access_token = self._get_access_token()
        return self.serialize(self._make_wopi_url(resource_id, access_token))

    def _make_wopi_url(self, resource_id, access_token):
        """Создать URL из базового урла к API и resource_id.

        :type resource_id: str
        :type access_token: str
        :rtype: str
        """
        # TODO
        # Исправить циклический импорт.
        # Например, MpfsProxyHandler вынести в отдельный модуль
        from mpfs.platform.v1.wopi.handlers import CheckFileInfoHandler
        return self.router.get_link(
            CheckFileInfoHandler,
            {'resource_id': resource_id, 'access_token': access_token},
            force_request_mode=tags.platform.EXTERNAL
        )

    @staticmethod
    def _get_access_token():
        """Сгенерить/вернуть access_token.

        :rtype: str
        """
        # TODO
        # Написать правильную реализацию когда будет сделана авторизация
        return '123'


class ImportMailAttachHandler(MpfsProxyHandler):
    """Импортировать прикреплённый в почте документ в Диск."""
    service_method = 'GET'
    permissions = DiskWritePermission() | WebDavPermission()
    service_url = ('/json/import_attach_to_disk/?'
                   'uid=%(uid)s&service_id=mail&dst=%(path)s&mail_mid=%(mail_mid)s&mail_hid=%(mail_hid)s&'
                   'overwrite=%(overwrite)d&autosuffix=%(autosuffix)d')
    query = fields.QueryDict({
        'overwrite': fields.BooleanField(default=False, required=False,
                                         help_text=u'Перезаписывать файл, если он существует.'),
        'autosuffix': fields.BooleanField(
            default=False, required=False,
            help_text=u'Добавить суффикс к имени файла, если файл с таким именем уже существует.'),
        'path': MpfsPathField(required=True, help_text=u'Путь, по которому файл будет сохранён в Диске.'),
        'service_file_id': MailAttachServiceFileIDField(
            required=True,
            help_text=u'Идентификатор аттача в почте.'
        ),
    })

    error_map = {
        409: exceptions.DiskResourceAlreadyExistsError,
        403: ForbiddenError,
        API_RESTRICTED_FOR_OVERDRAFT_USER: exceptions.DiskAPIDisabledForOverdraftUserError,
    }

    def get_context(self, context=None):
        context = super(ImportMailAttachHandler, self).get_context(context=context)
        mail_mid, mail_hid = context['service_file_id'].split(MailAttachServiceFileIDField.service_file_id_sep)
        context['mail_mid'] = mail_mid
        context['mail_hid'] = mail_hid
        return context


class GetResourceDetailHandler(CustomPreviewSizeHandlerMixin, MpfsProxyHandler):
    """Получить ресурс по его идентификатору."""
    service_method = 'POST'
    hidden = True
    permissions = DiskReadPermission() | WebDavPermission()
    kwargs = fields.QueryDict({
        'resource_id': ResourceIdField(required=True),
    })
    enabled_areas = (DISK_AREA_PATH, PHOTOUNLIM_AREA_PATH)
    serializer_cls = ResourceSerializer
    error_map = {
        403: exceptions.DiskUserBlockedError,
    }

    def initialize(self, *args, **kwargs):
        self.service_url = (
            '/json/bulk_info_by_resource_ids?enable_service_ids='
            + ','.join(self.enabled_areas)
            + '&uid=%(uid)s&meta='
            + self.serializer_cls.get_raw_mpfs_meta(extra_meta=['numchildren']))
        super(GetResourceDetailHandler, self).initialize(*args, **kwargs)

    @_auto_initialize_user
    def handle(self, request, *args, **kwargs):
        resource_id = request.kwargs['resource_id']
        raw_resource_id = resource_id.serialize()
        context = self.get_context()

        if JAVA_DJFS_API_PROXY_PLATFORM_BULK_INFO_ENABLED:
            # djfs-specific logic
            params = {'enable_service_ids': ','.join(self.enabled_areas),
                      'uid': context['uid']}
            meta = self.serializer_cls.get_raw_mpfs_meta(extra_meta=['numchildren']).split(',')
            resp = djfs_api.bulk_info_by_resource_ids(params, meta, '["%s"]' % raw_resource_id)
        else:
            # MPFS-specific logic
            url = self.get_url(context)
            resp = self.request_service(
                url,
                method=self.service_method,
                data=[raw_resource_id],
                headers={'Content-Type': 'application/json'},
            )

        if not resp:
            raise DiskNotFoundError()
        if len(resp) > 1:
            raise ValueError()

        resp = self.serialize(resp[0])
        return resp


class DeleteResourceByResourceIdHandler(DeleteResourceHandler):
    """Удалить файл или папку по его идентификатору"""
    hidden = True
    permissions = WebDavPermission()

    error_map = {
        FOLDER_DELETION_BY_RESOURCE_ID_FORBIDDEN: exceptions.DiskFolderDeletionByResourceIdForbiddenError,
    }

    query = fields.QueryDict({
        'path': None,
        'delete_all': fields.BooleanField(help_text=u'Удалить все ресурсы, попадающие под фильтр.', default=False),
        'md5': fields.StringField(help_text=u'Фильтрация удаляемых ресурсов по значению md5.'),
        'sha256': fields.StringField(help_text=u'Фильтрация удаляемых ресурсов по значению sha256.'),
        'size': fields.IntegerField(help_text=u'Фильтрация удаляемых ресурсов по размеру.'),
    })

    kwargs = fields.QueryDict({
        'resource_id': ResourceIdField(required=True),
    })

    def get_operation_url(self):
        if self.request.query.get('permanently'):
            return '/json/rm_by_resource_id?uid=%(uid)s&resource_id=%(resource_id)s&rm_all=%(rm_all)s&files_only=1'
        else:
            return '/json/trash_append_by_resource_id?uid=%(uid)s&resource_id=%(resource_id)s&append_all=%(append_all)s&files_only=1'

    def get_async_operation_url(self):
        raise NotImplemented()

    def handle(self, request, *args, **kwargs):
        mpfs_delete_all = 1 if request.query.get('delete_all') else 0
        context = self.urlencode_context(self.get_context({
            'rm_all': mpfs_delete_all,
            'append_all': mpfs_delete_all,
        }))

        url = self.get_operation_url() % context
        if context.get('md5'):
            url += '&md5=%s' % context['md5']
        if context.get('sha256'):
            url += '&sha256=%s' % context['sha256']
        if context.get('size'):
            url += '&size=%s' % context['size']
        self.request_service('%s%s' % (self.service.base_url, url))


class GetNotesResourceDetailHandler(GetResourceDetailHandler):
    """Получить ресурс аттачмента к заметке по его идентификатору."""
    hidden = True
    permissions = WebDavPermission() | MobileMailPermission() | (NotesReadPermission() & NotesWritePermission())
    enabled_areas = (NOTES_AREA_PATH,)


class GetResourceDimensionsHandler(MpfsProxyHandler):
    """Получить ширину и высоту изображения."""
    hidden = True
    permissions = DiskReadPermission() | WebDavPermission()
    kwargs = fields.QueryDict({
        'resource_id': ResourceIdField(required=True),
    })
    service_base_exception = errors.APIError
    serializer_cls = ResourceDimensionsSerializer
    error_map = {
        503: ServiceUnavailableError,
        404: NotFoundError,
        API_RESTRICTED_FOR_OVERDRAFT_USER: exceptions.DiskAPIDisabledForOverdraftUserError,
    }

    @_auto_initialize_user
    def handle(self, request, *args, **kwargs):
        resource_id = request.kwargs['resource_id']
        file_id = resource_id.file_id
        # если здесь будет не 200, то получим MpfsProxyBadResponse
        resp = self.raw_request_service(
            url=self.build_url('/json/image_dimensions_by_file_id?uid=%s&file_id=%s' % (request.user.uid, file_id)),
            method='GET',
            headers={'Content-Type': 'application/json'}
        )
        info = from_json(resp[1])
        return 200, self.serialize(info)


class GetVersionsHandler(MpfsProxyHandler):
    """Получить версии ресурса."""
    hidden = True
    permissions = AllowByClientIdPermission(PLATFORM_DISK_APPS_IDS)
    service_url = '/json/versioning_get_checkpoints?uid=%(uid)s&resource_id=%(resource_id)s&iteration_key=%(iteration_key)s'
    kwargs = fields.QueryDict({
        'resource_id': ResourceIdField(required=True),
    })
    query = fields.QueryDict({
        'iteration_key': fields.StringField(help_text=u'Ключ для получения следующей пачки версий.',
                                            default=''),
    })
    service_base_exception = errors.APIError
    serializer_cls = ResourceVersionsSerializer


class GetFoldedVersionsHandler(MpfsProxyHandler):
    """Получить свернутые версии ресурса."""
    hidden = True
    permissions = AllowByClientIdPermission(PLATFORM_DISK_APPS_IDS)
    service_url = '/json/versioning_get_folded?uid=%(uid)s&resource_id=%(resource_id)s&iteration_key=%(iteration_key)s'
    kwargs = fields.QueryDict({
        'resource_id': ResourceIdField(required=True),
    })
    query = fields.QueryDict({
        'iteration_key': fields.StringField(help_text=u'Ключ для получения следующей пачки версий.',
                                            default=''),
    })
    service_base_exception = errors.APIError
    serializer_cls = ResourceVersionsSerializer
    error_map = {
        400: BadRequestError,
    }


class GetVersionHandler(MpfsProxyHandler):
    """Получить версию ресурса."""
    hidden = True
    permissions = AllowByClientIdPermission(PLATFORM_DISK_APPS_IDS)
    service_url = '/json/versioning_get_version?uid=%(uid)s&resource_id=%(resource_id)s&version_id=%(version_id)s'
    kwargs = fields.QueryDict({
        'resource_id': ResourceIdField(required=True),
        'version_id': fields.StringField(required=True, help_text=u'ID версии.'),
    })
    service_base_exception = errors.APIError
    serializer_cls = ResourceVersionSerializer
    error_map = {
        409: VersionNotFoundError,
    }


class RestoreVersionHandler(MpfsProxyHandler):
    """Восстановить ресурс по версии."""
    hidden = True
    permissions = AllowByClientIdPermission(PLATFORM_DISK_APPS_IDS)
    service_url = '/json/versioning_restore?uid=%(uid)s&resource_id=%(resource_id)s&version_id=%(version_id)s&meta=' + \
                  ResourceSerializer.get_raw_mpfs_meta()
    resp_status_code = 201
    kwargs = fields.QueryDict({
        'resource_id': ResourceIdField(required=True),
        'version_id': fields.StringField(required=True, help_text=u'ID версии.'),
    })
    service_base_exception = errors.APIError
    serializer_cls = ResourceSerializer
    error_map = {
        409: VersionNotFoundError,
        423: exceptions.DiskResourceLockedError,
        GROUP_NOT_PERMIT: DiskNoWritePermissionForSharedFolderError,
    }


class CopyFromVersionHandler(MpfsProxyHandler):
    """Скопировать ресурс по версии."""
    hidden = True
    permissions = AllowByClientIdPermission(PLATFORM_DISK_APPS_IDS)
    service_url = '/json/versioning_save?uid=%(uid)s&resource_id=%(resource_id)s&version_id=%(version_id)s&meta=' + \
                  ResourceSerializer.get_raw_mpfs_meta()
    resp_status_code = 201
    kwargs = fields.QueryDict({
        'resource_id': ResourceIdField(required=True),
        'version_id': fields.StringField(required=True, help_text=u'ID версии.'),
    })
    service_base_exception = errors.APIError
    serializer_cls = ResourceSerializer
    error_map = {
        409: VersionNotFoundError,
        423: exceptions.DiskResourceLockedError,
        GROUP_NOT_PERMIT: DiskNoWritePermissionForSharedFolderError,
    }


class GetImageMetadataHandler(GetResourceDimensionsHandler):
    permissions = WebDavPermission()

    @_auto_initialize_user
    def handle(self, request, *args, **kwargs):
        resource_id = request.kwargs['resource_id']
        file_id = resource_id.file_id
        # если здесь будет не 200, то получим MpfsProxyBadResponse
        resp = self.raw_request_service(
            url=self.build_url('/json/image_metadata_by_file_id?uid=%s&file_id=%s&orientation=1' % (request.user.uid, file_id)),
            method='GET',
            headers={'Content-Type': 'application/json'}
        )
        info = from_json(resp[1])
        return 200, self.serialize(info)


class PutSnapshotHandler(MpfsProxyHandler):
    """Получить снепшот ресурсов пользователя.
    """
    error_map = {
        REQUESTS_LIMIT_EXCEEDED_429: TooManyRequestsError,
    }

    service_method = 'POST'

    rate_limiter = PerUserRateLimiter('cloud_api_user_snapshot')
    if SNAPSHOT_PER_HANDLER_RATE_LIMITER_ENABLED:
        rate_limiter = PerHandlerRandomUserRateLimiter(
            'cloud_api_random_user_snapshot',
            users_count=SNAPSHOT_PER_HANDLER_RATE_LIMITER_USERS_COUNT)
    hidden = True
    permissions = WebDavPermission()
    service_url = '/json/snapshot?uid=%(uid)s'
    serializer_cls = SnapshotSerializer
    body_serializer_cls = SnapshotIterationKeySerializer
    service_timeout = 600

    def get_context(self, context=None):
        context = super(PutSnapshotHandler, self).get_context(context)
        session_id = UserAgentParser.get_session_id(self.request.raw_headers.get('user-agent', ''))
        context['session_id'] = session_id
        return context

    def get_url(self, context=None, base_url=None):
        url = self.service_url
        if context.get('session_id', None):
            url += '&session_id=%(session_id)s' % self.urlencode_context({'session_id': context['session_id']})
        return self.build_url(url, context, base_url)

    @_auto_initialize_user
    def handle(self, request, *args, **kwargs):
        context = self.get_context()
        url = self.get_url(context)
        data = self.request_service(url,
                                    method=self.service_method,
                                    data=self.request.body,
                                    headers={'Content-type': 'application/json'})
        return self.serialize(data)


class GetDeltasHandler(MpfsProxyHandler):
    """Получить изменения начиная с переданной ревизии."""
    rate_limiter = PerPercentUserHandlerRateLimiter('cloud_api_user_deltas', DELTAS_PER_USER_HANDLER_LIMITER_PERCENT, exceeded_limit_exception_cls=ServiceUnavailableError)
    if DELTAS_PER_HANDLER_RATE_LIMITER_ENABLED:
        rate_limiter = PerHandlerRandomUserRateLimiter(
            'cloud_api_random_user_deltas',
            users_count=DELTAS_PER_HANDLER_RATE_LIMITER_USERS_COUNT)
    hidden = True
    permissions = WebDavPermission()
    service_url = ('/json/deltas?uid=%(uid)s&base_revision=%(base_revision)s&'
                   'meta=' + DeltaResourceSerializer.get_raw_mpfs_meta())
    serializer_cls = DeltasSerializer
    query = fields.QueryDict({
        'base_revision': fields.IntegerField(required=False, help_text=u'Номер текущей ревизии клиента.'),
        'iteration_key': fields.StringField(required=False, default='', help_text=u'Номер текущей ревизии клиента.'),
        'allow_moved_deltas': fields.BooleanField(
            required=False, default=False, help_text=u'Разрешить дельты о перемещении папки.'),
    })
    error_map = {
        404: exceptions.DiskRevisionNotFoundError,
        API_RESTRICTED_FOR_OVERDRAFT_USER: exceptions.DiskAPIDisabledForOverdraftUserError,
    }

    def get_url(self, context=None, base_url=None):
        url = self.service_url
        if context.get('allow_moved_deltas', False):
            url += '&allow_quick_move_deltas=1'
        return self.build_url(url, context, base_url)


class SearchResourcesIterationKey(object):
    _format_tmpl = '%i;%i'

    def __init__(self, limit, offset):
        self.limit = limit
        self.offset = offset

    @classmethod
    def parse(cls, raw_iteration_key):
        parts = raw_iteration_key.split(';')
        if len(parts) != 2:
            raise ValueError('Bad raw iteration_key %s' % raw_iteration_key)
        parts = [int(p) for p in parts]
        return cls(*parts)

    def serialize(self):
        return self._format_tmpl % (self.limit, self.offset)


class AbstractSearchResourceHandler(GetResourceHandler):
    """Базовый класс для поиска ресурсов"""
    hidden = True

    serializer_cls = SearchResourceListSerializer


    def handle(self, request, *args, **kwargs):
        context = self.get_context()
        raw_iteration_key = context['iteration_key']
        if raw_iteration_key:
            iteration_key = SearchResourcesIterationKey.parse(raw_iteration_key)
        else:
            iteration_key = SearchResourcesIterationKey(context['limit'], context['offset'])

        context['limit'] = iteration_key.limit
        context['offset'] = iteration_key.offset
        url = self.get_url(context)
        resp = self.request_service(url, method=self.service_method)

        lost_results_count = int(resp.get('lost_results_count', 0))
        # mpfs всегда возвращает root папку, поэтому -1
        if len(resp['results']) - 1 + lost_results_count == iteration_key.limit:
            next_iteration_key = SearchResourcesIterationKey(iteration_key.limit,
                                                             iteration_key.offset + iteration_key.limit)
            resp['iteration_key'] = next_iteration_key.serialize()
        return self.serialize(resp)

    def serialize(self, obj, *args, **kwargs):
        # убираем корневой ресурс, который всегда добавляется в new_search
        obj['results'] = obj['results'][1:]
        new_obj = {
            'limit': self.request.query['limit'],
            'offset': self.request.query['offset'],
            'items': obj['results']
        }
        if 'iteration_key' in obj:
            new_obj['iteration_key'] = obj['iteration_key']
        result = super(AbstractSearchResourceHandler, self).serialize(new_obj, *args, **kwargs)

        if 'path' in result:
            del result['path']

        return result


class SearchResourcesHandler(AbstractSearchResourceHandler):
    permissions = (
        WebDavPermission() | DiskSearchPermission()
    )
    query = fields.QueryDict({
        'path': None,
        'query': fields.StringField(required=True),
        'sort': None,
        'force': None,
        'iteration_key': fields.StringField(required=False, default='', help_text=u'Ключ для получения следующей страницы.'),
    })

    service_url = (
        '/json/new_search?uid=%(uid)s&path=/disk&count_lost_results=1&'
        'query=%(query)s&amount=%(limit)s&offset=%(offset)s&'
        'meta=' + SearchResourceSerializer.get_raw_mpfs_meta()
    )


class GeoSearchResourceHandler(AbstractSearchResourceHandler):
    permissions = (
        DiskReadPermission()
    )

    query = fields.QueryDict({
        'path': None,
        'latitude': fields.FloatField(required=False, help_text=u'Широта точки поиска.'),
        'longitude': fields.FloatField(required=False, help_text=u'Долгота точки поиска.'),
        'distance': fields.IntegerField(required=False, help_text=u'Максимальная удаленность от точки поиска.'),
        'start_date': fields.DateTimeToTSField(required=True, help_text=u'Нижняя граница времени для поиска.'),
        'end_date': fields.DateTimeToTSField(required=True, help_text=u'Верхняя граница времени для поиска.'),
        'iteration_key': fields.StringField(required=False, default='', help_text=u'Ключ для получения следующей страницы.'),
    })

    service_url = (
        '/json/geo_search?uid=%(uid)s&'
        'start_date=%(start_date)s&end_date=%(end_date)s&'
        'count_lost_results=1&amount=%(limit)s&offset=%(offset)s&'
        'meta=' + SearchResourceSerializer.get_raw_mpfs_meta()
    )

    def clean_query(self, raw_query):
        ret = super(AbstractSearchResourceHandler, self).clean_query(raw_query)

        latitude = ret.get('latitude')
        longitude = ret.get('longitude')
        distance = ret.get('distance')
        if (latitude is None or longitude is None or distance is None) \
            and (latitude is not None or longitude is not None or distance is not None):
            raise FieldsAllOrNoneValidationError(fields_names=['latitude', 'longitude', 'distance'])

        return ret

    def get_url(self, context = None):
        url = super(AbstractSearchResourceHandler, self).get_url(context)
        distance = context['distance']
        longitude = context['longitude']
        latitude = context['latitude']
        if context and latitude and longitude and distance:
            url = self.patch_url_params(url, {'latitude': latitude, 'longitude': longitude, 'distance': distance})
        return url


class UploadAttachHandler(UploadResourceHandler):
    """Загрузить аттач."""
    hidden = True
    permissions = DiskAttachWritePermission()
    service_url = '/json/attach_store?uid=%(uid)s&path=%(path)s&force=%(overwrite)d&' \
                  'md5=%(md5)s&sha256=%(sha256)s&size=%(size)d'
    query = fields.QueryDict({
        'path': MpfsPathField(
            required=True, allowed_areas=('attach',), default_area='attach',
            help_text=u'Путь к ресурсу в папке аттачей.'
        ),
    })


class UploadClientHandler(UploadResourceHandler):
    """Загрузить информацию из ФОС."""
    hidden = True
    auto_create_app_folder = False
    auto_resolve_app_folder = False
    permissions = WebDavPermission()
    service_url = '/json/store?uid=%(uid)s&path=%(path)s&fos_reply_email=%(reply_email)s&' \
                  'fos_app_version=%(app_version)s&fos_os_version=%(os_version)s&' \
                  'fos_subject=%(subject)s&fos_expire_seconds=%(expire_seconds)s&' \
                  'fos_recipient_type=%(recipient_type)s&skip_check_space=%(skip_check_space)s&' \
                  'skip_speed_limit=%(skip_speed_limit)s'
    service_method = 'POST'
    service_headers = {'Content-Type': 'application/json'}
    body_serializer_cls = ClientBodySerializer
    error_map = {
        CLIENT_BAD_REQUEST: exceptions.DiskClientBadRequest,
        API_RESTRICTED_FOR_OVERDRAFT_USER: exceptions.DiskAPIDisabledForOverdraftUserError,
    }
    query = fields.QueryDict({
        'app_version': fields.StringField(required=True, help_text=u'Версия приложения: номер и beta/prod'),
        'reply_email': fields.EmailField(required=True, help_text=u'Email, на который прислать ответ'),
        'recipient_type': fields.ChoiceField(required=True, choices=('testers', 'supports'), help_text=u'Получатель обращения'),
        'expire_seconds': fields.IntegerField(default=864000, help_text=u'Время доступности файла по ссылке'),
        'os_version': fields.StringField(required=True, help_text=u'Версия ОС на устройстве'),
        'subject': fields.StringField(required=True, help_text=u'Тема, выбранная из списка жалоб', unquote=True),
        'path': None,
    })

    def get_context(self, context=None):
        c = super(UploadClientHandler, self).get_context(context=context)
        c['path'] = '/client/Report-%s.zip' % datetime.now().strftime('%Y-%m-%d-%H-%M')
        c['skip_check_space'] = '1'
        c['skip_speed_limit'] = '1'
        return c

    def handle(self, request, *args, **kwargs):
        context = self.get_context()
        service_url = self.build_url(self.service_url, context=context)
        store_response = self.request_service(service_url,
                                              method=self.service_method,
                                              data=self.request.data,
                                              headers=self.service_headers)
        if store_response.get('status') == 'hardlinked':
            resource_link = self.get_resource_link()
            return 201, LinkSerializer(resource_link).data
        upload_link = ('PUT', store_response.get('upload_url'), False,)
        operation_link = self.router.get_link(GetOperationStatusHandler, {'operation_id': store_response.get('oid')})
        return self.serialize({'upload_link': upload_link, 'operation_link': operation_link})


def load_disk_android_videounlim_alert_uids():
    result = set()
    try:
        with open(DISK_ANDROID_VIDEOUNLIM_ALERT_FILE_PATH, 'r') as fh:
            for uid in fh:
                uid = uid.strip()
                if uid:
                    result.add(uid)
    except IOError:
        pass
    return result


class GetExperimentsHandler(BasePlatformHandler):
    auth_required = False
    auth_user_required = False
    hidden = True
    permissions = AllowAllPermission()
    serializer_cls = ExperimentsSerializer
    query = fields.QueryDict({
        'new': fields.IntegerField(required=False, default=0, help_text=u'Использовать новый uaas'),
    })

    disk_android_videounlim_alert_uids = load_disk_android_videounlim_alert_uids()

    def handle(self, request, *args, **kwargs):
        raw_user_agent = request.raw_headers.get('user-agent', '')

        uid = None
        if request.user:
            uid = request.user.uid

        is_mpfs_experiment = None

        remote_ip = request.get_real_remote_addr()
        additional_headers = {'X-Forwarded-For-Y': remote_ip}
        try:
            experiments = list(new_uaas.get_disk_experiments(raw_user_agent, uid, additional_headers=additional_headers))
            if not is_enabled_users_experiment(experiments, 'disk_forbidden_video_unlim', '190458'):
                experiments.append({
                    'CONTEXT': {
                        'DISK': {
                            'flags': ['disk_forbidden_video_unlim'], 'testid': ['190458']
                        }
                    },
                    'HANDLER': 'DISK'
                })
        except errors.UAASRequestInvalidClient:
            raise exceptions.DiskExperimentsUnsupportedDevice('Unsupported client device')

        if self.is_telemost_exp_needed_for(uid, experiments):
            experiments.append({'CONTEXT': {'DISK': {'flags': ['disk_telemost'],
                                                     'testid': ['241028']}},
                                'HANDLER': 'DISK'})

        if uid in self.disk_android_videounlim_alert_uids:
            experiments.append({'CONTEXT': {'DISK': {'flags': ['disk_android_videounlim_alert'],
                                                     'testid': ['269976']}},
                                'HANDLER': 'DISK'})

        # remove specific experiments listed in config
        filtered_experiments = []
        for exp in experiments:
            if self.is_experiment_excluded_for(exp, uid):
                continue
            filtered_experiments.append(exp)
        experiments = filtered_experiments

        log_data = to_json({'uid': uid, 'is_mpfs_experiment': is_mpfs_experiment, 'uas_exp_flags': experiments})
        default_log.info(log_data)
        return self.serialize({'uas_exp_flags': experiments})

    @staticmethod
    def is_telemost_exp_needed_for(uid, experiments):
        if not uid:
            return False

        user_info = passport.userinfo(uid)

        if bool(user_info.get('has_telemost')) or bool(user_info.get('has_staff')):
            return True

        if 'ru' != user_info.get('language', '').lower():
            return False

        experiment_testids = GetExperimentsHandler._get_testids_from_experiments(experiments)

        for testid in PLATFORM_TELEMOST_ENABLED_EXPERIMENTS_TEST_IDS:
            if testid in experiment_testids:
                return True

        return False

    @staticmethod
    def _get_testids_from_experiments(experiments):
        experiment_testids = [exp.get('CONTEXT', {}).get('DISK', {}).get('testid', []) for exp in experiments]
        flatten_experiment_testids = []
        for testids in experiment_testids:
            for testid in testids:
                flatten_experiment_testids.append(testid)
        return flatten_experiment_testids

    @staticmethod
    def is_experiment_excluded_for(exp, uid):
        for disabled_exp_settings in PLATFORM_EXCLUDE_EXPS:
            try:
                for testid in disabled_exp_settings['testids']:
                    if (testid in exp['CONTEXT']['DISK']['testid'] and
                            uid in disabled_exp_settings['for_uids']):
                        return True
            except Exception:
                pass
        return False


class GetUserFeatureTogglesHandler(MpfsProxyHandler):
    """Получить статус фич пользователя"""
    need_auto_initialize_user = False
    hidden = True
    permissions = AllowByClientIdPermission(PLATFORM_DISK_APPS_IDS) | AllowByClientIdPermission(PLATFORM_PS_APPS_IDS)
    service_url = '/json/user_feature_toggles?uid=%(uid)s'
    serializer_cls = UserFeatureTogglesSerializer


class GetUserActivityInfoHandler(MpfsProxyHandler):
    """Получить информацию об активности пользователя"""
    need_auto_initialize_user = False
    hidden = True
    permissions = WebDavPermission()
    service_url = '/json/user_activity_info?uid=%(uid)s'
    serializer_cls = UserActivityInfoSerializer


class GetClientsInstallerHandler(MpfsProxyHandler):
    """Получить ссылку на инсталлятор ПО"""
    hidden = True
    auth_required = False
    permissions = AllowAllPermission()
    serializer_cls = ClientsInstallerSerializer
    kwargs = fields.QueryDict({
        'platform_id': fields.ChoiceField(
            choices=SOFTWARE_INSTALLER_PATH.keys(),
            required=True,
            help_text=u"ID платформы ПО"),
    })
    query = fields.QueryDict({
        'build': fields.ChoiceField(
            choices=SOFTWARE_INSTALLER_PATH['win64'].keys(),
            required=False,
            default='stable',
            help_text=u"Тип инсталлера: alpha - тестовая, beta - внутри компании, stable - внешняя"),
    })
    service_url = '/json/info?uid=%(uid)s&path=%(path)s&meta=version|urn:yandex:disk:dist,cdn|urn:yandex:disk:dist:url,sha256,md5,file_url,size'
    hbf_service = HbfService()
    installers_path = SOFTWARE_INSTALLER_PATH

    def get_context(self, context=None):
        result_context = super(GetClientsInstallerHandler, self).get_context(context=context)
        result_context['uid'] = PUBLIC_UID
        return result_context

    def _handle(self, context=None, base_url=None, installer_type='stable'):
        if JAVA_DJFS_API_PROXY_PLATFORM_INFO_ENABLED and installer_type == 'stable':
            params = {'uid': context['uid'],
                      'path': context['path']}
            request_meta = ['version|urn:yandex:disk:dist', 'cdn|urn:yandex:disk:dist:url', 'sha256', 'md5', 'file_url',
                            'size']
            result = djfs_api.info(params, request_meta)
        else:
            url = self.get_url(context=context, base_url=base_url)
            result = self.request_service(url)
        meta = result['meta']
        file_url = meta['file_url']
        download_url = meta.get('cdn|urn:yandex:disk:dist:url', file_url)
        response = {'download_url': download_url,
                    'version': meta['version|urn:yandex:disk:dist'],
                    'md5': meta['md5'],
                    'sha256': meta['sha256'],
                    'size': meta['size']}
        return self.serialize(response)

    def handle(self, request, *args, **kwargs):
        context = self.get_context()
        platform_id = self.request.kwargs['platform_id']
        installer_type = 'stable'
        base_url = None
        if request.get_real_remote_addr() in self.hbf_service.networks[SOFTWARE_INSTALLER_ALLOWED_NETWORK_MACROS]:
            installer_type = self.request.query['build']
            if installer_type != 'stable':
                base_url = SOFTWARE_INSTALLER_TEST_BASE_URL

        context['path'] = self.installers_path[platform_id][installer_type]

        if base_url is not None:
            try:
                return self._handle(context=context, base_url=base_url, installer_type=installer_type)
            except errors.APIError:
                pass

        return self._handle(context)


class GetClientsInstallerWithAutologonHandler(MpfsProxyHandler):
    """Получить ссылку на инсталлятор ПО с автологином"""
    hidden = True
    permissions = AllowAllPermission()
    auth_required = False
    auth_user_required = False
    serializer_cls = ClientsInstallerWithAutologonSerializer

    query = fields.QueryDict({
        'build': fields.ChoiceField(
            choices=SOFTWARE_INSTALLER_WITH_AUTOLOGON_PATH['win'].keys(),
            required=False,
            default='stable',
            help_text=u'Тип инсталлера: alpha - тестовая, beta - внутри компании, stable - внешняя.'),
    })
    service_url = '/json/list_installers?uid=%(uid)s&al=%(al)s'
    installer_paths = SOFTWARE_INSTALLER_WITH_AUTOLOGON_PATH['win']

    def get_context(self, context=None):
        result_context = super(GetClientsInstallerWithAutologonHandler, self).get_context(context=context)
        result_context['al'] = '1'
        if not self.request.user:
            result_context['uid'] = PUBLIC_UID
            result_context['al'] = '0'
        return result_context

    def handle(self, request, *args, **kwargs):
        context = self.get_context()
        url = self.get_url(context)

        resp = self.request_service(url)
        # если в ответе нет нашего инстолятора, то падаем в 5xx, т.к. такого не должно быть
        download_url = [item['meta']['file_url']
                        for item in resp
                        if item['id'] == self.installer_paths[context['build']]][0]

        return self.serialize({'file': download_url})


class GDPRElectrichkiHandler(MpfsProxyWithAppFolderHandler):
    """Получить данные про электрички"""
    auth_required = True
    auth_methods = [TVM2Auth()]
    service_url = '/json/gdpr_takeout_electrichki?uid=%(uid)s'

    def serialize(self, obj, *args, **kwargs):
        link = ('GET', obj, False)
        return super(GDPRElectrichkiHandler, self).serialize(link, *args, **kwargs)


class OnlyOfficeCallbackHandler(MpfsProxyHandler):
    permissions = AllowAllPermission()
    auth_required = False
    hidden = True
    service_url = '/json/office_only_office_callback?uid=%(uid)s&oo_key=%(key)s&token=%(token)s&subdomain=%(subdomain)s'
    body_serializer_cls = OnlyOfficeCallbackBodySerializer
    query = fields.QueryDict({
        'token': fields.StringField(required=True, help_text=u'token'),
        'uid': fields.StringField(required=True, help_text=u'uid'),
        'subdomain': fields.StringField(required=True, help_text=u'subdomain')
    })
    kwargs = fields.QueryDict({
        'key': fields.StringField(help_text=u'Идентификатор редактирования документа в OnlyOffice'),
    })
    rate_limiter = None

    def handle(self, request, *args, **kwargs):
        context = self.get_context()
        service_url = self.build_url(self.service_url, context=context)
        if 'oid' in request.args:
            service_url = self.patch_url_params(service_url, {'oid': request.args['oid']})
        return self.request_service(service_url,
                                    method=self.service_method,
                                    data=self.request.data,
                                    headers=self.service_headers)


class GetDocsFileFilters(MpfsProxyHandler):
    """Получить фильтры на файлы для просмотра и редактирования."""
    hidden = True
    permissions = WebDavPermission()
    service_url = '/json/office_get_file_filters'
    serializer_cls = OfficeFileFiltersSerializer


class GetDocsFileURLsHandler(MpfsProxyHandler):
    """Получить ссылки на редактор и просмотр файла."""
    hidden = True
    permissions = WebDavPermission()
    service_url = '/json/office_get_file_urls?uid=%(uid)s&tld=%(tld)s&resource_id=%(resource_id)s'
    serializer_cls = OfficeFileURLsSerializer
    kwargs = fields.QueryDict({
        'resource_id': ResourceIdField(required=True),
    })
    query = fields.QueryDict({
        'tld': fields.StringField(
            default=USER_DEFAULT_TLD,
            help_text=u'Домен верхнего уровня для генерируемой ссылки.'
        )
    })

    error_map = {
        415: UnsupportedMediaTypeError
    }


class GetPublicSettingsHandler(MpfsProxyHandler):
    """Получить настройки публичных ссылок"""

    hidden = True
    permissions = WebDavPermission()
    service_url = '/json/get_public_settings?uid=%(uid)s&path=%(path)s'
    query = fields.QueryDict({
        'path': MpfsPathField(required=True, allowed_areas=('disk', 'app', PHOTOUNLIM_AREA), help_text=u'Путь к публикуемому ресурсу.',
                              only_official_clients_areas=(PHOTOUNLIM_AREA,)),
    })


class SetPublicSettingsHandler(MpfsProxyHandler):
    """Задать настройки публичных ссылок"""

    hidden = True
    permissions = WebDavPermission()
    service_url = '/json/set_public_settings?uid=%(uid)s&path=%(path)s'
    body_serializer_cls = SetPublicSettingsSerializer
    query = fields.QueryDict({
        'path': MpfsPathField(required=True, allowed_areas=('disk', 'app', PHOTOUNLIM_AREA), help_text=u'Путь к публикуемому ресурсу.',
                              only_official_clients_areas=(PHOTOUNLIM_AREA,)),
    })

    def handle(self, request, *args, **kwargs):
        context = self.get_context()
        url = self.get_url(context)
        req_body = self.request.body
        if 'available_until' in req_body and isinstance(req_body['available_until'], dict):
            available_until = req_body.get('available_until')
            if available_until.get('enabled') is True:
                req_body['available_until'] = int(available_until['value'])
            elif available_until.get('enabled') is False:
                req_body['available_until'] = None

        resp = self.request_service(url, method=self.service_method, data=req_body,
                                    headers={'Content-type': 'application/json'})
        return self.serialize(resp)


class OfficeSetAccessStateHandler(MpfsProxyHandler):
    """Задать настройки публичных ссылок офиса"""

    hidden = True
    permissions = WebDavPermission()
    service_url = '/json/office_set_access_state?uid=%(uid)s&resource_id=%(resource_id)s&access_state=%(access_state)s&set_office_selection_strategy=%(set_office_selection_strategy)s&connection_id=%(connection_id)s&region=%(region)s'
    kwargs = fields.QueryDict({
        'resource_id': ResourceIdField(required=True),
    })
    query = fields.QueryDict({
        'access_state': fields.StringField(required=True, help_text=u'Режим доступа'),
        'set_office_selection_strategy': fields.StringField(required=False, help_text=u'Стратегия выбора редактора'),
        'connection_id': fields.StringField(required=False, help_text=u'Идентификатор соединения'),
        'region': fields.StringField(required=False, help_text=u'Регион'),
    })
