# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import urllib
import urlparse

from mpfs.common.util.mobile_client import MobileClientVersion
from mpfs.config import settings
from mpfs.core.services.lenta_loader_service import lenta_loader
from mpfs.common.util.filetypes import builtin_extensions
from mpfs.common.util.urls import urlencode
from mpfs.common.util.user_agent_parser import UserAgentParser, UserAgent
from mpfs.core.user.constants import PHOTOUNLIM_AREA
from mpfs.platform import fields
from mpfs.platform.auth import PassportCookieAuth
from mpfs.platform.common import PlatformResponse
from mpfs.platform.v1.disk.fields import MpfsPathField
from mpfs.platform.exceptions import FieldOneRequiredValidationError, ServiceUnavailableError
from mpfs.platform.handlers import BasePlatformHandler, ETagHandlerMixin
from mpfs.platform.permissions import AllowByClientIdPermission
from mpfs.platform.v1.disk.handlers import MpfsProxyHandler, ServiceProxyHandler
from mpfs.platform.v1.disk.serializers import ResourceSerializer, PhotoAlbumWithItemsSerializer, PhotoAlbumSerializer
from mpfs.platform.v1.disk.lenta.exceptions import LentaEmptyBlockError, LentaTooLargeBlockSizeError
from mpfs.platform.v1.disk.lenta.serializers import LentaResourceSerializer, LentaBlockPublicLinkBodySerializer


PLATFORM_DISK_APPS_IDS = settings.platform['disk_apps_ids']

SUPPORTED_IOS_VERSION = MobileClientVersion.build_from_version(settings.promo['unlimited_autoupload']['ios_version_support'])
SUPPORTED_ANDROID_VERSION = MobileClientVersion.build_from_version(settings.promo['unlimited_autoupload']['android_version_support'])


class LentaResourcesHandler(ETagHandlerMixin, MpfsProxyHandler):
    """
    Получить содержимое блока Ленты.
    """
    hidden = True
    service_method = 'GET'
    permissions = AllowByClientIdPermission(PLATFORM_DISK_APPS_IDS)
    MEDIA_TYPE_CHOICES = sorted(builtin_extensions.keys())
    serializer_cls = LentaResourceSerializer
    service_url = (
        '/json/lenta_block_list?uid=%(uid)s&amount=%(limit)s&offset=%(offset)s'
        '&mtime_gte=%(modified_gte)s'
        '&meta=' + serializer_cls.get_raw_mpfs_meta(extra_meta=['numchildren', 'total_results_count'])
    )
    query = fields.QueryDict({
        'modified_gte': fields.DateTimeToTSField(
            required=True, help_text='Минимальное значение ```modified``` файлов, которые попадут в листинг.'
        ),
        'modified_lte': fields.DateTimeToTSField(
            help_text='Максимальное значение ```modified``` файлов, которые попадут в листинг.'
        ),
        'path': MpfsPathField(allowed_areas=('disk', PHOTOUNLIM_AREA), help_text='Путь к папке, по которой требуется получить листинг.'),
        'resource_id': fields.StringField(help_text='Идентификатор ресурса.'),
        'media_type': fields.MultipleChoicesField(
            choices=MEDIA_TYPE_CHOICES,
            help_text='Фильтр по типу файлов (поддерживается список через запятую).'
        ),
        'limit': fields.IntegerField(default=100,
                                     help_text='Максимальное количество записей на странице (по умолчанию 100).'),
        'offset': fields.IntegerField(default=0, help_text='Отступ от начала выборки.'),
        'iteration_key': fields.StringField(help_text='Ключ итерации'),
        'lenta_block_id': fields.StringField(help_text='Идентификатор блока в Ленте.'),
        'modified_by_uid': fields.StringField(help_text='Идентификатор пользователя, изменившего ресурс последним.')
    })

    def clean_query(self, raw_query):
        ret = super(LentaResourcesHandler, self).clean_query(raw_query)

        if not (ret.get('path') or ret.get('resource_id')):
            raise FieldOneRequiredValidationError(fields_names=['path', 'resource_id'])

        return ret

    def get_url(self, context=None):
        url = super(LentaResourcesHandler, self).get_url(context=context)

        url_chunks = list(urlparse.urlparse(url))
        query_params = urlparse.parse_qs(url_chunks[4].encode('ascii'))  # its safe because URL contains only ascii

        path = self.request.query.get('path')
        resource_id = self.request.query.get('resource_id')
        if path:
            query_params['path'] = path
        elif resource_id:
            query_params['resource_id'] = resource_id

        media_type = self.request.query.get('media_type')
        if media_type:
            query_params['media_type'] = ','.join(media_type)

        iteration_key = self.request.query.get('iteration_key')
        if iteration_key:
            query_params['iteration_key'] = iteration_key

        lenta_block_id = self.request.query.get('lenta_block_id')
        if lenta_block_id:
            query_params['block_id'] = lenta_block_id

        modify_uid = self.request.query.get('modified_by_uid')
        if modify_uid:
            query_params['modify_uid'] = modify_uid

        modified_lte = self.request.query.get('modified_lte')
        if modified_lte:
            query_params['mtime_lte'] = modified_lte

        raw_user_agent = self.request.raw_headers.get('user-agent')
        user_agent = UserAgentParser.parse(raw_user_agent)
        if user_agent.is_mobile_client():
            query_params['filter_photounlim'] = int(not self._does_mobile_client_support_photounlim(user_agent))

        url_chunks[4] = urllib.urlencode(query_params, doseq=True)
        return urlparse.urlunparse(url_chunks)

    def serialize(self, obj, *args, **kwargs):
        """
        Предварительно отсериализовать.

        В случае, если на вход подан список, то превращает его в дикт с вложенными элементами.
        """
        if isinstance(obj, list):
            res = obj[0]
            assert isinstance(res, dict)
            if '_embedded' not in res:
                res['_embedded'] = {}
            embedded = {
                'items': obj[1:],
                'offset': self.request.query['offset'],
                'limit': self.request.query['limit'],
                'total': res['meta'].get('total_results_count'),
                'iteration_key': res['meta'].get('iteration_key')
            }
            res['_embedded'].update(embedded)
        else:
            res = obj
        return super(LentaResourcesHandler, self).serialize(res, *args, **kwargs)

    @staticmethod
    def _does_mobile_client_support_photounlim(user_agent):
        version = user_agent.get_version()
        if not version:
            return True
        if (user_agent.is_android() and version >= SUPPORTED_ANDROID_VERSION
                or user_agent.is_ios() and version >= SUPPORTED_IOS_VERSION):
            return True
        return False


class LentaReportBadBlockHandler(BasePlatformHandler):
    """Сообщить о плохом блоке в Ленте."""
    permissions = AllowByClientIdPermission(PLATFORM_DISK_APPS_IDS)
    auth_methods = [PassportCookieAuth()]

    def handle(self, request, *args, **kwargs):
        qs = urllib.urlencode(request.args, doseq=True)
        query = urlparse.parse_qs(qs.encode('ascii'))
        query['uid'] = [str(request.user.uid)]
        qs = urllib.urlencode(query, doseq=True)
        status_code, content, headers = lenta_loader.delete_empty_block(qs)
        return PlatformResponse(status_code=status_code, content=content, headers=headers)


class LentaDeleteBlockHandler(ServiceProxyHandler):
    """Удалить блок в Ленте."""
    hidden = True
    permissions = AllowByClientIdPermission(PLATFORM_DISK_APPS_IDS)
    service_base_exception = lenta_loader.api_error
    error_map = {
        500: ServiceUnavailableError
    }
    auth_methods = [PassportCookieAuth()]

    kwargs = fields.QueryDict({
        'lenta_block_id': fields.StringField(required=True, help_text='Идентификатор блока в Ленте.')
    })

    def get_service_error_code(self, exception):
        return getattr(exception, 'status_code', None)

    def handle(self, request, *args, **kwargs):
        uid = self.request.user.uid
        block_id = self.request.kwargs['lenta_block_id']
        lenta_loader.delete_block(uid=uid, block_id=block_id)
        return 204, None


class LentaCreateAlbumFromBlockHandler(LentaResourcesHandler):
    permissions = AllowByClientIdPermission(PLATFORM_DISK_APPS_IDS)
    serializer_cls = PhotoAlbumWithItemsSerializer
    service_url = (
        '/json/lenta_create_album_from_block?'
        'uid=%(uid)s&amount=%(limit)s&offset=%(offset)s'
        '&mtime_gte=%(modified_gte)s'
        '&title=%(title)s&no_items=0&layout=%(layout)s'  # настройки для альбома
        '&meta=' + ResourceSerializer.get_raw_mpfs_meta()
    )
    # flags в настоящий момент в альбомы не прокидываются, будут использованы дефолтные
    query = fields.QueryDict({
        'title': fields.StringField(
            required=True, help_text='Название для создаваемого альбома.'
        ),
        'layout': fields.ChoiceField(choices=['rows', 'squares', 'waterfall', 'fit_width'],
                                     default='waterfall',
                                     help_text='Вид обложки альбома.')
    })
    error_map = {
        205: LentaTooLargeBlockSizeError,
        206: LentaEmptyBlockError
    }

    def get_url(self, context=None):
        url = super(LentaCreateAlbumFromBlockHandler, self).get_url(context=context)
        return self.patch_url_params(url, {k: v for (k, v) in {
            'if_not_exists': self.request.query.get('if_not_exists'),
            'preview_size': self.request.query.get('preview_size'),
            'preview_crop': self.request.query.get('preview_crop'),
            'preview_quality': self.request.query.get('preview_quality'),
            'preview_allow_big_size': self.request.query.get('preview_allow_big_size')
        }.iteritems() if v is not None})


class LentaGenerateBlockHandler(ServiceProxyHandler):
    hidden = True
    permissions = AllowByClientIdPermission(PLATFORM_DISK_APPS_IDS)
    service = lenta_loader
    service_base_exception = lenta_loader.api_error
    service_method = 'GET'
    service_url = '/api/generate-block?uid=%(uid)s&count=%(items_limit)i'
    resp_status_code = 204

    query = fields.QueryDict({
        'items_limit': fields.IntegerField(default=10, help_text='Максимально число элементов в блоке.'),
    })
    def handle(self, request, *args, **kwargs):
        super(LentaGenerateBlockHandler, self).handle(request, *args, **kwargs)
        return ''


class LentaCreateBlockHandler(ServiceProxyHandler):
    """Создать блок в Ленте."""
    hidden = True
    permissions = AllowByClientIdPermission(PLATFORM_DISK_APPS_IDS)
    service = lenta_loader
    service_base_exception = lenta_loader.api_error
    service_method = 'POST'
    service_url = '/api/blocks/types/%(lenta_block_type)s?uid=%(uid)s'
    auth_methods = [PassportCookieAuth()]

    kwargs = fields.QueryDict({
        'lenta_block_type': fields.StringField(required=True, help_text='Тип блока Ленты.')
    })

    def get_service_error_code(self, exception):
        return getattr(exception, 'status_code', None)

    def get_url(self, context=None):
        url = super(LentaCreateBlockHandler, self).get_url(context=context)
        # пробрасываем в Ленту всё содержимое query string, но следим чтоб честный
        # uid подставлялся
        url_chunks = self.request.request_uri.split('?', 1)
        if len(url_chunks) == 2:
            url += '&%s' % url_chunks[1]
        url = self.patch_url_params(url, {'uid': self.request.user.uid})
        return url

    def raw_request_service(self, *args, **kwargs):
        status_code, content, headers = super(LentaCreateBlockHandler, self).raw_request_service(*args, **kwargs)
        if status_code != 200:
            raise ServiceUnavailableError()
        return status_code, content, headers


class LentaBlockPublicLinkHandler(MpfsProxyHandler):
    permissions = AllowByClientIdPermission(PLATFORM_DISK_APPS_IDS)
    serializer_cls = PhotoAlbumSerializer
    body_serializer_cls = LentaBlockPublicLinkBodySerializer
    service_url = '/json/lenta_block_public_link?uid=%(uid)s'

    def handle(self, request, *args, **kwargs):
        url = self.get_url(self.get_context())
        resp = self.request_service(url, method=self.service_method, data=request.body)
        resp = self.serialize(resp)
        return resp
