# -*- coding: utf-8 -*-
from mpfs.common.util import to_json
from mpfs.common.errors import NotifierBadResponse
from mpfs.common.util.user_agent_parser import UserAgentParser, UserAgent
from mpfs.core.services.notifier_service import notifier
from mpfs.core.services.push_service import XivaSubscribeNotifierService
from mpfs.platform import fields
from mpfs.platform.auth import PassportCookieAuth
from mpfs.platform.fields import QueryDict
from mpfs.platform.handlers import ServiceProxyHandler
from mpfs.platform.permissions import AllowByClientIdPermission
from mpfs.platform.v1.data import handlers


class BaseNotificationsHandler(ServiceProxyHandler):
    permissions = AllowByClientIdPermission(['disk_verstka_ext'])
    auth_methods = [PassportCookieAuth()]
    service = notifier
    service_base_exception = NotifierBadResponse

    def get_service_error_code(self, exception):
        # пока нам ни чего не известно о списке ошибок, так что пятисотим в любой непонятной ситуации
        return None

    kwargs = QueryDict({
        'resource_id': fields.StringField(help_text=u'Идентификатор ресурса.', required=True),
    })

    query = QueryDict({
        'notification_entity_type': fields.StringField(help_text=u'Тип сущности нотификации.'),
    })

    def get_url(self, context=None):
        url = super(BaseNotificationsHandler, self).get_url(context=context)
        optional_params = {}

        notification_groups = context.get('notification_groups')
        if notification_groups:
            optional_params['notification_groups'] = ','.join(notification_groups)

        notification_entity_type = context.get('notification_entity_type')
        if notification_entity_type:
            optional_params['resource_type'] = notification_entity_type

        if optional_params:
            url = self.patch_url_params(url, optional_params)

        return url

    def serialize(self, obj, *args, **kwargs):
        obj = obj.get('result', {})
        return super(BaseNotificationsHandler, self).serialize(obj, *args, **kwargs)


class ManageResourceNotificationsHandler(BaseNotificationsHandler):
    query = QueryDict({
        'notification_groups': fields.MultipleChoicesField(help_text=u'Список групп нотификаций.',
                                                           choices=notifier.NOTIFICATION_GROUPS),
    })


class EnableResourceNotificationsHandler(ManageResourceNotificationsHandler):
    """Включить отправку нотификаций для ресурса."""
    service_url = '/enable-resource-notifications?owner_uid=%(uid)s&resource_id=%(resource_id)s'


class DisableResourceNotificationsHandler(ManageResourceNotificationsHandler):
    """Выключить отправку нотификаций для ресурса."""
    service_url = '/disable-resource-notifications?owner_uid=%(uid)s&resource_id=%(resource_id)s'


class GetResourceNotificationsSettingsHandler(BaseNotificationsHandler):
    """Получить параметры нотификаций"""
    service_url = '/get-resource-notification-settings?owner_uid=%(uid)s&resource_id=%(resource_id)s'


class CreateAppSubscriptionHandler(handlers.CreateAppSubscriptionHandler):
    """Подписать приложение на уведомления об изменении баз DataSync."""
    query = fields.QueryDict({
        'events': fields.ListField(
            required=True,
            help_text=(u'Список event-ов, '
                       u'на уведомления об изменениях которых клиент хочет подписаться.')
        ),
        'databases_ids': None,
        'tags': fields.ListField(
            required=False,
            help_text=u'Список тегов, с помощью которых осуществляется дополнительная фильтрация пушей.'
        ),
        'bright': fields.BooleanField(
            required=False, default=False, help_text=u'Зарегистрировать "bright" пуш, поддерживается только для iOs'
        )
    })
    subscribe_to_service = 'disk-notifier'
    service = XivaSubscribeNotifierService()

    def build_event_filter(self, events, tags=None):
        assert isinstance(events, list)
        assert events

        if tags is None:
            tags = []
        device_id = None
        user_agent = UserAgentParser.parse(self.request.raw_headers.get('user-agent'))
        if user_agent.is_mobile_client():
            tags += [user_agent.get_os_family().lower()]
            device_id = 'device_id.%s' % user_agent.id
            device_id = device_id.lower().replace('-', '')

        result = {
            'vars': {
                'CommonEvents': {'$event': events},
            },
            'rules': [],
        }

        if tags:
            result['vars']['Tags'] = {'$has_tags': list(set(tags))}
            result['rules'].append({'if': 'CommonEvents & Tags', 'do': 'send_bright'})
            if self.get_context()['bright']:
                bright_tags = [t if t != UserAgent.IOS.lower() else 'ios_bright' for t in tags]

                # подписываемся на ios_bright
                result['vars']['BrightEnabled'] = {'$has_tags': list(set(bright_tags))}
                result['rules'].append({'if': 'CommonEvents & BrightEnabled', 'do': 'send_bright'})
        else:
            result['rules'].append({'if': 'CommonEvents', 'do': 'send_bright'})

        if device_id:
            result['vars']['DeviceIdTag'] = {'$has_tags': [device_id]}
            result['rules'].append({'if': 'CommonEvents & DeviceIdTag', 'do': 'send_bright'})

        mobile_client_version = user_agent.get_version()
        if mobile_client_version:
            result['vars']['Version'] = {'$has_tags': [mobile_client_version.get_without_build()]}
            if 'BrightEnabled' in result['vars']:
                if_rule = 'CommonEvents & BrightEnabled & Version'
            else:
                if_rule = 'CommonEvents & Version'
            result['rules'].append({'if': if_rule, 'do': 'send_bright'})

        result['rules'].append({'do': 'skip'})
        return result

    def get_xiva_filter(self):
        context = self.get_context()
        return self.build_event_filter(context['events'], tags=context['tags'])

    def get_device_id(self):
        user_agent = UserAgentParser.parse(self.request.raw_headers.get('user-agent'))
        if user_agent.is_mobile_client():
            return user_agent.id
        return None

    def get_service_with_tags(self):
        return self.subscribe_to_service


class DeleteAppSubscriptionHandler(handlers.DeleteAppSubscriptionHandler):
    service = XivaSubscribeNotifierService()
