# -*- coding: utf-8 -*-

import collections

from mpfs.platform import fields
from mpfs.platform.resources import BaseResource, adhoc, NamespaceResource
from mpfs.platform.v1.albums.handlers import MergePhotoAlbumsHandler
from mpfs.platform.v1.disk import handlers
from mpfs.platform.v1.disk.billing import handlers as billing_handlers
from mpfs.platform.v1.disk.billing.inapp.handlers import ProcessInappReceiptHandler, ListInappProductsHandler, ListInAppTariffsHandler
from mpfs.platform.v1.disk.deletion_log.handlers import ReadDeletionLogHandler
from mpfs.platform.v1.disk.notifications.handlers import (
    EnableResourceNotificationsHandler, DisableResourceNotificationsHandler, GetResourceNotificationsSettingsHandler,
    DeleteAppSubscriptionHandler, CreateAppSubscriptionHandler
)
from mpfs.platform.v1.disk.promo_codes.handlers import PromoCodeActivateHandler
from mpfs.platform.v1.disk.restore.resources import RestoreResource
from mpfs.platform.v1.disk.rostelecom.handlers import (
    RostelecomCheckUserHandler,
    RostelecomActivateServiceHandler,
    RostelecomDeactivateServiceHandler,
    RostelecomUnfreezeServiceHandler,
    RostelecomFreezeServiceHandler,
    RostelecomServiceStatusesHandler,
)
from mpfs.platform.v1.disk.source_ids.handlers import (
    AddSourceIdHandler,
    CheckSourceIdHandler,
    BulkCheckSourceIdsHandler,
)
from mpfs.platform.v1.wopi.resources import WopiNamespace
from mpfs.platform.v1.disk.event_history.resources import EventHistoryNamespace
from mpfs.platform.v1.disk.lenta.resources import LentaNamespace

class PublicResourceResource(BaseResource):
    """Публичные файлы и папки"""

    class PublicSettingsResource(BaseResource):
        relations = {
            'GET': handlers.GetPublicSettingsHandler,
            'POST': handlers.SetPublicSettingsHandler
        }

    relations = {
        'GET': handlers.GetPublicResourceHandler,
        'download': adhoc('GET', handlers.GetPublicResourceDownloadLinkHandler),
        'save-to-disk': adhoc('POST', handlers.SaveToDiskPublicResourceHandler),
        'public-settings': PublicSettingsResource,
    }


class PublicVideoResource(BaseResource):
    """Публичное видео"""
    hidden = True
    relations = {
        'streams': adhoc('GET', handlers.ListPublicVideoStreamsHandler),
    }


class OldPublicResources(PublicResourceResource):
    """Старый ресурс публичных папок и файлов для обратной совместимости"""
    hidden = True


class UploadResource(BaseResource):
    """Загрузка файлов в Диск"""
    relations = {
        'GET': handlers.GetResourceUploadLinkHandler,
        'PUT': handlers.UploadResourceHandler,
        'POST': handlers.UploadExternalResourceHandler,
    }


class SearchResouce(BaseResource):
    """Поиск ресурсов"""
    relations = {
        'GET': handlers.SearchResourcesHandler,
        'ugc': adhoc('GET', handlers.GeoSearchResourceHandler)
    }


class ImportResource(BaseResource):
    """Импорт файлов в Диск"""
    relations = {
        'mail': adhoc('POST', handlers.ImportMailAttachHandler, hidden=True)
    }


class SourceIdCheckResource(BaseResource):
    """Проверка по source_id"""
    relations = {
        'HEAD': CheckSourceIdHandler,
        'POST': BulkCheckSourceIdsHandler,
    }


class SourceIdsResource(BaseResource):
    """Работа с source id"""
    relations = {
        'PUT': AddSourceIdHandler,
        'check': SourceIdCheckResource,
    }


class DeletionLogResource(BaseResource):
    """Работа с deletion log"""
    relations = {
        'GET': ReadDeletionLogHandler,
    }


class NotificationsResource(BaseResource):
    relations = {
        'enable': adhoc('PUT', EnableResourceNotificationsHandler, hidden=True),
        'disable': adhoc('PUT', DisableResourceNotificationsHandler, hidden=True),
        'settings': adhoc('GET', GetResourceNotificationsSettingsHandler, hidden=True),
    }


class VersionResource(BaseResource):
    """Работа с версией ресурса."""
    relations = {
        'GET': handlers.GetVersionHandler,
        'copy': adhoc('POST', handlers.CopyFromVersionHandler),
        'restore': adhoc('PUT', handlers.RestoreVersionHandler),
    }


class VersionsResource(BaseResource):
    """Работа с версиями ресурса"""
    relations = collections.OrderedDict([
        ('checkpoints', adhoc('GET', handlers.GetVersionsHandler)),
        ('folded', adhoc('GET', handlers.GetFoldedVersionsHandler)),
        ('(?P<version_id>[^/]+)', VersionResource)
    ])


class ResourceDetailResource(BaseResource):
    """Файл и папка."""
    relations = {
        'GET': handlers.GetResourceDetailHandler,
        'DELETE': handlers.DeleteResourceByResourceIdHandler,
        'notifications': NotificationsResource,
        'dimensions': adhoc('GET', handlers.GetResourceDimensionsHandler),
        'image_metadata': adhoc('GET', handlers.GetImageMetadataHandler),
        'download': adhoc('GET', handlers.GetResourceDownloadLinkByResourceIDHandler),
        'versions': VersionsResource,
    }


class ResourcesResource(BaseResource):
    """Файлы и папки"""
    relations = collections.OrderedDict([
        ('GET', handlers.GetResourceHandler),
        ('PUT', handlers.CreateResourceHandler),
        ('DELETE', handlers.DeleteResourceHandler),
        ('PATCH', handlers.UpdateResourceHandler),
        ('search', SearchResouce),
        ('snapshot', adhoc('PUT', handlers.PutSnapshotHandler, hidden=True)),
        ('deltas', adhoc('GET', handlers.GetDeltasHandler)),
        ('upload', UploadResource),
        ('download', adhoc('GET', handlers.GetResourceDownloadLinkHandler)),
        ('copy', adhoc('POST', handlers.CopyResourceHandler)),
        ('move', adhoc('POST', handlers.MoveResourceHandler)),
        ('publish', adhoc('PUT', handlers.PublishResourceHandler)),
        ('unpublish', adhoc('PUT', handlers.UnpublishResourceHandler)),
        ('online-editor', adhoc('GET', handlers.GetOnlineEditorURLHandler)),
        ('exclude-from-generated-album', adhoc('POST', handlers.AddGeneratedAlbumExclusionHandler, hidden=True)),
        ('last-uploaded', adhoc('GET', handlers.GetLastUploadedFilesListHandler)),
        ('files', adhoc('GET', handlers.GetFlatFilesListHandler)),
        ('public', adhoc('GET', handlers.ListPublicResourcesHandler)),
        ('wopi-url', adhoc('GET', handlers.WOPIURLResourcesHandler, hidden=True)),
        ('import', ImportResource),
        ('(?P<resource_id>[^/]+)', ResourceDetailResource),
    ])


class NotesDetailResource(BaseResource):
    """Аттач к заметке"""
    relations = {
        '(?P<resource_id>[^/]+)': adhoc('GET', handlers.GetNotesResourceDetailHandler, hidden=True),
    }


class TrashResource(BaseResource):
    """Корзина"""
    relations = {
        'GET': handlers.GetTrashResourceHandler,
        'DELETE': handlers.ClearTrashHandler,
        'restore': adhoc('PUT', handlers.RestoreFromTrashHandler),
    }


class PhotounlimResource(BaseResource):
    """Безлимитный раздел для фотографий"""
    relations = {
        'last-modified': adhoc('GET', handlers.GetPhotounlimLastModifiedFilesListHandler),
        'download': adhoc('GET', handlers.GetPhotounlimResourceDownloadLinkHandler),
    }


class AttachResource(BaseResource):
    """Аттачи"""
    relations = {
        'upload': adhoc('PUT', handlers.UploadAttachHandler),
    }


class OperationsResource(BaseResource):
    """Асинхронная операция"""
    relations = {
        'GET': handlers.GetOperationStatusHandler,
    }


# За этим ендпоинтом закреплены два хендлера
class AlternateHandlersOperationsResource(BaseResource):
    """Список активных асинхронных операций."""
    hidden = True

    query = fields.QueryDict({
        'limit': fields.IntegerField(default=20, help_text=u'Количество выводимых операций.'),
        'offset': fields.IntegerField(default=0, help_text=u'Смещение от начала списка операций.'),
    })

    def __init__(self, *args, **kwargs):
        super(AlternateHandlersOperationsResource, self).__init__(*args, **kwargs)
        self._operation_status_handler = handlers.OldGetOperationStatusHandler(*args, **kwargs)
        self._list_operations_handler = handlers.ListActiveOperationsHandler(*args, **kwargs)

    def dispatch(self, request, *args, **kwargs):
        if request.args.get('id'):
            # Для обратной совместимости
            return self._operation_status_handler.dispatch(request, *args, **kwargs)
        return self._list_operations_handler.dispatch(request, *args, **kwargs)


class OperationsCollection(NamespaceResource):
    """Асинхронные операции"""
    hidden = True
    relations = {
        'GET': AlternateHandlersOperationsResource,
        '(?P<operation_id>[^/]+)': OperationsResource,
    }


class PartnerServiceResource(BaseResource):
    """Партнерские продукты"""
    relations = collections.OrderedDict([
        ('GET', billing_handlers.GetServiceInfoByProductHandler),
        ('POST', billing_handlers.CreateServiceByProductHandler),
        ('prolongate_by_product', adhoc('PUT', billing_handlers.ProlongateServiceByProductHandler)),
        ('remove_by_product', adhoc('DELETE', billing_handlers.DeleteServiceByProductHandler)),
        ('(?P<service_id>[^/]+)', adhoc('DELETE', billing_handlers.DeleteServiceByServiceIdHandler)),
    ])


class VideoResource(BaseResource):
    """стримминг видео"""
    relations = {
        'streams': adhoc('GET', handlers.ListVideoStreamsHandler),
    }


class PhotosliceResource(BaseResource):
    """Операции с фотосрезом"""
    relations = {
        'POST': handlers.PhotosliceInitSnapshotHandler,
        '(?P<photoslice_id>[^/]+)': adhoc('GET', handlers.PhotosliceGetSnapshotHandler),
        '(?P<photoslice_id>[^/]+)/deltas': adhoc('GET', handlers.PhotosliceGetDeltaListHandler),
    }


class PhotoAlbumItemResource(BaseResource):
    """Элемент фотоальбома"""
    relations = {
        'GET': handlers.GetPhotoAlbumItemHandler,
        'DELETE': handlers.DeletePhotoAlbumItemHandler,
    }


class PhotoAlbumsItemsResource(BaseResource):
    """Элементы фотоальбома"""
    relations = {
        'GET': handlers.ListPhotoAlbumItemsHandler,
        'POST': handlers.CreatePhotoAlbumItemHandler,
        'bulk\-create': adhoc('POST', handlers.BulkCreatePhotoAlbumItemHandler),
        '(?P<item_id>[^/]+)': PhotoAlbumItemResource
    }


class PhotoAlbumResource(BaseResource):
    """Операции с фотоальбомом"""
    relations = {
        'GET': handlers.GetPhotoAlbumHandler,
        'PATCH': handlers.ChangePhotoAlbumHandler,
        'DELETE': handlers.DeletePhotoAlbumHandler,
    }


class PhotoAlbumsResource(BaseResource):
    """Операции с фотоальбомами"""
    relations = collections.OrderedDict([
        ('GET', handlers.ListPhotoAlbumsHandler),
        ('POST', handlers.CreatePhotoAlbumHandler),
        ('share_providers', adhoc('GET', handlers.ListPhotoAlbumShareProvidersHandler)),
        ('merge', adhoc('POST', MergePhotoAlbumsHandler)),
        ('(?P<album_id>[^/]+)/items', PhotoAlbumsItemsResource),
        ('(?P<album_id>[^/]+)/share/(?P<provider_id>[^/]+)', adhoc('POST', handlers.SharePhotoAlbumHandler)),
        ('(?P<album_id>[^/]+)', PhotoAlbumResource),
    ])


class OrganizationResourcesResource(BaseResource):
    relations = {
        'GET': handlers.GetOrganizationResourceHandler,
        'PUT': handlers.CreateOrganizationFolderHandler,
        'upload': adhoc('GET', handlers.GetOrganizationUploadLinkHandler),
        'download': adhoc('GET', handlers.GetOrganizationDownloadLinkHandler),
        'grant_access': adhoc('PUT', handlers.GrantAccessToOrganizationFolderHandler),
    }


class OrganizationResource(BaseResource):
    relations = {
        'GET': handlers.GetOrganizationHandler,
        'resources': OrganizationResourcesResource,
    }


class OrganizationsCollection(BaseResource):
    """Диск для организаций"""
    relations = {
        'GET': handlers.ListOrganizationsHandler,
        'POST': handlers.CreateOrganizationHandler,
        '(?P<organization_id>[^/]+)': OrganizationResource,
    }


class AppSubscriptionsResource(BaseResource):
    """Подписки и отписки на уведомления для мобильных приложений"""
    relations = {
        'PUT': CreateAppSubscriptionHandler,
        '(?P<subscription_id>[^/]+)': adhoc('DELETE', DeleteAppSubscriptionHandler, hidden=True),
    }


class BillingSubscriptionsResource(BaseResource):
    """Работа с подписками"""
    relations = {
        'GET': billing_handlers.ListSubscriptionsHandler,
    }


class BillingProductsResource(BaseResource):
    """Работа с продуктами"""
    relations = {
        'GET': billing_handlers.ListProductsHandler,
    }


class BillingInappResource(BaseResource):
    """Работа с продуктами"""
    relations = {
        'receipt': adhoc('PUT', ProcessInappReceiptHandler, hidden=True),
        'products': adhoc('GET', ListInappProductsHandler, hidden=True),
        'tariffs': adhoc('GET', ListInAppTariffsHandler, hidden=True),
    }


class BillingResource(BaseResource):
    """API для работы с Дисковым биллингом"""
    relations = {
        'products': BillingProductsResource,
        'subscriptions': BillingSubscriptionsResource,
        'in-app': BillingInappResource,
    }


class PromoCodesResource(BaseResource):
    """API для работы с промокодами"""
    relations = {
        'activate': adhoc('PUT', PromoCodeActivateHandler, hidden=True),
    }


class ClientsResource(BaseResource):
    """Работа с клиентами ПО"""
    relations = collections.OrderedDict([
        ('fos/resources/upload', adhoc('PUT', handlers.UploadClientHandler)),
        ('config', adhoc('GET', handlers.GetClientsConfigHandler)),
        ('features', adhoc('GET', handlers.GetUserFeatureTogglesHandler)),
        ('activity', adhoc('GET', handlers.GetUserActivityInfoHandler)),
        ('win/installer', adhoc('GET', handlers.GetClientsInstallerWithAutologonHandler)),
        ('(?P<platform_id>[^/]+)/installer', adhoc('GET', handlers.GetClientsInstallerHandler)),
    ])


class ExperimentsResource(BaseResource):
    """Получение параметров экспериментов"""
    relations = {
        'GET': handlers.GetExperimentsHandler
    }


class RostelecomCloudPlatformResources(BaseResource):
    """API для партерства с Ростелекомом (безлимиты)

    https://wiki.yandex-team.ru/disk/mpfs/meps/mep-039/
    """
    relations = {
        'check': adhoc('GET', RostelecomCheckUserHandler),
        'activate': adhoc('PUT', RostelecomActivateServiceHandler),
        'deactivate': adhoc('PUT', RostelecomDeactivateServiceHandler),
        'on': adhoc('PUT', RostelecomUnfreezeServiceHandler),
        'off': adhoc('PUT', RostelecomFreezeServiceHandler),
        'status': adhoc('GET', RostelecomServiceStatusesHandler),
    }


class GDPResources(BaseResource):
    """
    https://st.yandex-team.ru/CHEMODAN-62809
    """
    relations = {
        'takeout/electrichki': adhoc('GET', handlers.GDPRElectrichkiHandler)
    }


class DocsResource(BaseResource):
    """Работа с документами"""
    relations = {
        'resources/(?P<resource_id>[^/]+)': adhoc('GET', handlers.GetDocsFileURLsHandler, hidden=True),
        'filters': adhoc('GET', handlers.GetDocsFileFilters),
        'resources/(?P<resource_id>[^/]+)/office-access-state': adhoc('POST', handlers.OfficeSetAccessStateHandler),
    }


class DiskNamespace(BaseResource):
    """Диск пользователя"""
    relations = {
        'GET': handlers.GetDiskHandler,
        'public/resources': PublicResourceResource,
        'public/video': PublicVideoResource,
        'partners/(?P<partner>[^/]+)/services': PartnerServiceResource,
        'public-resources': OldPublicResources,  # эта штука тут для обратной совместимости
        'trash/resources': TrashResource,
        'photounlim/resources': PhotounlimResource,
        'attach/resources': AttachResource,
        'resources': ResourcesResource,
        'video': VideoResource,
        'operations': OperationsCollection,
        'photoslice': PhotosliceResource,
        'photoalbums': PhotoAlbumsResource,
        'organizations': OrganizationsCollection,
        'wopi': WopiNamespace,
        'event-history': EventHistoryNamespace,
        'lenta': LentaNamespace,
        'notifications/subscriptions/app': AppSubscriptionsResource,
        'billing': BillingResource,
        'clients': ClientsResource,
        'restore': RestoreResource,
        'promo-codes': PromoCodesResource,
        'notes': NotesDetailResource,
        'experiments': ExperimentsResource,
        'rostelecom/cloud-platform': RostelecomCloudPlatformResources,
        'source-ids': SourceIdsResource,
        'deletion-log': DeletionLogResource,
        'gdpr': GDPResources,
        'only-office/(?P<key>[^/]+)': adhoc('POST', handlers.OnlyOfficeCallbackHandler),
        'docs': DocsResource,
    }
