# -*- coding: utf-8 -*-
"""API для партерства с Ростелекомом (безлимиты)

https://wiki.yandex-team.ru/disk/mpfs/meps/mep-039/
"""
from mpfs.common.static import codes
from mpfs.common.static.tags.billing import ROSTELECOM_UNLIM
from mpfs.common.util import type_to_bool
from mpfs.config import settings
from mpfs.core.rostelecom_unlim.constants import ROSTELECOM_PID_TO_SERVICE_KEY, ROSTELECOM_SUBSTITUTE_PRODUCTS
from mpfs.platform import fields
from mpfs.platform.auth import OAuthAuth
from mpfs.platform.exceptions import ForbiddenError
from mpfs.platform.permissions import AllowAllPermission
from mpfs.platform.v1.disk.billing.handlers import BasePartnerServiceHandler
from mpfs.platform.v1.disk.exceptions import DiskUserNotFoundError
from mpfs.platform.v1.disk.handlers import MpfsProxyHandler
from mpfs.platform.v1.disk.rostelecom.exceptions import (
    RostelecomUnableToInitializeUserError,
    RostelecomUserBlockedError,
    RostelecomIncorrectServiceKeyError, RostelecomServiceActiveError, RostelecomServiceBlockedError,
    RostelecomServiceDeactivatedError, RostelecomUserDoesNotExistError)
from mpfs.platform.v1.disk.rostelecom.permissions import (
    RostelecomCheckUserPermission,
    RostelecomManageServicesPermission,
)
from mpfs.platform.v1.disk.rostelecom.serializers import (
    RostelecomCheckResponseSerializer,
    RostelecomServiceStatusSerializer,
    RostelecomServiceListSerializer,
)

ROSTELECOM_UNLIM_ENABLED_FOR_UIDS = settings.rostelecom_unlim['enabled_for_uids']


class RostelecomCheckUserHandler(MpfsProxyHandler):
    auth_methods = [OAuthAuth()]
    permissions = AllowAllPermission() | RostelecomCheckUserPermission()
    need_auto_initialize_user = False
    can_init_user_url = '/json/can_init_user?uid=%(uid)s'
    user_info_url = '/json/user_info?uid=%(uid)s'
    user_check_url = '/json/user_check?uid=%(uid)s'
    serializer_cls = RostelecomCheckResponseSerializer

    error_map = {
        codes.USER_BLOCKED: RostelecomUserBlockedError,
    }

    def handle(self, request, *args, **kwargs):
        context = self.get_context()
        user_check_url = self.build_url(self.user_check_url, context)
        need_init = type_to_bool(self.request_service(user_check_url, self.service_method)['need_init'])
        if need_init:
            can_init_user_url = self.build_url(self.can_init_user_url, context)
            can_init_results = type_to_bool(self.request_service(can_init_user_url, self.service_method)['can_init'])
            if not can_init_results:
                raise RostelecomUnableToInitializeUserError()
        else:
            user_info_url = self.build_url(self.user_info_url, context)
            self.request_service(user_info_url, self.service_method)
        return self.serialize({'uid': request.user.uid})


class RostelecomManageServicesBaseHandler(BasePartnerServiceHandler):
    auth_methods = [OAuthAuth()]
    permissions = RostelecomManageServicesPermission()
    serializer_cls = RostelecomServiceStatusSerializer
    default_product_line = ROSTELECOM_UNLIM
    unavailable_product_exception = RostelecomIncorrectServiceKeyError

    query = fields.QueryDict({
        'uid': fields.StringField(required=True, help_text=u'Идентификатор пользователя.'),
        'service_key': fields.StringField(required=False, help_text=u'Идентификатор услуги'),
        'rostelecom_uid': fields.StringField(required=False, help_text=u'Идентификатор пользователя в РТК', default=''),
        'product_id': None,
        'email': None,
    })

    kwargs = fields.QueryDict({
        'partner': None,
    })

    error_map = {
        codes.ROSTELECOM_UNLIM_INCORRECT_SERVICE_KEY: RostelecomIncorrectServiceKeyError,
        codes.ROSTELECOM_UNLIM_SERVICE_IS_ACTIVATED: RostelecomServiceActiveError,
        codes.ROSTELECOM_UNLIM_SERVICE_IS_BLOCKED: RostelecomServiceBlockedError,
        codes.ROSTELECOM_UNLIM_SERVICE_IS_DEACTIVATED: RostelecomServiceDeactivatedError,
    }

    def load_product_check_permissions(self, *args, **kwargs):
        context = self.get_context()
        partner = 'rostelecom_unlim'
        # в ручку /status не передается service_key, поэтмоу используем проихвольную для проверки
        service_key = context.get('service_key', ROSTELECOM_UNLIM)
        product_id = ROSTELECOM_SUBSTITUTE_PRODUCTS.get(service_key, service_key)
        self.request.partner_product = self.get_partner_product(partner, product_id)
        self.check_permissions_for_product(partner, self.request.partner_product)

    def handle(self, request, *args, **kwargs):
        if ROSTELECOM_UNLIM_ENABLED_FOR_UIDS and request.user.uid not in ROSTELECOM_UNLIM_ENABLED_FOR_UIDS:
            raise ForbiddenError()
        try:
            return super(RostelecomManageServicesBaseHandler, self).handle(request, *args, **kwargs)
        except DiskUserNotFoundError:
            # попадаем сюда, если юзера нет в диске и не хотим автоматически инициализировать пользователя
            raise RostelecomUserDoesNotExistError()


class RostelecomActivateServiceHandler(RostelecomManageServicesBaseHandler):
    need_auto_initialize_user = True
    service_url = '/json/rostelecom_activate?uid=%(uid)s&service_key=%(service_key)s&rostelecom_uid=%(rostelecom_uid)s'

    def serialize(self, obj, *args, **kwargs):
        return super(RostelecomActivateServiceHandler, self).serialize({'status': 'active'})


class RostelecomDeactivateServiceHandler(RostelecomManageServicesBaseHandler):
    need_auto_initialize_user = False
    service_url = '/json/rostelecom_deactivate?uid=%(uid)s&service_key=%(service_key)s&rostelecom_uid=%(rostelecom_uid)s'

    def serialize(self, obj, *args, **kwargs):
        return super(RostelecomDeactivateServiceHandler, self).serialize({'status': 'inactive'}, *args, **kwargs)


class RostelecomUnfreezeServiceHandler(RostelecomManageServicesBaseHandler):
    need_auto_initialize_user = False
    service_url = '/json/rostelecom_unfreeze?uid=%(uid)s&service_key=%(service_key)s&rostelecom_uid=%(rostelecom_uid)s'

    def serialize(self, obj, *args, **kwargs):
        return super(RostelecomUnfreezeServiceHandler, self).serialize({'status': 'active'})


class RostelecomFreezeServiceHandler(RostelecomManageServicesBaseHandler):
    need_auto_initialize_user = False
    service_url = '/json/rostelecom_freeze?uid=%(uid)s&service_key=%(service_key)s&rostelecom_uid=%(rostelecom_uid)s'

    def serialize(self, obj, *args, **kwargs):
        return super(RostelecomFreezeServiceHandler, self).serialize({'status': 'blocked'}, *args, **kwargs)


class RostelecomServiceStatusesHandler(RostelecomManageServicesBaseHandler):
    auth_methods = [OAuthAuth()]
    permissions = RostelecomManageServicesPermission()
    serializer_cls = RostelecomServiceListSerializer
    need_auto_initialize_user = False
    service_url = '/json/rostelecom_list_services?uid=%(uid)s'

    query = fields.QueryDict({
        'uid': fields.StringField(required=True, help_text=u'Идентификатор пользователя.'),
        'service_key': None,
    })

    def serialize(self, obj, *args, **kwargs):
        statuses = {
            'inactive': [],
            'blocked': [],
            'active': [],
        }
        for service, status in obj.items():
            statuses[status].append(service)

        services = {}
        for status in ('inactive', 'blocked', 'active'):
            for pid in statuses[status]:
                services[ROSTELECOM_PID_TO_SERVICE_KEY.get(pid, pid)] = status

        result = {'items': []}
        for service, status in services.items():
            result['items'].append({'status': status, 'service': {'key': service}})

        return super(RostelecomServiceStatusesHandler, self).serialize(result, *args, **kwargs)
