# -*- coding: utf-8 -*-
from mpfs.common import errors
from mpfs.common.util import from_json
from mpfs.core.user.constants import PHOTOUNLIM_AREA
from mpfs.platform.fields import (BaseField, StringField, SerializerField, IntegerField,
                                  ParentMethodField, HalLinkField, DateTimeToTSField, BooleanField, ChoiceField,
                                  JsonObjectField, ListField, DateField, FloatField, LatitudeField, LongitudeField,
                                  UrlField)
from mpfs.platform.serializers import BaseSerializer, ListSerializer, ErrorSerializer
from mpfs.platform.v1.disk.fields import (
    MpfsPathField, PublicPathField, OrganizationPathField, RightsField, AntiVirusStatusField, AutouploadFlagField
)


class ExifSerializer(BaseSerializer):
    """Сериализатор EXIF

    Все новые поля в объекте exif именуем по правилу:
        * Название поля в спецификации http://www.exif.org/Exif2-2.PDF
        * приведённое к нижнему регистру
        * разделённое нижними подчёркиваниями "_" на отдельные слова.
    """
    fields = {
        'date_time': DateTimeToTSField(source='etime', help_text=u'Дата съёмки.'),
        'gps_latitude': LatitudeField(source='ext_coordinates', help_text=u'Координата съемки (широта).'),
        'gps_longitude': LongitudeField(source='ext_coordinates', help_text=u'Координата съемки (долгота).'),
    }
    visible_fields = fields.keys()

    def prepare_object(self, obj):
        obj = super(ExifSerializer, self).prepare_object(obj)
        if 'etime' in obj:
            etime = obj.get('etime')
            if isinstance(etime, basestring):
                obj['etime'] = int(etime)
        return obj


class UserPublicInformationSerializer(BaseSerializer):
    fields = {
        'uid': StringField(help_text=u'Идентификатор пользователя.'),
        'login': StringField(help_text=u'Логин.'),
        'display_name': StringField(help_text=u'Отображаемое имя пользователя.')
    }
    visible_fields = fields.keys()


class SystemFoldersSerializer(BaseSerializer):
    visible_fields = [
        'downloads', 'applications', 'screenshots', 'photostream', 'social',
        'facebook', 'google', 'instagram', 'vkontakte', 'mailru', 'odnoklassniki',
        'scans'
    ]
    fields = {
        'downloads': MpfsPathField(help_text=u'Путь к папке "Загрузки".'),
        'applications': MpfsPathField(help_text=u'Путь к папке "Приложения".'),
        'screenshots': MpfsPathField(help_text=u'Путь к папке "Скриншоты".'),
        'photostream': MpfsPathField(help_text=u'Путь к папке "Фотокамера".'),
        'social': MpfsPathField(help_text=u'Путь к папке "Социальные сети".'),
        'facebook': MpfsPathField(help_text=u'Путь к папке "Социальные сети/Facebook".'),
        'google': MpfsPathField(help_text=u'Путь к папке "Социальные сети/Google+".'),
        'instagram': MpfsPathField(help_text=u'Путь к папке "Социальные сети/Instagram".'),
        'vkontakte': MpfsPathField(help_text=u'Путь к папке "Социальные сети/ВКонтакте".'),
        'mailru': MpfsPathField(help_text=u'Путь к папке "Социальные сети/Мой Мир".'),
        'odnoklassniki': MpfsPathField(help_text=u'Путь к папке "Социальные сети/Одноклассники".'),
        'scans': MpfsPathField(help_text=u'Путь к папке "Сканы".'),
    }


class DiskSerializer(BaseSerializer):
    visible_fields = ['total_space', 'used_space', 'trash_size', 'max_file_size', 'paid_max_file_size', 'system_folders', 'is_paid',
                      'revision', 'user', 'unlimited_autoupload_enabled', 'unlimited_video_autoupload_enabled',
                      'unlimited_photo_autoupload_enabled', 'unlimited_video_autoupload_reason',
                      'unlimited_video_autoupload_allowed', 'unlimited_photo_autoupload_allowed',
                      'overdraft_lite_date', 'overdraft_hard_date', 'overdraft_block_date', 'overdraft_status',
                      'office_online_editor_name']
    native_client_fields = ['unlimited_video_autoupload_enabled', 'unlimited_photo_autoupload_enabled',
                            'unlimited_video_autoupload_reason', 'unlimited_video_autoupload_allowed',
                            'unlimited_photo_autoupload_allowed', 'office_online_editor_name', 'overdraft_lite_date',
                      'overdraft_hard_date', 'overdraft_block_date', 'overdraft_status']

    fields = {
        'total_space': IntegerField(source='user_info.space.limit', pbid=1, help_text=u'Общий объем диска (байт)'),
        'used_space': IntegerField(source='user_info.space.used', pbid=2, help_text=u'Используемый объем диска (байт)'),
        'trash_size': IntegerField(source='user_info.space.trash', pbid=3,
                                   help_text=u'Общий размер файлов в Корзине (байт). Входит в used_space.'),
        'max_file_size': IntegerField(source='user_info.space.filesize_limit', pbid=4,
                                      help_text=u'Максимальный поддерживаемый размер файла.'),
        'paid_max_file_size': IntegerField(source='user_info.space.paid_filesize_limit',
                                           help_text=u'Максимальный поддерживаемый размер файла для платного аккаунта.'),
        'system_folders': SerializerField(SystemFoldersSerializer, source='user_info.default_folders',
                                          help_text=u'Адреса системных папок в Диске пользователя.'),
        'is_paid': BooleanField(boolify=True, source='user_info.paid', help_text=u'Признак наличия купленного места.'),
        'revision': IntegerField(source='user_info.version', help_text=u'Текущая ревизия Диска'),
        'user': SerializerField(lambda: UserSerializer, help_text=u'Владелец Диска'),
        'unlimited_autoupload_enabled': BooleanField(boolify=True, source='user_info.unlimited_autoupload_enabled',
                                                     help_text=u'Признак включенной безлимитной автозагрузки с '
                                                               u'мобильных устройств.'),
        'unlimited_video_autoupload_enabled': BooleanField(
            boolify=True, source='user_info.unlimited_video_autoupload_enabled',
            help_text=u'Признак включенной безлимитной автозагрузки видео с мобильных устройств.'),
        'unlimited_photo_autoupload_enabled': BooleanField(
            boolify=True, source='user_info.unlimited_photo_autoupload_enabled',
            help_text=u'Признак включенной безлимитной автозагрузки фото с мобильных устройств.'),
        'unlimited_video_autoupload_reason': ChoiceField(
            required=False, source='user_info.unlimited_video_autoupload_reason',
            help_text=u'Причина состояния безлимитной автозагрузки видео с телефона.'),
        'unlimited_video_autoupload_allowed': BooleanField(
            boolify=True, source='user_info.unlimited_video_autoupload_allowed',
            help_text=u'Доступность безлимитной автозагрузки видео с мобильных устройств.'),
        'unlimited_photo_autoupload_allowed': BooleanField(
            boolify=True, source='user_info.unlimited_photo_autoupload_allowed',
            help_text=u'Доступность безлимитной автозагрузки фото с мобильных устройств.'),
        'office_online_editor_name': StringField(
            source='user_info.office_online_editor_name', help_text=u'Редактор'),
        'overdraft_lite_date': DateField(source='user_info.overdraft_lite_date', required=False),
        'overdraft_hard_date': DateField(source='user_info.overdraft_hard_date', required=False),
        'overdraft_block_date': DateField(source='user_info.overdraft_block_date', required=False),
        'overdraft_status': IntegerField(source='user_info.overdraft_status', required=False)
    }


class ShareInfoSerializer(BaseSerializer):
    visible_fields = ['rights', 'is_owned', 'is_root']

    fields = {
        'is_owned': BooleanField(boolify=True,
                                 help_text=u'Признак, что текущий пользователь является владельцем общей папки'),
        'is_root': BooleanField(boolify=True, help_text=u'Признак того, что папка является корневой в группе'),
        'rights': RightsField(required=True, default='660', help_text=u'Права доступа'),
    }


class RootShareInfoSerializer(ShareInfoSerializer):
    visible_fields = ['rights', 'is_owned']


class CommentIdsSerializer(BaseSerializer):
    visible_fields = ['public_resource', 'private_resource']
    fields = {
        'public_resource': StringField(source='public_resource', help_text=u'Идентификатор комментариев для публичных ресурсов.'),
        'private_resource': StringField(source='private_resource', help_text=u'Идентификатор комментариев для приватных ресурсов.'),
    }


class VideoMetadataSerializer(BaseSerializer):
    visible_fields = ['duration']

    fields = {
        'duration': IntegerField(help_text=u'Длительность видео в миллисекундах'),
    }


class ImageMetadataSerializer(BaseSerializer):
    visible_fields = ['width', 'height', 'angle', 'beauty']

    fields = {
        'width': IntegerField(help_text=u'Высота'),
        'height': IntegerField(help_text=u'Ширина'),
        'angle': IntegerField(help_text=u'Угол поворота изображения'),
        'beauty': FloatField(source='aesthetics',
                             help_text=u'Красивость изображения по компьютерному зрению'),
    }


class ResourceSerializer(BaseSerializer):
    visible_fields = [
        'path', 'type', 'name', 'created', 'modified', 'size', 'mime_type', 'md5', 'sha256', 'preview',
        '_links', 'public_key', 'public_url', '_embedded', 'media_type',
        'custom_properties', 'resource_id', 'share', 'revision', 'comment_ids', 'exif', 'antivirus_status',
        'file', 'autouploaded', 'photoslice_time', 'source_ids', 'photoslice_album_type', 'video_metadata',
        'image_metadata', 'albums_exclusions', 'office_access_state', 'office_online_url', 'office_online_sharing_url',
    ]

    native_client_fields = [
        'source_ids', 'photoslice_album_type', 'albums_exclusions', 'video_metadata', 'image_metadata',
        'office_access_state', 'office_online_url', 'office_online_sharing_url',
    ]

    internal_only_fields = [
        'autouploaded',
    ]

    fields = {
        'path': MpfsPathField(required=True, allowed_areas=('disk', 'app', PHOTOUNLIM_AREA), pbid=1, help_text=u'Путь к ресурсу'),
        'type': StringField(required=True, pbid=2, help_text=u'Тип'),
        'name': StringField(required=True, pbid=3, help_text=u'Имя'),
        'created': DateTimeToTSField(required=True, pbid=4, source='ctime', help_text=u'Дата создания'),
        'modified': DateTimeToTSField(required=True, pbid=5, source='mtime', help_text=u'Дата изменения'),

        'size': IntegerField(source='meta.size', pbid=6, help_text=u'Размер файла'),
        'mime_type': StringField(source='meta.mimetype', pbid=7, help_text=u'MIME-тип файла'),
        'md5': StringField(source='meta.md5', pbid=8, help_text=u'MD5-хэш'),
        'sha256': StringField(source='meta.sha256', help_text=u'SHA256-хэш'),
        'preview': StringField(source='meta.custom_preview', pbid=9, help_text=u'URL превью файла'),
        'public_key': StringField(source='meta.public_hash', pbid=10, help_text=u'Ключ опубликованного ресурса'),
        'public_url': StringField(source='meta.short_url', pbid=11, help_text=u'Публичный URL'),
        '_embedded': SerializerField(lambda: ResourceListSerializer, pbid=13, help_text=u'Список вложенных ресурсов'),
        'media_type': StringField(source='meta.media_type', pbid=15, help_text=u'Определённый Диском тип файла'),
        'file': StringField(source='meta.file_url', pbid=17, help_text=u'URL для скачивания файла'),
        'resource_id': StringField(source='meta.resource_id', help_text=u'Идентификатор ресурса'),
        'share': SerializerField(ShareInfoSerializer, source='meta.group', help_text=u'Информация об общей папке'),
        'video_metadata': SerializerField(VideoMetadataSerializer,
                                          source='meta.video_info',
                                          help_text=u'Метаданные видео-файла',
                                          request_explicitly=True),
        'image_metadata': SerializerField(ImageMetadataSerializer,
                                          source='meta',
                                          help_text=u'Метаданные файла изображения',
                                          request_explicitly=True),
        'revision': IntegerField(source='meta.revision',
                                 help_text=u'Ревизия Диска в которой этот ресурс был изменён последний раз'),
        'comment_ids': SerializerField(lambda: CommentIdsSerializer, source='meta.comment_ids', help_text=u'Идентификаторы комментариев'),
        'custom_properties': JsonObjectField(source='meta.custom_properties', max_nesting_level=0,
                                             allow_list_attributes=False, max_size=1024,
                                             help_text=u'Пользовательские атрибуты ресурса'),
        'exif': SerializerField(lambda: ExifSerializer, source='meta', help_text=u'Метаданные медиафайла (EXIF)'),
        'antivirus_status': AntiVirusStatusField(source='meta.drweb', help_text=u'Статус проверки антивирусом'),
        'autouploaded': AutouploadFlagField(source='meta.autouploaded', help_text=u'Признак автозагруженности'),
        'photoslice_time': DateTimeToTSField(source='meta.photoslice_time', help_text=u'Дата создания фото или видео файла'),
        'source_ids': ListField(source='meta.source_ids', help_text=u'Список source id.'),
        'photoslice_album_type': StringField(source='meta.photoslice_album_type', help_text=u'Тип фотосрезного альбома'),
        'albums_exclusions': ListField(source='meta.albums_exclusions',
                                       help_text=u'Список исключений файла из сгенерированных альбомов'),
        'office_access_state': StringField(source='meta.office_access_state'),
        'office_online_url': StringField(source='meta.office_online_url'),
        'office_online_sharing_url': StringField(source='meta.office_online_sharing_url')
    }

    explicitly_requested_meta_fields = {'source_ids', }
    """Значения, которые содержатся в meta, но должны запрашиваться явно. Чтобы получить все дефолтные поля плюс
    дополнительное, можно просто передать +<имя_поля> в fields
    """

    @classmethod
    def get_mpfs_meta(cls, extra_meta=None):
        """
        Из fields получает поля, которые нужно указать в qs параметр `meta` при запросе в MPFS
        """
        if extra_meta is None:
            mpfs_meta_fields = set()
        else:
            mpfs_meta_fields = set(extra_meta)

        for field_name, field in cls.fields.iteritems():
            # Если source=meta, то обрабатываем source внутренних полей
            if field.source == 'meta' and isinstance(field, SerializerField):
                for nested_field in field.serializer.get_fields().values():
                    mpfs_meta_fields.add(nested_field.source
                                         if nested_field.source is not None
                                         else nested_field.name)
                continue

            if not field or not field.source:
                continue
            if field_name in cls.explicitly_requested_meta_fields:
                continue
            source_chunks = field.source.split('.')
            if len(source_chunks) != 2 or source_chunks[0] != 'meta':
                continue
            mpfs_meta_fields.add(source_chunks[1])

        return mpfs_meta_fields

    @classmethod
    def get_raw_mpfs_meta(cls, extra_meta=None):
        return ','.join(cls.get_mpfs_meta(extra_meta=extra_meta))

    def get_links(self):
        from mpfs.platform.v1.disk import handlers
        ret = {
            'self': HalLinkField(handlers.GetResourceHandler),
            'delete': HalLinkField(handlers.DeleteResourceHandler, context={'permanently': True}),
            'copy': HalLinkField(handlers.CopyResourceHandler, map={'from': 'path'}, exclude=['path']),
            'move': HalLinkField(handlers.MoveResourceHandler, map={'from': 'path'}, exclude=['path']),
            'trash': HalLinkField(handlers.DeleteResourceHandler),
            'download': HalLinkField(handlers.GetResourceDownloadLinkHandler),
        }
        if self._object:
            if self._object.get('type', None) == 'file':
                ret['upload'] = HalLinkField(handlers.GetResourceUploadLinkHandler)
            meta = self._object.get('meta', {})
            if meta.get('public_hash', None):
                ret['unpublish'] = HalLinkField(handlers.UnpublishResourceHandler)
            else:
                ret['publish'] = HalLinkField(handlers.PublishResourceHandler)
        return ret


class SearchResourceSerializer(ResourceSerializer):
    visible_fields = ResourceSerializer.visible_fields + ['search_scope']
    fields = {
        'search_scope': StringField(source='meta.search_scope', help_text=u'Механизм индексации документа'),
        '_embedded': SerializerField(lambda: SearchResourceSerializer, pbid=13, help_text=u'Список вложенных ресурсов'),
    }


class TrashResourceSerializer(ResourceSerializer):
    visible_fields = ResourceSerializer.visible_fields + ['origin_path', 'deleted']
    fields = {
        'origin_path': MpfsPathField(source='meta.original_id', allowed_areas=('disk', 'app', PHOTOUNLIM_AREA), pbid=12, help_text=u'Путь откуда был удалён ресурс'),
        'deleted': DateTimeToTSField(pbid=16, source='meta.append_time', help_text=u'Дата добавления в корзину(для ресурсов в корзине)'),
        '_embedded': SerializerField(lambda: TrashResourceListSerializer, pbid=13, help_text=u'Список вложенных ресурсов'),
    }


class OrganizationResourceSerializer(ResourceSerializer):
    fields = {
        'path': OrganizationPathField(required=True, pbid=1, help_text=u'Путь к ресурсу'),
        '_embedded': SerializerField(lambda: OrganizationResourceListSerializer, pbid=13, help_text=u'Список вложенных ресурсов'),
    }


class ResourcePatchSerializer(ResourceSerializer):
    is_patch = True
    visible_fields = ['custom_properties']


class SnapshotIterationKeySerializer(BaseSerializer):
    visible_fields = ['iteration_key']
    fields = {
        'iteration_key': StringField(pbid=1, help_text=u'Ключ итерации')
    }


class SnapshotResourceSerializer(ResourceSerializer):
    visible_fields = [
        'revision', 'type', 'path', 'resource_id', 'md5', 'sha256', 'size', 'modified', 'public_url',
        'share', 'discsw_symlink']

    fields = {
        'share': SerializerField(RootShareInfoSerializer, source='meta.group', help_text=u'Информация об общей папке'),
        'discsw_symlink': StringField(source='meta.discsw_symlink', help_text=u'Disk software symlink'),
    }

    def __init__(self, *args, **kwargs):
        super(SnapshotResourceSerializer, self).__init__(self, *args, **kwargs)
        for field_name in self.visible_fields:
            if field_name not in ('md5', 'sha256', 'size', 'public_url', 'share', 'discsw_symlink'):
                field = self.fields[field_name]
                field.required = True


class SnapshotSerializer(BaseSerializer):
    visible_fields = ['revision', 'iteration_key', 'items']
    fields = {
        'revision': IntegerField(required=True, pbid=1, help_text=u'Ревизия Диска'),
        'iteration_key': StringField(required=True, pdib=2, help=u"Ключ итерации"),
        'items':  SerializerField(lambda: SnapshotResourceSerializer,
                                  many=True, init={'fail_on_missing_required': True},
                                  pbid=3, help_text=u'Элементы снепшота'),
    }


class DeltaResourceSerializer(ResourceSerializer):
    visible_fields = ['modified', 'md5', 'sha256', 'size', 'public_url']


class DeltaSerializer(BaseSerializer):
    fields = {
        'change_type': StringField(required=True, help_text=u'Тип изменения'),
        'type': StringField(required=True, help_text=u'Тип ресурса'),
        'path': MpfsPathField(required=True, allowed_areas=('disk', 'app', PHOTOUNLIM_AREA), help_text=u'Путь к ресурсу'),
        'new_path': MpfsPathField(required=False, allowed_areas=('disk', 'app', PHOTOUNLIM_AREA), help_text=u'Новый путь к ресурсу'),
        'resource_id': StringField(required=True, help_text=u'Идентификатор ресурса'),
        'revision': IntegerField(required=True, help_text=u'Ревизия'),
        'discsw_symlink': StringField(help_text=u'Disk software symlink'),
        'share': SerializerField(RootShareInfoSerializer, source='group', help_text=u'Информация об общей папке'),
        'resource': SerializerField(DeltaResourceSerializer, help_text=u'Ресурс, для которого произошло изменение'),
    }
    visible_fields = fields.keys()


class DeltasSerializer(BaseSerializer):
    fields = {
        # это заглушка, на случай, если будем делать пагинацию
        'iteration_key': StringField(required=True, default='', help=u"Ключ итерации"),
        'revision': IntegerField(required=True, help_text=u'Ревизия Диска'),
        'items':  SerializerField(lambda: DeltaSerializer, required=True, many=True, help_text=u'Список изменений'),
    }
    visible_fields = fields.keys()


class PublicResourceSerializer(ResourceSerializer):
    """Опубликованный ресурс"""
    visible_fields = ResourceSerializer.visible_fields + ['comment_ids', 'views_count', 'owner']

    fields = {
        'public_key': ParentMethodField('get_public_key', required=True, source='path', field_type=StringField, pbid=14,
                                        help_text=u'Ключ опубликованного ресурса'),
        'path': PublicPathField(required=True, pbid=1, help_text=u'Путь опубликованного ресурса'),
        '_embedded': SerializerField(lambda: PublicResourceListSerializer, pbid=13,
                                     help_text=u'Список вложенных ресурсов'),
        'comment_ids': SerializerField(lambda: CommentIdsSerializer, source='meta.comment_ids', help_text=u'Идентификаторы комментариев'),
        'views_count': IntegerField(source='meta.views_counter', help_text=u'Счетчик просмотров публичного ресурса'),
        'group': None,
        'custom_properties': None,
        'owner': SerializerField(UserPublicInformationSerializer, source='meta.user',
                                 help_text=u'Владелец опубликованного ресурса')
    }

    def get_links(self):
        from mpfs.platform.v1.disk import handlers
        ret = {
            'self': HalLinkField(handlers.GetPublicResourceHandler),
            'save_to_disk': HalLinkField(handlers.SaveToDiskPublicResourceHandler),
            'download': HalLinkField(handlers.GetPublicResourceDownloadLinkHandler),
        }
        return ret

    def get_public_key(self, val):
        return val.split(':')[0]


class BaseOperationSerializer(BaseSerializer):
    """Базовый сериализатор операции, инкапсулирующий логику формирования значения ```status```."""
    default_status = 'success'
    STATUS_MAPPING = {
        'WAITING': 'in-progress',
        'EXECUTING': 'in-progress',
        'DONE': 'success',
        'FAILED': 'failed',
        'ABORTED': 'failed',
        'REJECTED': 'failed',
    }

    fields = {
        'status': ParentMethodField('get_status', required=True, field_type=StringField, pbid=1,
                                    help_text=u'Статус операции'),
    }

    def get_status(self, value):
        return self.STATUS_MAPPING.get(value) or self.default_status


class ResourceListSerializer(ListSerializer):
    visible_fields = ['_links', 'path', 'limit', 'offset', 'sort', 'total', 'items']

    fields = dict(
        path=MpfsPathField(required=True, allowed_areas=('disk', 'app', PHOTOUNLIM_AREA), pbid=5, help_text=u'Путь к ресурсу, для которого построен список'),
        sort=StringField(pbid=6, help_text=u'Поле, по которому отсортирован список'),
        items=SerializerField(lambda: ResourceSerializer, many=True, required=True, pbid=1, help_text=u'Элементы списка'),
    )

    @property
    def handler_cls(self):
        from mpfs.platform.v1.disk.handlers import GetResourceHandler
        return GetResourceHandler


class SearchResourceListSerializer(ResourceListSerializer):
    visible_fields = ['_links', 'path', 'limit', 'offset', 'sort', 'total', 'items', 'iteration_key']

    fields = dict(
        iteration_key=StringField(required=False, help_text=u'Поле, по которому отсортирован список'),
        items=SerializerField(lambda: SearchResourceSerializer, many=True, required=True, pbid=1,
                                 help_text=u'Элементы списка'),
    )


class TrashResourceListSerializer(ResourceListSerializer):
    fields = {
        'items': SerializerField(lambda: TrashResourceSerializer, many=True, required=True, pbid=1, help_text=u'Элементы списка'),
    }


class OrganizationResourceListSerializer(ResourceListSerializer):
    fields = {
        'path': OrganizationPathField(required=True, pbid=5, help_text=u'Путь к ресурсу, для которого построен список'),
        'items': SerializerField(lambda: OrganizationResourceSerializer, many=True, required=True, pbid=1, help_text=u'Элементы списка'),
    }


class PublicResourcesListSerializer(ListSerializer):
    """Сериализуют ответ mpfs-ой ручки `/json/list_public`"""
    visible_fields = ['_links', 'limit', 'offset', 'items', 'type']

    fields = {
        # в mpfs-ой ручке `/json/list_public`, отдаются "обычные" ресурсы, а не публичные
        'items': SerializerField(ResourceSerializer, many=True, required=True, pbid=1, help_text=u'Элементы списка'),
        'type': StringField(pbid=5, help_text=u'Значение фильтра по типу ресурсов')
    }

    @property
    def handler_cls(self):
        from mpfs.platform.v1.disk.handlers import ListPublicResourcesHandler
        return ListPublicResourcesHandler


class LinkSerializer(BaseSerializer):
    """Сериализатор объектов возвращаемых методом ```Router.get_link()```."""
    visible_fields = ['method', 'href', 'templated']
    fields = {
        'method': StringField(required=True, pbid=1, help_text=u'HTTP-метод'),
        'href': StringField(required=True, pbid=2, help_text=u'URL'),
        'templated': BooleanField(pbid=3, help_text=u'Признак шаблонизированного URL'),
    }

    def prepare_object(self, obj):
        """
        :param obj: Кортеж из 3х элементов, возвращаемый методом ```Router.get_link()```.
        """
        return dict(zip(type(self).visible_fields, obj))


class ResourceUploadLinkSerializer(LinkSerializer):
    visible_fields = ['method', 'href', 'templated', 'operation_id']
    fields = {
        'operation_id': StringField(required=True, pbid=4, help_text=u'Идентификатор операции загрузки файла')
    }


class UploadResourceSerializer(BaseSerializer):
    """Сериализует ссылки на статус операции и на загрузку файла."""
    visible_fields = ['upload_link', 'operation_link']
    fields = {
        'operation_link': SerializerField(lambda: LinkSerializer, pbid=1, required=True,
                                          help_text=u'Ссылка на получение статуса операции.'),
        'upload_link': SerializerField(lambda: LinkSerializer, pbid=2, required=True,
                                       help_text=u'URL файла.')
    }


class OwnerNoFreeSpaceErrorSerializer(ErrorSerializer):
    visible_fields = ErrorSerializer.visible_fields + ['owner']

    fields = {
        'owner': SerializerField(lambda: UserSerializer, pbid=4, help_text=u'Данные о владельце ОП')
    }


class OperationStatusSerializer(BaseOperationSerializer):
    """Сериализует объект, возвращаемый MPFS-ручкой ```status```."""
    visible_fields = ['status']
    is_failed_with_owner_no_space = False
    """Признак фейла операции из-за ошибки нехватки места у Владельца ОП"""

    def __init__(self, *args, **kwargs):
        super(OperationStatusSerializer, self).__init__(*args, **kwargs)
        obj = kwargs.get('obj')
        # Добавляем детали об ошибке, если это ошибка нехватки места у Владельца
        if (obj and
                    'error' in obj and
                    'code' in obj['error'] and
                    obj['error']['code'] == errors.OwnerHasNoFreeSpace.code):
            self.is_failed_with_owner_no_space = True
            self.visible_fields += ['error_data']
            self.fields['error_data'] = SerializerField(lambda: OwnerNoFreeSpaceErrorSerializer, pbid=2, required=True,
                                                        help_text=u'Данные об ошибке, из-за которой операция не '
                                                                  u'завершилась успешно')

    def prepare_object(self, obj):
        if self.is_failed_with_owner_no_space:
            from mpfs.platform.v1.disk.exceptions import DiskOwnerStorageQuotaExhaustedError
            error_data = DiskOwnerStorageQuotaExhaustedError()
            # Формируем данные от MPFS для сериализаторов
            if 'title' in obj['error']:
                obj['error']['details'] = from_json(obj['error']['title'])
                error_data.owner = obj['error']['details'].get('owner')
            obj['error_data'] = error_data
        return super(OperationStatusSerializer, self).prepare_object(obj)


class BaseOperationDataSerializer(BaseSerializer):
    _operation_type = None

    @classmethod
    def get_operation_type(cls):
        return cls._operation_type


class CopyOnDiskOperationDataSerializer(BaseOperationDataSerializer):
    visible_fields = ['from', 'path', 'from_resource_id']

    fields = {
        'from': MpfsPathField(source='data.source', required=True, allowed_areas=('disk', 'app', PHOTOUNLIM_AREA), pbid=1, help_text=u'Путь к копируемому ресурсу'),
        'path': MpfsPathField(source='data.target', required=True, pbid=2, help_text=u'Путь к создаваемому ресурсу'),
        'from_resource_id': StringField(source='data.source_resource_id', required=True, pbid=3,
                                        help_text=u'Идентификатор копируемого ресурса'),
    }
    _operation_type = 'copy'


class MoveOnDiskOperationDataSerializer(BaseOperationDataSerializer):
    visible_fields = ['from', 'path', 'from_resource_id']

    fields = {
        'from': MpfsPathField(source='data.source', required=True, pbid=1, help_text=u'Путь к перемещаемому ресурсу'),
        'path': MpfsPathField(source='data.target', required=True, pbid=2, help_text=u'Путь к создаваемому ресурсу'),
        'from_resource_id': StringField(source='data.source_resource_id', required=True,
                                        pbid=3, help_text=u'Идентификатор перемещаемого ресурса'),
    }
    _operation_type = 'move'


class DeleteOperationDataSerializer(BaseOperationDataSerializer):
    visible_fields = ['path', 'resource_id', 'permanently']

    fields = {
        'path': MpfsPathField(source='data.path', required=True, allowed_areas=('disk', 'app', PHOTOUNLIM_AREA), pbid=1, help_text=u'Путь к удаляемому ресурсу'),
        'resource_id': StringField(source='data.source_resource_id', required=True, pbid=2,
                                   help_text=u'Идентификатор удаляемого ресурса'),
        'permanently': ParentMethodField('is_permanent', source='type', required=True, pbid=3, field_type=BooleanField,
                                         help_text=u'Ресурс удален без помещения в корзину.'),
    }
    _operation_type = 'delete'

    def is_permanent(self, type_):
        if type_ == 'remove':
            return True
        return False


class UploadOperationDataSerializer(BaseOperationDataSerializer):
    visible_fields = ['path']

    fields = {
        'path': MpfsPathField(source='data.path', required=True, allowed_areas=('disk', 'app', PHOTOUNLIM_AREA), pbid=1, help_text=u'Путь создаваемого ресурса'),
    }
    _operation_type = 'upload'


class DefaultOperationDataSerializer(BaseOperationDataSerializer):
    """Класс сериализатора, который отдает вместо ```data``` пустой словарь.

    Используется по-умолчанию, если не найден конкретный сериализатор."""
    _operation_type = 'default'


class OperationSerializer(BaseOperationSerializer):
    """Сериализует объект, возвращаемый MPFS-ручкой ```active_operations```."""
    visible_fields = ['status', 'operation_id', 'link', 'type', 'data']

    fields = {
        'operation_id': StringField(source='id', required=True, pbid=2, help_text=u'Идентификатор операции'),
        'link': ParentMethodField('get_operation_status_link', required=True, source='id',
                                  field_type=StringField, pbid=3, help_text=u''),
        'type': ParentMethodField('get_operation_type', source='type', field_type=StringField, required=True, pbid=4,
                                  help_text=u'Тип операции'),
        'data': ParentMethodField('get_operation_data', pbid=5, help_text=u'Данные операции')
    }
    # Наивный подход.
    # Возможно, нужны будут дополнительные критерии для определения типа операции.
    _data_serializers_map = {
        ('copy', 'disk_disk'): CopyOnDiskOperationDataSerializer,
        ('copy', 'copy'): CopyOnDiskOperationDataSerializer,
        ('move', 'disk_disk'): MoveOnDiskOperationDataSerializer,
        ('trash', 'append'): DeleteOperationDataSerializer,
        ('remove', 'disk'): DeleteOperationDataSerializer,
        ('store', ''): UploadOperationDataSerializer
    }

    def get_operation_status_link(self, value):
        from mpfs.platform.v1.disk import handlers
        field = HalLinkField(handler=handlers.GetOperationStatusHandler)
        field.parent = self
        return field.from_native({'operation_id': value})

    def get_data_serializer(self):
        """Получить класс сериализатора обработчика тела операции."""
        type_ = self.object['type']
        subtype = self.object['subtype']
        serializer_cls = self._data_serializers_map.get((type_, subtype), DefaultOperationDataSerializer)
        if serializer_cls == DefaultOperationDataSerializer:
            # Ищем сериализатор общий, без подтипа.
            serializer_cls = self._data_serializers_map.get((type_, ''), DefaultOperationDataSerializer)
        return serializer_cls

    def get_operation_type(self, type_):
        """Получить тип операции.

        :param type_: значение поля `type`
        :rtype: str
        """
        serializer_cls = self.get_data_serializer()
        if serializer_cls == DefaultOperationDataSerializer:
            return type_
        return serializer_cls.get_operation_type()

    def get_operation_data(self, _):
        data_serializer_cls = self.get_data_serializer()
        return data_serializer_cls(self.object).data


class OperationListSerializer(ListSerializer):
    visible_fields = ['items']
    fields = {
        'items': SerializerField(lambda: OperationSerializer, many=True, required=True, pbid=1,
                                 help_text=u'Элементы списка')
    }


class PublicResourceListSerializer(ResourceListSerializer):
    """Список подресурсов опубликованного ресурса"""
    fields = {
        'public_key': StringField(required=True, pbid=7, help_text=u'Ключ опубликованного ресурса'),
        'path': PublicPathField(required=True, pbid=5, help_text=u'Путь опубликованного ресурса'),
        'items': SerializerField(lambda: PublicResourceSerializer, many=True, required=True, pbid=1,
                                 help_text=u'Элементы списка'),
    }

    def get_visible_fields(self):
        return super(PublicResourceListSerializer, self).get_visible_fields() + ['public_key']


class LastUploadedResourceListSerializer(BaseSerializer):
    visible_fields = ['limit', 'items']
    fields = dict(
        items=SerializerField(ResourceSerializer, many=True, required=True, pbid=1, help_text=u'Элементы списка'),
        limit=IntegerField(pbid=3, help_text=u'Количество элементов на странице'),
        media_type=StringField(required=False, pbid=5, help_text=u'Фильтр по медиа типу')
    )


class PhotounlimLastModifiedResourceListSerializer(ListSerializer):
    visible_fields = ['items', 'iteration_key']
    fields = dict(
        items=SerializerField(ResourceSerializer, many=True, required=True, pbid=1, help_text=u'Элементы списка'),
        iteration_key=StringField(required=False, pbid=5, help_text=u'Ключ для получения следующей страницы списка.')
    )


class FilesResourceListSerializer(ListSerializer):
    visible_fields = ['limit', 'items', 'offset']
    fields = {
        'items': SerializerField(ResourceSerializer, many=True, required=True, pbid=1, help_text=u'Элементы списка'),
    }


class VideoLinksSerializer(BaseSerializer):
    visible_fields = ['https', 'http']
    fields = {
        'https': StringField(pbid=1, required=False, help_text=u'Ссылка по протоколу https'),
        'http': StringField(pbid=2, required=False, help_text=u'Ссылка по протоколу http')
    }


class VideoStreamSerializer(BaseSerializer):
    CONTAINER_CHOICES = ['mp4', 'hls', 'hls-adaptive']
    RESOLUTION_CHOICES = ['240p', '360p', '480p', '720p']
    visible_fields = ['container', 'links', 'bitrate', 'audio_codec', 'video_codec', 'resolution', 'width', 'height']
    fields = {
        'container': ChoiceField(required=True, pbid=1, choices=CONTAINER_CHOICES, help_text=u'Тип контейнера. Например mp4, hls'),
        'links': SerializerField(lambda: VideoLinksSerializer, required=True, pbid=2, help_text=u'Список ссылок на поток по разным протоколам'),
        'bitrate': IntegerField(pbid=3, help_text=u'Суммарный битрейт, если известен'),
        'audio_codec': StringField(pbid=4, help_text=u'Аудио кодек. Обычно AAC'),
        'video_codec': StringField(pbid=5, help_text=u'Видео кодек. Обычно H.264'),
        'resolution': ChoiceField(pbid=6, choices=RESOLUTION_CHOICES, help_text=u'Разрешение видео'),
        'width': IntegerField(pbid=7, help_text=u'Ширина видео'),
        'height': IntegerField(pbid=8, help_text=u'Высота видео'),
    }


class VideoStreamListSerializer(ListSerializer):
    visible_fields = ['duration', 'items', 'total', 'stream_id']
    fields = {
        'items': SerializerField(lambda: VideoStreamSerializer, many=True, required=True, pbid=1, help_text=u'Описание различных видеопотоков'),
        'duration': StringField(required=True, pbid=5, source='duration', help_text=u'Длительность видео в миллисекундах'),
        'stream_id': StringField(required=True, pbid=6, source='stream_id', help_text=u'Идентификатор видеопотока'),
    }


"""
Фотосрез
"""

class PhotosliceSnapshotLinkSerializer(LinkSerializer):
    fields = {
        'revision': IntegerField(required=True, pbid=4, source='result.revision', help_text=u'Ревизия снапшота'),
        'photoslice_id': StringField(required=True, pbid=5, source='result.photoslice_id', help_text=u'Уникальный идентификатор фотосреза'),
    }

    def prepare_object(self, obj):
        obj.update(super(PhotosliceSnapshotLinkSerializer, self).prepare_object(obj['link']))
        return obj

    def get_visible_fields(self):
        return super(PhotosliceSnapshotLinkSerializer, self).get_visible_fields() + ['revision', 'photoslice_id']


class PhotoLocalizedNameSerializer(BaseSerializer):
    visible_fields = ['ru', 'en', 'uk', 'tr']
    fields = {
        "ru": StringField(required=True, pbid=1, help_text=u'Название на русском языке'),
        "en": StringField(required=True, pbid=2, help_text=u'Название на английском языке'),
        "uk": StringField(required=True, pbid=3, help_text=u'Название на украинском языке'),
        "tr": StringField(required=True, pbid=4, help_text=u'Название на турецком языке'),
    }


class PhotosliceIndexItemSerializer(BaseSerializer):
    visible_fields = ['cluster_id', 'items_count', 'from', 'to', 'locality', 'places', 'albums']
    fields = {
        'cluster_id': StringField(required=True, pbid=1, help_text=u'Уникальный идентификатор кластера'),
        'items_count': IntegerField(required=True, pbid=2, help_text=u'Количество фоток в кластере'),
        'from': DateTimeToTSField(required=False, milliseconds=True, pbid=3, help_text=u'Дата первой фотографии в кластере'),
        'to': DateTimeToTSField(required=False, milliseconds=True, pbid=4, help_text=u'Дата последней фотографии в кластере'),
        'locality': SerializerField(lambda: PhotoLocalizedNameSerializer, required=False, many=False, pbid=5, help_text=u'Город'),
        'places': SerializerField(lambda: PhotoLocalizedNameSerializer, required=False, many=True, pbid=6, help_text=u'Места'),
        'albums': JsonObjectField(required=False, max_nesting_level=0, help_text=u'Альбомы')
    }


class PhotosliceIndexListSerializer(BaseSerializer):
    visible_fields = ['items']
    fields = {
        'items': SerializerField(lambda: PhotosliceIndexItemSerializer, required=True, many=True, pbid=1, help_text=u'Элементы индекса фотосреза'),
    }


class PhotosliceClusterItemSerializer(BaseSerializer):
    visible_fields = ['item_id', 'path', 'width', 'height', 'beauty', 'albums']
    fields = {
        'item_id': StringField(required=True, pbid=1, help_text=u'идентификатор фотографии'),
        'path': MpfsPathField(required=True, allowed_areas=('disk', 'app', PHOTOUNLIM_AREA), pbid=2, help_text=u'путь к файлу'),
        'width': IntegerField(required=False, help_text=u'ширина фотографии'),
        'height': IntegerField(required=False, help_text=u'высота фотографии'),
        'beauty': FloatField(required=False, help_text=u'красивость фотографии'),
        'albums': ListField(required=False, help_text=u'альбомы'),
    }


class PhotosliceClusterSerializer(BaseSerializer):
    visible_fields = ['cluster_id', 'items']
    fields = {
        'cluster_id': StringField(required=True, pbid=1, help_text=u'идентификатор кластера'),
        'items': SerializerField(lambda: PhotosliceClusterItemSerializer, required=True, many=True, pbid=2, help_text=u'Элементы кластера фотосреза'),
    }


class PhotosliceClustersListSerializer(BaseSerializer):
    visible_fields = ['items']
    fields = {
        'items': SerializerField(lambda: PhotosliceClusterSerializer, required=True, many=True, pbid=1, help_text=u'Кластера фотосреза'),
    }


class PhotosliceAlbumSerializer(BaseSerializer):
    visible_fields = ['album', 'previews', 'count']
    fields = {
        'album': StringField(required=False, help_text=u'Идентификатор альбома'),
        'previews': ListField(required=False, help_text=u'Список путей файлов используемых для превью альбома'),
        'count': IntegerField(required=False,  help_text=u'Количество фото в альбоме'),
    }


class PhotosliceAlbumsListSerializer(BaseSerializer):
    visible_fields = ['items']
    fields = {
        'items': SerializerField(lambda: PhotosliceAlbumSerializer, required=False, many=True, pbid=1, help_text=u'Альбомы'),

    }

class PhotosliceSnapshotSerializer(BaseSerializer):
    CLUSTERIZATION_TYPE_CHOICES = ['geo']
    visible_fields = ['clusterization_type', 'photoslice_id', 'revision', 'index', 'clusters', 'albums']
    fields = {
        'clusterization_type': ChoiceField(required=True, pbid=1, choices=CLUSTERIZATION_TYPE_CHOICES, source='result.clusterization_type',
                                           help_text=u'Тип кластеризации, например, geo'),
        'photoslice_id': StringField(required=True, pbid=2, source='result.photoslice_id', help_text=u'Уникальный идентификатор фотосреза'),
        'revision': IntegerField(required=True, pbid=3, source='result.revision', help_text=u'Ревизия снапшота'),
        'index': SerializerField(lambda: PhotosliceIndexListSerializer, source='result.index', many=False, required=False, pbid=4, help_text=u'Индекс кластеров фотографий'),
        'clusters': SerializerField(lambda: PhotosliceClustersListSerializer, source='result.clusters', many=False, required=False, pbid=5, help_text=u'Кластера фотографий'),
        'albums': SerializerField(lambda: PhotosliceAlbumsListSerializer, source='result.albums', required=False, pbid=6, help_text=u'Альбомы')
    }


class PhotosliceIndexPlaceChangeItemSerializer(BaseSerializer):
    CHANGE_TYPES = ['insert', 'update', 'delete']
    visible_fields = ['change_type', 'place_index', 'data']
    fields = {
        'change_type': ChoiceField(required=True, pbid=1, choices=CHANGE_TYPES, help_text=u'Тип изменения'),
        'place_index': IntegerField(required=True, pbid=2, help_text=u'Номер в списке мест'),
        'data': SerializerField(lambda: PhotoLocalizedNameSerializer, required=False, many=False, pbid=3, help_text=u'Новое название места'),
    }


class PhotosliceIndexLocalityChangeSerializer(BaseSerializer):
    CHANGE_TYPES = ['insert', 'update', 'delete']
    visible_fields = ['change_type', 'data']
    fields = {
        'change_type': ChoiceField(required=True, pbid=1, choices=CHANGE_TYPES, help_text=u'Тип изменения'),
        'data': SerializerField(lambda: PhotoLocalizedNameSerializer, required=False, many=False, pbid=2, help_text=u'Новое название города'),
    }


class PhotosliceIndexAlbumsChangeSerializer(BaseSerializer):
    CHANGE_TYPES = ['insert', 'update', 'delete']
    visible_fields = ['change_type', 'count', 'album']
    fields = {
        "album": StringField(required=False, pbid=1, help_text=u'Идентификатор альбома'),
        "change_type":  ChoiceField(required=True, pbid=2, choices=CHANGE_TYPES, help_text=u'Тип изменения'),
        "count": IntegerField(required=False, pbid=3, help_text=u'Количество фото в альбоме'),
    }


class PhotosliceIndexChangeItemSerializer(BaseSerializer):
    visible_fields = ['items_count', 'from', 'to', 'locality', 'places', 'albums']
    fields = {
        'items_count': IntegerField(required=False, pbid=1, help_text=u'Новое количество фотографий в кластере'),
        'from': DateTimeToTSField(required=False, milliseconds=True, pbid=2, help_text=u'Новая дата первой фотографии в кластере'),
        'to': DateTimeToTSField(required=False, milliseconds=True, pbid=3, help_text=u'Новая дата последней фотографии в кластере'),
        'locality': SerializerField(lambda: PhotosliceIndexLocalityChangeSerializer, required=False, many=False, pbid=4, help_text=u'Изменение города'),
        'places': SerializerField(lambda: PhotosliceIndexPlaceChangeItemSerializer, required=False, many=True, pbid=5,
                                  help_text=u'Изменения в списке мест'),
        'albums': SerializerField(lambda: PhotosliceIndexAlbumsChangeSerializer, many=True, pbid=6, required=False)
    }


class PhotosliceIndexChangesSerializer(BaseSerializer):
    CHANGE_TYPES = ['insert', 'update', 'delete']
    visible_fields = ['change_type', 'cluster_id', 'data']
    fields = {
        'change_type': ChoiceField(required=True, pbid=1, choices=CHANGE_TYPES, help_text=u'Тип изменения'),
        'cluster_id': StringField(required=True, pbid=2, help_text=u'Идентификатор кластера'),
        'data': SerializerField(lambda: PhotosliceIndexChangeItemSerializer, many=False, required=False, pbid=3,
                                help_text=u'Изменения полей кластера в индексе'),
    }


class PhotosliceItemDataSerializer(BaseSerializer):
    visible_fields = ['path', 'width', 'height', 'beauty', 'albums']
    fields = {
        'path': MpfsPathField(required=True, allowed_areas=('disk', 'app', PHOTOUNLIM_AREA), pbid=1, help_text=u'Путь к файлу'),
        'width': IntegerField(required=False, help_text=u'ширина фотографии'),
        'height': IntegerField(required=False, help_text=u'высота фотографии'),
        'beauty': FloatField(required=False, help_text=u'красивость фотографии'),
        'albums': ListField(required=False, help_text=u'список альбомов в которые входит фотография'),
    }


class PhotosliceItemChangesSerializer(BaseSerializer):
    CHANGE_TYPES = ['insert', 'update', 'delete']
    visible_fields = ['change_type', 'cluster_id', 'item_id', 'data']
    fields = {
        'change_type': ChoiceField(required=True, pbid=1, choices=CHANGE_TYPES, help_text=u'Тип изменения'),
        'cluster_id': StringField(required=True, pbid=2, help_text=u'Идентификатор кластера'),
        'item_id': StringField(required=True, pbid=3, help_text=u'Идентификатор фотографии'),
        'data': SerializerField(lambda: PhotosliceItemDataSerializer, many=False, required=False, pbid=4,
                                help_text=u'Изменения полей кластера'),
    }


class PhotosliceAlbumItemDataChangesSerializer(BaseSerializer):
    visible_fields = ['previews', 'count']
    fields = {
        'previews': ListField(required=False, help_text=u'Список путей до файлов с превью альбома'),
        'count': IntegerField(required=False, help_text=u'Количество'),
    }


class PhotosliceAlbumChangesSerializer(BaseSerializer):
    CHANGE_TYPES = ['insert', 'update', 'delete', 'set']
    visible_fields = ['change_type', 'album', 'data']
    fields = {
        'change_type': ChoiceField(required=False, choices=CHANGE_TYPES, help_text=u'Тип изменения'),
        'album': StringField(required=False, help_text=u'Идентификатор альбома.'),
        'data': SerializerField(lambda: PhotosliceAlbumItemDataChangesSerializer, many=False, required=False,
                                help_text=u'Изменения в альбомах'),

    }


class PhotosliceDeltaSerializer(BaseSerializer):
    visible_fields = ['base_revision', 'revision', 'index_changes', 'items_changes', 'albums_changes']
    fields = {
        'base_revision': IntegerField(required=True, pbid=1, help_text=u'Исходная ревизия дельты'),
        'revision': IntegerField(required=True, pbid=2, help_text=u'Ревизия дельты'),
        'index_changes': SerializerField(lambda: PhotosliceIndexChangesSerializer, many=True, required=False, pbid=3,
                                         help_text=u'Изменения в индексе кластеров'),
        'items_changes': SerializerField(lambda: PhotosliceItemChangesSerializer, many=True, required=False, pbid=4,
                                         help_text=u'Изменения в кластерах'),
        'albums_changes': SerializerField(lambda: PhotosliceAlbumChangesSerializer, many=True, required=False, pbid=5,
                                          help_text=u'Изменения в альбомах'),
    }


class PhotosliceDeltaListSerializer(BaseSerializer):
    visible_fields = ['revision', 'total', 'limit', 'items']
    fields = {
        'revision': IntegerField(required=True, pbid=1, source='result.revision', help_text=u'Текущая ревизия фотосреза'),
        'total': IntegerField(required=True, pbid=2, source='result.total', help_text=u'Количество дельт до текущей ревизии'),
        'limit': IntegerField(required=True, pbid=3, source='result.limit', help_text=u'Ограничение по количеству дельт'),
        'items': SerializerField(lambda: PhotosliceDeltaSerializer, source='result.items', many=True, required=False, pbid=4, help_text=u'Список дельт'),
    }


"""
Фотоальбомы
"""

class PhotoAlbumItemSerializer(BaseSerializer):
    """Сериализатор элемента-файла альбома"""
    visible_fields = ['album_id', 'item_id', 'resource']
    fields = {
        'album_id': StringField(required=True, help_text=u'Идентификатор фотоальбома.'),
        'item_id': StringField(required=True, source='id', help_text=u'Идентификатор элемента фотоальбома.'),
        'resource': SerializerField(ResourceSerializer, required=True, source='object', help_text=u'Ресурс, на который ссылается элемент фотоальбома.'),
    }


class PhotoAlbumSerializer(BaseSerializer):
    """Сериализатор альбома"""
    LAYOUTS = ['rows', 'squares', 'waterfall', 'fit_width']
    visible_fields = ['album_id', 'title', 'created', 'layout', 'public_url', 'is_public', 'cover', 'is_empty', 'views_count']
    fields = {
        'album_id': StringField(required=True, source='id', help_text=u'Идентификатор фотоальбома.'),
        'title': StringField(required=True, help_text=u'Название фотоальбома.'),
        'created': DateTimeToTSField(required=True, source='ctime', help_text=u'Дата и время создания фотоальбома.'),
        'layout': ChoiceField(required=True, choices=LAYOUTS, help_text=u'Тип вёрстки фотоальбома.'),
        'public_url': StringField(source='public.short_url', help_text=u'Публичная ссылка на фотоальбом.'),
        'is_public': BooleanField(help_text=u'Признак публичности фотоальбома.'),
        'cover': SerializerField(ResourceSerializer, help_text=u'Ресурс, на который ссылается элемент фотоальбома.'),
        'is_empty': BooleanField(help_text=u'Признак пустоты фотоальбома.'),
        'views_count': IntegerField(source='public.views_count', help_text=u'Счетчик просмотров публичного фотоальбома.'),
    }

    def prepare_object(self, obj):
        cover = obj.pop('cover', None)
        if cover and 'object' in cover:
            obj['cover'] = cover['object']
        return super(PhotoAlbumSerializer, self).prepare_object(obj)


class ExclusionsFromGeneratedAlbumsItemsSerializer(BaseSerializer):
    visible_fields = ['items']
    fields = {
        'items': ListField(
            item_field_type=StringField,
            help_text=u'Список исключений из автосгенерированных альбомов.'
        )
    }


class OfficeOnlineEditorURLSerializer(BaseSerializer):
    visible_fields = ['edit_url']
    fields = {
        'edit_url': StringField(required=True, help_text=u'Ссылка на редактирование файла.'),
    }


class PhotoAlbumWithItemsSerializer(PhotoAlbumSerializer):
    visible_fields = PhotoAlbumSerializer.visible_fields + ['items']
    fields = {
        'items': SerializerField(PhotoAlbumItemSerializer, many=True, help_text=u'Список элементов фотоальбома.')
    }


class PhotoAlbumsListSerializer(ListSerializer):
    """Сериализатор списка альбомов"""
    visible_fields = ['limit', 'offset', 'total', 'items']
    fields = {
        'items': SerializerField(PhotoAlbumSerializer, many=True, required=True, pbid=1, help_text=u'Элементы списка.'),
    }


class PhotoAlbumItemResourceNewSerializer(BaseSerializer):
    """Сериализатор содержимого ресурса, присылаемого пользователем"""
    visible_fields = ['path']
    fields = {
        'path': MpfsPathField(required=True, allowed_areas=('disk', 'app', PHOTOUNLIM_AREA), help_text=u'Путь к ресурсу, который следует добавить в альбом.'),
    }


class PhotoAlbumItemsNewSerializer(BaseSerializer):
    """Сериализатор списка элементов альбома, присылаемого пользователем"""
    visible_fields = ['resources']
    fields = {
        'resources': SerializerField(PhotoAlbumItemResourceNewSerializer, many=True, required=True, help_text=u'Список ресурсов.'),
    }


class PhotoAlbumItemNewSerializer(BaseSerializer):
    """Сериализатор элемента альбома, присылаемого пользователем"""
    visible_fields = ['resource']
    fields = {
        'resource': SerializerField(PhotoAlbumItemResourceNewSerializer, required=True, help_text=u'Ресурс.'),
    }

    def restore_object(self, attrs, instance=None):
        """Приведение формата элементов альбома к MPFS-ному формату"""
        attrs = super(PhotoAlbumItemNewSerializer, self).restore_object(attrs, instance=instance)
        return {'type': 'resource', 'path': attrs['resource']['path']}


class PhotoAlbumNewSerializer(BaseSerializer):
    """Сериализатор тела пользовательского запроса на создание альбома"""
    visible_fields = ['title', 'layout', 'is_public', 'cover', 'is_empty', 'items']
    fields = {
        'title': StringField(help_text=u'Название фотоальбома.'),
        'layout': ChoiceField(choices=PhotoAlbumSerializer.LAYOUTS, default=PhotoAlbumSerializer.LAYOUTS[0], help_text=u'Тип вёрстки фотоальбома.'),
        'is_public': BooleanField(default=True, help_text=u'Признак публичности фотоальбома.'),
        'cover': StringField(default='0', help_text=u'Номер элемента или id элемента фотоальбома, который станет обложкой.'),
        'is_empty': BooleanField(default=True, help_text=u'Признак пустоты фотоальбома.'),
        'items': SerializerField(PhotoAlbumItemNewSerializer, many=True, help_text=u'Ресурсы, добавляемые в альбом.'),
    }


class PhotoAlbumItemsListSerializer(BaseSerializer):
    """Сериализатор списка элементов альбома"""
    DEFAULT_LIMIT = 20
    visible_fields = ['limit', 'last_item_id', 'items']
    fields = {
        'items': SerializerField(PhotoAlbumItemSerializer, many=True, required=True, help_text=u'Список элементов фотоальбома.'),
        'limit': IntegerField(default=DEFAULT_LIMIT, help_text=u'Количество элементов на странице'),
        'last_item_id': StringField(help_text=u'Идентификатор последнего элемента полученного на предыдущей странице.'),
    }


class PhotoAlbumPatchSerializer(PhotoAlbumNewSerializer):
    """Сериализатор тела пользовательского запроса на изменение аттрибутов альбома"""
    is_patch = True
    visible_fields = ['title', 'layout', 'is_public', 'cover']


class OrganizationSerializer(BaseSerializer):
    """Сериализатор организации"""
    visible_fields = ['organization_id']
    fields = {
        'organization_id': StringField(required=True, help_text=u'Идентификатор организации.')
    }


class OrganizationsListSerializer(BaseSerializer):
    """Сериализатор списка организаций"""
    visible_fields = ['items']
    fields = {
        'items': SerializerField(OrganizationSerializer, many=True, required=True, help_text=u'Список организаций.'),
    }


class ShareProviderSerializer(BaseSerializer):
    visible_fields = ['provider_id']
    fields = {
        'provider_id': StringField(required=True, help_text=u'Идентификатор социальной сети.'),
    }


class ShareProvidersListSerializer(ListSerializer):
    visible_fields = ['items']
    fields = {
        'items': SerializerField(ShareProviderSerializer, required=True, many=True,
                                 help_text=u'Список социальных сетей, в которых пользователь может публиковать альбомы.'),
    }


class InternalUserSerializer(BaseSerializer):
    fields = {
        'uid': StringField(help_text=u'Идентификатор пользователя'),
        'login': StringField(help_text=u'Логин'),
        'first_name': StringField(source='firstname', help_text=u'Имя'),
        'last_name': StringField(source='lastname', help_text=u'Фамилия'),
        'sex': StringField(help_text=u'Пол'),
    }
    visible_fields = fields.keys()


class ClientsConfigSerializer(BaseSerializer):
    fields = {
        'config_db_prefix': StringField(help_text=u'Префикс базы c конфигурацией ПО'),
    }
    visible_fields = fields.keys()


class UserSerializer(InternalUserSerializer):
    fields = {
        'display_name': StringField(help_text=u'Отображаемое имя'),
        'country': StringField(help_text=u'Страна'),
        'avatar_url': UrlField(source='avatar', help_text=u'Ссылка на аватарку'),
        'is_yandex_staff': BooleanField(source='has_staff',
                                        help_text=u'Признак того что пользователь привязан к Стаффу')
    }
    visible_fields = ['uid', 'login', 'display_name', 'country', 'avatar_url', 'is_yandex_staff']
    native_client_fields = ['avatar_url', 'is_yandex_staff']


class ResourceDimensionsSerializer(BaseSerializer):
    fields = {
        'width': IntegerField(help_text=u'Ширина изображения.'),
        'height': IntegerField(help_text=u'Высота изображения.')
    }
    visible_fields = fields.keys()


class ResourceVersionSerializer(BaseSerializer):
    fields = {
        'id': StringField(required=True, help_text=u'ID версии'),
        'type': StringField(required=True, help_text=u'Тип версии'),
        'uid_created': StringField(required=True, help_text=u'uid пользователя создавшего версию'),
        'platform_created': StringField(required=True, help_text=u'Название платформы, из которой создали версию'),
        'can_be_restored': BooleanField(required=True, help_text=u'Может ли версия быть восстановлена'),
        'size': IntegerField(required=False, help_text=u'Размер версии (только для бинарных)'),
        'md5': StringField(required=False, help_text=u'MD5-хэш'),
        'sha256': StringField(required=False, help_text=u'SHA256-хэш'),
        'file': StringField(source='file_url', help_text=u'URL для скачивания файла'),
        'created': StringField(required=True, source='date_created', help_text=u'Дата создания'),
        'folded_items_iteration_key': StringField(required=False,
                                                  help_text=u'iteration_key для получения свернутых версию'),
    }
    visible_fields = fields.keys()


class ResourceVersionsSerializer(BaseSerializer):
    fields = {
        'iteration_key': StringField(help_text=u'Ключ итерирования по списку версий.'),
        'items':  SerializerField(lambda: ResourceVersionSerializer,
                                  required=True,
                                  many=True,
                                  source='versions',
                                  help_text=u'Список версий.'),
    }
    visible_fields = fields.keys()


class ExperimentsSerializer(BaseSerializer):
    fields = {
        'uas_exp_flags': ListField(item_field_type=JsonObjectField,
                                   help_text=u'Список параметров экспериментов, полученых от UaaS'),
    }
    visible_fields = fields.keys()


class FeatureToggleSerializer(BaseSerializer):
    visible_fields = ['enabled']
    fields = {
        'enabled': BooleanField('enabled', required=True)
    }


class PlatformActivityInfoSerializer(BaseSerializer):
    fields = {
        'first_activity': DateField(source='first_activity', required=True),
        'last_activity': DateField(source='last_activity', required=True),
    }
    visible_fields = fields.keys()


class UserFeatureTogglesSerializer(BaseSerializer):
    fields = {
        'priority_support': SerializerField(
            FeatureToggleSerializer, source='priority_support', required=True,
            help_text=u'Обращения в саппорт обрабатываются в отдельной очереди (true) или в общей (false)'
        ),
        'versioning_extended_period': SerializerField(
            FeatureToggleSerializer, source='versioning_extended_period', required=True,
            help_text=u'Версии пользователя хранятся 90 дней (true) или 14 (false)'
        ),
        'antifo': SerializerField(
            FeatureToggleSerializer, source='antifo', required=True,
            help_text=u'Применять к публичным файлам пользователя antifo-ограничения (true), не применять (false)'
        ),
        'advertising': SerializerField(
            FeatureToggleSerializer, source='advertising', required=True,
            help_text=u'Показывать рекламу (true) или нет (false)'
        ),
        'disk_pro': SerializerField(
            FeatureToggleSerializer, source='disk_pro', required=True,
            help_text=u'Диск.PRO включен (true) или нет (false)'
        ),
        'disk_pro_without_ps_billing': SerializerField(
            FeatureToggleSerializer, source='disk_pro_without_ps_billing', required=False,
            help_text=u'Признак того, есть ли у пользователя диск.PRO. при этом, если у пользователя подключены только услуги ps billing, то считается что про у него нет'
        ),
        'online_editor': SerializerField(
            FeatureToggleSerializer, source='online_editor', required=True,
            help_text=u'Редактирование офисных файлов онлайн доступно (true) или нет (false)'
        ),
        'desktop_folder_autosave': SerializerField(
            FeatureToggleSerializer, source='desktop_folder_autosave', required=True,
            help_text=u'Автосорханение обычных десктопных папок в облако включено (true) или нет (false)'
        ),
        'unlimited_video_autouploading': SerializerField(
            FeatureToggleSerializer, source='unlimited_video_autouploading', required=True,
            help_text=u'Загрузка безлимитного видео'
        ),
        'unlimited_photo_autouploading': SerializerField(
            FeatureToggleSerializer, source='unlimited_photo_autouploading', required=True,
            help_text=u'Загрузка безлимитных фотографий'
        ),
        'promote_mail360': SerializerField(
            FeatureToggleSerializer, source='promote_mail360', required=False,
            help_text=u'Промоутинг Почты 360 включен (true) или нет (false)'
        ),
        'public_settings': SerializerField(
            FeatureToggleSerializer, source='public_settings', required=False,
            help_text=u'Настройки публичных ссылок доступны (true) или нет (false)'
        )
    }
    visible_fields = fields.keys()


class UserActivityInfoSerializer(BaseSerializer):
    fields = {
        'web': SerializerField(
            PlatformActivityInfoSerializer, source='web', help_text=u'Даты активности в web-интерфейсе диска'
        ),
        'ios': SerializerField(
            PlatformActivityInfoSerializer, source='ios', help_text=u'Даты активности в приложении диска для iOS'
        ),
        'android': SerializerField(
            PlatformActivityInfoSerializer, source='android', help_text=u'Даты активности в приложении диска для Android'
        ),
        'mac': SerializerField(
            PlatformActivityInfoSerializer, source='mac', help_text=u'Даты активности в приложении диска для Mac'
        ),
        'windows': SerializerField(
            PlatformActivityInfoSerializer, source='windows', help_text=u'Даты активности в приложении диска для Windows'
        ),
        'search_app': SerializerField(
            PlatformActivityInfoSerializer, source='search_app', help_text=u'Даты активности в поисковом приложении'
        ),
    }
    visible_fields = fields.keys()


class ClientsInstallerSerializer(BaseSerializer):
    fields = {
        'file': StringField(source='download_url', help_text=u'URL для скачивания файла'),
        'version': StringField(source='version', help_text=u'Версия ПО'),
        'md5': StringField(source='md5', help_text=u'MD5-хэш'),
        'sha256': StringField(source='sha256', help_text=u'SHA256-хэш'),
        'size': IntegerField(source='size', help_text=u'Размер файла'),
    }
    visible_fields = fields.keys()


class ClientsInstallerWithAutologonSerializer(ClientsInstallerSerializer):
    fields = {
        'file': StringField(help_text=u'URL для скачивания установщика'),
    }
    visible_fields = fields.keys()


class ClientBodySerializer(BaseSerializer):
    fields = {
        'fos_support_text': StringField(default='', help_text=u'Текст обращения пользователя из формы обратной связи')
    }
    visible_fields = ['fos_support_text']


class OnlyOfficeCallbackBodySerializer(BaseSerializer):
    fields = {
        'key': StringField(default='', help_text=u'Идентификатор редактирования документа в OnlyOffice'),
        'status': StringField(default='', help_text=u'Статус редактирования документа в OnlyOffice'),
        'url': StringField(default='', help_text=u'URL для скачивания отредактированного документа'),
    }
    visible_fields = ['key', 'status', 'url']


class OfficeTypeSerializer(BaseSerializer):
    fields = {
        'supported_types': ListField(item_field_type=StringField, help_text=u'Список доступных форматов'),
        'size_limit': IntegerField(source='size_limit', help_text=u'Максимальный размер файла'),
    }
    visible_fields = fields.keys()


class OfficeFileFiltersSerializer(BaseSerializer):
    fields = {
        'all_supported_types': ListField(item_field_type=StringField, help_text=u'Список всех доступных форматов'),
        'text': SerializerField(OfficeTypeSerializer, help_text=u'Параметры для Word.'),
        'presentation': SerializerField(OfficeTypeSerializer, help_text=u'Параметры для PowerPoint.'),
        'spreadsheet': SerializerField(OfficeTypeSerializer,
                                        help_text=u'Параметры для Excel.')
    }
    visible_fields = fields.keys()


class OfficeFileURLsSerializer(BaseSerializer):
    visible_fields = ['edit_url', 'view_url']
    fields = {
        'edit_url': StringField(help_text=u'Ссылка на редактирование файла.'),
        'view_url': StringField(help_text=u'Ссылка на просмотр файла.'),
    }


class AvailableUntilSerializer(BaseSerializer):
    visible_fields = ['enabled', 'value']
    fields = {
        'enabled': BooleanField('enabled', required=True, help_text=u'Признак включенности настройки.'),
        'value': IntegerField(required=False, help_text=u'Время жизни ссылки.'),
    }


class SetPublicSettingsSerializer(BaseSerializer):
    visible_fields = ['available_until', 'read_only', 'available_until_verbose', 'external_organization_id']
    fields = {
        'available_until': IntegerField(required=False, help_text=u'Время жизни ссылки.'),
        'read_only': BooleanField(required=False, help_text=u'Признак того, что ссылка доступна только на чтение.'),
        'available_until_verbose': SerializerField(
            AvailableUntilSerializer, source='available_until', required=False, help_text=u'Время жизни ссылки.'
        ),
    }
