# -*- coding: utf-8 -*-
from mpfs.common.static.codes import UPLOAD_RECORD_NOT_FOUND
from mpfs.config import settings
from mpfs.platform import fields
from mpfs.platform.exceptions import NotFoundError, PayloadTooLargeError
from mpfs.platform.permissions import AllowByClientIdPermission
from mpfs.platform.v1.case.handlers import PLATFORM_MOBILE_APPS_IDS
from mpfs.platform.v1.disk.handlers import MpfsProxyHandler
from mpfs.platform.v1.disk.source_ids.serializers import (
    AddSourceIdBodySerializer,
    BulkCheckSourceIdsListBodySerializer,
    BulkSourceIdExistenceCheckListSerializer,
)


GLOBAL_GALLERY_MAX_BULK_CHECK_SOURCE_IDS_SIZE = settings.global_gallery['max_bulk_check_source_ids_size']


class AddSourceIdHandler(MpfsProxyHandler):
    service_method = 'POST'
    permissions = AllowByClientIdPermission(PLATFORM_MOBILE_APPS_IDS)
    body_serializer_cls = AddSourceIdBodySerializer
    hidden = True
    service_url = '/json/add_source_ids?uid=%(uid)s&md5=%(md5)s&sha256=%(sha256)s&size=%(size)s' \
                  '&is_live_photo=%(is_live_photo)s'

    error_map = {
        UPLOAD_RECORD_NOT_FOUND: NotFoundError,
    }

    def get_context(self, context=None):
        c = super(AddSourceIdHandler, self).get_context(context)
        request_body = self.request.body
        c['md5'] = request_body['md5']
        c['sha256'] = request_body['sha256']
        c['size'] = request_body['size']
        c['is_live_photo'] = '1' if request_body.get('is_live_photo', False) else '0'
        return c

    def handle(self, request, *args, **kwargs):
        context = self.get_context()
        url = self.get_url(context)
        source_ids = {'source_ids': [x['source_id'] for x in self.request.body['items']]}
        data = self.request_service(url,
                                    method=self.service_method,
                                    data=source_ids,
                                    headers={'Content-type': 'application/json'})
        return self.serialize(data)


class CheckSourceIdHandler(MpfsProxyHandler):
    permissions = AllowByClientIdPermission(PLATFORM_MOBILE_APPS_IDS)
    hidden = True
    service_url = '/json/check_source_id?uid=%(uid)s&source_id=%(source_id)s'

    error_map = {
        UPLOAD_RECORD_NOT_FOUND: NotFoundError,
    }

    query = fields.QueryDict({
        'source_id': fields.StringField(required=True, help_text=u'Искомый source id ресурса.'),
    })


class BulkCheckSourceIdsHandler(MpfsProxyHandler):
    permissions = AllowByClientIdPermission(PLATFORM_MOBILE_APPS_IDS)
    hidden = True
    service_url = '/json/check_source_ids?uid=%(uid)s'

    body_serializer_cls = BulkCheckSourceIdsListBodySerializer
    serializer_cls = BulkSourceIdExistenceCheckListSerializer

    def handle(self, request, *args, **kwargs):
        context = self.get_context()
        url = self.get_url(context)
        if len(self.request.body['source_ids']) > GLOBAL_GALLERY_MAX_BULK_CHECK_SOURCE_IDS_SIZE:
            raise PayloadTooLargeError('Size of source_ids chunk is %s' % len(self.request.body['source_ids']))
        source_ids = {'source_ids': [x['source_id'] for x in self.request.body['source_ids']]}
        data = self.request_service(url,
                                    method=self.service_method,
                                    data=source_ids,
                                    headers={'Content-type': 'application/json'})
        return self.serialize(data)

    def serialize(self, obj, *args, **kwargs):
        obj = {
            'items':
                [
                    {
                        'source_id': {
                            'source_id': i['source_id']
                        },
                        'found': i['found']
                    }
                    for i in obj['items']
                ]
        }
        return super(BulkCheckSourceIdsHandler, self).serialize(obj, *args, **kwargs)
