# -*- coding: utf-8 -*-
import urllib

from mpfs.config import settings
from mpfs.common.errors import NotesBadResponse
from mpfs.common.util.ordereddict import OrderedDict
from mpfs.core.services.notes_service import notes
from mpfs.platform import fields
from mpfs.platform.auth import PassportCookieAuth
from mpfs.platform.common import PlatformResponse
from mpfs.platform.formatters import UnicodeStringFormatter
from mpfs.platform.handlers import ServiceProxyHandler
from mpfs.platform.v1.disk.permissions import WebDavPermission, MobileMailPermission
from mpfs.platform.v1.comments.handlers import CommentsProxyHandler
from mpfs.platform.v1.notes.exceptions import (
    NotesBadRequestError,
    NotesBadRequestBodyError,
    NotesNotAuthenticatedError,
    NotesForbiddenError,
    NotesNotFoundError,
    NotesMethodNotAllowedError,
    NotesConflictError,
)
from mpfs.platform.v1.notes.permissions import NotesReadPermission, NotesWritePermission


PLATFORM_DISK_APPS_IDS = settings.platform['disk_apps_ids']


class NotesProxyHandler(CommentsProxyHandler):
    auth_required = True
    auth_methods = [PassportCookieAuth()]

    permissions = WebDavPermission() | MobileMailPermission() | (NotesReadPermission() & NotesWritePermission())

    content_types = OrderedDict({ServiceProxyHandler.CONTENT_TYPE_JSON: UnicodeStringFormatter()})

    service = notes
    service_base_exception = NotesBadResponse

    kwargs = fields.QueryDict({
        'proxied_path': fields.StringField(required=True, help_text='Проксируемый путь в сервис заметок.')
    })

    error_map = {
        'validate': NotesBadRequestError,
        'cant-parse': NotesBadRequestBodyError,
        'not-authenticated': NotesNotAuthenticatedError,
        'forbidden': NotesForbiddenError,
        'not-found': NotesNotFoundError,
        'method-not-allowed': NotesMethodNotAllowedError,
        'revision-conflict': NotesConflictError,
    }

    def get_base_url(self):
        base_url = super(NotesProxyHandler, self).get_base_url()
        if base_url.endswith('/'):
            return base_url
        else:
            return base_url + '/'

    def handle(self, request, *args, **kwargs):
        query = {
            name: value for name, value in request.args.iteritems()
            if name not in ('__uid', 'uid')
        }
        query['__uid'] = request.user.uid

        url = self.get_base_url() + kwargs['proxied_path'] + '?' + urllib.urlencode(query)

        headers = self.get_all_forwardable_headers(request)
        headers.update({
            'Accept': 'application/json',
            'Content-Type': 'application/json'
        })

        try:
            code, content, headers = self.raw_request_service(
                url, method=request.method, headers=headers, data=request.body
            )
            return PlatformResponse(status_code=code, content=content, headers=headers)
        except NotesBadResponse as e:
            code, content, headers = e.data['code'], e.data['text'], e.data['headers']
            if code == 304:
                return PlatformResponse(status_code=code, content=content, headers=headers)
            raise

    def get_platform_exception(self, exception):
        result = super(NotesProxyHandler, self).get_platform_exception(exception)
        if isinstance(result, NotesConflictError):
            result.headers = exception.data['headers']
        return result
