# -*- coding: utf-8 -*-
from mpfs.common.util import urls
from mpfs.config import settings
from mpfs.common.errors import NotifierBadResponse
from mpfs.common.errors.common import APIError
from mpfs.core.services.notifier_service import notifier
from mpfs.platform import fields
from mpfs.platform.auth import PassportCookieAuth
from mpfs.platform.handlers import ServiceProxyHandler
from mpfs.platform.v1.notifier import exceptions
from mpfs.platform.permissions import AllowByClientIdPermission, DenyAllPermission
from mpfs.platform.v1.notifier.permissions import NotifierNotificationMarkAsReadPremission

PLATFORM_DISK_APPS_IDS = settings.platform['disk_apps_ids']


class NotifierProxyHandler(ServiceProxyHandler):
    """Прокидываем в бэкенд запрос как есть"""
    auth_required = False
    auth_user_required = False
    permissions = DenyAllPermission()
    hidden = True
    service = notifier
    kwargs = fields.QueryDict({
        'notifier_path_part': fields.StringField(required=True),
    })
    service_base_exception = NotifierBadResponse

    def get_base_url(self):
        url = super(NotifierProxyHandler, self).get_base_url()
        if not url.endswith('/'):
            return url + '/'
        return url

    def handle(self, request, *args, **kwargs):
        url = '%s%s?%s' % (self.get_base_url(), kwargs['notifier_path_part'], urls.urlencode(request.args))
        headers = self.get_all_forwardable_headers(request)
        return self.service.open_url(url, method=request.method, headers=headers, pure_data=request.data,
                                     retry=False, return_status=True, timeout=self.service.timeout)

    def handle_exception(self, exception):
        if isinstance(exception, self.service_base_exception):
            data = getattr(exception, 'data', {})
            status_code = data.get('code', None)
            content = data.get('text', None)
            if status_code:
                return status_code, content, {}
        return super(NotifierProxyHandler, self).handle_exception(exception)

    def _clean_body(self, raw_data):
        return None


class BaseNotifierHandler(ServiceProxyHandler):
    permissions = AllowByClientIdPermission(PLATFORM_DISK_APPS_IDS) | NotifierNotificationMarkAsReadPremission()
    auth_methods = [PassportCookieAuth()]
    service = notifier
    service_base_exception = NotifierBadResponse

    def get_service_error_code(self, exception):
        try:
            return exception.data['code']
        except (AttributeError, KeyError):
            return None


class MarkNotificationAsReadHandler(BaseNotifierHandler):
    """Отметить нотификацию как прочитанную"""
    hidden = True
    service_url = ('/mark-as-read?owner_uid=%(uid)s&record_type=%(notification_entity_type)s&'
                   'group_key=%(notification_id)s&service=%(service)s')
    kwargs = fields.QueryDict({
        'notification_entity_type': fields.StringField(required=True),
        'notification_id': fields.StringField(required=True),
        'service': fields.StringField(required=True),
    })
    resp_status_code = 200
    error_map = {
        400: exceptions.BadNotificationEntityTypeError,
    }

    def serialize(self, *args, **kwargs):
        return {}
