# -*- coding: utf-8 -*-

import urlparse

from mpfs.common.util import from_json, to_json
from mpfs.common.util.urls import urlencode
from mpfs.core.services.big_brother_service import big_brother
from mpfs.core.services.data_api_profile_service import data_api_profile
from mpfs.core.services.data_api_service import data_api
from mpfs.platform.v1.batch.batch_processors import BaseBatchRequestProcessor
from mpfs.platform.v1.personality.handlers import (
    ProfileBaseHandler,
    ListOldAddressesDataApiHandler,
    SaveOldAddressDataApiHandler, GetOldAddressDataApiHandler,
    DeleteOldAddressDataApiHandler,
    GetActualFlightsListHandler, CreateActualFlightHandler, GetFlightHandler,
    SaveActualFlightHandler, DeleteActualFlightHandler,
    GetYaTicketsOrdersListHandler, GetYaTicketsOrderHandler, SaveYaTicketsOrderHandler, DeleteYaTicketsOrderHandler,
    GenericObjectHandler
)
from mpfs.platform.v2.personality.handlers import (
    GetNewAddressHandler, ListNewAddressesHandler, CreateNewAddressHandler,
    PatchNewAddressHandler, DeleteNewAddressHandler, SaveNewAddressHandler,
    TouchNewAddressHandler, TagNewAddressHandler, UntagNewAddressHandler
)


class ProfileBatchRequestProcessor(BaseBatchRequestProcessor):
    """
    Обрабатывает залогиновые запросы в personality/profile.
    """
    support_handlers = {
        # v1
        ListOldAddressesDataApiHandler, SaveOldAddressDataApiHandler, GetOldAddressDataApiHandler,
        DeleteOldAddressDataApiHandler,
        GetActualFlightsListHandler, CreateActualFlightHandler, GetFlightHandler, SaveActualFlightHandler,
        DeleteActualFlightHandler,
        GetYaTicketsOrdersListHandler, GetYaTicketsOrderHandler, SaveYaTicketsOrderHandler, DeleteYaTicketsOrderHandler,
        GenericObjectHandler,

        # v2
        ListNewAddressesHandler, CreateNewAddressHandler,
        GetNewAddressHandler, PatchNewAddressHandler, DeleteNewAddressHandler,
        TouchNewAddressHandler, TagNewAddressHandler, UntagNewAddressHandler
    }

    URL_TEMPLATE = '/api/batch'

    def check_request(self, batch_request):
        """Дополнительно к проверке по умолчанию проверяет, что запрос делает залогиновый пользователь."""
        if super(ProfileBatchRequestProcessor, self).check_request(batch_request):
            native_request = self.native_requests.get(batch_request)
            uid_type, _ = ProfileBaseHandler.get_typed_uid(native_request.user.uid)
            return uid_type == 'uid'
        return False

    def _get_handler_kwargs_args(self, batch_request):
        native_request = self.native_requests.get(batch_request)
        match = self.dispatcher.router.resolve(native_request)
        resource = match.handler
        return resource.relations.get(native_request.method), match.kwargs, match.args

    def _get_handler(self, batch_request):
        handler, kwargs, args = self._get_handler_kwargs_args(batch_request)
        return handler

    def construct_dataapi_batch_request_item(self, request, uid):
        """
        Преобразовать наш реквест в батчевой пачке в item, который мы вставим
        внутрь тела для батчевого запроса в dataapi.
        """

        handler, kwargs, args = self._get_handler_kwargs_args(request)
        handler_cls = type(handler)

        relative_url_context = {'uid': uid}
        relative_url_context.update(kwargs)

        if isinstance(request.body, basestring):
            formatted_body = request.body.strip('"')
        else:
            formatted_body = to_json(request.body or {})

        item = {
            'method': handler.service_method,
            'headers': request.headers,
            'body': formatted_body,
        }

        prefix_path = urlparse.urlparse(handler.service.base_url).path.rstrip('/')
        if handler_cls == GenericObjectHandler:
            _, resource_path = request.url.split('personality/profile/')
            parsed_url = urlparse.urlparse(resource_path)
            relative_url_context.update({
                'http_method': request.method,
                'uid': uid,
                'resource_path': parsed_url.path
            })
            args = urlparse.parse_qs(parsed_url.query)
            args.pop('uid', None)
            args.pop('__uid', None)
            item['relative_url'] = handler.service_url % relative_url_context + '&' + urlencode(args)

        elif handler_cls in (
            # v1
            GetYaTicketsOrdersListHandler, ListOldAddressesDataApiHandler, DeleteActualFlightHandler,
            GetFlightHandler, GetActualFlightsListHandler, SaveYaTicketsOrderHandler,
            DeleteYaTicketsOrderHandler, GetYaTicketsOrderHandler,

            # v2
            GetNewAddressHandler, ListNewAddressesHandler, CreateNewAddressHandler, DeleteNewAddressHandler,
            PatchNewAddressHandler, SaveNewAddressHandler
        ):
            item['relative_url'] = prefix_path + handler.service_url % relative_url_context

        elif handler_cls in (
            GetOldAddressDataApiHandler, DeleteOldAddressDataApiHandler, SaveOldAddressDataApiHandler
        ):
            relative_url_context.update(handler.get_additional_context(request, *args, **kwargs))
            item['relative_url'] = prefix_path + handler.service_url % relative_url_context

        elif handler_cls in (SaveActualFlightHandler, CreateActualFlightHandler):
            item['relative_url'] = prefix_path + handler.service_url % relative_url_context
            item['body'] = to_json(handler.preprocess_flight_times(request.body))
        else:
            raise NotImplementedError()

        return item

    def process(self, batch_requests):
        items = []
        uid = self.main_request.user.uid
        for batch_request in batch_requests:
            item = self.construct_dataapi_batch_request_item(batch_request, uid)
            items.append(item)

        resp = data_api.open_url(
            '%s%s' % (data_api.base_url, self.URL_TEMPLATE),
            pure_data=to_json({'items': items}),
            headers={'Content-type': 'application/json'},
            method='POST'
        )
        data_api_resp = from_json(resp)

        result = {}
        for request_number, batch_request in enumerate(batch_requests):
            data = {'invocationInfo': data_api_resp['invocationInfo']}

            handler_cls = type(self._get_handler(batch_request))

            if handler_cls == ListOldAddressesDataApiHandler:
                data['result'] = [
                    data_api_resp['result'][request_number]['result'][0],
                    data_api_resp['result'][request_number]['result'][1]
                ]
            else:
                patch = data_api_resp['result'][request_number]
                data.update(patch)

            url_pattern = '.*%s' % urlparse.urlparse(items[request_number]['relative_url']).path
            status_code = data_api_resp['result'][request_number]['code']

            data = to_json(data)

            patched_service = data_api_profile
            if handler_cls == GenericObjectHandler:
                patched_service = data_api

            patched_service.clean_response_patches()
            patched_service.set_response_patch(
                items[request_number]['method'],
                url_pattern,
                status_code,
                data=data
            )
            result[batch_request] = self.dispatcher.dispatch(self.raw_requests.get(batch_request),
                                                             check_rate_limit=False)

        return result
