# -*- coding: utf-8 -*-
import sys
import urllib
from copy import deepcopy

from mpfs.common.errors import ProfileError, DataApiProfileError, DataApiBadResponse
from mpfs.common.util import from_json, ctimestamp
from mpfs.config import settings
from mpfs.core.services.big_brother_service import big_brother
from mpfs.core.services.data_api_profile_service import data_api_profile
from mpfs.core.services.data_api_service import data_api
from mpfs.core.services.profile_service import profile
from mpfs.platform import fields
from mpfs.platform.auth import PassportCookieAuth
from mpfs.platform.exceptions import ValidationError, CloudApiError, BadRequestError
from mpfs.platform.handlers import ServiceProxyHandler, BasePlatformHandler
from mpfs.platform.permissions import DenyAllPermission
from mpfs.platform.rate_limiters import DataApiPerClientIdRateLimiter
from mpfs.platform.v1.disk.permissions import WebDavPermission
from mpfs.platform.v1.disk.serializers import LinkSerializer
from mpfs.platform.v1.personality.permissions import GenericObjectsPermission, DocsObjectPermission
from mpfs.platform.v1.personality.exceptions import (
    PersonalityAddressesUidMustBeIntegerError,
    PersonalityAddressesNotFoundError,
    PersonalityHomeRegionNotFoundError,
    PersonalityEventNotFoundError,
    PersonalityYaTicketsOrderNotFoundError,
    PersonalityObjectValidationError,
    PersonalityMethodNotAllowedError,
    PersonalityObjectNotFoundError,
    PersonalityRewriteIfConditionError,
    PersonalityUnknownObjectTypeError,
    PersonalityUserNotFoundError,
)
from mpfs.platform.v1.personality.serializers import (
    FlightsListSerializer,
    FlightSerializer,
    DataApiAddressSerializer,
    DataApiAddressListSerializer,
    DataApiGeopointSerializer,
    YaTicketsOrdersListSerializer,
    YaTicketsOrderSerializer,
    NotificationsCountSerializer,
    DocsObjectSerializer
)

DOCS_COLLECTION_SIZE = settings.docs['collection_size']
DOCS_DATABASES = settings.docs['databases']


def is_geo_point(id):
    return id.startswith("work=home=") or id.startswith("home=work=")


class ProfileBaseHandler(ServiceProxyHandler):
    uid_prefixes = {
        'uid': 'uid',
        'yaid': 'yandexuid',
        'device': 'device_id'
    }
    profile_id_separator = '-'
    profile_id_separator_replacement = ':'
    service = profile
    consumer = 'cloud-api'
    service_base_exception = ProfileError
    error_map = {
        'integer': PersonalityAddressesUidMustBeIntegerError,
        'toolong': ValidationError,
        'empty': ValidationError,
        'toofew': ValidationError,
        'validate': ValidationError,
    }

    @classmethod
    def get_typed_uid(cls, uid):
        type = 'uid'
        uid_chunks = uid.split(cls.profile_id_separator, 1)
        if len(uid_chunks) > 1 and uid_chunks[0] in cls.uid_prefixes:
            type = cls.uid_prefixes[uid_chunks[0]]
            uid = uid_chunks[1]
        return type, uid

    def get_context(self, context=None):
        c = super(ProfileBaseHandler, self).get_context(context=context)
        # Используем типизированный UID
        # Подробности: https://wiki.yandex-team.ru/users/akinfold/cloud-api-personality-interface#identifikacijapolzovatelejjvovnutrennemapi
        id_type, id_value = None, None
        if self.request.user is not None:
            id_type, id_value = self.get_typed_uid(self.request.user.uid)
        if id_type in ('yandexuid', 'device_id'):
            profile_id = self.profile_id_separator_replacement.join((id_type, id_value))
        else:
            profile_id = id_value

        if id_type is not None:
            c['uid_type'] = id_type
        if profile_id is not None:
            c['uid'] = profile_id

        c['consumer'] = self.consumer
        return c

    def get_platform_exception(self, exception):
        code = 'default'
        text = exception.data.get('text', None)
        if text:
            try:
                err_data = from_json(text)
            except:
                pass
            else:
                errors = err_data.get('errors', None)
                if errors:
                    error = errors[0]
                    code = error.get('code', None)
        return self.get_error_map().get(code, None)


def _auto_initialize_user(func):
    def wrapper(handler, *args, **kwargs):
        try:
            return func(handler, *args, **kwargs)
        except handler.service_base_exception, e:
            if handler.get_service_error_code(e) == 'user-not-found':
                handler.init_user()
                return func(handler, *args, **kwargs)
            raise
    return wrapper


class NewProfileBaseHandler(BasePlatformHandler):
    permissions = DenyAllPermission()
    uid_prefixes = {
        'uid': 'uid',
        'yaid': 'yandexuid',
        'device': 'device_id'
    }
    uid_separator = '-'

    def get_typed_uid(self, uid):
        type = 'uid'
        uid_chunks = uid.split(self.uid_separator, 1)
        if len(uid_chunks) > 1 and uid_chunks[0] in self.uid_prefixes:
            type = self.uid_prefixes[uid_chunks[0]]
            uid = uid_chunks[1]
        return type, uid

    def get_empty_response(self):
        return {}

    def yandexuid_request_service(self, uid):
        return self.get_empty_response()

    def passport_request_service(self, uid):
        return self.get_empty_response()

    def device_id_request_service(self, uid):
        return self.get_empty_response()

    def handle(self, request, *args, **kwargs):
        uid_type, uid = self.get_typed_uid(self.request.user.uid)
        if uid_type == 'yandexuid':
            resp = self.yandexuid_request_service(uid=uid)
        elif uid_type == 'uid':
            resp = self.passport_request_service(uid=uid)
        elif uid_type == 'device_id':
            resp = self.device_id_request_service(uid=uid)
        else:
            raise NotImplementedError()

        return self.serialize(resp)


class DataApiProfileBaseHandler(ProfileBaseHandler):
    service_base_exception = DataApiProfileError
    service = data_api_profile
    permissions = DenyAllPermission()
    rate_limiter = DataApiPerClientIdRateLimiter()
    empty_response = {}

    def get_empty_response(self):
        return self.empty_response

    @staticmethod
    def _get_result_from_data_api_response(resp):
        return resp.get('result')

    @_auto_initialize_user
    def handle(self, request, data=None, *args, **kwargs):
        context = kwargs.get('context')
        url = self.get_url(self.get_context(context=context))
        headers = {'Accept': 'application/json', 'Content-type': 'application/json'}
        resp = self.request_service(url, method=self.service_method, headers=headers, data=data)
        resp = self.serialize(resp)
        return resp

    def request_service(self, url, method=None, headers=None, data=None, **kwargs):
        uid_type = None
        if self.auth_user_required:
            uid_type, _ = self.get_typed_uid(self.request.user.uid)

        if not self.auth_user_required or uid_type in ('uid', 'device_id', 'yandexuid'):
            ret = super(DataApiProfileBaseHandler, self).request_service(url, method, headers, data)
            return self._get_result_from_data_api_response(ret)
        else:
            return self.get_empty_response()

    def get_platform_exception(self, exception):
        return super(ProfileBaseHandler, self).get_platform_exception(exception)

    def get_service_error_code(self, exception):
        data = getattr(exception, 'data', {})
        raw_data_api_resp = data.get('text', '{}')
        if raw_data_api_resp:
            try:
                data_api_resp = from_json(raw_data_api_resp)
            except (UnicodeDecodeError, ValueError):
                # если нам в ответ приехала фигня
                return None
            else:
                error = data_api_resp.get('error', {})
                return error.get('name', None)

    def handle_exception(self, exception):
        if isinstance(exception, CloudApiError):
            raise exception
        if isinstance(exception, self.service_base_exception):
            platform_exception = self.get_platform_exception(exception)
            if platform_exception:
                # re-raise exception preserving original stack trace and value
                raise platform_exception, None, sys.exc_info()[2]
        return super(ServiceProxyHandler, self).handle_exception(exception)

    def init_user(self):
        """Инициализирует пользователя Data API если тот ещё не проинициализирован"""
        user = self.request.user
        # Дёргаем Data API чтоб проинициализировал пользователя
        url = "%s%s" %(data_api.base_url, '/api/register_user?__uid=%s' % user.uid)
        data_api.open_url(url, method='PUT')
        self.request.user_initialized = True


def preprocess_time(obj):
    if 'time' in obj:
        obj['time'] = (obj['time'], obj['tzoffset'])
        del obj['tzoffset']


def preprocess_flight_time(obj):
    if 'departure' in obj:
        preprocess_time(obj['departure'])

    if 'arrival' in obj:
        preprocess_time(obj['arrival'])


class GetActualFlightsListHandler(DataApiProfileBaseHandler):
    """Получить список актуальных перелетов"""
    service_url = '/events/flights/actual/?__uid=%(uid)s'
    serializer_cls = FlightsListSerializer

    query = fields.QueryDict({
        'limit': fields.IntegerField(default=20, help_text=u'Количество выводимых ресурсов.'),
        'offset': fields.IntegerField(default=0, help_text=u'Смещение от начала списка ресурсов.'),
    })

    def serialize(self, obj, *args, **kwargs):
        offset = self.request.query['offset']
        limit = self.request.query['limit']
        obj['limit'] = limit
        obj['offset'] = offset
        items = obj.get('items', [])
        obj['total'] = len(items)
        obj['items'] = items[offset:offset+limit]

        for flight in obj['items']:
            preprocess_flight_time(flight)

        return super(GetActualFlightsListHandler, self).serialize(obj, *args, **kwargs)


class GetFlightHandler(DataApiProfileBaseHandler):
    """Получить один перелет"""
    service_url = '/events/flights/actual/%(event_id)s?__uid=%(uid)s'
    serializer_cls = FlightSerializer

    error_map = {
        'not-found': PersonalityEventNotFoundError
    }

    kwargs = fields.QueryDict({
        'event_id': fields.StringField(required=True, help_text=u'Идентификатор события.')
    })

    def serialize(self, obj, *args, **kwargs):
        preprocess_flight_time(obj)

        return super(GetFlightHandler, self).serialize(obj, *args, **kwargs)


class ChangeActualFlightHandlerBase(DataApiProfileBaseHandler):

    def preprocess_time(self, c):
        if 'time' in c and isinstance(c['time'], tuple):
            ts, tz_offset = c['time']
            c['time'] = ts
            c['tzoffset'] = tz_offset

    def preprocess_flight_times(self, flight):
        if 'departure' in flight:
            self.preprocess_time(flight['departure'])
        if 'arrival' in flight:
            self.preprocess_time(flight['arrival'])

    @_auto_initialize_user
    def handle(self, request, *args, **kwargs):
        self.preprocess_flight_times(self.request.body)
        return super(ChangeActualFlightHandlerBase, self).handle(request, data=self.request.body, *args, **kwargs)


class CreateActualFlightHandler(ChangeActualFlightHandlerBase):
    """Создать новый перелет"""
    service_method = 'POST'
    service_url = '/events/flights/actual/?__uid=%(uid)s'
    body_serializer_cls = FlightSerializer

    error_map = {
        'invalid-schema': BadRequestError
    }


class SaveActualFlightHandler(ChangeActualFlightHandlerBase):
    """Сохранить существующий или новый перелет"""
    service_method = 'PUT'
    service_url = '/events/flights/actual/%(event_id)s?__uid=%(uid)s'
    body_serializer_cls = FlightSerializer

    error_map = {
        'invalid-schema': BadRequestError
    }

    kwargs = fields.QueryDict({
        'event_id': fields.StringField(required=True, help_text=u'Идентификатор события.')
    })


class DeleteActualFlightHandler(DataApiProfileBaseHandler):
    """Удалить существующий перелет"""
    service_method = 'DELETE'
    service_url = '/events/flights/actual/%(event_id)s?__uid=%(uid)s'

    # invalid-delta нужно обрабатывать из-за внутреннего устройства data-api
    error_map = {
        'not-found': PersonalityEventNotFoundError,
        'invalid-delta': PersonalityEventNotFoundError
    }

    kwargs = fields.QueryDict({
        'event_id': fields.StringField(required=True, help_text=u'Идентификатор события.')
    })



class SaveOldAddressDataApiHandler(DataApiProfileBaseHandler):
    """Сохранить адрес пользователя"""
    service_url = '/%(data_type)s/%(data_key)s?__uid=%(uid)s'
    service_method = 'PUT'
    body_serializer_cls = DataApiAddressSerializer
    serializer_cls = LinkSerializer

    @staticmethod
    def get_additional_context(request, *args, **kwargs):
        data_key = request.body['address_id']
        context = {}
        if is_geo_point(data_key):
            context['data_key'] = data_key
            context['data_type'] = 'geopoints'
        else:
            context['data_key'] = data_key
            context['data_type'] = 'addresses'

        return context

    @_auto_initialize_user
    def handle(self, request, *args, **kwargs):
        context = self.get_additional_context(request, *args, **kwargs)
        address_id = context['data_key']
        self.request.kwargs['address_id'] = address_id

        super(SaveOldAddressDataApiHandler, self).handle(request, data=self.request.body, context=context, *args, **kwargs)
        params = {
            'data_source': 'morda',  # we don't need source more
            'data_key': address_id,
        }
        link = self.router.get_link(GetOldAddressDataApiHandler, params=params)
        return self.serialize(link)


class DeleteOldAddressDataApiHandler(DataApiProfileBaseHandler):
    """Удалить адрес"""
    service_url = '/%(data_type)s/%(data_key)s?__uid=%(uid)s'
    resp_status_code = 204
    service_method = 'DELETE'

    error_map = {
        'not-found': PersonalityAddressesNotFoundError,
        'invalid-delta': PersonalityAddressesNotFoundError
    }

    kwargs = fields.QueryDict({
        'data_key': fields.StringField(required=True, help_text=u'Идентификатор адреса.'),
    })

    @staticmethod
    def get_additional_context(request, *args, **kwargs):
        data_key = kwargs['data_key']
        context = {}
        if is_geo_point(data_key):
            context['data_type'] = 'geopoints'
        else:
            context['data_type'] = 'addresses'

        return context

    def handle(self, request, *args, **kwargs):
        context = self.get_additional_context(request, *args, **kwargs)
        return super(DeleteOldAddressDataApiHandler, self).handle(request, *args, context=context, **kwargs)


class GetOldAddressDataApiHandler(DataApiProfileBaseHandler):
    """Получить адрес"""

    service_url = '/%(data_type)s/%(data_key)s?__uid=%(uid)s'
    error_map = {
        'not-found': PersonalityAddressesNotFoundError,
        'invalid-delta': PersonalityAddressesNotFoundError
    }

    kwargs = fields.QueryDict({
        'data_key': fields.StringField(required=True, help_text=u'Идентификатор адреса.'),
    })

    @staticmethod
    def get_additional_context(request, *args, **kwargs):
        data_key = kwargs['data_key']
        context = {}
        if is_geo_point(data_key):
            context['data_type'] = 'geopoints'
            context['serializer_cls'] = DataApiGeopointSerializer
        else:
            context['data_type'] = 'addresses'
            context['serializer_cls'] = DataApiAddressSerializer

        return context

    def handle(self, request, *args, **kwargs):
        context = self.get_additional_context(request, *args, **kwargs)
        self.serializer_cls = context.pop('serializer_cls')
        return super(GetOldAddressDataApiHandler, self).handle(request, *args, context=context, **kwargs)


class ListOldAddressesDataApiHandler(DataApiProfileBaseHandler):
    """Получить список адресов"""
    service_url = '/batch?actions=addresses,geopoints&__uid=%(uid)s'
    serializer_cls = DataApiAddressListSerializer

    query = fields.QueryDict({
        'limit': fields.IntegerField(default=20, help_text=u'Количество выводимых ресурсов.'),
        'offset': fields.IntegerField(default=0, help_text=u'Смещение от начала списка ресурсов.'),
    })

    def serialize(self, obj, *args, **kwargs):
        res = {}
        offset = self.request.query['offset']
        limit = self.request.query['limit']
        res['limit'] = limit
        res['offset'] = offset
        addresses = []
        for result in obj:
            items = result.get('result').get('items', [])
            addresses.extend(items)

        # Hack to properly serialize geopoint ids
        for address in addresses:
            if "geopoint_id" in address:
                address["address_id"] = address["geopoint_id"]

        res['total'] = len(addresses)
        res['addresses'] = addresses[offset:offset+limit]
        return super(ListOldAddressesDataApiHandler, self).serialize(res, *args, **kwargs)


class GetYaTicketsOrdersListHandler(DataApiProfileBaseHandler):
    """Получить список заказов в Я.Билетах"""
    service_method = 'GET'
    service_url = '/ya-tickets/orders?__uid=%(uid)s'
    serializer_cls = YaTicketsOrdersListSerializer

    query = fields.QueryDict({
        'limit': fields.IntegerField(default=20, help_text=u'Количество выводимых ресурсов.'),
        'offset': fields.IntegerField(default=0, help_text=u'Смещение от начала списка ресурсов.'),
    })

    def request_service(self, url, method=None, headers=None, data=None, **kwargs):
        res = super(GetYaTicketsOrdersListHandler, self).request_service(url, method, headers, data, **kwargs)

        res['limit'] = self.request.query['limit']
        res['offset'] = self.request.query['offset']

        return res


class GetYaTicketsOrderHandler(DataApiProfileBaseHandler):
    """Получить заказ из Я.Билетов"""
    service_method = 'GET'
    service_url = '/ya-tickets/orders/%(order_id)s?__uid=%(uid)s'
    serializer_cls = YaTicketsOrderSerializer

    error_map = {
        'not-found': PersonalityYaTicketsOrderNotFoundError
    }

    kwargs = fields.QueryDict({
        'order_id': fields.StringField(required=True, help_text=u'Идентификатор заказа.')
    })


class SaveYaTicketsOrderHandler(DataApiProfileBaseHandler):
    """Сохранить данные про новый или существующий заказ из Я.Билетов"""
    service_method = 'PUT'
    service_url = '/ya-tickets/orders/%(order_id)s?__uid=%(uid)s'
    serializer_cls = YaTicketsOrderSerializer
    body_serializer_cls = YaTicketsOrderSerializer

    error_map = {
        'bad-request': BadRequestError
    }

    kwargs = fields.QueryDict({
        'order_id': fields.StringField(required=True, help_text=u'Идентификатор заказа.')
    })

    def handle(self, request, data=None, *args, **kwargs):
        return super(SaveYaTicketsOrderHandler, self).handle(request, data=self.request.body, *args, **kwargs)


class DeleteYaTicketsOrderHandler(DataApiProfileBaseHandler):
    """Удалить из хранилища данные про заказ в Я.Билетах"""
    service_method = 'DELETE'
    service_url = '/ya-tickets/orders/%(order_id)s?__uid=%(uid)s'
    resp_status_code = 204

    # invalid-delta нужно обрабатывать из-за внутреннего устройства data-api
    error_map = {
        'not-found': PersonalityYaTicketsOrderNotFoundError,
        'invalid-delta': PersonalityYaTicketsOrderNotFoundError
    }

    kwargs = fields.QueryDict({
        'order_id': fields.StringField(required=True, help_text=u'Идентификатор заказа.')
    })


class GenericObjectHandler(DataApiProfileBaseHandler):
    """Обработать запрос к типизованному объекту"""
    auth_methods = [PassportCookieAuth()]
    service_base_exception = DataApiBadResponse
    service = data_api
    service_method = 'POST'
    service_url = '/api/process_generic_data/?http_method=%(http_method)s&resource_path=%(resource_path)s&__uid=%(uid)s'
    resp_status_code = 200
    permissions = GenericObjectsPermission()

    kwargs = fields.QueryDict({
        'resource_path': fields.StringField(required=True, help_text=u'Путь до ресурса')
    })

    error_map = {
        'not-found': PersonalityObjectNotFoundError,
        'method-not-allowed': PersonalityMethodNotAllowedError,
        'invalid-schema': PersonalityObjectValidationError,
        'unknown-generic-object-type-name': PersonalityUnknownObjectTypeError,
        'outdated-change': PersonalityRewriteIfConditionError,
        'passport-user-not-found': PersonalityUserNotFoundError,
    }

    def handle(self, request, *args, **kwargs):
        # эта ручка не использует сериализатор, вместо этого пробрасываем запрос/ответ на сторону явы без изменений
        context = self.get_context()
        context['http_method'] = request.method
        url = self.get_url(context)
        # FIXME: Выглядит небезопасно, тк могут uid извне передать.
        url += '&' + urllib.urlencode(request.args)
        headers = {'Accept': 'application/json', 'Content-type': 'application/json'}
        resp = self.request_service(url, method=self.service_method, headers=headers, data=self.request.body)
        return resp

    def _clean_body(self, data):
        # ничего не делаем т.к будем пробрасывать данные до DataApi без какой-либо обработки
        return data


class GetNotificationsCountHandler(DataApiProfileBaseHandler):
    """Получить количество непросмотренных сообщений в колоколе"""
    auth_methods = [PassportCookieAuth()]
    service_method = 'GET'
    service = data_api
    service_url = '/v1/personality/profile/notifications/unread-count?__uid=%(uid)s'
    serializer_cls = NotificationsCountSerializer

    @staticmethod
    def _get_result_from_data_api_response(resp):
        return resp

    def request_service(self, url, method=None, headers=None, data=None, **kwargs):
        headers['X-Uid'] = self.request.user.uid
        return super(GetNotificationsCountHandler, self).request_service(url, method, headers, data, **kwargs)


class DocsSaveObjectHandler(DataApiProfileBaseHandler):
    auth_methods = [PassportCookieAuth()]
    service_base_exception = DataApiBadResponse
    body_serializer_cls = DocsObjectSerializer
    service = data_api
    service_method = 'POST'
    service_url = '/api/process_generic_data/?http_method=%(http_method)s&resource_path=%(resource_path)s&__uid=%(uid)s'
    resp_status_code = 200
    permissions = DocsObjectPermission() | WebDavPermission()

    kwargs = fields.QueryDict({
        'resource_path': fields.ChoiceField(required=True, choices=DOCS_DATABASES, help_text=u'Путь до ресурса')
    })

    error_map = {
        'not-found': PersonalityObjectNotFoundError,
        'method-not-allowed': PersonalityMethodNotAllowedError,
        'invalid-schema': PersonalityObjectValidationError,
        'unknown-generic-object-type-name': PersonalityUnknownObjectTypeError,
        'outdated-change': PersonalityRewriteIfConditionError,
        'passport-user-not-found': PersonalityUserNotFoundError,
    }

    def handle(self, request, *args, **kwargs):
        headers = {'Accept': 'application/json', 'Content-type': 'application/json'}
        context = self.get_context()
        body = self.request.body

        # Получаем список документов пользователя
        items = self._get_all_docs(context, headers)

        current_doc = self._get_doc(items, body)
        current_doc['ts'] = ctimestamp()

        # В зависимости от того есть ли уже документ в коллекции делаем PUT или POST с удалением старых записей
        upsert_context = deepcopy(context)
        if current_doc.get('id'):
            upsert_context['http_method'] = 'PUT'
            upsert_context['resource_path'] += '/%s' % current_doc['id']
        else:
            old_docs = self._get_old_docs(items)
            for doc in old_docs:
                delete_context = deepcopy(context)
                delete_context['http_method'] = 'DELETE'
                delete_context['resource_path'] += '/%s' % doc['id']
                url = self.get_url(delete_context)
                try:
                    self.request_service(url, method=self.service_method, headers=headers)
                except DataApiBadResponse:
                    continue
            upsert_context['http_method'] = 'POST'

        url = self.get_url(upsert_context)
        resp = self.request_service(url, method=self.service_method, headers=headers, data=current_doc)
        return resp

    @staticmethod
    def _get_doc(items, body):
        resource_id, office_online_sharing_url = body.get('resource_id'), body.get('office_online_sharing_url')
        for doc in items:
            if resource_id:
                if doc.get('resource_id') == resource_id:
                    return doc
            elif office_online_sharing_url:
                if doc.get('office_online_sharing_url') == office_online_sharing_url:
                    return doc
            else:
                raise PersonalityObjectValidationError

        return body

    def _get_all_docs(self, context, headers):
        list_context = deepcopy(context)
        list_context['http_method'] = 'GET'
        url = self.get_url(list_context)
        resp = self.request_service(url, method=self.service_method, headers=headers)
        return resp["items"]

    @staticmethod
    def _get_old_docs(items):
        """ Получаем самые старые документы, если общее кол-во в коллекции больше лимита DOCS_COLLECTION_SIZE """
        if len(items) < DOCS_COLLECTION_SIZE:
            return []
        sorted_items = sorted(items, key=lambda k: k.get('ts', 0))
        return sorted_items[:len(items) - DOCS_COLLECTION_SIZE + 1]
