# -*- coding: utf-8 -*-
import re

from mpfs.platform.permissions import ClientHasScopesPermission, DynamicScopesPermission


class ProfileReadPremission(ClientHasScopesPermission):
    """Доступ на чтение профиля пользователя."""
    scopes = ('cloud_api:profile.read',)


class ProfileWritePremission(ClientHasScopesPermission):
    """Доступ на запись профиля пользователя."""
    scopes = ('cloud_api:profile.write',)


class GenericObjectsPermission(DynamicScopesPermission):
    """
    Доступом к динамическим объектам API Персонализации.

    Проверяет наличие у клиентов скоупов вида "cloud_api.profile:generic.<resource_path>[.read|.write]".
    Для доступа к методам GET, HEAD, OPTIONS достаточно скоупа с суффиксом ".read".
    Для доступа ко всем остальным методам, необходим скоуп с суффиксом ".write".
    Для полного доступа необходимы оба скоупа.

    Например:
        Допустим у нас есть ресурс-коллекция "/v1/personality/profile/videosearch/likes"
        и ресурс-объект "/v1/personality/profile/videosearch/likes/{id}".

        Для readonly доступа к ресурам достаточно чтобы клиент имел скоуп
        "cloud_api.profile:generic.videosearch.likes.read".

        Для доступа на запись достаточно скоупа "cloud_api.profile:generic.videosearch.likes.write".

        Для полного доступа потребуются оба скоупа: "cloud_api.profile:generic.videosearch.likes.read"
        и "cloud_api.profile:generic.videosearch.likes.write".
    """
    REQUEST_OBJ_SCOPE_RX = re.compile(r'^/v\d/personality/profile/(?P<request_scope>.*)')
    SCOPE_PREFIX = 'cloud_api.profile:generic.'
    READONLY_HTTP_METHODS = {'GET', 'HEAD', 'OPTIONS'}

    def get_required_scopes(self, request):
        match = self.REQUEST_OBJ_SCOPE_RX.match(request.path)
        if not match:
            return []
        generic_obj_scope = match.groupdict().get('request_scope')
        generic_obj_scope = generic_obj_scope.strip('/').replace('/', '.')
        generic_obj_scope = '%s%s' % (self.SCOPE_PREFIX, generic_obj_scope)

        # Фильтруем клиентские скоупы в зависимости от HTTP-метода и наличия префикса скоупов динамических объектов.
        action_type = '.read' if request.method in self.READONLY_HTTP_METHODS else '.write'
        client_generic_obj_scopes = []
        for client_scope in request.client.scopes:
            if client_scope.startswith(self.SCOPE_PREFIX) and client_scope.endswith(action_type):
                client_generic_obj_scopes.append(client_scope)

        for client_scope in client_generic_obj_scopes:
            if generic_obj_scope.startswith(client_scope.rsplit('.', 1)[0]):
                # если у клиента есть необходимый для выплонения запроса скоуп, то возвращаем его
                return [client_scope]
        return []


class DocsObjectPermission(GenericObjectsPermission):
    REQUEST_OBJ_SCOPE_RX = re.compile(r'^/v\d/personality/profile/docs/(?P<request_scope>.*)')
