# -*- coding: utf-8 -*-
from collections import OrderedDict
from mpfs.platform import fields
from mpfs.platform.serializers import BaseSerializer, ListSerializer, FormatOptions


class DataApiAddressSerializer(BaseSerializer):
    visible_fields = ['data_key', 'title', 'tags', 'latitude', 'longitude', 'address_line',
                      'address_line_short', 'entrance_number', 'custom_metadata']

    fields = {
        'data_key': fields.StringField(required=True, source='address_id', pbid=1,
                                       help_text=u'Идентификатор адреса'),
        'title': fields.StringField(required=True, pbid=2, help_text=u'Название адреса'),
        'tags': fields.ListField(pbid=3, help_text=u'Список тэгов'),
        'latitude': fields.FloatField(pbid=4, help_text=u'Широта'),
        'longitude': fields.FloatField(pbid=5, help_text=u'Долгота'),
        'address_line': fields.StringField(pbid=6, help_text=u'Полная запись адреса'),
        'address_line_short': fields.StringField(pbid=7, help_text=u'Короткая запись адреса'),
        'entrance_number': fields.StringField(pbid=8, help_text=u'Номер подъезда'),
        'custom_metadata': fields.StringField(pbid=9, help_text=u'json с кастомными метаданными'),
    }


class DataApiAddressListSerializer(ListSerializer):
    visible_fields = ['total', 'limit', 'offset', 'items', '_links']
    fields = {
        'items': fields.SerializerField(DataApiAddressSerializer, required=True, many=True, source='addresses', pbid=1),
    }

    @property
    def handler_cls(self):
        from mpfs.platform.v1.personality.handlers import DataApiAddressListSerializer
        return DataApiAddressListSerializer


class DataApiGeopointSerializer(BaseSerializer):
    visible_fields = ['data_key', 'title', 'tags', 'latitude', 'longitude']

    fields = {
        'data_key': fields.StringField(required=True, source='geopoint_id', pbid=1,
                                       help_text=u'Идентификатор адреса'),
        'title': fields.StringField(required=True, pbid=2, help_text=u'Название адреса'),
        'tags': fields.ListField(pbid=3, help_text=u'Список тэгов'),
        'latitude': fields.FloatField(pbid=4, help_text=u'Широта'),
        'longitude': fields.FloatField(pbid=5, help_text=u'Долгота'),
    }


class RegionSerializer(BaseSerializer):
    visible_fields = ['geo_id', 'slug']

    fields = {
        'geo_id': fields.IntegerField(required=True, source='geo_id', pbid=1,
                                      help_text=u'Идентификатор региона в геобазе'),
        'slug': fields.StringField(required=False, source='slug', pbid=2,
                                   help_text=u'Тип региона. Например home'),
    }


class AirlineSerializer(BaseSerializer):
    visible_fields = ['iata_code', 'icao_code', 'sirena_code', 'name', 'ya_schedule_id']
    fields = {
        'iata_code': fields.StringField(required=False, source='iata_code', pbid=1, help_text=u'IATA код'),
        'icao_code': fields.StringField(required=False, source='icao_code', pbid=2, help_text=u'ICAO код'),
        'sirena_code': fields.StringField(required=False, source='sirena_code', pbid=3, help_text=u'Sirena код'),
        'name': fields.StringField(required=False, source='name', pbid=4, help_text=u'Название'),
        'ya_schedule_id': fields.StringField(required=False, source='ya_schedule_id', pbid=5, help_text=u'Идентификатор на Я.Расписаниях'),
    }


class AirportSerializer(BaseSerializer):
    visible_fields = ['iata_code', 'icao_code', 'sirena_code', 'name', 'ya_schedule_id']
    fields = {
        'iata_code': fields.StringField(required=False, source='iata_code', pbid=1, help_text=u'IATA код'),
        'icao_code': fields.StringField(required=False, source='icao_code', pbid=2, help_text=u'ICAO код'),
        'sirena_code': fields.StringField(required=False, source='sirena_code', pbid=3, help_text=u'Sirena код'),
        'name': fields.StringField(required=False, source='name', pbid=4, help_text=u'Название'),
        'ya_schedule_id': fields.StringField(required=False, source='ya_schedule_id', pbid=5, help_text=u'Идентификатор на Я.Расписаниях'),
    }


class FlightPlaceSerializer(BaseSerializer):
    visible_fields = ['time', 'geo_id', 'airport', 'city_name']
    fields = {
        'time': fields.DateTimeToTSWithTimezoneField(milliseconds=True, required=False, source='time', pbid=1, help_text=u'Время в локальной таймзоне'),
        'geo_id': fields.IntegerField(required=False, source='geo_id', pbid=2, help_text=u'Идертификатор в геобазе'),
        'airport': fields.SerializerField(required=False, serializer_cls=AirportSerializer, source='airport', pbid=3, help_text=u'Информация про аэропорт'),
        'city_name': fields.StringField(required=False, source='city_name', pbid=4, help_text=u'Название города в случае когда geo_id неизвестен'),
    }


class FlightSerializer(BaseSerializer):
    visible_fields = ['id', 'flight_number', 'checkin_url', 'airline', 'departure', 'arrival', 'data_source']

    fields = {
        'id': fields.StringField(required=False, source='id', pbid=1, help_text=u'Уникальный идентификатор'),
        'flight_number': fields.StringField(required=True, source='flight_number', pbid=2, help_text=u'Номер рейса'),
        'checkin_url': fields.StringField(required=False, source='checkin_url', pbid=3, help_text=u'Ссылка на электронную регистрацию'),
        'airline': fields.SerializerField(required=False, serializer_cls=AirlineSerializer, source='airline', pbid=4, help_text=u'Авиакомпания'),
        'departure': fields.SerializerField(required=True, serializer_cls=FlightPlaceSerializer, source='departure', pbid=5, help_text=u'Информация про вылет'),
        'arrival': fields.SerializerField(required=False, serializer_cls=FlightPlaceSerializer, source='arrival', pbid=6, help_text=u'Информация про посадку'),
        'data_source': fields.StringField(required=False, source='data_source', pbid=7, help_text=u'Идентификатор источника данных'),
    }

    def prepare_object(self, obj):
        if 'data_source' not in obj:
            # this hack will be removed after total migration
            # and field 'data_source' we be made required
            obj['data_source'] = 'mail'
        return obj


class FlightsListSerializer(ListSerializer):
    visible_fields = ['total', 'limit', 'offset', 'items', '_links', 'flags']
    fields = {
        'items': fields.SerializerField(FlightSerializer, required=True, many=True, source='items', pbid=1)
    }


class CoordinatesSerializer(BaseSerializer):
    visible_fields = ['latitude', 'longitude']

    fields = {
        'latitude': fields.FloatField(required=True, pbid=1),
        'longitude': fields.FloatField(required=True, pbid=2)
    }


class SubwayLineSerializer(BaseSerializer):
    visible_fields = ['hex_color']

    fields = {
        'hex_color': fields.StringField(required=True, pbid=1, help_text=u'hex-значение цвета линии'),
    }


class SubwayStationSerializer(BaseSerializer):
    visible_fields = ['title', 'lines']

    fields = {
        'title': fields.StringField(required=True, pbid=1, help_text=u'Название станции'),
        'lines': fields.SerializerField(SubwayLineSerializer, required=True, pbid=2, many=True, help_text=u'Список линий, проходящих через станцию'),
    }


class YaTicketsVenueSerializer(BaseSerializer):
    visible_fields = ['id', 'types', 'name', 'address', 'region_id', 'coordinates', 'smartpass', 'subway']

    fields = {
        'id': fields.IntegerField(required=True, pbid=1, help_text=u'Идентификатор'),
        'types': fields.ListField(required=True, pbid=2, help_text=u'Список типов мес'),
        'name': fields.StringField(required=True, pbid=3, help_text=u'Название'),
        'address': fields.StringField(required=True, pbid=4, help_text=u'Адрес площадки'),
        'region_id': fields.IntegerField(required=True, pbid=5, help_text=u'Идентификатор региона в геобазе, где находится кинотеатр'),
        'coordinates': fields.SerializerField(CoordinatesSerializer, required=True, pbid=6, help_text=u'Координаты площадки'),
        'smartpass': fields.BooleanField(required=True, pbid=7, help_text=u'Установлены ли в кинотеатре сканеры Smartpass'),
        'subway': fields.SerializerField(SubwayStationSerializer, required=False, pbid=8, many=True, help_text=u'Список ближайших станций метро'),
    }


class YaTicketsEventSerializer(BaseSerializer):
    visible_fields = ['id', 'type', 'name']

    fields = {
        'id': fields.IntegerField(required=True, pbid=1, help_text=u'Идентификатор'),
        'type': fields.StringField(required=False, pbid=2, help_text=u'Тип'),
        'name': fields.StringField(required=True, pbid=3, help_text=u'Название'),
    }


class YaTicketsHallSerializer(BaseSerializer):
    visible_fields = ['name']

    fields = {
        'name': fields.StringField(required=False, pbid=1, help_text=u'Название зала'),
    }


class YaTicketsSessionSerializer(BaseSerializer):
    visible_fields = ['id', 'date', 'previous_date', 'duration', 'event', 'venue', 'hall']

    fields = {
        'id': fields.IntegerField(required=True, pbid=1, help_text=u'Идентификатор'),
        'date': fields.StringField(required=True, pbid=2, help_text=u'Дата проведения'),
        'previous_date': fields.StringField(required=False, pbid=3, help_text=u'Предидущая дата проведения, если менялась'),
        'duration': fields.IntegerField(required=False, pbid=4, help_text=u'продолжительность сеанса в минутах'),
        'event': fields.SerializerField(YaTicketsEventSerializer, required=True, pbid=5, help_text=u'Описание мероприятия'),
        'venue': fields.SerializerField(YaTicketsVenueSerializer, required=True, pbid=6, help_text=u'Описание площадки проведения'),
        'hall': fields.SerializerField(YaTicketsHallSerializer, required=True, pbid=7, help_text=u'Описание зала'),
    }


class YaTicketsTicketSerializer(BaseSerializer):
    visible_fields = ['place', 'row', 'ticket_number', 'admission', 'level_name', 'category_name', 'entrance_name']

    fields = {
        'place': fields.StringField(required=False, pbid=1, help_text=u'Место в ряду'),
        'row': fields.StringField(required=False, pbid=2, help_text=u'Ряд'),
        'ticket_number': fields.StringField(required=False, pbid=3, help_text=u'Номер билета для системы контроля'),
        'admission': fields.BooleanField(required=False, pbid=4, default=False, help_text=u'Признак входного билета'),
        'level_name': fields.StringField(required=False, pbid=5, help_text=u'Название уровня'),
        'category_name': fields.StringField(required=False, pbid=6, help_text=u'Название категории'),
        'entrance_name': fields.StringField(required=False, pbid=7, help_text=u'Название подъезда'),
    }


class YaTicketsBarcodeSerializer(BaseSerializer):
    visible_fields = ['url', 'height', 'width']

    fields = {
        'url': fields.StringField(required=True, pbid=1, help_text=u'URL изображения'),
        'height': fields.IntegerField(required=True, pbid=2, help_text=u'Высота'),
        'width': fields.IntegerField(required=True, pbid=3, help_text=u'Ширина'),
    }


class YaTicketsOrderSerializer(BaseSerializer):
    visible_fields = ['order_id', 'session', 'ticket_count', 'order_number', 'presentation_order_number', 'code_word', 'barcode', 'pkpass_barcode', 'tickets']

    fields = {
        'order_id': fields.StringField(required=True, pbid=1, help_text=u'Идентификатор заказа'),
        'session': fields.SerializerField(YaTicketsSessionSerializer, required=True, pbid=2, help_text=u'Описание сеанса'),
        'ticket_count': fields.IntegerField(required=True, pbid=3, help_text=u'Количество билетов в заказе'),
        'order_number': fields.StringField(required=False, pbid=4, help_text=u'Номер заказа'),
        'presentation_order_number': fields.StringField(required=False, pbid=5, help_text=u'Номер заказа'),
        'code_word': fields.StringField(required=False, pbid=6, help_text=u'Кодовое слово'),
        'barcode': fields.SerializerField(YaTicketsBarcodeSerializer, required=False, pbid=7, help_text=u'Картинка со штрихкодом номера заказа'),
        'pkpass_barcode': fields.SerializerField(YaTicketsBarcodeSerializer, required=False, pbid=8, help_text=u'Картинка со штрихкодом ссылки на билет для Passbook'),
        'tickets': fields.SerializerField(YaTicketsTicketSerializer, required=False, pbid=9, many=True, help_text=u'Список билетов'),
    }


class YaTicketsOrdersListSerializer(ListSerializer):
    visible_fields = ['items', 'total', 'limit', 'offset']

    fields = {
        'items': fields.SerializerField(YaTicketsOrderSerializer, required=True, many=True, source='items', pbid=1),
        'total': fields.IntegerField(pbid=2, help_text=u'Общее количество элементов в списке'),
        'limit': fields.IntegerField(pbid=3, help_text=u'Количество элементов на странице'),
        'offset': fields.IntegerField(pbid=4, help_text=u'Смещение от начала списка'),
    }

    def prepare_object(self, obj):
        obj = super(YaTicketsOrdersListSerializer, self).prepare_object(obj)

        offset = obj.get('offset')
        limit = obj.get('limit')
        total = len(obj.get('items'))

        obj['total'] = total
        obj['items'] = obj['items'][offset:offset+limit]

        return obj


class NotificationsCountSerializer(BaseSerializer):
    visible_fields = ['count']

    fields = {
        'count': fields.IntegerField(required=True, pbid=1, help_text=u'Количество непросмотренных сообщений'),
    }


class DocsObjectSerializer(BaseSerializer):
    visible_fields = ['resource_id', 'office_online_sharing_url']

    fields = {
        'resource_id': fields.StringField(required=False, pbid=1, help_text=u'Id ресурса'),
        'office_online_sharing_url': fields.StringField(required=False, pbid=2, help_text=u'Ссылка на редактирование'),
    }
