# -*- coding: utf-8 -*-

from mpfs.common.errors import (
    PsBillingError,
)
from mpfs.core.services.ps_billing_service import ps_billing
from mpfs.platform.auth import PassportCookieAuth
from mpfs.platform.exceptions import (
    NotFoundError,
    ForbiddenError,
)
from mpfs.platform.handlers import RequestsPoweredServiceProxyHandler
from mpfs.platform.permissions import AllowByClientIdPermission
from mpfs.platform.v1.disk.permissions import WebDavPermission
from mpfs.platform.v1.psbilling.exceptions import (
    PsBillingBadRequestError, PsBillingInappReceiptIsInvalidError, PsBillingInappUidMismatchError,
    PsBillingInappApplicationMismatchError, PsBillingInappYandexAccountMismatchError,
    PsBillingInappStoreAccountMismatchError, PsBillingPromocodeAlreadyActivated,
)
from mpfs.platform.v1.psbilling.permissions import PsBillingInappPermission
from mpfs.config import settings

PLATFORM_PS_APPS_IDS = settings.platform['ps_apps_ids']


class PsBillingAPIProxyHandler(RequestsPoweredServiceProxyHandler):
    service = ps_billing
    permissions = AllowByClientIdPermission(PLATFORM_PS_APPS_IDS) | WebDavPermission() | PsBillingInappPermission()
    auth_methods = [PassportCookieAuth()]
    service_base_exception = PsBillingError
    hidden = True

    error_map = {
        'invalid-receipt': PsBillingInappReceiptIsInvalidError,
        'uid-mismatch': PsBillingInappUidMismatchError,
        'multiuser_store': PsBillingInappStoreAccountMismatchError,
        'wrong_app': PsBillingInappApplicationMismatchError,
        'multiuser_yandex': PsBillingInappYandexAccountMismatchError,
        'bad-request': PsBillingBadRequestError,
        'promocode-already-activated': PsBillingPromocodeAlreadyActivated,
        '400': PsBillingBadRequestError,
        '403': ForbiddenError,
        '404': NotFoundError
    }

    def get_url(self, context=None):
        url = super(PsBillingAPIProxyHandler, self).get_url(context=context)

        lang = self.request.language
        if '-' in lang:
            lang = lang.split('-')[0]
        if lang:
            url = self.patch_url_params(url, {'lang': lang.split('-')[0]})

        return url

    def get_service_error_code(self, exception):
        if not isinstance(exception, PsBillingError) or not hasattr(exception, 'code'):
            return None
        return exception.code
