# -*- coding: utf-8 -*-

from mpfs.platform import fields
from mpfs.platform.rate_limiters import PerUserRateLimiter
from mpfs.platform.v1.psbilling.handlers import PsBillingAPIProxyHandler
from mpfs.platform.v1.psbilling.permissions import PsBillingReadPermission
from mpfs.platform.v1.psbilling.users.serializers import GetUserServicesSerializer, GetProductsForUserSerializer, \
    GetSubscriptionsStatusesForPassportSerializer, SubscribeSerializer


class GetUserServicesHandler(PsBillingAPIProxyHandler):
    serializer_cls = GetUserServicesSerializer
    service_method = 'GET'
    service_url = '/v1/users/services?__uid=%(uid)s'
    resp_status_code = 200

    rate_limiter = PerUserRateLimiter('cloud_api_user_psbilling_user_services')

    query = fields.QueryDict({
        'product_owner': fields.StringField(required=False,
                                            help_text=u'Идентификатор владельца продуктов, услуги которых интересуют.'),
        'status': fields.StringField(required=False,
                                     help_text=u'Статус сервиса которых интересуют.'),
    })

    def get_url(self, context=None):
        url = super(GetUserServicesHandler, self).get_url(context=context)
        params = {}

        product_owner = self.request.query.get('product_owner', None)
        if product_owner:
            params['product_owner'] = product_owner

        status = self.request.query.get('status', None)
        if status:
            params['status'] = status

        return self.patch_url_params(url, params)


class GetProductsForUserHandler(PsBillingAPIProxyHandler):
    serializer_cls = GetProductsForUserSerializer
    service_method = 'GET'
    service_url = '/v2/productsets/%(key)s/products?__uid=%(uid)s'
    resp_status_code = 200

    kwargs = fields.QueryDict({
        'key': fields.StringField(required=True, help_text=u'Ключ продукт сета.')
    })

    query = fields.QueryDict({
        'currency': fields.StringField(required=False,
                                       help_text=u'Валюта, в которой будут отображаться цены на продукты.'),
    })

    rate_limiter = PerUserRateLimiter('cloud_api_user_psbilling_user_products')

    def get_url(self, context=None):
        url = super(GetProductsForUserHandler, self).get_url(context=context)
        params = {}

        currency = self.request.query.get('currency', None)
        if currency:
            params['currency'] = currency

        return self.patch_url_params(url, params)


class GetSubscriptionsStatusesForPassportHandler(PsBillingAPIProxyHandler):
    serializer_cls = GetSubscriptionsStatusesForPassportSerializer
    service_method = 'GET'
    service_url = '/v1/users/passport/subscriptions_statuses?__uid=%(uid)s'
    resp_status_code = 200
    permissions = PsBillingReadPermission()
    rate_limiter = PerUserRateLimiter('cloud_api_user_psbilling_user_passport')


class SubscribeHandler(PsBillingAPIProxyHandler):
    serializer_cls = SubscribeSerializer
    service_method = 'POST'
    service_url = '/v1/users/subscribe?__uid=%(uid)s&price_id=%(price_id)s'
    resp_status_code = 200

    rate_limiter = PerUserRateLimiter('cloud_api_user_psbilling_user_services')

    query = fields.QueryDict({
        'price_id': fields.StringField(required=True,
                                       help_text=u'Идентификатор продукта.'),
        'currency': fields.StringField(required=False,
                                       help_text=u'Валюта.'),
        'domain_suffix': fields.StringField(required=False,
                                            help_text=u'Суффикс домена.'),
        'return_path': fields.StringField(required=False,
                                          help_text=u'URL для возврата после оплаты заказа.'),
        'form_type': fields.ChoiceField(required=False, choices=['mobile', 'desktop'],
                                        help_text=u'Тип формы.'),
        'real_user_ip': fields.StringField(required=False,
                                           help_text=u'Ip пользователя.'),
        'disable_trust_header': fields.BooleanField(required=False, default=None,
                                                    help_text=u'Скрываем шапку в форме.'),
        'login_id': fields.StringField(required=False,
                                      help_text=u'Идентификатор логина.'),
        'use_template': fields.BooleanField(required=False, default=None,
                                                    help_text=u'Использовать дефолтный шаблон в форме оплаты.'),
    })

    def get_url(self, context=None):
        url = super(SubscribeHandler, self).get_url(context=context)
        params = {}

        currency = self.request.query.get('currency', None)
        if currency:
            params['currency'] = currency

        return_path = self.request.query.get('return_path', None)
        if return_path:
            params['return_path'] = return_path

        domain_suffix = self.request.query.get('domain_suffix', None)
        if domain_suffix:
            params['domain_suffix'] = domain_suffix

        form_type = self.request.query.get('form_type', None)
        if form_type:
            params['form_type'] = form_type

        real_user_ip = self.request.query.get('real_user_ip', None)
        if real_user_ip:
            params['real_user_ip'] = real_user_ip

        disable_trust_header = self.request.query.get('disable_trust_header', None)
        if disable_trust_header:
            params['disable_trust_header'] = str(disable_trust_header).lower()

        login_id = self.request.query.get('login_id', None)
        if login_id:
            params['login_id'] = login_id

        use_template = self.request.query.get('use_template', None)
        if use_template:
            params['use_template'] = str(use_template).lower()

        return self.patch_url_params(url, params)
