# -*- coding: utf-8 -*-
from mpfs.platform.formatters import PlainTextFormatter
import re
import mpfs
from mpfs.common.static import tags
from mpfs.platform.exceptions import NotFoundError
from mpfs.platform import fields
from mpfs.platform.handlers import BasePlatformHandler
from mpfs.platform.permissions import AllowAllPermission
from mpfs.platform.rate_limiters import PerHandlerRateLimiter
from mpfs.platform.resources import BaseResource
from mpfs.platform.v1.schema.serializers import (ResourceListingSerializer, ApiDeclarationSerializer,
                                                 ProtoPackageSerializer)


class BaseSchemaHandler(BasePlatformHandler):
    auth_required = False
    rate_limiter = PerHandlerRateLimiter('cloud_api_schema')

    def get_api_root_resource(self):
        return self.router.root_resource

    def get_parent_resources(self):
        if self.request.mode == tags.platform.INTERNAL:
            return self.parent.resources + getattr(self.parent, 'internal_resources', [])
        else:
            return self.parent.resources

    def get_root_resources(self, api=None):
        root = self.get_api_root_resource()
        assert isinstance(root, BaseResource)
        subresources = root.get_subresources()
        resources = []
        for p in self.get_parent_resources():
            resource_candidates = []
            for r in subresources:
                if re.match(p, r.path):
                    resource_candidates.append(r)
            if resource_candidates:
                resource = min(resource_candidates, key=lambda r: len(r.path))
                resources.append(resource)
        return resources


class GetSchemaIndexHandler(BaseSchemaHandler):
    serializer_cls = ResourceListingSerializer
    query = fields.QueryDict({
        'api': fields.StringField(help_text=u'API для которого требуется сгенерировать схему.'),
    })
    permissions = AllowAllPermission()

    def handle(self, request, *args, **kwargs):
        api = request.query.get('api', None)
        resources = self.get_root_resources(api=api)
        return self.serialize(resources, api=api)


class GetSchemaApiHandler(BaseSchemaHandler):
    serializer_cls = ApiDeclarationSerializer
    kwargs = fields.QueryDict({
        'path': fields.StringField(help_text=u'Path ресурса API для которого требуется сгенерировать схему.'),
    })
    permissions = AllowAllPermission()

    def handle(self, request, *args, **kwargs):
        path = request.kwargs.get('path', None)
        path = '/%s' % path

        for p in self.get_parent_resources():
            if re.match(p, path):
                for resolver in self.router.resolvers:
                    match = resolver.resolve(path)
                    if match:
                        return self.serialize(match.handler, pattern=p)
        raise NotFoundError()


class ProtobufHandler(BaseSchemaHandler):
    permissions = AllowAllPermission()

    def get_content_types(self):
        return {
            'text/plain': PlainTextFormatter()
        }

    def get_proto_header(self, package=None):
        api = self.request.query.get('api')
        path = self.request.kwargs.get('path')
        params = {}
        if api:
            params['api'] = api
        if path:
            params['path'] = path
        context = {
            'source': self.router.get_link(self, params=params, rfc6570=True)[1],
            'version': mpfs.__version__,
            'java_package': '.'.join(filter(None, ['net.yandex.cloud_api', package])),
            'package': '.'.join(filter(None, ['cloud_api', package])),
        }
        ret = ('// Protocol Buffers for cloud-api.yandex.net\n'
               '// Source: %(source)s\n'
               '// Version: %(version)s\n'
               'package %(package)s;\n'
               'option java_package = "%(java_package)s";\n\n') % context
        return ret

    def get_proto_packages(self, api=None):
        """
        Возвращает список API для которых активирован protobuf.

        :param api: Фильтр по API
        :rtype: list
        :return: ['v1/<api-1>', 'v1/api-2', ...]
        """
        packages = ['v1/data']
        # resources = self.get_root_resources(api=api)
        # packages = []
        # for r in resources:
        #     p = '%s' % '/'.join(r.path.strip('/').split('/', 2)[:2])
        #     if p not in packages:
        #         packages.append(p)

        return packages


class GetProtoIndexHandler(ProtobufHandler):
    """Получить список .proto-файлов."""
    query = fields.QueryDict({
        'api': fields.StringField(help_text=u'API для которого требуется сгенерировать список.'),
    })

    def handle(self, request, *args, **kwargs):
        api = request.query.get('api', None)
        packages = self.get_proto_packages(api=api)
        ret = self.get_proto_header()
        for p in packages:
            ret += 'import "%s.proto";\n' % (p,)
        return ret


class GetProtoFileHandler(ProtobufHandler):
    """Получить .proto-файл"""
    kwargs = fields.QueryDict({
        'path': fields.StringField(help_text=u'Путь .proto-файла.')
    })

    def handle(self, request, *args, **kwargs):
        path = request.kwargs.get('path')
        if path in self.get_proto_packages():
            pkg = path.replace('/', '.')
            ret = self.get_proto_header(package=pkg)

            # resources = filter(lambda r: r.path.startswith('/%s' % (path,)), self.get_root_resources())
            # patterns = self.get_parent_resources()
            resources = filter(None, self.get_api_root_resource().get_subresources(depth=3))
            patterns = ['^/v1/data/{context}/databases.*']

            schema = ProtoPackageSerializer(resources, patterns=patterns, router=self.router,
                                            lang=self.request.language).data
            models = sorted(schema['models'], key=lambda m: m['id'])
            for model in models:
                ret += 'message %s {\n' % (model['id'],)
                props = [(k, v) for k, v in model['properties'].iteritems()]
                props = sorted(props, key=lambda p: p[1]['pbid'])
                for name, p in props:
                    if p['pbid'] is not None:
                        ref = p.get('$ref')
                        enum = p.get('enum')
                        if ref:
                          type = ref
                        elif enum:
                            indexes = p['pbenum']
                            enum_name = ''.join(map(lambda s: s.capitalize(), name.split('_')))
                            ret += '    enum %s {\n' % (enum_name,)
                            for i in range(len(enum)):
                                ret += '        %s = %s;\n' % (enum[i].upper(), indexes[i])
                            ret += '    }\n'
                            type = enum_name
                        else:
                            type = p.get('pbtype', p.get('type'))

                        if p.get('type') == 'array':
                            ret += '    repeated '
                            ret += '%s' % (p['items'].get('$ref') or p['items'].get('type'))
                        else:
                            if name in model['required']:
                                ret += '    required '
                            else:
                                ret += '    optional '
                            ret += '%s' % type

                        ret += ' %s = %s;  // %s\n' % (name, p['pbid'], p['description'])
                ret += '}\n\n'
            return ret
        raise NotFoundError()
