# -*- coding: utf-8 -*-
"""
Сериализаторы для генерации схемы API.

Все сериализаторы здесь унаследованы от 2х базовых классов SwaggerSerializer и SchemaSerializer.

Такое разделение сделано так, как JSON Schema является более универсальным стандартом нежели Swagger.
Swagger использует JSON Schema для описания моделей и данных, но сама по себе JSON Schema может быть использована и
без Swagger'а. Поэтому для сериализаторов генерирующих данные специфичные для Swagger'а используется один базовый класс,
а для сериализаторов генерирующих схему не зависящую от Swagger'а -- другой.
"""
from inspect import isclass
import urlparse
import re

from mpfs.common.static import tags

try:
    from markdown import markdown
except Exception:
    def markdown(text, *args, **kwargs):
        """Stub for markdown"""
        return text

from mpfs.common.util import normalize_unicode
from mpfs.platform.fields import (ParentMethodField, ParentAttrField, SerializerField, StringField, BaseField,
                                  BooleanField, IntegerField, FloatField, DateTimeField, ListField, ChoiceField,
                                  Base64ToBinaryField, JsonObjectField, IfNoneMatchField)
from mpfs.platform.handlers import BasePlatformHandler, ServiceProxyHandler
from mpfs.platform.serializers import BaseSerializer
from mpfs.core.services.tanker_service import TankerHelper
from mpfs.platform.utils import model_name_for_serializer, trim_docstring, split_docstring
from mpfs.platform.common import HTTP_VERBS


I18N_PROJECT_ID = 'tech'
I18N_KEYSET_ID = 'desktop.blocks:swagger_for_disk'


class SchemaSerializer(BaseSerializer):
    tanker_helper = TankerHelper(I18N_PROJECT_ID, I18N_KEYSET_ID, 'ru')

    def __init__(self, *args, **kwargs):
        self.lang = kwargs.get('lang', 'ru')
        self.tanker_helper.lang = self.lang
        super(SchemaSerializer, self).__init__(*args, **kwargs)

    def get_fields(self):
        for f in self.fields.values():
            if isinstance(f, SerializerField):
                f.init.update({'lang': self.lang})
        return super(SchemaSerializer, self).get_fields()

    def is_internal(self):
        """
        Проверяет для внутреннего ли API строится схема.

        Алгоритм:
        1. Проверяет что у сериализатора есть роутер (при запроса хэндлер его туда прописывает).
        2. Если роутера нет, значит что-то происходит не тсандартное и для пущей безопасности считаем,
            что рендерим схему для внешнего API.
        3. Проверяем что у роутера есть запрос (в нормальной ситуации его туда прописывает диспатчер).
        4. Если запроса в роутере нет, то что-то идёт не по плану и для безопасности считаем,
            что рендерим схему для внешнего API.
        5. Проверяем, что запрос вызвавший рендеринг сделан во внутреннее API.
        """
        # в люблой непонятной ситуации считаем, что схема строится для внешнего API
        router = self.router
        if not router:
            return False
        request = router.request
        if not request:
            return False
        return request.mode == tags.platform.INTERNAL


class SwaggerSerializer(SchemaSerializer):
    """
    Базовый класс для сериализаторов генерирующих Swagger-совместимую структуру API.

    Подробнее см.: https://github.com/wordnik/swagger-core/wiki
    """
    swagger_version = '1.2'

    fields = dict(
        swaggerVersion=ParentAttrField('swagger_version', help_text=u"Совместимая версия Swagger'а."),
        basePath=ParentMethodField('get_base_path', help_text=u'Путь по которому доступны русурсы API.'),
        apiVersion=ParentMethodField('get_api_version', help_text=u'Текущая версия API.'),
    )

    def get_base_path(self, val):
        raise NotImplemented()

    def get_api_version(self, val):
        raise NotImplemented()


class JsonSchemaSerializer(SchemaSerializer):
    """
    Базовый класс для сериализаторов генерирующих JSON Schema.

    Используется для получения JSON Schema ресурсов нашего API. И попутно для описания моделей для Swagger.

    По сути наследники этого класса сериализуют сериализаторы и филдсеты.

    Сам по себе класс содержит базовые утилиты для преобразования сериализаторов и филдов в правильную схему.

    Филды преобразуются в примитивы:
        https://github.com/wordnik/swagger-core/wiki/datatypes#wiki-primitives
    С помощью сериализаторов можно описывать сложные (complex) типы данных:
        https://github.com/wordnik/swagger-core/wiki/datatypes#wiki-complex-types
    """
    FIELD_TO_PRIMITIVE_MAP = {  # {field: (<type>, <format>), ...}
        StringField: ('string', ''),
        ChoiceField: ('string', ''),
        BooleanField: ('boolean', ''),
        IntegerField: ('integer', 'int64'),
        FloatField: ('number', ''),
        DateTimeField: ('string', 'date-time'),
        JsonObjectField: ('object', ''),
        IfNoneMatchField: ('string', ''),
    }

    FIELD_TO_PROTOBUF_MAP = {
        StringField: ('string',),
        ChoiceField: ('string',),
        BooleanField: ('bool',),
        IntegerField: ('int64',),
        FloatField: ('double',),
        DateTimeField: ('uint64',),
        Base64ToBinaryField: ('bytes',),
        IfNoneMatchField: ('string',),
    }

    def get_rformat(self, val):
        """
        Возвращает формат данных, возвращаемых сериализатором или филдом, пригодный для вставки в поле format схемы.

        :param val: Сериализатор, экземпляр сериализатора или экземпляр филда.
        :return: Имя типа данных. None если не удалось найти ничего подходящего.
        :rtype: str | None
        """
        ret = self._get_rprimitive(val)
        return ret[1] if ret else ''

    def get_rtype(self, val, map=FIELD_TO_PRIMITIVE_MAP):
        """
        Возвращает тип данных, возвращаемых сериализатором или филдом, пригодный для вставки в поле type схемы.

        :param val: Сериализатор, экземпляр сериализатора или экземпляр филда.
        :return: Имя типа данных. None если не удалось найти ничего подходящего.
        :rtype: str | None
        """
        ret = self._get_rprimitive(val, map=map)
        return ret[0] if ret else ''

    def _get_serializer_rtype(self, serializer_cls):
        return model_name_for_serializer(serializer_cls)

    def _get_rprimitive(self, val, map=FIELD_TO_PRIMITIVE_MAP):
        # Если передали филд.
        if isinstance(val, BaseField):
            # Для SerializerField выдёргиваем инкапсулированный сериализатор и возвращаем соответствующий результат.
            if isinstance(val, SerializerField):
                val = val.serializer_cls
                return self._get_serializer_rtype(val), ''

            if isinstance(val, (ParentMethodField, ParentAttrField)):
                val = getattr(val, 'field_type', None)
                if val and isclass(val):
                    val = val()
            elif isinstance(val, ListField):
                val = getattr(val, 'item_field', None)

            # Сначала пытаемся найти непосредственно соответствие поля и примитива.
            ret = map.get(type(val), None)
            # Если прямого соответствия нет, то пытаемся найти соответствия по родительским классам.
            if not ret:
                for c, t in map.iteritems():
                    if isinstance(val, c):
                        ret = map.get(c, None)
                        break
            return ret

        # Если передали сериализатор.
        else:
            if isinstance(val, BaseSerializer):
                val = type(val)

            if isclass(val) and issubclass(val, BaseSerializer):
                # Имя сложного типа представленного сериализатором получается путём ампутации суффикса "Serializer".
                return self._get_serializer_rtype(val), None

        # Если ничего не удалось подобрать.
        return ''


class BaseFieldSerializer(JsonSchemaSerializer):
    MD_PARAGRAPH_RE = re.compile(r'^<p>(.*)</p>$')

    def get_description(self, val):
        ret = self.tanker_helper.get_key_for_object(self.object, 'title') or markdown(getattr(self.object, 'help_text', ''))
        return self.MD_PARAGRAPH_RE.sub(r'\1', ret)

    def get_enum(self, val):
        if isinstance(self.object, ChoiceField):
            return self.object.choices.keys() if isinstance(self.object.choices, dict) else self.object.choices

    def get_datatype(self, val):
        return self.get_rtype(self.object)

    def get_format(self, val):
        return self.get_rformat(self.object)


class ApiParameterSerializer(BaseFieldSerializer):
    visible_fields = ['paramType', 'name', 'description', 'type', 'format', 'required', 'enum']

    fields = dict(
        paramType=ParentMethodField('get_param_type', field_type=StringField,
                                  help_text=u'Тип параметра: path, query, body, header или form.'),
        required=ParentMethodField('get_required', field_type=BooleanField),
        name=ParentMethodField('get_name', field_type=StringField, help_text=u'Имя параметра'),
        description=ParentMethodField('get_description', field_type=StringField),
        type=ParentMethodField('get_datatype', field_type=StringField),
        format=ParentMethodField('get_format', field_type=StringField),
        enum=ParentMethodField('get_enum', field_type=ListField),
    )

    def get_required(self, val):
        if not isclass(self.object):
            query_dict = self.object.parent
            handler = query_dict.parent
            if getattr(handler, 'kwargs', None) is query_dict:
                return True  # все path параметры обязательны не зависимо от того что там в них написано
        return val

    def get_param_type(self, val):
        # в случае body значение параметра val будет установлено и ни какой магии, по крайней мере здесь, не потребуется
        if not val:
            query_dict = self.object.parent
            handler = query_dict.parent
            for handler_attr, param_type in {'query': 'query', 'headers': 'header', 'kwargs': 'path'}.iteritems():
                if query_dict is getattr(handler, handler_attr, None):
                    return param_type
        return val

    def get_name(self, val):
        return val

    def prepare_object(self, obj):
        if isclass(obj) and issubclass(obj, BaseSerializer):
            return {
                'paramType': 'body',
                'name': 'body',
                'type': self.get_datatype(self._object),
                'required': True,
                'allowMultiple': False,
            }
        return super(ApiParameterSerializer, self).prepare_object(obj)


class PermissionSerializer(SwaggerSerializer):
    """Сериализует экземпляр класса ClientHasScopesPermission в Swagger представление."""
    visible_fields = ['scope', 'description']
    fields = {
        'scope': ParentMethodField('get_scope', source='scopes', field_type=StringField),
        'description': ParentMethodField('get_description', source='__doc__', field_type=StringField),
    }

    def get_description(self, val):
        return self.tanker_helper.get_key_for_object(self.object, 'title') or val

    def get_scope(self, val):
        return val[0] if len(val) > 0 else None


class AuthorizationSerializer(SwaggerSerializer):
    """Сериализует доступные способы авторизации."""
    visible_fields = ['oauth2']
    fields = {
        'oauth2': SerializerField(PermissionSerializer, many=True)
    }


class ResponseObjectSerializer(JsonSchemaSerializer):
    """Сериализует исключения API в Response Message"""
    TEMPLATED_MESSAGE_ATTR_RE = re.compile(r'%\((\w+)\)[sbcdoxXneEfFgGn%]')
    visible_fields = ['code', 'message', 'responseModel']
    fields = {
        'code': IntegerField(required=True, source='status_code'),
        'message': ParentMethodField('get_reason', required=True, source='reason', field_type=StringField),
        'responseModel': ParentMethodField('get_rtype', source='response_cls', field_type=StringField),
    }

    def get_reason(self, val):
        def match_param(match):
            param_name = match.group(1)
            return '{%s}' % param_name
        return self.TEMPLATED_MESSAGE_ATTR_RE.sub(match_param, val)


class ApiHandlerSerializer(JsonSchemaSerializer):
    """
    Сериализует объект хэндлера API в swagger представление.
    """
    visible_fields = ['method', 'nickname', 'type', 'format', 'summary', 'notes', 'parameters', 'consumes',
                      'authorizations', 'responseMessages', 'produces', 'consumes']

    fields = dict(
        method=StringField(help_text=u'HTTP метод'),
        nickname=ParentMethodField('get_nickname', field_type=StringField),
        type=ParentMethodField('get_rtype', source='serializer_cls', field_type=StringField),
        format=ParentMethodField('get_rformat', source='serializer_cls', field_type=StringField),
        summary=StringField(),
        notes=StringField(),
        parameters=SerializerField(ApiParameterSerializer, many=True),
        responseMessages=SerializerField(ResponseObjectSerializer, many=True),
        produces=ParentMethodField('get_produces', help_text=u'Поддерживаемые MIME-типы.'),
        consumes=ParentMethodField('get_consumes', help_text=u'Поддерживаемые MIME-типы.'),
        # Отпиливаем пока OAuth скоупы из схемы, ибо после поддержки папок приложения не понятно как их выводить.
        # authorizations=SerializerField(AuthorizationSerializer),
    )

    def prepare_object(self, obj):
        from mpfs.platform.routers import URLPattern
        data = super(ApiHandlerSerializer, self).prepare_object(obj)

        doc = obj.__doc__
        doc = trim_docstring(doc) if isinstance(doc, (str, unicode)) else ''
        if doc:
            summary, notes = split_docstring(doc)
            data['summary'] = self.tanker_helper.get_key_for_object(obj, 'title') or summary
            data['notes'] = self.tanker_helper.get_key_for_object(obj, 'description') or markdown(normalize_unicode(notes))

        if getattr(obj, 'body_serializer_cls', None):
            data['consumes'] = ['application/json']

        # Отпиливаем пока OAuth скоупы из схемы, ибо после поддержки папок приложения не понятно как их выводить.
        # permissions = [p for p in getattr(obj, 'permissions', []) if issubclass(p, ClientHasScopesPermission)]
        # if permissions:
        #     data['authorizations'] = {'oauth2': permissions}

        data['parameters'] = []
        if not isinstance(obj, BasePlatformHandler):
            return data
        else:
            data['parameters'] += obj.query.get_fields().values()
            data['parameters'] += obj.headers.get_fields().values()

            # фильтруем параметры которые есть и в URL и поддерживаются хэндлером
            url_params = URLPattern.get_parameters(obj.pattern)
            kwargs_fields = obj.kwargs.get_fields()
            data['parameters'] += [kwargs_fields.get(p) for p in url_params if p in kwargs_fields]

            # фильтруем параметры
            data['parameters'] = filter(lambda p: (not p.hidden or self.is_internal()), data['parameters'])
            # сортируем параметры
            data['parameters'] = sorted(data['parameters'], key=lambda p: (not p.required, p.name))

            # добавляем body параметр елси надо
            body_serializer_cls = getattr(obj, 'body_serializer_cls', None)
            if body_serializer_cls:
                data['parameters'].append(body_serializer_cls)

            response_objects = data.get('get_response_objects', lambda *args, **kwargs: [])()
            # фильтруем уникальные ответы по reason
            seen = set()
            response_objects = [o for o in response_objects if not o.reason or not (o.reason in seen or seen.add(o.reason))]
            if response_objects:
                data['responseMessages'] = sorted(response_objects, key=lambda o: o.status_code)

            # получаем представлние хэндлера
            return data

    def get_nickname(self, val):
        nickname = type(self._object).__name__
        return re.sub(r'Handler$', '', nickname)

    def get_produces(self, val):
        obj = self.object
        assert isinstance(obj, BasePlatformHandler)
        return obj.get_content_types().keys()

    def get_consumes(self, value):
        obj = self.object
        assert isinstance(obj, BasePlatformHandler)
        return [ct for ct in obj.get_content_types().keys() if ct != 'application/hal+json']


class ApiResourceSerializer(SwaggerSerializer):
    """
    Сериализует ресурс API в swagger.

    Хитрый сериализатор, принимающий в качестве сериализуемого объекта не традиционный объект,
    а строку - идентифицирующую ресурс API, т.е. path. По этому самому path сериализатор добывает из роутера
    все связанные с ним хэндлеры и формирует правильное swagger представление ресурса.
    """
    visible_fields = ['path', 'description', 'operations']

    fields = dict(
        path=StringField(help_text=u'Путь до ресурса.'),
        description=StringField(default=''),  # Пока всегда пустое поле ибо у нас нет ни какой документации по ресурсам.
        operations=SerializerField(ApiHandlerSerializer, many=True, help_text=u'Список поддерживаемых операций.'),
    )

    def _from_native(self, obj):
        handlers = [resource
                    for relation, resource in obj.relations.iteritems()
                    if relation in HTTP_VERBS and (not resource.hidden or self.is_internal())]
        if not handlers:
            return {}
        link = self.router.get_link(handlers[0], rfc6570=True)
        parsed_url = urlparse.urlparse(link[1])
        data = {'path': parsed_url.path, 'operations': handlers}
        return super(ApiResourceSerializer, self)._from_native(data)


class SchemaPropertySerializer(BaseFieldSerializer):
    """Сериализует филд сериализатора в правильную схему."""
    visible_fields = ['$ref', 'type', 'items', 'format', 'enum', 'description']

    def prepare_object(self, obj):
        if isinstance(obj, SerializerField):
            ret = {'$ref': self.get_rtype(obj.serializer_cls)}
            if obj.many:
                ret = {'type': 'array', 'items': ret}
        elif isinstance(obj, ListField):
            ret = {'type': 'array', 'items': {'type': self.get_rtype(obj)}}

        else:
            ret = {}

            rtype = self.get_rtype(obj)
            if rtype:
                ret['type'] = rtype

            rformat = self.get_rformat(obj)
            if rformat:
                ret['format'] = rformat
            if isinstance(obj, ChoiceField):
                ret['enum'] = self.get_enum(obj)
        ret['description'] = self.get_description(None)
        return ret


class SerializerSerializer(JsonSchemaSerializer):
    """Сериализует в JSON Schema переданный сериализатор."""
    visible_fields = ['id', 'required', 'properties']
    fields = {
        'id': ParentMethodField('get_id', required=True),
        'required': ParentMethodField('get_required', required=True),
    }
    property_serializer_cls = SchemaPropertySerializer

    def get_id(self, val):
        return self.get_rtype(self.object)

    def get_required(self, val):
        return [name for name, f in self.object.fields.iteritems() if f.required]

    def prepare_object(self, obj):
        obj_dict = super(SerializerSerializer, self).prepare_object(obj)
        properties = {}
        visible_fields = obj().get_visible_fields()
        excluded_fields = obj().get_excluded_fields_of_request()
        fields = [(k, v) for k, v in obj.fields.iteritems() if k in visible_fields and k not in excluded_fields]
        for name, field in fields:
            properties[name] = self.property_serializer_cls(field, router=self.router, hal=self.hal, lang=self.lang).data
        obj_dict['properties'] = properties
        return obj_dict


class ModelsSerializer(JsonSchemaSerializer):
    """
    Сериализует переданный список сериализаторов в JSON Schema совместимую со swagger.

    Хитрый сериализатор, который принимает в качестве объекта список сериализаторов
    и сам генерит на лету состав своих филдов.
    """
    def _from_native(self, obj):
        ret = {}
        for serializer in obj:
            name = self.get_rtype(serializer)
            ret[name] = SerializerSerializer(serializer, router=self.router, hal=self.hal, lang=self.lang).data
        return ret


class ApiModuleSerializer(SwaggerSerializer):
    def get_serializers(self, resource, pattern):
        # Достаём хэндлеры, чтоб достать из них сериализаторы.
        patterns = reduce(lambda ret, r: ret + r.get_patterns([resource.path]), resource.url_resolvers, [])
        patterns.sort(key=lambda p: p.expand())
        handlers = [p.handler for p in patterns if re.match(pattern, p.handler.path)]
        # Теперь для каждого хэндлера достаём сериализатор и достаём вложенные сериализаторы.
        serializers = []
        for h in handlers:
            root_serializers = [h.serializer_cls, h.body_serializer_cls]
            if isinstance(h, ServiceProxyHandler):
                root_serializers += [e.serializer_cls for e in h.error_map.values()]
            for serializer_cls in filter(None, root_serializers):
                if serializer_cls:
                    serializers.append(serializer_cls)
                    serializers += serializer_cls.get_subserializers()
        # Фильтруем None
        serializers = filter(None, serializers)
        # Фильтруем дубликаты
        serializers = set(serializers)
        return serializers


class ApiDeclarationSerializer(ApiModuleSerializer):
    """
    Создаёт swagger представление модуля API.

    Подробнее см.: https://github.com/wordnik/swagger-core/wiki/API-Declaration
    """
    visible_fields = ['apiVersion', 'swaggerVersion', 'basePath', 'description', 'apis', 'resourcePath', 'models',
                      'produces', 'consumes']
    fields = {
        'resourcePath': StringField(source='path', help_text=u'Путь по которому доступны русурсы API.'),
        'description': ParentMethodField(method_name='get_description', source='__doc__', field_type=StringField,
                                         help_text=u'Описание API.'),
        'produces': ParentMethodField('get_produces', help_text=u'Поддерживаемые MIME-типы.'),
        'consumes': ParentMethodField('get_consumes', help_text=u'Поддерживаемые MIME-типы.'),
        'apis': SerializerField(ApiResourceSerializer, many=True, help_text=u'Список ресурсов API.'),
        'models': SerializerField(ModelsSerializer),
    }

    def __init__(self, obj=None, pattern=None, *args, **kwargs):
        """
        :param pattern: Задаёт фильтр хэндлеров которые следует выводить.
        """
        super(ApiDeclarationSerializer, self).__init__(obj, *args, **kwargs)
        self.pattern = pattern

    def _from_native(self, obj):
        """
        :param obj: Список ресурсов
        :rtype: dict
        """
        data = self.get_dict(obj)
        apis = filter(lambda r: set(r.relations.keys()) & set(HTTP_VERBS),
                      sorted([obj] + obj.get_subresources(hidden=self.is_internal()), key=lambda r: r.path))

        apis = filter(lambda r: (not r.hidden or self.is_internal()), apis)
        if self.pattern is not None:
            apis = filter(lambda r: re.match(self.pattern, r.path), apis)
        data['apis'] = apis

        serializers = self.get_serializers(obj, pattern=self.pattern)
        data['models'] = serializers
        return super(ApiDeclarationSerializer, self)._from_native(data)

    def get_description(self, val):
        return self.tanker_helper.get_key_for_object(self.object, 'title') or val

    def get_api_version(self, val):
        return self.object.path.strip('/').split('/')[0]

    def get_base_path(self, val):
        return self.router.base_url

    def get_produces(self, value):
        return BasePlatformHandler.DEFAULT_CONTENT_TYPES.keys()

    def get_consumes(self, value):
        return [ct for ct in BasePlatformHandler.DEFAULT_CONTENT_TYPES.keys() if ct != 'application/hal+json']


class ResourceListingItemSerializer(SwaggerSerializer):
    """Создаёт swagger представление русурса для списка ресурсов."""
    visible_fields = ['path', 'description']

    def __init__(self, obj=None, api=None, *args, **kwargs):
        super(ResourceListingItemSerializer, self).__init__(obj, *args, **kwargs)
        self.api = api

    fields = dict(
        path=StringField(help_text=u'Путь по которому доступны API.'),
        description=ParentMethodField(method_name='get_description', source='__doc__', field_type=StringField,
                                      help_text=u'Описание API.'),
    )

    def get_description(self, val):
        return self.tanker_helper.get_key_for_object(self.object, 'title') or val


class ResourceListingSerializer(SwaggerSerializer):
    """
    Создаёт swagger представление списка доступных API.

    Подробнее см.: https://github.com/wordnik/swagger-core/wiki/Resource-Listing
    """
    visible_fields = ['apiVersion', 'swaggerVersion', 'basePath', 'apis']
    fields = {
        'apis': SerializerField(ResourceListingItemSerializer, many=True, help_text=u'Список поддерживаемых API.'),
    }

    def __init__(self, obj=None, api=None, *args, **kwargs):
        super(ResourceListingSerializer, self).__init__(obj, *args, **kwargs)
        self.api = api

    def get_api_version(self, val):
        return 'v1'

    def get_base_path(self, val):
        from mpfs.platform.v1.schema.handlers import GetSchemaIndexHandler
        _, url, _ = self.router.get_link(GetSchemaIndexHandler, rfc6570=True)
        url_chunks = list(urlparse.urlsplit(url))
        url_chunks[2] = '%s/resources' % url_chunks[2]
        url = urlparse.urlunsplit(url_chunks)
        return url

    def _from_native(self, obj):
        """
        :param obj: Список корневых ресурсов (нэймспэйсов) API.
        :rtype: dict
        """
        if self.api:
            prefix = '/%s/%s' % (self.get_api_version(None), self.api)
            obj = [r for r in obj if r.path.startswith(prefix)]
        # пробрасываем во вложенный сеиализатор значение параметра api
        self.get_fields()['apis'].init.update({'api': self.api})
        data = {'apis': obj}
        return super(ResourceListingSerializer, self)._from_native(data)


class ProtoPropertySerializer(SchemaPropertySerializer):
    def get_visible_fields(self):
        return super(ProtoPropertySerializer, self).get_visible_fields() + ['pbid', 'pbtype', 'pbenum']

    def prepare_object(self, obj):
        ret = super(ProtoPropertySerializer, self).prepare_object(obj)
        ret['pbid'] = obj.pbid
        ret['pbtype'] = self.get_rtype(obj, map=self.FIELD_TO_PROTOBUF_MAP)
        if isinstance(obj, ChoiceField):
            ret['pbenum'] = obj.get_num_choices()
        return ret


class ProtoSerializerSerializer(SerializerSerializer):
    property_serializer_cls = ProtoPropertySerializer


class ProtoPackageSerializer(ApiModuleSerializer):
    visible_fields = ['models']
    fields = {
        'models': SerializerField(ProtoSerializerSerializer, many=True)
    }

    def __init__(self, obj=None, patterns=None, *args, **kwargs):
        """
        :param pattern: Задаёт фильтр хэндлеров которые следует выводить.
        """
        super(ProtoPackageSerializer, self).__init__(obj, *args, **kwargs)
        self.patterns = patterns

    def prepare_object(self, obj):
        serializers = []
        for r in obj:
            serializers += reduce(lambda ret, p: ret + list(self.get_serializers(r, p)), self.patterns, [])
        # избавляемся от дублей
        serializers = sorted(set(serializers), key=lambda s: type(s).__name__)
        data = {'models': serializers}
        return data
