# -*- coding: utf-8 -*-
import hashlib
import urllib
import re
from collections import OrderedDict

from uuid import uuid4

from mpfs.common.errors import (
    ClckNoResponse,
    TelemostApiNotFound,
    TelemostApiForbidden,
    TelemostApiBadRequest,
    TelemostApiGone,
    UAASRequestInvalidClient
)
from mpfs.common.static.codes import (
    CLIENT_BAD_REQUEST,
)
from mpfs.common.util import ctimestamp, to_json, from_json, filter_uid_by_percentage
from mpfs.common.util.crypt import AesCbcCryptAgent
from mpfs.common.util.user_agent_parser import UserAgentParser
from mpfs.config import settings
from mpfs.core.services.clck_service import clck
from mpfs.core.services.passport_service import passport, passport_api, PassportSID
from mpfs.core.services.telemost_api_service import TelemostViaTCMService
from mpfs.core.services.uaas_service import uaas
from mpfs.engine.process import get_default_log, get_error_log
from mpfs.platform import fields
from mpfs.platform.auth import (
    PassportCookieAuth,
    TVM2Auth,
    TVM2TicketsOnlyAuth,
    InternalAuth,
    InternalTokenAuth,
    InternalConductorAuth,
    YaTeamCookieAuth,
    YaTeamOAuthAuth,
)
from mpfs.platform.common import logger, PlatformResponse
from mpfs.platform.exceptions import ForbiddenError, PayloadTooLargeError, BadRequestError
from mpfs.platform.formatters import PlainTextFormatter
from mpfs.platform.handlers import BasePlatformHandler
from mpfs.platform.permissions import AllowAllPermission
from mpfs.platform.rate_limiters import PerUserRateLimiter, PerUserWithSeparateAnonymousRateLimiter
from mpfs.platform.utils.uid_formatters import is_yateam_uid
from mpfs.platform.utils.user_agent_detector import UserAgentDetector, UserAgentDetails
from mpfs.platform.v1.disk.exceptions import DiskExperimentsUnsupportedDevice
from mpfs.platform.v1.disk.handlers import (
    GetClientsInstallerHandler,
    MpfsProxyHandler,
    GetClientsInstallerWithAutologonHandler,
)
from mpfs.platform.v1.disk.permissions import WebDavPermission, MobileMailPermission
from mpfs.platform.v1.disk import exceptions
from mpfs.platform.v1.telemost.exceptions import (
    TelemostPasswordTooLongError,
    TelemostClientBadRequest,
    TelemostConferenceNotFoundError,
    TelemostPasswordTypeError,
    TelemostInviteLinkExpiredError,
    BadBodyError,
)
from mpfs.platform.v1.telemost.permissions import (
    TelemostPermission,
)
from mpfs.platform.v1.telemost.serializers import (
    ConferenceConnectionSerializer,
    ConferenceSerializer,
    CreateConferenceBodySerializer,
    FOSBodySerializer,
    GetConferenceConnectionBodySerializer,
    StatReportBodySerializer,
    AuthorizationURISerializer,
    ExperimentsSerializer,
)
from mpfs.platform.v2.telemost.handlers import TelemostAPIProxyHandler


PLATFORM_TELEMOST_STAFF_ONLY = settings.platform['telemost']['staff_only']
PLATFORM_TELEMOST_ENABLE_FOR_UIDS = settings.platform['telemost']['enable_for_uids']
PLATFORM_TELEMOST_ENABLE_STAFF_ONLY_CONFERENCES_FOR_UIDS = settings.platform['telemost']['enable_staff_only_conferences_for_uids']
PLATFORM_TELEMOST_DEFAULT_DISPLAY_NAME = settings.platform['telemost']['default_display_name']
PLATFORM_TELEMOST_CONF_SERVER = settings.platform['telemost']['conf_server']
PLATFORM_TELEMOST_UNLIMITED_UIDS = settings.platform['telemost']['xmpp_unlimited_uids']
PLATFORM_TELEMOST_XMPP_DOMAIN = settings.platform['telemost']['xmpp_domain']
PLATFORM_TELEMOST_XMPP_MUC = settings.platform['telemost']['xmpp_muc']
PLATFORM_TELEMOST_INVITE_EXPIRE_SECONDS = settings.platform['telemost']['invite_expire_seconds']
PLATFORM_TELEMOST_YANDEX_TEAM_PASSPORT_AUTH_LINK_TMPL = settings.platform['telemost']['yandex_team_passport_auth_link_tmpl']

PLATFORM_TELEMOST_DISABLE_ACCESS_TO_CREATE_FOR_UIDS_PERCENT = settings.platform['telemost']['disable_access_to_create_for_uids_percent']

PLATFORM_TELEMOST_XMPP_USERS = settings.platform['telemost']['xmpp']['users']
PLATFORM_TELEMOST_API_ENABLED = settings.platform['telemost']['telemost-api']['enabled']

TELEMOST_SOFTWARE_INSTALLER_PATH = settings.platform['software_installer']['telemost_path']
PLATFORM_SOFTWARE_INSTALLER_WITH_AUTO_LOGON_TELEMOST_PATH = settings.platform['software_installer']['with_autologon']['telemost_path']


default_log = get_default_log()
error_log = get_error_log()


class XMPPLimitTypes(object):
    DEFAULT = 'default'
    STAFF = 'staff'
    UNLIM = 'unlim'


class ConferenceKey(object):
    def __init__(self, conf_id=None, conf_pwd=None, staff_only=False, limit_type=XMPPLimitTypes.DEFAULT, timestamp=None, salt=None):
        self.conf_id = conf_id or self.generate_conf_id()
        self.conf_pwd = conf_pwd or self.generate_conf_pwd()
        self.staff_only = staff_only
        self.limit_type = limit_type
        self.timestamp = timestamp or ctimestamp()
        self.salt = salt or uuid4().hex

    @classmethod
    def generate(cls, staff_only=False, limit_type=XMPPLimitTypes.DEFAULT):
        return cls(staff_only=staff_only, limit_type=limit_type)

    @classmethod
    def from_str(cls, raw_conf_key):
        data = from_json(raw_conf_key)

        conf_id = data.get('conf_id')
        conf_pwd = data.get('conf_pwd')
        staff_only = data.get('staff_only')
        # Поддержка старых ссылок с безлимитным доступом
        # раньше был unlimited параметр, если он True, то безлимитный доступ
        old_limit_param = data.get('unlimited')
        default_limit = XMPPLimitTypes.DEFAULT
        if old_limit_param:
            default_limit = XMPPLimitTypes.UNLIM

        limit_type = data.get('limit_type', default_limit)
        timestamp = data.get('timestamp')
        salt = data.get('salt')
        if (any(item is None
                for item in (conf_id, conf_pwd, staff_only, timestamp, salt))):
            raise TelemostConferenceNotFoundError('Bad raw conf_key %s' % raw_conf_key)

        return cls(conf_id, conf_pwd, staff_only=staff_only, limit_type=limit_type, timestamp=timestamp, salt=salt)

    def serialize(self):
        return to_json({'conf_id': self.conf_id,
                        'conf_pwd': self.conf_pwd,
                        'staff_only': self.staff_only,
                        'limit_type': self.limit_type,
                        'timestamp': self.timestamp,
                        'salt': self.salt})

    @staticmethod
    def generate_conf_id():
        return uuid4().hex

    @staticmethod
    def generate_conf_pwd():
        return uuid4().hex


class ConferenceURIParser(object):
    CONF_JOIN_URL_TMPL = settings.platform['telemost']['conf_join_url_tmpl']
    CONF_JOIN_URL_RE = re.compile(settings.platform['telemost']['conf_join_url_re'])
    CONF_JOIN_SHORT_URL_RE = re.compile(settings.platform['telemost']['conf_join_short_url_re'])

    @classmethod
    def parse(cls, conf_uri):
        conf_uri = urllib.unquote(conf_uri)
        if not cls.CONF_JOIN_SHORT_URL_RE.match(conf_uri):
            raise TelemostConferenceNotFoundError('Bad conference short URI: %s' % conf_uri)

        conf_uri = clck.short_url_to_full_url(conf_uri)

        match = cls.CONF_JOIN_URL_RE.match(conf_uri)
        if match:
            conf_key = match.group('conf_key')
            return urllib.unquote(conf_key)

        raise TelemostConferenceNotFoundError('Bad conference URI: %s' % conf_uri)

    @classmethod
    def build(cls, encrypted_conf_key):
        link = cls.CONF_JOIN_URL_TMPL % urllib.quote(encrypted_conf_key, safe='')
        _, short_link = clck.generate(link, mode='telemost-join')
        return short_link


class ConferencesCryptAgent(object):
    USER_SECRET_MAX_LENGHT = 16
    SECRET_KEY = settings.platform['telemost']['secret_key']

    @classmethod
    def get_user_secret(cls, password):
        if not isinstance(password, basestring):
            raise TelemostPasswordTypeError(type=type(password))

        if len(password) > cls.USER_SECRET_MAX_LENGHT:
            raise TelemostPasswordTooLongError(max_length=cls.USER_SECRET_MAX_LENGHT)

        return password + (cls.USER_SECRET_MAX_LENGHT - len(password)) * '\0'

    @classmethod
    def encrypt_conf_key(cls, conf_key, password=''):
        user_secret = cls.get_user_secret(password)
        aes_cbc = AesCbcCryptAgent(user_secret + cls.SECRET_KEY, urlsafe=True)

        return aes_cbc.encrypt(conf_key)

    @classmethod
    def decrypt_conf_key(cls, encrypted_conf_key, password=''):
        user_secret = cls.get_user_secret(password)
        aes_cbc = AesCbcCryptAgent(user_secret + cls.SECRET_KEY, urlsafe=True)
        try:
            conf_key = aes_cbc.decrypt(encrypted_conf_key)
        except Exception:
            logger.error_log.exception('Failed to decrypt: %s' % encrypted_conf_key)
            raise TelemostConferenceNotFoundError()
        return conf_key


class BaseConferenceHandler(BasePlatformHandler):
    hidden = True

    def _get_display_name(self, user):
        if not user:
            return PLATFORM_TELEMOST_DEFAULT_DISPLAY_NAME

        if is_yateam_uid(user.uid):
            # для yt-oauth базовую инфу о пользователе спрашиваем при проверке токена
            # поэтому можем сразу получить имя
            if user.details and user.details.get('firstname'):
                return user.details['firstname']
            return PLATFORM_TELEMOST_DEFAULT_DISPLAY_NAME

        user_info = passport.userinfo(user.uid)
        if 'firstname' in user_info and user_info['firstname'] is not None:
            return user_info['firstname']
        if 'public_name' in user_info and user_info['public_name'] is not None:
            return user_info['public_name']

        return PLATFORM_TELEMOST_DEFAULT_DISPLAY_NAME

    def _get_xmpp_user_id(self, limit_type):
        if limit_type == XMPPLimitTypes.UNLIM:
            return PLATFORM_TELEMOST_XMPP_USERS[XMPPLimitTypes.UNLIM]['user_id']
        elif limit_type == XMPPLimitTypes.STAFF:
            return PLATFORM_TELEMOST_XMPP_USERS[XMPPLimitTypes.STAFF]['user_id']

        return PLATFORM_TELEMOST_XMPP_USERS[XMPPLimitTypes.DEFAULT]['user_id']

    def _get_xmpp_password(self, limit_type):
        if limit_type == XMPPLimitTypes.UNLIM:
            return PLATFORM_TELEMOST_XMPP_USERS[XMPPLimitTypes.UNLIM]['password']
        elif limit_type == XMPPLimitTypes.STAFF:
            return PLATFORM_TELEMOST_XMPP_USERS[XMPPLimitTypes.STAFF]['password']

        return PLATFORM_TELEMOST_XMPP_USERS[XMPPLimitTypes.DEFAULT]['password']

    def log_details(self, event_type, conf_id, conf_uri):
        try:
            hashed_url = hashlib.sha256(conf_uri).hexdigest()
        except Exception:
            error_log.exception('failed to hash conf_uri')
            hashed_url = '-'
        details = {'event_type': event_type,
                   'user-agent': self.request.raw_headers.get('user-agent', '-'),
                   'uid': '-',
                   'hashed_url': hashed_url,
                   'conf_id': conf_id}
        if self.request.user:
            details['uid'] = self.request.user.uid

        default_log.info(u'event_type: %(event_type)s; uid: %(uid)s; user-agent: %(user-agent)s; conf_id: %(conf_id)s; '
                         u'hashed_url: sha256-%(hashed_url)s' % details)

    def check_sid_and_subscribe(self, user):
        if not user:
            return
        if is_yateam_uid(user.uid):
            return
        try:
            user_info = passport.userinfo(user.uid)
            if (user_info.get('has_telemost') or
                    user_info.get('is_mailish')):
                return

            result = passport_api.subscribe(user.uid, sid=PassportSID.TELEMOST)
            if result['status'] != 'ok':
                error_log.info('failed to subscribe on sid=telemost: %s' % result)
        except Exception:
            error_log.exception('failed to subscribe on sid=telemost')


class CreateConferenceHandler(BaseConferenceHandler):
    permissions = WebDavPermission() | TelemostPermission() | MobileMailPermission()
    auth_methods = [PassportCookieAuth(), TVM2Auth(allow_yateam=True), TVM2TicketsOnlyAuth(allow_yateam=True)]
    serializer_cls = ConferenceSerializer
    body_serializer_cls = CreateConferenceBodySerializer
    resp_status_code = 201

    rate_limiter = PerUserRateLimiter('cloud_api_user_telemost_create')

    def check_permissions(self, request):
        super(CreateConferenceHandler, self).check_permissions(request)
        if (request.user and
                request.user.uid in PLATFORM_TELEMOST_ENABLE_FOR_UIDS):
            return
        if (request.user and
                filter_uid_by_percentage(request.user.uid, PLATFORM_TELEMOST_DISABLE_ACCESS_TO_CREATE_FOR_UIDS_PERCENT)):
            self.permission_denied(request)

    def handle(self, request, *args, **kwargs):
        staff_only = request.body.get('staff_only')
        password = request.body.get('password', '')
        is_permanent = request.body.get('is_persistent')
        external_meeting = None
        if request.user and is_yateam_uid(request.user.uid):
            # создаем вечную встречу (telemost.yandex.ru ссылка)
            is_permanent = True
            external_meeting = True
            if staff_only:
                # сбрасываем настройки на обычную встречу для yt:<uid> (telemost.yandex-team.ru ссылка)
                is_permanent = None
                external_meeting = False
                staff_only = False

        conf_key_data = self._get_conference_data_provider(
            staff_only, request.user, is_permanent, password, external_meeting=external_meeting
        ).generate()

        conf_id = conf_key_data['conf_id']
        conf_uri = conf_key_data['uri']

        self.log_details(event_type='create', conf_id=conf_id, conf_uri=conf_uri)

        self.check_sid_and_subscribe(request.user)

        limit_type = conf_key_data['limit_type']

        result = {
            'uri': conf_uri,
            'conf_server': PLATFORM_TELEMOST_CONF_SERVER,
            'conf_id': conf_id,
            'conf_pwd': conf_key_data['conf_pwd'],
            'display_name': self._get_display_name(request.user),
            'xmpp_user_id': self._get_xmpp_user_id(limit_type),
            'xmpp_password': self._get_xmpp_password(limit_type),
            'xmpp_domain': PLATFORM_TELEMOST_XMPP_DOMAIN,
            'xmpp_muc': PLATFORM_TELEMOST_XMPP_MUC
        }

        if conf_key_data.get('client_configuration'):
            result.update({'client_configuration': conf_key_data['client_configuration']})

        return self.serialize(result)

    def _get_conference_data_provider(self, staff_only, user=None, is_permanent=False, password=None,
                                      external_meeting=None):
        if PLATFORM_TELEMOST_API_ENABLED:
            return TelemostApiConferenceKeyGenerator(staff_only, user, is_permanent, external_meeting=external_meeting)
        return ClckConferenceKeyGenerator(staff_only, user, password)


class GetConferenceConnectionHandler(BaseConferenceHandler):
    permissions = AllowAllPermission()
    auth_methods = [PassportCookieAuth(), YaTeamOAuthAuth()]
    serializer_cls = ConferenceConnectionSerializer
    body_serializer_cls = GetConferenceConnectionBodySerializer
    auth_user_required = False
    auth_required = False

    rate_limiter = PerUserWithSeparateAnonymousRateLimiter('cloud_api_user_telemost_join',
                                                           anonymous_group='cloud_api_user_telemost')

    kwargs = fields.QueryDict({
        'uri': fields.StringField(required=True,
                                  help_text=u'URI конференции, для которой запрашиваем данные соединения.')
    })

    def handle(self, request, *args, **kwargs):
        conf_uri = request.kwargs['uri']
        # обрабатываем символы, которые вставляет в ссылки Яндекс.Заметки
        conf_uri = urllib.unquote(conf_uri).rstrip(u' \t\xe2\x80\x8b')
        password = request.body.get('password', '')

        conf_data = self._get_conference_data(self._get_conference_providers(conf_uri, request.user, password))

        conf_id = conf_data['conf_id']
        self.log_details(event_type='join', conf_id=conf_id, conf_uri=conf_uri)

        self.check_sid_and_subscribe(request.user)

        limit_type = conf_data['limit_type']

        result = {
            'uri': conf_data['uri'],
            'conf_server': PLATFORM_TELEMOST_CONF_SERVER,
            'conf_id': conf_id,
            'conf_pwd': conf_data['conf_pwd'],
            'display_name': self._get_display_name(request.user),
            'xmpp_user_id': self._get_xmpp_user_id(limit_type),
            'xmpp_password': self._get_xmpp_password(limit_type),
            'xmpp_domain': PLATFORM_TELEMOST_XMPP_DOMAIN,
            'xmpp_muc': PLATFORM_TELEMOST_XMPP_MUC
        }

        if conf_data.get('client_configuration'):
            result.update({'client_configuration': conf_data['client_configuration']})

        return self.serialize(result)

    def _get_conference_providers(self, uri, user, password):
        if PLATFORM_TELEMOST_API_ENABLED:
            return [TelemostApiConferenceKeyProvider(uri, user), ClckConferenceKeyProvider(uri, password, user)]
        return [ClckConferenceKeyProvider(uri, password, user), TelemostApiConferenceKeyProvider(uri, user)]

    def _get_conference_data(self, providers):
        for provider in providers:
            conf_data = provider.get_conference()
            if conf_data:
                return conf_data
        raise TelemostConferenceNotFoundError()


class GetConferenceAuthorizationURIHandler(TelemostAPIProxyHandler):
    permissions = AllowAllPermission()
    service_method = 'POST'
    service_url = '/v1/yandex-team/conferences/%(uri)s/authorized-uri'
    default_auth_methods = [InternalTokenAuth(), InternalConductorAuth(), InternalAuth()]
    auth_methods = [YaTeamCookieAuth(), YaTeamOAuthAuth()]
    serializer_cls = AuthorizationURISerializer
    resp_status_code = 200
    auth_user_required = False
    auth_required = False

    rate_limiter = PerUserWithSeparateAnonymousRateLimiter('cloud_api_user_telemost_join',
                                                           anonymous_group='cloud_api_user_telemost')

    query = fields.QueryDict({
        'return_redirect': fields.BooleanField(default=False, help_text=u'true, если нужно ответить с переадресацией.'),
    })

    kwargs = fields.QueryDict({
        'uri': fields.StringField(required=True,
                                  help_text=u'yandex-team URI конференции, для которой запрашиваем данные соединения.')
    })

    error_map = {
        TelemostApiNotFound: TelemostConferenceNotFoundError,
        TelemostApiGone: TelemostInviteLinkExpiredError,
    }

    def handle(self, request, *args, **kwargs):
        context = self.get_context()
        url = self.get_url(context)

        user_agent_details = UserAgentDetector.parse_from_headers(dict(request.raw_headers))

        if (context['return_redirect'] and
                not request.user and
                not (user_agent_details.get_browser_name() != UserAgentDetails.UNKNOWN and
                         user_agent_details.is_mobile())):
            passport_link = PLATFORM_TELEMOST_YANDEX_TEAM_PASSPORT_AUTH_LINK_TMPL % urllib.quote(context['uri'], safe='')
            return PlatformResponse(status_code=302, content='', headers={'Location': passport_link})

        uid = None
        if self.request.user and self.request.user.uid:
            uid = self.request.user.uid
        telemost_service = TelemostViaTCMService.get_telemost_api(uid, uri=context['uri'])
        _, resp_body, _ = self.raw_request_service(url, service=telemost_service)

        result = self.serialize(from_json(resp_body))

        if context['return_redirect']:
            return PlatformResponse(status_code=302, content='', headers={'Location': result['uri']})

        return result

    def get_url(self, context=None, base_url=None):
        url = super(GetConferenceAuthorizationURIHandler, self).get_url(context)

        if self.request.user and self.request.user.uid:
            url += '?uid=%s' % self.request.user.uid

        return url


class LogStatisticsHandler(BasePlatformHandler):
    hidden = True
    permissions = AllowAllPermission()
    auth_methods = [PassportCookieAuth()]
    body_serializer_cls = StatReportBodySerializer
    auth_user_required = False
    auth_required = False

    rate_limiter = PerUserWithSeparateAnonymousRateLimiter('cloud_api_user_telemost_stats',
                                                           anonymous_group='cloud_api_user_telemost_stats_anonymous')

    def handle(self, request, *args, **kwargs):
        if not request.body:
            raise BadBodyError(details='empty body')

        report = request.body.get('stats_report', {})
        default_log.info(to_json(report))


class BandwidthBenchmarkHandler(TelemostAPIProxyHandler):
    hidden = True
    permissions = AllowAllPermission()
    auth_methods = [PassportCookieAuth()]
    body_serializer_cls = StatReportBodySerializer
    auth_user_required = False
    auth_required = False
    query = fields.QueryDict({
        'response_size': fields.IntegerRangeField(
            required=False,
            default=10,
            bottom_border=0,
            upper_border=1024*1024,
            help_text=u'Желаемый Размер ответа. '
            u'Нужен для измерение пропускной способности '
            u'downstream канала по времени скачивания')
    })
    content_types = dict([
        ('application/x-www-form-urlencoded', PlainTextFormatter()),
        ('application/octet-stream', PlainTextFormatter()),
        ('text/plain', PlainTextFormatter())
        ])

    rate_limiter = PerUserWithSeparateAnonymousRateLimiter('cloud_api_user_telemost_stats',
                                                           anonymous_group='cloud_api_user_telemost_stats_anonymous')

    def handle(self, request, *args, **kwargs):
        context = self.get_context()
        response_size = context['response_size']

        content = ['x']*response_size

        return PlatformResponse(content=''.join(content), headers={'Content-Type': 'text/plain; charset=utf-8'})


class StaffUserHelper(object):

    def __init__(self, user):
        self.user = user

    def _is_staff_user(self):
        if not self.user:
            return False

        if self.user.uid in PLATFORM_TELEMOST_ENABLE_STAFF_ONLY_CONFERENCES_FOR_UIDS:
            return True

        if is_yateam_uid(self.user.uid):
            return True

        user_info = passport.userinfo(self.user.uid)
        return bool(user_info.get('has_staff'))


class ClckConferenceKeyGenerator(StaffUserHelper):

    def __init__(self, staff_only, user, password):
        StaffUserHelper.__init__(self, user)
        self.staff_only = staff_only
        self.password = password

    def generate(self):
        limit_type = self._get_limit_type()
        conf_key = ConferenceKey.generate(staff_only=self.staff_only, limit_type=limit_type)
        encrypted_conf_key = ConferencesCryptAgent.encrypt_conf_key(conf_key.serialize(), password=self.password)
        conf_uri = ConferenceURIParser.build(encrypted_conf_key)
        return {'uri': conf_uri,
                'conf_id': conf_key.conf_id,
                'conf_pwd': conf_key.conf_pwd,
                'limit_type': limit_type}

    def _get_limit_type(self):
        if (self.user and
                self.user.uid in PLATFORM_TELEMOST_UNLIMITED_UIDS):
            return XMPPLimitTypes.UNLIM
        if self._is_staff_user():
            return XMPPLimitTypes.STAFF

        # Лимит для всех такой же, как для staff на MVP1 (https://st.yandex-team.ru/TELEMOST-99)
        return XMPPLimitTypes.STAFF


class TelemostApiConferenceKeyGenerator(object):

    def __init__(self, staff_only, user=None, is_permanent=None, external_meeting=None):
        self.staff_only = staff_only
        self.uid = user.uid if user else None
        self.is_permanent = is_permanent
        self.external_meeting = external_meeting

    def generate(self):
        response = TelemostViaTCMService.get_telemost_api_for_create(self.uid).create_conference(
            self.uid, self.staff_only, self.is_permanent,
            external_meeting=self.external_meeting
        )
        return {'uri': response['uri'],
                'conf_id': response['conf_id'],
                'conf_pwd': response['conf_pwd'],
                'limit_type': response['limit_type'].lower(),
                'client_configuration': response.get('client_configuration', {})
            }


class ClckConferenceKeyProvider(StaffUserHelper):

    def __init__(self, uri, password, user):
        StaffUserHelper.__init__(self, user)
        self.uri = uri
        self.password = password

    def get_conference(self):
        try:
            encrypted_conf_key = ConferenceURIParser.parse(self.uri)
        except ClckNoResponse:
            logger.error_log.exception('Failed to resolve short url: %s' % self.uri)
            return None

        raw_conf_key = ConferencesCryptAgent.decrypt_conf_key(encrypted_conf_key, password=self.password)

        try:
            conf_key = ConferenceKey.from_str(raw_conf_key)
        except (ValueError, TypeError):
            logger.error_log.exception('Failed to parse conf_key: %s' % raw_conf_key)
            return None

        current_time = ctimestamp()
        if current_time - conf_key.timestamp > PLATFORM_TELEMOST_INVITE_EXPIRE_SECONDS:
            raise TelemostInviteLinkExpiredError()

        if (conf_key.staff_only and
                not self._is_staff_user()):
            raise ForbiddenError()

        return {
            'uri': self.uri,
            'conf_id': conf_key.conf_id,
            'conf_pwd': conf_key.conf_pwd,
            'limit_type': conf_key.limit_type
        }


class TelemostApiConferenceKeyProvider(object):

    def __init__(self, uri, user):
        self.uri = uri
        self.uid = user.uid if user else None

    def get_conference(self):
        try:
            response = TelemostViaTCMService.get_telemost_api(
                self.uid, uri=self.uri
            ).get_conference(self.uri, self.uid)
            return {'uri': response['uri'],
                    'conf_id': response['conf_id'],
                    'conf_pwd': response['conf_pwd'],
                    'limit_type': response['limit_type'].lower(),
                    'client_configuration': response.get('client_configuration', {})}
        except (TelemostApiNotFound, TelemostApiBadRequest):
            return None
        except TelemostApiForbidden:
            raise ForbiddenError()
        except TelemostApiGone:
            raise TelemostInviteLinkExpiredError()


class GetTelemostClientsInstallerHandler(GetClientsInstallerHandler):
    installers_path = TELEMOST_SOFTWARE_INSTALLER_PATH
    auth_user_required = False
    kwargs = fields.QueryDict({
        'platform_id': fields.ChoiceField(
            choices=TELEMOST_SOFTWARE_INSTALLER_PATH.keys(),
            required=True,
            help_text=u"ID платформы ПО"),
    })


class GetTelemostClientsInstallerWithAutologonHandler(GetClientsInstallerWithAutologonHandler):
    query = fields.QueryDict({
        'build': fields.ChoiceField(
            choices=PLATFORM_SOFTWARE_INSTALLER_WITH_AUTO_LOGON_TELEMOST_PATH['win'].keys(),
            required=False,
            default='stable',
            help_text=u'Тип инсталлера: alpha - тестовая, beta - внутри компании, stable - внешняя.'),
    })
    installer_paths = PLATFORM_SOFTWARE_INSTALLER_WITH_AUTO_LOGON_TELEMOST_PATH['win']


class PutTelemostFOSHandler(MpfsProxyHandler):
    """Отправить информацию из ФОС."""
    hidden = True
    permissions = AllowAllPermission()
    auth_user_required = False
    auth_required = False
    service_url = '/json/telemost_fos?uid=%(uid)s&ip=%(ip)s&reply_email=%(reply_email)s&' \
                  'app_version=%(app_version)s&os_version=%(os_version)s&' \
                  'subject=%(subject)s&recipient_type=%(recipient_type)s'

    service_method = 'POST'
    service_headers = {'Content-Type': 'application/json'}
    body_serializer_cls = FOSBodySerializer
    rate_limiter = PerUserWithSeparateAnonymousRateLimiter('fos_telemost_send_email',
                                                           anonymous_group='fos_telemost_send_email')
    error_map = {
        CLIENT_BAD_REQUEST: TelemostClientBadRequest
    }
    query = fields.QueryDict({
        'app_version': fields.StringField(required=True, help_text=u'Версия приложения: номер и beta/prod'),
        'reply_email': fields.EmailField(required=True, help_text=u'Email, на который прислать ответ'),
        'recipient_type': fields.ChoiceField(required=True, choices=('testers', 'supports'), help_text=u'Получатель обращения'),
        'os_version': fields.StringField(required=True, help_text=u'Версия ОС на устройстве'),
        'subject': fields.StringField(required=True, help_text=u'Тема, выбранная из списка жалоб', unquote=True),
    })

    def get_context(self, context=None):
        context = super(PutTelemostFOSHandler, self).get_context(context=context)
        context['ip'] = self.request.get_real_remote_addr()
        if not self.request.user:
            context['uid'] = ''
        return context

    def handle(self, request, *args, **kwargs):
        context = self.get_context()

        service_url = self.build_url(self.service_url, context=context)
        self.request_service(service_url,
                             method=self.service_method,
                             data=self.request.data,
                             headers=self.service_headers)


class GetExperimentsHandler(BasePlatformHandler):
    auth_required = False
    auth_user_required = False
    hidden = True
    permissions = AllowAllPermission()
    serializer_cls = ExperimentsSerializer

    def handle(self, request, *args, **kwargs):
        raw_user_agent = request.raw_headers.get('user-agent', None)
        parsed = UserAgentParser.parse(raw_user_agent, user_agent_prefix='Yandex.Telemost') if raw_user_agent else None
        uid = request.user.uid if request.user else None
        remote_ip = request.get_real_remote_addr()
        additional_headers = {'X-Forwarded-For-Y': remote_ip}

        try:
            experiments = list(uaas.get_telemost_experiments(parsed, uid, additional_headers=additional_headers))
        except UAASRequestInvalidClient:
            raise DiskExperimentsUnsupportedDevice('Unsupported client device')
        return self.serialize({'uas_exp_flags': experiments})
