# -*- coding: utf-8 -*-
import base64
import os
import posixpath
import re
import urllib

from urlparse import urlparse, urlunparse

import mpfs.engine.process
from mpfs.common.static.tags import experiment_names
from mpfs.common.util.experiments.logic import experiment_manager

from mpfs.config import settings
from mpfs.common.errors import ResourceNotFound
from mpfs.core.office.util import parse_resource_id, parse_session_context, OfficeDocShortID
from mpfs.core.services.passport_service import passport
from mpfs.core.user.constants import OFFICE_SERVICE_BRAND_NAME, OFFICE_SERVICE_FOLDER_NAME
from mpfs.platform import fields
from mpfs.platform.auth import InternalAuth, InternalTokenAuth, InternalConductorAuth
from mpfs.platform.v1.disk.handlers import MpfsProxyHandler
from mpfs.platform.v1.disk.fields import StringField
from mpfs.platform.v1.wopi.auth import WOPIAuth
from mpfs.platform.v1.wopi.exceptions import WopiLockMismatchError, WopiNotAllowedActionError
from mpfs.platform.v1.wopi.permissions import AccessResourceIdPermission
from mpfs.platform.exceptions import NotImplementedError, NotFoundError, BadRequestError, CloudApiError
from mpfs.platform.v1.wopi.serializers import CheckFileInfoSerializer, RenameFileSerializer

log = mpfs.engine.process.get_default_log()
OFFICE_CHECK_INFO_CUSTOM_RESPONSE_FIELDS = dict(settings.office['check_info_custom_response_fields'])
OFFICE_FILENAME_ENCODING = settings.office['filename_encoding']
OFFICE_BREADCRUMB_DEFAULT_ORIGIN = settings.office['breadcrumb_default_origin']
OFFICE_BREADCRUMB_DISK_URL = settings.office['breadcrumb_disk_url']
OFFICE_BREADCRUMB_FOLDER_URL = settings.office['breadcrumb_folder_url']
USER_DEFAULT_TLD = settings.user['default_tld']


class WOPIHandlerMixin(object):
    default_auth_methods = [InternalTokenAuth(), InternalConductorAuth(), InternalAuth()]
    auth_methods = [WOPIAuth()]
    permissions = AccessResourceIdPermission()


def empty_content_on_exception(fn):
    def wrapper(self, *args, **kwargs):
        try:
            return fn(self, *args, **kwargs)
        except Exception, e:
            try:
                self.handle_exception(e)
            except CloudApiError, e:
                self.set_response_headers(e)
                return e.status_code, '', e.headers
            raise
    return wrapper


class CheckFileInfoHandler(WOPIHandlerMixin, MpfsProxyHandler):
    service_url = '/json/info_by_file_id?uid=%(uid)s&file_id=%(file_id)s&owner_uid=%(owner_uid)s&meta=size,wh_version,file_url,group'
    sharing_url_service_url = '/json/office_info_by_office_doc_short_id?uid=%(uid)s&office_doc_short_id=%(office_doc_short_id)s&owner_uid=%(owner_uid)s&meta=size,wh_version,file_url,group'
    service_method = 'GET'
    serializer_cls = CheckFileInfoSerializer
    kwargs = fields.QueryDict({
        'resource_id': fields.StringField(required=True, help_text=u'Идентификатор офисного файла.'),
    })
    query = fields.QueryDict({
        'access_token': fields.StringField(required=True, help_text=u'Идентификатор прав и доступа.'),
        'sc': fields.StringField(required=False, help_text=u'Контекст сессии.'),
    })
    headers = fields.QueryDict({
        'X-WOPI-SessionContext': StringField(help_text=u'Значение параметра sc, переданного в запросе.'),
    })

    PM_ORIGIN_REGEX = re.compile(r'^https://[a-z0-9.-]+.yandex.(ru|com|com.tr|net|ua)$')

    @empty_content_on_exception
    def handle(self, request, *args, **kwargs):
        owner_uid, file_id = parse_resource_id(kwargs['resource_id'])
        params = {'uid': request.user.uid,
                  'owner_uid': owner_uid,
                  'file_id': file_id}
        service_url = self.service_url

        if OfficeDocShortID.is_office_doc_short_id(file_id):
            parsed = OfficeDocShortID.parse(file_id)
            if parsed:
                params['office_doc_short_id'] = parsed.office_doc_short_id
                service_url = self.sharing_url_service_url

        url = self.build_url(service_url, params)

        info = self.request_service(url)
        if info['type'] != u'file':
            raise ResourceNotFound()

        user_info = passport.userinfo(request.user.uid)
        login = user_info['login']

        sc = self._get_session_context()
        pm_origin, locale = parse_session_context(sc)
        if not re.match(self.PM_ORIGIN_REGEX, pm_origin):
            pm_origin = ''

        origin = pm_origin or OFFICE_BREADCRUMB_DEFAULT_ORIGIN

        response = {
            'BaseFileName': info['name'],
            'OwnerId': owner_uid,
            'Size': info['meta']['size'],
            'Version': info['meta']['wh_version'],

            'AllowExternalMarketplace': False,
            'CloseButtonClosesWindow': True,
            'FileNameMaxLength': settings.address['max_name_length'],

            'BreadcrumbBrandUrl': OFFICE_BREADCRUMB_DISK_URL % {'origin': origin},
            'BreadcrumbDocName': info['name'],
            'BreadcrumbFolderUrl': self._get_breadcrumb_folder_url(info['id'], origin),

            'DownloadUrl': self._get_contents_url(),
            'UserId': login,
            'UserFriendlyName': login,
            'UserCanNotWriteRelative': True,

            'SupportsCoauth': False
        }

        if pm_origin:
            response['PostMessageOrigin'] = pm_origin
        if locale:
            response['BreadcrumbBrandName'] = OFFICE_SERVICE_BRAND_NAME[locale]
            response['BreadcrumbFolderName'] = OFFICE_SERVICE_FOLDER_NAME[locale]
        if experiment_manager.is_feature_active(experiment_names.MS_OFFICE_FILE_SHARING_POST_MESSAGE):
            response['FileSharingPostMessage'] = True
            response['FileSharingUrl'] = 'https://disk.yandex.ru/client/disk'

        group = info['meta'].get('group', {})
        if 'rights' in group:
            if group['rights'] == 640:
                return self.serialize(response)

        response.update({
            'SupportsLocks': True,
            'UserCanWrite': True,
            'SupportsUpdate': True,
            'SupportsRename': True,
            'UserCanRename': True,
        })

        return self.serialize(response)

    def serialize(self, obj, *args, **kwargs):
        result = super(CheckFileInfoHandler, self).serialize(obj, *args, **kwargs)
        result.update(OFFICE_CHECK_INFO_CUSTOM_RESPONSE_FIELDS)
        return result

    def _get_session_context(self):
        """
        :rtype: str
        """
        sc = self.request.headers.get('X-WOPI-SessionContext', '')
        if sc:
            try:
                sc = base64.b64decode(sc)
            except TypeError:
                pass
        return sc

    @staticmethod
    def _get_breadcrumb_folder_url(path, origin):
        if isinstance(path, unicode):
            path = path.encode('utf-8')
        folder_path = os.path.dirname(path)[1:]
        return OFFICE_BREADCRUMB_FOLDER_URL % {'origin': origin,
                                               'folder_path': urllib.quote(folder_path),
                                               'file_path': urllib.quote(path[1:])}

    def _get_contents_url(self):
        parsed = list(urlparse(self.request.url))
        path = parsed[2]
        parsed[2] = posixpath.join(path, 'contents')
        return urlunparse(parsed)


class PostFileHandler(WOPIHandlerMixin, MpfsProxyHandler):
    serializer_cls = RenameFileSerializer

    kwargs = fields.QueryDict({
        'resource_id': fields.StringField(required=True, help_text=u'Идентификатор файла.'),
    })
    query = fields.QueryDict({
        'access_token': fields.StringField(required=True, help_text=u'Идентификатор прав и доступа.'),
        'sc': fields.StringField(required=False, help_text=u'Контекст сессии.'),
    })
    headers = fields.QueryDict({
        'X-WOPI-SessionContext': StringField(help_text=u'Значение параметра sc, переданного в запросе.'),
        'X-WOPI-Lock': StringField(help_text=u'Новое значение лока.'),
        'X-WOPI-OldLock': StringField(help_text=u'Текущее значение лока.'),
        'X-WOPI-Override': StringField(help_text=u'Действие, которое надо выполнить над локом.'),
        'X-WOPI-RequestedName': StringField(help_text=u'Новое имя файла в кодировке UTF-7, без расширения.'),
    })

    error_map = {
        403: WopiNotAllowedActionError,
    }

    def _validate_filename(self, filename):
        if not filename or len(filename) > settings.address['max_name_length'] or '/' in filename:
            return False
        return True

    def _rename_file(self, resource_id, lock_value, new_file_name):
        try:
            new_file_name = new_file_name.decode(OFFICE_FILENAME_ENCODING)
        except ValueError:
            # раскомментировать, если нужно будет возвращать текст ошибки:
            # return 400, None, {'X-WOPI-InvalidFileNameError': u'Ошибка кодировки имени файла.'}
            raise BadRequestError()

        if not self._validate_filename(new_file_name):
            # раскомментировать, если нужно будет возвращать текст ошибки:
            # return 400, None, {'X-WOPI-InvalidFileNameError': u'Имя файла содержит недопустимые символы.'}
            raise BadRequestError()

        owner_uid, file_id = self._validate_resource_id(resource_id)

        rename_file_url = '/json/office_rename?uid=%(uid)s&owner_uid=%(owner_uid)s&office_lock_id=%(office_lock_id)s&' \
                          'file_id=%(file_id)s&new_name=%(new_name)s'
        url = self.build_url(rename_file_url, self.get_context({'office_lock_id': lock_value, 'owner_uid': owner_uid,
                                                                'file_id': file_id, 'new_name': new_file_name}))
        resp = self.request_service(url)
        if not resp['success']:
            raise WopiLockMismatchError(resp['office_lock_id'])

        response = {'Name': new_file_name}
        return self.serialize(response)

    @staticmethod
    def _validate_resource_id(resource_id):
        parse_result = parse_resource_id(resource_id)
        if parse_result is None:
            raise NotFoundError()
        owner_uid, file_id = parse_result
        return owner_uid, file_id

    def _lock_file(self, resource_id, lock_value):
        owner_uid, file_id = self._validate_resource_id(resource_id)

        lock_file_url = '/json/office_lock?uid=%(uid)s&owner_uid=%(owner_uid)s&office_lock_id=%(office_lock_id)s&' \
                        'file_id=%(file_id)s'
        url = self.build_url(lock_file_url, self.get_context({'owner_uid': owner_uid, 'office_lock_id': lock_value,
                                                              'file_id': file_id}))

        return self._process_lock(url)

    def _unlock_and_relock_file(self, resource_id, old_lock_value, new_lock_value):
        owner_uid, file_id = self._validate_resource_id(resource_id)

        lock_file_url = '/json/office_lock?uid=%(uid)s&owner_uid=%(owner_uid)s&office_lock_id=%(office_lock_id)s&' \
                        'old_office_lock_id=%(old_office_lock_id)s&file_id=%(file_id)s'
        url = self.build_url(lock_file_url,
                             self.get_context({'owner_uid': owner_uid, 'office_lock_id': new_lock_value,
                                               'old_office_lock_id': old_lock_value, 'file_id': file_id}))

        return self._process_lock(url)

    def _unlock_file(self, resource_id, lock_value):
        owner_uid, file_id = self._validate_resource_id(resource_id)

        lock_file_url = '/json/office_unlock?uid=%(uid)s&owner_uid=%(owner_uid)s&' \
                        'office_lock_id=%(office_lock_id)s&file_id=%(file_id)s'
        url = self.build_url(lock_file_url, self.get_context({'owner_uid': owner_uid, 'office_lock_id': lock_value,
                                                              'file_id': file_id}))

        return self._process_lock(url)

    def _refresh_lock(self, resource_id, lock_value):
        owner_uid, file_id = self._validate_resource_id(resource_id)

        lock_file_url = '/json/office_lock?uid=%(uid)s&owner_uid=%(owner_uid)s&office_lock_id=%(office_lock_id)s&' \
                        'old_office_lock_id=%(old_office_lock_id)s&file_id=%(file_id)s'
        url = self.build_url(lock_file_url, self.get_context({'owner_uid': owner_uid, 'office_lock_id': lock_value,
                                                              'old_office_lock_id': lock_value, 'file_id': file_id}))

        return self._process_lock(url)

    def _process_lock(self, url):
        resp = self.request_service(url)

        if not resp['success']:
            raise WopiLockMismatchError(resp['office_lock_id'])

        return 200, None, {}

    @empty_content_on_exception
    def handle(self, request, *args, **kwargs):
        action = request.headers['X-WOPI-Override']
        lock_value = request.headers['X-WOPI-Lock']
        old_lock_value = request.headers['X-WOPI-OldLock']
        resource_id = kwargs['resource_id']

        if action == 'LOCK':
            if old_lock_value:
                return self._unlock_and_relock_file(resource_id, old_lock_value, lock_value)
            else:
                return self._lock_file(resource_id, lock_value)
        elif action == 'UNLOCK':
            return self._unlock_file(resource_id, lock_value)
        elif action == 'REFRESH_LOCK':
            return self._refresh_lock(resource_id, lock_value)
        elif action == 'RENAME_FILE':
            new_file_name = request.headers['X-WOPI-RequestedName']
            return self._rename_file(resource_id, lock_value, new_file_name)
        else:
            raise NotImplementedError()
