# -*- coding: utf-8 -*-
from mpfs.common.util import to_json
from mpfs.config import settings
from mpfs.core.services.djfs_api_service import djfs_api
from mpfs.platform import fields
from mpfs.platform.permissions import AllowByClientIdPermission
from mpfs.platform.v1.disk.handlers import MpfsProxyHandler
from mpfs.platform.v1.disk.permissions import WebDavPermission
from mpfs.platform.v1.disk.serializers import ResourceSerializer, ResourceListSerializer
from mpfs.platform.v1.case.handlers import PLATFORM_MOBILE_APPS_IDS
from mpfs.platform.v2.disk.permissions import DiskV2ResourcesReadPermission


JAVA_DJFS_API_PROXY_PLATFORM_BULK_INFO_ENABLED = settings.java_djfs_api['proxy_platform']['bulk_info']['enabled']


class BulkInfoByResourceIdsHandler(MpfsProxyHandler):
    """Получить информацию о списке ресурсов"""
    hidden = True
    permissions = WebDavPermission() | AllowByClientIdPermission(PLATFORM_MOBILE_APPS_IDS) | DiskV2ResourcesReadPermission()

    service_method = 'POST'

    serializer_cls = ResourceListSerializer

    service_headers = {'Content-Type': 'application/json'}
    service_url = '/json/bulk_info_by_resource_ids?enable_service_ids=%(enable_service_ids)s&uid=%(uid)s&meta=%(meta)s&' \
                  'preview_size=%(preview_size)s&preview_crop=%(preview_crop)s&preview_quality=%(preview_quality)s&' \
                  'preview_allow_big_size=%(preview_allow_big_size)s'

    query = fields.QueryDict({
        'enable_service_ids': fields.StringField(required=False, default='/disk,/attach,/trash,/photounlim',
                                                 help_text=u'Список областей для поиска.'),
        'preview_size': fields.StringField(required=False),
        'preview_crop': fields.IntegerField(required=False, default=0),
        'preview_quality': fields.IntegerField(required=False, default=0),
        'preview_allow_big_size': fields.BooleanField(required=False, default=False),
    })

    def serialize(self, obj, *args, **kwargs):
        items = obj
        new_obj = {
            'total': len(items),
            'items': items
        }
        return super(BulkInfoByResourceIdsHandler, self).serialize(new_obj, *args, **kwargs)

    def get_context(self, context=None):
        c = super(BulkInfoByResourceIdsHandler, self).get_context(context)
        c['meta'] = ResourceSerializer.get_raw_mpfs_meta()
        return c

    def handle(self, request, *args, **kwargs):
        context = self.get_context()
        service_url = self.build_url(self.service_url, context=context)
        source_ids = request.args['resource_ids'].split(',')

        if JAVA_DJFS_API_PROXY_PLATFORM_BULK_INFO_ENABLED:
            # djfs-specific logic
            params = {}
            for param in ('enable_service_ids', 'uid', 'preview_size', 'preview_crop',
                          'preview_quality', 'preview_allow_big_size'):
                params[param] = context[param]
            meta = context['meta'].split(',')
            response = djfs_api.bulk_info_by_resource_ids(params, meta, to_json(source_ids))
        else:
            # MPFS-specific logic
            response = self.request_service(service_url,
                                            method=self.service_method,
                                            data=source_ids,
                                            headers=self.service_headers)

        return self.serialize(response)
