# -*- coding: utf-8 -*-

from mpfs.platform import fields
from mpfs.platform.v1.disk.fields import QuotedListField
from mpfs.platform.v1.disk.serializers import LinkSerializer
from mpfs.platform.v1.personality.exceptions import (PersonalityAddressesNotFoundError,
                                                     PersonalityAddressExistsError)
from mpfs.platform.v1.personality.handlers import DataApiProfileBaseHandler
from mpfs.platform.v2.personality.serializers import (DataApiNewAddressListSerializer,
                                                      DataApiNewAddressSerializer,
                                                      DataApiNewAddressPatchSerializer)


class GetNewAddressHandler(DataApiProfileBaseHandler):
    """Получить адрес по id"""

    service_method = 'GET'
    service_url = '/addresses/%(address_id)s?__uid=%(uid)s'
    serializer_cls = DataApiNewAddressSerializer

    error_map = {
        'not-found': PersonalityAddressesNotFoundError,
    }

    kwargs = fields.QueryDict({
        'address_id': fields.StringField(required=True, help_text=u'Идентификатор адреса.')
    })


class DeleteNewAddressHandler(DataApiProfileBaseHandler):
    """Удалить адрес"""
    resp_status_code = 204
    service_method = 'DELETE'
    service_url = '/addresses/%(address_id)s?__uid=%(uid)s'

    error_map = {
        'not-found': PersonalityAddressesNotFoundError,
        'invalid-delta': PersonalityAddressesNotFoundError
    }

    kwargs = fields.QueryDict({
        'address_id': fields.StringField(required=True, help_text=u'Идентификатор адреса.'),
    })


class NewAddressLinkResultBaseHandler(DataApiProfileBaseHandler):

    def handle(self, request, *args, **kwargs):
        super(NewAddressLinkResultBaseHandler, self).handle(request, *args, **kwargs)
        params = {
            'address_id': self.request.kwargs['address_id']
        }
        link = self.router.get_link(GetNewAddressHandler, params=params)
        return self.serialize(link)


class SaveNewAddressHandler(NewAddressLinkResultBaseHandler):
    """Сохранить адрес"""

    service_method = 'PUT'
    service_url = '/addresses/%(address_id)s?__uid=%(uid)s'
    serializer_cls = LinkSerializer
    body_serializer_cls = DataApiNewAddressSerializer

    kwargs = fields.QueryDict({
        'address_id': fields.StringField(required=True, help_text=u'Идентификатор адреса.')
    })

    def handle(self, request, *args, **kwargs):
        self.request.body['address_id'] = self.request.kwargs['address_id']

        return super(SaveNewAddressHandler, self).handle(request, data=self.request.body, *args, **kwargs)


class CreateNewAddressHandler(DataApiProfileBaseHandler):
    """Создать адрес"""

    service_method = 'POST'
    service_url = '/addresses?__uid=%(uid)s'
    serializer_cls = DataApiNewAddressSerializer
    body_serializer_cls = DataApiNewAddressSerializer
    resp_status_code = 201

    error_map = {
        'invalid-delta': PersonalityAddressExistsError
    }

    def handle(self, request, *args, **kwargs):
        return super(CreateNewAddressHandler, self).handle(request, data=self.request.body, *args, **kwargs)


class PatchNewAddressHandler(NewAddressLinkResultBaseHandler):
    """Изменить адрес"""

    service_method = 'PATCH'
    service_url = '/addresses/%(address_id)s?__uid=%(uid)s'
    serializer_cls = LinkSerializer
    body_serializer_cls = DataApiNewAddressPatchSerializer

    error_map = {
        'not-found': PersonalityAddressesNotFoundError,
        'invalid-delta': PersonalityAddressesNotFoundError
    }

    kwargs = fields.QueryDict({
        'address_id': fields.StringField(required=True, help_text=u'Идентификатор адреса.')
    })

    def handle(self, request, *args, **kwargs):
        return super(PatchNewAddressHandler, self).handle(request, data=self.request.body, *args, **kwargs)


class TouchNewAddressHandler(NewAddressLinkResultBaseHandler):
    """Обновить дату последнего использования адреса"""
    resp_status_code = 200
    service_method = 'PUT'
    service_url = '/addresses/%(address_id)s/touch?__uid=%(uid)s'
    serializer_cls = LinkSerializer

    error_map = {
        'not-found': PersonalityAddressesNotFoundError,
        'invalid-delta': PersonalityAddressesNotFoundError
    }

    kwargs = fields.QueryDict({
        'address_id': fields.StringField(required=True, help_text=u'Идентификатор адреса.'),
    })


class TagNewAddressHandler(NewAddressLinkResultBaseHandler):
    """Пометить адрес тэгами"""
    resp_status_code = 200
    service_method = 'PUT'
    service_url = '/addresses/%(address_id)s/tag?__uid=%(uid)s'
    serializer_cls = LinkSerializer

    error_map = {
        'not-found': PersonalityAddressesNotFoundError,
        'invalid-delta': PersonalityAddressesNotFoundError
    }

    query = fields.QueryDict({
        'tags': QuotedListField(help_text=u'Список тегов'),
    })

    kwargs = fields.QueryDict({
        'address_id': fields.StringField(required=True, help_text=u'Идентификатор адреса.'),
    })

    def get_url(self, context=None):
        url = super(TagNewAddressHandler, self).get_url(context)
        url += "&tag=" + ",".join(context['tags'])
        return url


class UntagNewAddressHandler(NewAddressLinkResultBaseHandler):
    """Убрать тэги с адреса"""
    resp_status_code = 200
    service_method = 'PUT'
    service_url = '/addresses/%(address_id)s/untag?__uid=%(uid)s'
    serializer_cls = LinkSerializer

    error_map = {
        'not-found': PersonalityAddressesNotFoundError,
        'invalid-delta': PersonalityAddressesNotFoundError
    }

    query = fields.QueryDict({
        'tags': QuotedListField(help_text=u'Список тегов'),
    })

    kwargs = fields.QueryDict({
        'address_id': fields.StringField(required=True, help_text=u'Идентификатор адреса.'),
    })

    def get_url(self, context=None):
        url = super(UntagNewAddressHandler, self).get_url(context)
        url += "&tag=" + ",".join(context['tags'])
        return url


class ListNewAddressesHandler(DataApiProfileBaseHandler):
    """Получить список адресов"""

    service_method = 'GET'
    service_url = '/addresses?__uid=%(uid)s'
    serializer_cls = DataApiNewAddressListSerializer

    query = fields.QueryDict({
        'limit': fields.IntegerField(default=20, help_text=u'Количество выводимых ресурсов.'),
        'offset': fields.IntegerField(default=0, help_text=u'Смещение от начала списка ресурсов.'),
    })

    def serialize(self, res, *args, **kwargs):
        offset = self.request.query['offset']
        limit = self.request.query['limit']

        res['limit'] = limit
        res['offset'] = offset
        res['total'] = len(res['items'])
        res['items'] = res['items'][offset:offset+limit]
        return super(ListNewAddressesHandler, self).serialize(res, *args, **kwargs)
