# -*- coding: utf-8 -*-
import hashlib
import urllib
import re

from mpfs.common.errors import (
    TelemostApiNotFound,
    TelemostApiError,
    TelemostApiGone,
    TelemostApiNotCome,
    TelemostApiBroadcastLinkExpired,
    TelemostApiStreamNotStarted,
    TelemostApiNoSuchBroadcastCreated,
    TelemostApiInvalidTranslatorToken,
    TelemostApiNoSuchUserInConference,
    TelemostApiForbiddenAccessToCreateBroadcast,
    TelemostApiForbiddenAccessToStartBroadcast,
    TelemostApiForbiddenToPrivate,
    UAASRequestInvalidClient)
from mpfs.common.util import to_json
from mpfs.config import settings
from mpfs.core.services.passport_service import passport
from mpfs.engine.process import get_default_log, get_error_log
from mpfs.core.services.telemost_api_service import (
    telemost_api,
    TelemostViaTCMService,
    TemplatedTelemostService,
    SEPARATE_TELEMOST_CLUSTERS,
    MultiTelemostRequestsPoweredServiceBase,
    FanOutToAllTelemostService,
)
from mpfs.platform import fields
from mpfs.platform.auth import (
    PassportCookieAuth,
    TVM2Auth,
    TVM2TicketsOnlyAuth,
    YaTeamOAuthAuth,
)
from mpfs.platform.exceptions import (
    NotFoundError,
    ForbiddenError,
)
from mpfs.platform.handlers import RequestsPoweredServiceProxyHandler, BasePlatformHandler
from mpfs.platform.permissions import AllowAllPermission
from mpfs.platform.rate_limiters import PerUserRateLimiter, PerUserWithSeparateAnonymousRateLimiter
from mpfs.platform.utils.uid_formatters import is_yateam_uid
from mpfs.platform.utils.user_agent_detector import UserAgentDetector, UserAgentDetails
from mpfs.platform.v1.disk.permissions import WebDavPermission, MobileMailPermission
from mpfs.platform.v1.telemost.exceptions import (
    TelemostConferenceNotFoundError,
    TelemostInviteLinkExpiredError,
    BadBodyError,
    TelemostActiveSessionNotFoundError,
    TelemostPeerTokenInvalidError,
    TelemostSessionAlreadyDisconnectedError,
    TelemostTooManyUsersError,
    TelemostServiceTemporaryOverloaded,
)
from mpfs.platform.v1.telemost.permissions import (
    TelemostPermission,
)

from mpfs.platform.v2.telemost.permissions import TelemostConferenceCreatePermission
from mpfs.platform.v2.telemost.permissions import TelemostBillingPermission

from mpfs.platform.v1.telemost.serializers import CreateConferenceBodySerializer
from mpfs.platform.v2.telemost import exceptions
from mpfs.platform.v2.telemost.exceptions import (
    TelemostYaTeamTokenAccessError,
    TelemostYaTeamConferenceForbiddenError,
    TelemostUnauthorizedError,
    TelemostInviteLinkNotComeError,
    TelemostStreamNotStartedError,
    TelemostBroadcastLinkExpiredError,
    TelemostNoSuchBroadcastCreatedError,
    TelemostInvalidTranslatorTokenError,
    TelemostNoSuchUserInConferenceError,
    TelemostForbiddenAccessToCreateBroadcastError,
    TelemostForbiddenAccessToStartBroadcastError,
    TelemostForbiddenToPrivateError,
)
from mpfs.platform.v2.telemost.serializers import (
    ConferenceConnectionSerializer,
    UserInfoSerializer,
    PeerListSerializer,
    PeerIDsBodySerializer,
    ConferenceInfoSerializer,
    LeaveRoomSerializer,
    UserFeatureTogglesSerializer,
    CreateChatSerializer,
    JoinToChatSerializer,
    GetChatHistorySerializer,
    CreateBroadcastBodySerializer,
    BroadcastDataSerializer,
    StreamDataSerializer,
    ViewerDataSerializer,
)

default_log = get_default_log()
error_log = get_error_log()


CONF_JOIN_SHORT_URL_RE = re.compile(settings.platform['telemost']['conf_join_short_url_re'])
CONF_YANDEX_TEAM_JOIN_URL_TMPL = settings.platform['telemost']['conf_yandex_team_join_url_tmpl']


USER_TICKET_HEADER = 'X-Ya-User-Ticket'


class TelemostAPIProxyHandler(RequestsPoweredServiceProxyHandler):
    service = telemost_api
    permissions = WebDavPermission() | TelemostPermission() | MobileMailPermission()
    auth_methods = [PassportCookieAuth()]
    service_base_exception = TelemostApiError
    hidden = True

    headers = fields.QueryDict({
        'Client-Instance-Id': fields.StringField(required=False,
                                                 help_text=u'Идентификатор клиента с которого осуществляется подключение к конференции.')
    })

    error_map = {
        'validate': exceptions.TelemostBadRequestError,
        'not-found': NotFoundError,
        'ActiveSessionNotFound': TelemostActiveSessionNotFoundError,
        'BadUid': exceptions.TelemostBadRequestError,
        'BadUri': exceptions.TelemostBadRequestError,
        'ConferenceLinkExpired': TelemostInviteLinkExpiredError,
        'BroadcastLinkExpired': TelemostBroadcastLinkExpiredError,
        'ConferenceLinkNotCome': TelemostInviteLinkNotComeError,
        'StreamNotStarted': TelemostStreamNotStartedError,
        'ForbiddenAccessToPrivateConference': TelemostForbiddenToPrivateError,
        'ConferenceNotAvailable': ForbiddenError,
        'ConferenceNotFound': TelemostConferenceNotFoundError,
        'InvalidPeerToken': TelemostPeerTokenInvalidError,
        'PeerNotFound': NotFoundError,
        'SessionAlreadyDisconnected': TelemostSessionAlreadyDisconnectedError,
        'TooManyUsers': TelemostTooManyUsersError,
        'TooMuchDataError': exceptions.TelemostBadRequestError,
        'YaTeamTokenAccessDenied': TelemostYaTeamTokenAccessError,
        'YaTeamConferenceNotAvailable': TelemostYaTeamConferenceForbiddenError,
        'UnauthorizedAccessToYaTeamDenied': TelemostUnauthorizedError,
        'TelemostOverload': TelemostServiceTemporaryOverloaded,
        'NoSuchBroadcastCreated': TelemostNoSuchBroadcastCreatedError,
        'InvalidTranslatorToken': TelemostInvalidTranslatorTokenError,
        'NoSuchUserInConference': TelemostNoSuchUserInConferenceError,
        'ForbiddenAccessToCreateBroadcast': TelemostForbiddenAccessToCreateBroadcastError,
        'ForbiddenAccessToStartBroadcast': TelemostForbiddenAccessToStartBroadcastError,
        '400': exceptions.TelemostBadRequestError,
        '403': ForbiddenError,
        '404': NotFoundError
    }

    def get_context(self, context=None):
        context = super(TelemostAPIProxyHandler, self).get_context(context)
        lang = self.request.language
        if '-' in lang:
            lang = lang.split('-')[0]
        context['language'] = lang
        return context

    def get_service(self, context=None):
        if context is not None:
            return TelemostViaTCMService.get_telemost_api(
                context.get('uid'), uri=context.get('uri'), room_id=context.get('room_id')
            )

        return self.service

    def get_service_error_code(self, exception):
        if not isinstance(exception, TelemostApiError) or not hasattr(exception, 'code'):
            return None
        return exception.code

    def log_details(self, event_type, conf_id, conf_uri):
        details = {'event_type': event_type,
                   'user-agent': self.request.raw_headers.get('user-agent', '-'),
                   'uid': '-',
                   'hashed_url': '-',
                   'conf_id': conf_id}
        try:
            details['hashed_url'] = hashlib.sha256(conf_uri).hexdigest()
        except Exception:
            error_log.exception('failed to hash conf_uri')

        if self.request.user:
            details['uid'] = self.request.user.uid

        default_log.info(u'event_type: %(event_type)s; uid: %(uid)s; user-agent: %(user-agent)s; conf_id: %(conf_id)s; '
                         u'hashed_url: sha256-%(hashed_url)s' % details)

    def get_client_instance_id(self, request):
        return self.request.raw_headers


class CreateConferenceHandler(TelemostAPIProxyHandler):
    service = telemost_api
    permissions = WebDavPermission() | TelemostPermission() | MobileMailPermission()\
                  | TelemostConferenceCreatePermission()
    auth_methods = [PassportCookieAuth(), TVM2Auth(allow_yateam=True), TVM2TicketsOnlyAuth(allow_yateam=True)]

    serializer_cls = ConferenceConnectionSerializer
    service_method = 'POST'
    service_url = '/v2/conferences?uid=%(uid)s&staff_only=%(staff_only)s&lang=%(language)s'
    body_serializer_cls = CreateConferenceBodySerializer
    resp_status_code = 201

    rate_limiter = PerUserRateLimiter('cloud_api_user_telemost_create')

    query = fields.QueryDict({
        'calendar_event_id': fields.StringField(required=False,
                help_text=u'Идентификатор события календаря, из которого создана конференция.')
    })

    def get_service(self, context=None):
        if context is not None:
            return TelemostViaTCMService.get_telemost_api_for_create(context.get('uid'))

        return self.service

    def get_context(self, context=None):
        context = super(CreateConferenceHandler, self).get_context(context)
        # В Telemost API нужно передавать параметрами, поэтому выбираем из тела нужные значения
        context['staff_only'] = self.request.body.get('staff_only', False)
        context['is_permanent'] = self.request.body.get('is_permanent', False)
        return context

    def handle(self, request, *args, **kwargs):
        result = super(CreateConferenceHandler, self).handle(request, *args, **kwargs)
        self.log_details(event_type='create', conf_id=result.get('room_id'), conf_uri=result.get('uri'))
        return result

    def get_url(self, context=None, base_url=None):
        url = super(CreateConferenceHandler, self).get_url(context)

        optional_params = {}
        context = {} if not context else context
        if context['Client-Instance-Id']:
            optional_params['client_instance_id'] = context['Client-Instance-Id']
        if self.request.query.get('calendar_event_id'):
            optional_params['calendar_event_id'] = self.request.query.get('calendar_event_id')

        if optional_params:
            url += '&%s' % urllib.urlencode(optional_params)

        return url


class BaseConferenceHandler(TelemostAPIProxyHandler):
    permissions = AllowAllPermission()
    auth_user_required = False
    auth_required = False

    rate_limiter = PerUserWithSeparateAnonymousRateLimiter('cloud_api_user_telemost_join',
                                                           anonymous_group='cloud_api_user_telemost')

    kwargs = fields.QueryDict({
        'uri': fields.StringField(required=True,
                                  help_text=u'URI конференции, для которой запрашиваем данные соединения.')
    })

    error_map = {
        TelemostApiNotFound: TelemostConferenceNotFoundError,
        TelemostApiGone: TelemostInviteLinkExpiredError,
        TelemostApiBroadcastLinkExpired: TelemostBroadcastLinkExpiredError,
        TelemostApiNotCome: TelemostInviteLinkNotComeError,
        TelemostApiStreamNotStarted: TelemostStreamNotStartedError,
        TelemostApiNoSuchBroadcastCreated: TelemostNoSuchBroadcastCreatedError,
        TelemostApiInvalidTranslatorToken: TelemostInvalidTranslatorTokenError,
        TelemostApiNoSuchUserInConference: TelemostNoSuchUserInConferenceError,
        TelemostApiForbiddenAccessToCreateBroadcast: TelemostForbiddenAccessToCreateBroadcastError,
        TelemostApiForbiddenAccessToStartBroadcast: TelemostForbiddenAccessToStartBroadcastError,
        TelemostApiForbiddenToPrivate: TelemostForbiddenToPrivateError,
    }

    def get_service_error_code(self, exception):
        error_code = super(BaseConferenceHandler, self).get_service_error_code(exception)

        if error_code == 'YaTeamTokenAccessDenied' and self.request.user is None:
            error_code = 'UnauthorizedAccessToYaTeamDenied'

        return error_code

    def get_exception_context(self, context=None, exception=None):
        exc_context = super(BaseConferenceHandler, self).get_exception_context(context)
        context = self.get_context()

        # Для мобильных не отдаем ссылку для авторизации на telemost.yandex-team.ru
        # (там front покажет заглушку про приложение)
        user_agent_details = UserAgentDetector.parse_from_headers(dict(self.request.raw_headers))
        if (user_agent_details.get_browser_name() == UserAgentDetails.UNKNOWN or
                user_agent_details.is_mobile()):
            return exc_context

        # Добавляем ссылку на yandex-team встречку, для ошибок, которые должны содержать эту ссылку
        # (подхватится ошибками в error_map)
        uri = context.get('uri', '')
        match = CONF_JOIN_SHORT_URL_RE.match(uri)
        conf_key = ''
        if match:
            conf_key = match.group('conf_key')
            conf_key = urllib.unquote(conf_key)
        exc_context['yandex_team_join_link'] = CONF_YANDEX_TEAM_JOIN_URL_TMPL % conf_key

        return exc_context

    @staticmethod
    def collect_uid_and_translator_token(context):
        result = {}

        for param in ['uid', 'translator_token']:
            if param in context and context[param]:
                result[param] = context[param]

        return result


class GetConferenceInfoHandler(BaseConferenceHandler):
    serializer_cls = ConferenceInfoSerializer
    service_method = 'GET'
    service_url = '/v2/conferences/%(uri)s'
    auth_methods = [PassportCookieAuth(), YaTeamOAuthAuth()]

    def get_url(self, context=None, base_url=None):
        url = super(BaseConferenceHandler, self).get_url(context)

        if self.request.user:
            if self.request.user.uid:
                url += '?uid=%s' % self.request.user.uid
            if USER_TICKET_HEADER not in self.request.headers and self.request.user.user_ticket:
                self.service_headers[USER_TICKET_HEADER] = self.request.user.user_ticket

        return url


class LinkCalendarEventHandler(TelemostAPIProxyHandler):
    permissions = WebDavPermission() | TelemostPermission() | MobileMailPermission() \
                  | TelemostConferenceCreatePermission()
    auth_methods = [PassportCookieAuth(), TVM2Auth(allow_yateam=True), TVM2TicketsOnlyAuth(allow_yateam=True)]

    serializer_cls = ConferenceConnectionSerializer
    service_method = 'POST'
    service_url = '/v2/conferences/%(uri)s/link_calendar_event?uid=%(uid)s&calendar_event_id=%(calendar_event_id)s'

    rate_limiter = PerUserRateLimiter('cloud_api_user_telemost_create')

    kwargs = fields.QueryDict({
        'uri': fields.StringField(
            required=True,
            help_text=u'URI конференции, которую требуется связать с событием календаря.')
    })

    query = fields.QueryDict({
        'calendar_event_id': fields.StringField(
            required=True,
            help_text=u'Идентификатор события календаря, которое требуется связать с конференцией.')
    })


class GetConferenceConnectionHandler(BaseConferenceHandler):
    serializer_cls = ConferenceConnectionSerializer
    service_method = 'GET'
    service_url = '/v2/conferences/%(uri)s/connection?lang=%(language)s'
    auth_methods = [PassportCookieAuth(), YaTeamOAuthAuth()]

    query = fields.QueryDict({
        'translator_token': fields.StringField(help_text=u'Токен, идентифицирующий транслятор при подключении.'),
        'display_name': fields.StringField(help_text=u'Отображаемое имя.'),
    })

    def get_url(self, context=None, base_url=None):
        url = super(GetConferenceConnectionHandler, self).get_url(context)

        optional_params = self.collect_uid_and_translator_token(context)

        if context['display_name']:
            optional_params['display_name'] = context['display_name'].encode('utf-8')
        elif (self.request.user and
                  is_yateam_uid(self.request.user.uid) and
                  self.request.user.details and
                  self.request.user.details.get('firstname')):
            # Если авторизация с yandex-team аккаунтом, то передадим в бэк Телемоста имя
            # т.к. прямо сейчас нет поддержки yandex-team Паспорта в бэке Телемоста
            optional_params['display_name'] = self.request.user.details['firstname'].encode('utf-8')
        if context['Client-Instance-Id']:
            optional_params['client_instance_id'] = context['Client-Instance-Id']

        if optional_params:
            url += '&%s' % urllib.urlencode(optional_params)

        return url

    def handle(self, request, *args, **kwargs):
        result = super(GetConferenceConnectionHandler, self).handle(request, *args, **kwargs)
        self.log_details(event_type='join', conf_id=result.get('room_id'), conf_uri=result.get('uri'))
        return result


class GetConferencePeersHandler(BaseConferenceHandler):
    serializer_cls = PeerListSerializer
    service_method = 'PUT'
    service_url = '/v2/conferences/%(uri)s/peers'
    service_headers = {'Accept': 'application/json', 'Content-type': 'application/json'}
    body_serializer_cls = PeerIDsBodySerializer
    auth_methods = [PassportCookieAuth(), YaTeamOAuthAuth()]

    def handle(self, request, *args, **kwargs):
        context = self.get_context()
        url = self.get_url(context)

        if not self.request.body:
            raise BadBodyError(details='empty body')

        uid = None
        if self.request.user and self.request.user.uid:
            uid = self.request.user.uid
        telemost_service = TelemostViaTCMService.get_telemost_api(uid, uri=context.get('uri'))

        resp = self.request_service(url, data=to_json(self.request.body), service=telemost_service)

        return self.serialize(resp)

    def get_url(self, context=None, base_url=None):
        url = super(GetConferencePeersHandler, self).get_url(context)

        if self.request.user and self.request.user.uid:
            url += '?uid=%s' % self.request.user.uid

        return url


class GetUserInfoHandler(TelemostAPIProxyHandler):
    serializer_cls = UserInfoSerializer
    service_method = 'GET'
    service_url = '/v2/users/%(uid)s'
    auth_methods = [PassportCookieAuth(), YaTeamOAuthAuth()]

    def get_service(self, context=None):
        if context is not None:
            return TelemostViaTCMService.get_telemost_api_for_general_info(context.get('uid'))

        return self.service

    def handle(self, request, *args, **kwargs):
        result = super(GetUserInfoHandler, self).handle(request, *args, **kwargs)
        result['login'] = self.request.user.login
        user_info = passport.userinfo(self.request.user.uid)
        field = 'country'
        if field in user_info and user_info[field] is not None:
            result[field] = user_info[field]
        return result


class GetUserFeatureTogglesHandler(BasePlatformHandler):
    serializer_cls = UserFeatureTogglesSerializer
    permissions = WebDavPermission() | TelemostPermission() | MobileMailPermission()
    auth_methods = [PassportCookieAuth()]
    hidden = True

    """Получить статус фич пользователя"""
    def handle(self, request, *args, **kwargs):
        promote_mail360 = not passport.userinfo(request.user.uid).get('has_mail360')
        result = {}
        for field_name, enabled in [('promote_mail360', promote_mail360)]:
            result[field_name] = {'enabled': enabled}
        return self.serialize(result)


class CreateChatHandler(TelemostAPIProxyHandler):
    serializer_cls = CreateChatSerializer
    service_method = 'PUT'
    service_url = '/v2/create_chat/%(room_id)s?user=%(user)s'
    service_headers = {'Accept': 'application/json', 'Content-type': 'application/json'}
    auth_methods = [PassportCookieAuth(), YaTeamOAuthAuth()]
    permissions = AllowAllPermission()
    auth_user_required = False
    auth_required = False

    rate_limiter = PerUserWithSeparateAnonymousRateLimiter('cloud_api_user_telemost_join',
                                                           anonymous_group='cloud_api_user_telemost')

    kwargs = fields.QueryDict({
        'room_id': fields.StringField(required=True,
                                      help_text=u'Идентификатор комнаты, для которой необходимо создать чат.')
    })

    query = fields.QueryDict({
        'user': fields.StringField(required=True,
                                   help_text=u'Уникальный идентификатор пользователя Мессенджера.')
    })

    def get_url(self, context=None, base_url=None):
        url = super(CreateChatHandler, self).get_url(context)
        if self.request.user:
            if self.request.user.uid:
                url += '&uid=%s' % self.request.user.uid
            if USER_TICKET_HEADER not in self.request.headers and self.request.user.user_ticket:
                self.service_headers[USER_TICKET_HEADER] = self.request.user.user_ticket

        return url


class JoinToChatHandler(TelemostAPIProxyHandler):
    serializer_cls = JoinToChatSerializer
    service_method = 'PUT'
    service_url = '/v2/join_to_chat/%(room_id)s?user=%(user)s'
    service_headers = {'Accept': 'application/json', 'Content-type': 'application/json'}
    auth_methods = [PassportCookieAuth(), YaTeamOAuthAuth()]
    permissions = AllowAllPermission()
    auth_user_required = False
    auth_required = False

    rate_limiter = PerUserWithSeparateAnonymousRateLimiter('cloud_api_user_telemost_join',
                                                           anonymous_group='cloud_api_user_telemost')

    kwargs = fields.QueryDict({
        'room_id': fields.StringField(required=True,
                                      help_text=u'Идентификатор комнаты, для которой необходимо создать чат.')
    })

    query = fields.QueryDict({
        'user': fields.StringField(required=True,
                                   help_text=u'Уникальный идентификатор пользователя Мессенджера.')
    })

    def get_url(self, context=None, base_url=None):
        url = super(JoinToChatHandler, self).get_url(context)
        if self.request.user:
            if self.request.user.uid:
                url += '&uid=%s' % self.request.user.uid
            if USER_TICKET_HEADER not in self.request.headers and self.request.user.user_ticket:
                self.service_headers[USER_TICKET_HEADER] = self.request.user.user_ticket

        return url


class GetChatHistoryHandler(TelemostAPIProxyHandler):
    serializer_cls = GetChatHistorySerializer
    service_method = 'PUT'
    service_url = '/v2/get_chat_history/%(room_id)s?user=%(user)s'
    service_headers = {'Accept': 'application/json', 'Content-type': 'application/json'}
    auth_methods = [PassportCookieAuth(), YaTeamOAuthAuth()]
    permissions = AllowAllPermission()
    auth_user_required = False
    auth_required = False

    rate_limiter = PerUserWithSeparateAnonymousRateLimiter('cloud_api_user_telemost_join',
                                                           anonymous_group='cloud_api_user_telemost')

    kwargs = fields.QueryDict({
        'room_id': fields.StringField(required=True,
                                      help_text=u'Идентификатор комнаты, для чата которой необходимо получить историю.')
    })

    query = fields.QueryDict({
        'user': fields.StringField(required=True,
                                   help_text=u'Уникальный идентификатор пользователя Мессенджера.'),
        'offset': fields.StringField(required=False,
                                     help_text=u'Смещение времени для получения раннего сообщения в timestamp.')
    })

    def get_url(self, context=None, base_url=None):
        url = super(GetChatHistoryHandler, self).get_url(context)

        optional_params = {}
        if self.request.query.get('offset'):
            optional_params['offset'] = self.request.query.get('offset')
        if optional_params:
            url += '&%s' % urllib.urlencode(optional_params)
        if self.request.user:
            if self.request.user.uid:
                url += '&uid=%s' % self.request.user.uid
            if USER_TICKET_HEADER not in self.request.headers and self.request.user.user_ticket:
                self.service_headers[USER_TICKET_HEADER] = self.request.user.user_ticket

        return url


class LeaveRoomHandler(TelemostAPIProxyHandler):
    permissions = AllowAllPermission()
    auth_user_required = False
    auth_required = False
    serializer_cls = LeaveRoomSerializer
    service_method = 'PUT'
    service_url = '/v2/rooms/%(room_id)s/leave?peer_id=%(peer_id)s&peer_token=%(peer_token)s&media_session_id=%(media_session_id)s'

    kwargs = fields.QueryDict({
        'room_id': fields.StringField(required=True,
                                      help_text=u'Идентификатор комнаты для которой пытаемся отключить клиента.')
    })

    query = fields.QueryDict({
        'peer_id': fields.StringField(required=True,
                                      help_text=u'Идентификатор пользователя видеовстречи.'),
        'peer_token': fields.StringField(required=True,
                                      help_text=u'Токен пользователя видеовстречи.'),
        'media_session_id': fields.StringField(required=True,
                                      help_text=u'Идентификитор медиа-сессии.')
    })

    def handle(self, request, *args, **kwargs):
        result = super(LeaveRoomHandler, self).handle(request, *args, **kwargs)
        self.log_details(event_type='leave', conf_id=None, conf_uri=None)
        return result


# ===============================================================================
# Commands
# ===============================================================================

class BaseCommandHandler(BaseConferenceHandler):
    auth_methods = [PassportCookieAuth(), YaTeamOAuthAuth()]
    permissions = WebDavPermission() | TelemostPermission() | MobileMailPermission()
    service_method = 'POST'
    auth_user_required = True
    auth_required = True

    kwargs = fields.QueryDict({
        'uri': fields.StringField(
            required=True,
            help_text=u'URI конференции, в которой отправляем команду.')
    })


class BaseModerationCommandHandler(BaseCommandHandler):
    query = fields.QueryDict({
        'peer_id': fields.StringField(required=True,
                                      help_text=u'Идентификатор пира-получателя команды.')})


class BaseAllModerationCommandHandler(BaseCommandHandler):
    query = fields.QueryDict({
        'exclude_peer_id': fields.StringField(required=True,
                                              help_text=u'Идентификатор пира который не получит команду.')})


class MuteMicrophoneCommandHandler(BaseModerationCommandHandler):
    service_url = '/v2/conferences/%(uri)s/commands/mute-microphone?peer_id=%(peer_id)s&uid=%(uid)s'


class MuteCameraCommandHandler(BaseModerationCommandHandler):
    service_url = '/v2/conferences/%(uri)s/commands/mute-camera?peer_id=%(peer_id)s&uid=%(uid)s'


class MuteDesktopCommandHandler(BaseModerationCommandHandler):
    service_url = '/v2/conferences/%(uri)s/commands/mute-desktop?peer_id=%(peer_id)s&uid=%(uid)s'


class MuteAllMicrophonesCommandHandler(BaseAllModerationCommandHandler):
    service_url = '/v2/conferences/%(uri)s/commands/mute-all-microphones?exclude_peer_id=%(exclude_peer_id)s&uid=%(uid)s'


class MuteAllCamerasCommandHandler(BaseAllModerationCommandHandler):
    service_url = '/v2/conferences/%(uri)s/commands/mute-all-cameras?exclude_peer_id=%(exclude_peer_id)s&uid=%(uid)s'


class MuteAllDesktopsCommandHandler(BaseAllModerationCommandHandler):
    service_url = '/v2/conferences/%(uri)s/commands/mute-all-desktops?exclude_peer_id=%(exclude_peer_id)s&uid=%(uid)s'


class SetRoleCommandHandler(BaseCommandHandler):
    service_url = '/v2/conferences/%(uri)s/commands/set-user-role?uid=%(uid)s&target_uid=%(target_uid)s&user_role=%(role)s'

    query = fields.QueryDict({
        'target_uid': fields.StringField(required=True, help_text=u'UID пользователя, чья роль будет изменена.'),
        'role': fields.ChoiceField(required=True, choices=['ADMIN', 'MEMBER'], help_text=u'Новая роль пользователя')})


# ===============================================================================
# Broadcasts
# ===============================================================================

class BaseBroadcastHandler(BaseConferenceHandler):
    auth_methods = [PassportCookieAuth(), YaTeamOAuthAuth(), TVM2Auth(allow_yateam=True), TVM2TicketsOnlyAuth(allow_yateam=True)]
    permissions = WebDavPermission() | TelemostPermission() | MobileMailPermission()
    serializer_cls = BroadcastDataSerializer
    auth_user_required = True
    auth_required = True

    kwargs = fields.QueryDict({
        'uri': fields.StringField(
            required=True,
            help_text=u'URI конференции, для которой выполняется управление трансляцией.')
    })


class CreateBroadcastHandler(BaseBroadcastHandler):
    service_method = 'PUT'
    service_url = '/v2/conferences/%(uri)s/broadcast?uid=%(uid)s'

    body_serializer_cls = CreateBroadcastBodySerializer

    def handle(self, request, *args, **kwargs):
        context = self.get_context()
        url = self.get_url(context)

        body_data = '{}'
        if self.request.body:
            body_data = to_json(self.request.body)

        uid = None
        if self.request.user and self.request.user.uid:
            uid = self.request.user.uid
        telemost_service = TelemostViaTCMService.get_telemost_api(uid, uri=context.get('uri'))

        resp = self.request_service(url, data=body_data, service=telemost_service,
                                    headers={'Content-type': 'application/json'})

        return self.serialize(resp)


# ===============================================================================
# Streams
# ===============================================================================

class BaseStreamHandler(BaseConferenceHandler):
    auth_methods = [PassportCookieAuth(), YaTeamOAuthAuth()]
    permissions = WebDavPermission() | TelemostPermission() | MobileMailPermission()
    serializer_cls = StreamDataSerializer
    auth_user_required = True
    auth_required = True
    service_timeout = 30

    kwargs = fields.QueryDict({
        'uri': fields.StringField(
            required=True,
            help_text=u'URI конференции, для которой выполняется управление стримом.'),
        'broadcast_uri': fields.StringField(
            required=True,
            help_text=u'URI трансляции, для которой выполняется управление стримом.')
    })


class StartStreamHandler(BaseStreamHandler):
    service_method = 'POST'
    service_url = '/v2/conferences/%(uri)s/broadcast/%(broadcast_uri)s/start?uid=%(uid)s'


class StopStreamHandler(BaseStreamHandler):
    service_method = 'POST'
    service_url = '/v2/conferences/%(uri)s/broadcast/%(broadcast_uri)s/stop'

    auth_user_required = False
    auth_required = False

    query = fields.QueryDict({
        'translator_token': fields.StringField(help_text=u'Токен, идентифицирующий транслятор.'),
    })

    def get_url(self, context=None, base_url=None):
        url = super(StopStreamHandler, self).get_url(context)

        url = self.patch_url_params(url, self.collect_uid_and_translator_token(context))

        return url


# ===============================================================================
# Viewers
# ===============================================================================

class BaseViewerHandler(BaseConferenceHandler):
    auth_methods = [PassportCookieAuth(), YaTeamOAuthAuth()]
    serializer_cls = ViewerDataSerializer

    kwargs = fields.QueryDict({
        'uri': None,
        'broadcast_uri': fields.StringField(
            required=True,
            help_text=u'URI трансляции, для которой получаются данные по стриму.')
    })


class GetStreamConnectionHandler(BaseViewerHandler):
    service_method = 'GET'
    service_url = '/v2/broadcast/%(broadcast_uri)s/connection'

# ===============================================================================
# Billing
# ===============================================================================

class BaseBillingHandler(TelemostAPIProxyHandler):
    permissions = WebDavPermission() | TelemostPermission() | TelemostBillingPermission()
    auth_methods = [PassportCookieAuth(), TVM2Auth(allow_yateam=True), TVM2TicketsOnlyAuth(allow_yateam=True)]

    service_method = 'PUT'

    auth_methods = [TVM2Auth(allow_yateam=True)]
    fanout_services = [TemplatedTelemostService(host)
                       for host in SEPARATE_TELEMOST_CLUSTERS['fallback_hosts']]
    service = FanOutToAllTelemostService(services=fanout_services)

    kwargs = fields.QueryDict({
        'uri': None
    })

    def get_service(self, context=None):
        return self.service


class BillingBroadcastEnableHandler(BaseBillingHandler):
    service_url = '/v2/billing/broadcast/enable?uid=%(uid)s'


class BillingBroadcastDisableHandler(BaseBillingHandler):
    service_url = '/v2/billing/broadcast/disable?uid=%(uid)s'
