#!/usr/bin/python
# -*- coding: utf-8 -*-
"""
MPFS Tools
Package Builder
"""
import ConfigParser
import json
import os
import re
import subprocess

from time import time

from build_common import copy_deb_related_files, remove_deb_related_files
from util import dist
from testing.report_tests import report_build


class SVNInfo(object):
    RELEASE_BRANCH_RELATIVE_URL = '\^/branches/disk/mpfs/releases/(release-.*)'
    RELEASE_BRANCH_REGEX = re.compile(r'^release-(\d+)\.(\d+)$')

    @staticmethod
    def get_svn_info():
        info = {}
        SEPARATOR = ': '

        res = subprocess.Popen(["svn", "info"], stdout=subprocess.PIPE)
        stdout = res.communicate()[0]
        for line in stdout.splitlines():
            if SEPARATOR not in line:
                continue
            key, value = line.split(SEPARATOR, 1)
            info[key] = value

        return info

    @staticmethod
    def get_last_release():
        release_versions = []
        SEPARATOR = ': '

        res = subprocess.Popen(["ya", "svn", "ls", "svn+ssh://arcadia.yandex.ru/arc/branches/disk/mpfs/releases"],
                               stdout=subprocess.PIPE)
        stdout = res.communicate()[0]
        for line in stdout.splitlines():
            if not line:
                continue
            branch = line.strip().strip('/')
            match = SVNInfo.RELEASE_BRANCH_REGEX.search(branch)
            if match:
                release_versions.append((int(match.group(1)), int(match.group(2))))

        return max(release_versions)

    @staticmethod
    def get_relative_url():
        info = SVNInfo.get_svn_info()
        return info.get('Relative URL')

    @staticmethod
    def get_release_branch():
        relative_url = SVNInfo.get_relative_url()
        if not relative_url:
            return None

        release_branch_pattern = re.compile(SVNInfo.RELEASE_BRANCH_RELATIVE_URL)
        matched = release_branch_pattern.match(relative_url)
        if not matched:
            return

        return matched.group(1)


class BuildConfig(object):
    CONFIG_PATH = 'tools/builder.conf'

    def __init__(self):
        self._config = ConfigParser.ConfigParser()
        self._config.optionxform = str
        self._config.read(self.CONFIG_PATH)

    def get_release_number(self):
        return self._config.get('global', 'release')

    def get_version(self):
        return self._config.get('global', 'version')

    def get_packages_list(self):
        return self._config.items('packages')

    def update(self, version=None, release=None):
        with open(self.CONFIG_PATH, 'wb') as configfile:
            if version is not None:
                self._config.set('global', 'version', str(version))
            if release is not None:
                self._config.set('global', 'release', str(release))
            self._config.write(configfile)


class MPFSRepo(object):
    """Работа с репо"""
    @classmethod
    def get_current_branch_version(cls):
        release_branch = SVNInfo.get_release_branch()
        version = release_branch.replace('release-', '')
        return version

    @classmethod
    def update_mpfs_version(cls, version, release=None):
        if release is None:
            release = 1
        cmd = "sed -ri \"s/__version__ = '[0-9\.]+'/__version__ = '%s.%s'/\" lib/mpfs/__init__.py" % (version, release)
        os.system(cmd)


class YaUpload(object):
    """
    Отправка ресурса в Sandbox
    """

    TTL_INF = 'inf'
    TASK_LINK_TMPL = 'https://sandbox.yandex-team.ru/task/%s/view'

    def __init__(self, source_dir):
        self.source_dir = source_dir

    def get_ya(self):
        ya_path = '/tmp/ya'
        if not os.path.isfile(ya_path):
            subprocess.check_call(
                'svn export svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/ya %s' % ya_path,
                shell=True
            )
        return ya_path

    def pack(self, package_name):
        """
        Запаковать tar.gz
        """
        subprocess.check_call(
            'tar -C %s/debian/%s/ -cvzf %s/debian/data.tar.gz --exclude=DEBIAN .'
            % (self.source_dir, package_name, self.source_dir),
            shell=True
        )

    def upload(self, package_name, package_version, days_to_live):
        """Загрузка в sandbox.

        :param package_name: Имя пакета
        :param package_version: Версия пакета
        :param days_to_live: Сколько дней хранить ресурс ("inf" - вечно)
        """
        package_full_name = "%s=%s" % (package_name, package_version)
        token = os.getenv('SANDBOX_TOKEN')
        if not token:
            raise ValueError('SANDBOX_TOKEN env var required')
        # Время существования ресурса (в днях)
        time_to_live_option = '--ttl=%s' % days_to_live
        command = [
            self.get_ya(),
            'upload',
            '--json-output',
            '-T=DISK_COMPRESSED_RESOURCE_APPLICATION',  # Общий тип для приложений в Диске
            '-a=linux',
            '--owner=DISK-ADMIN',
            '--token=%s' % token,
            time_to_live_option,
            '-d="%s"' % package_full_name,  # Описание ресурса в виде package_full_name
            '-A', package_full_name,  # Заводим аттрибут <package_name> со значением <package_version> для фильтрации
            '-A', "disk_app=%s" % package_name, # Заводим аттрибут disk_app для опередения типа приложения в ресурсе. Используется для фильтрации в деплойных правилах.
            '--prepare-yd-release', # Делаем возможным создание релизов из таска MDS_UPLOAD, после загрузки ресурса. Подробнее тут https://st.yandex-team.ru/DEPLOY-2252#60140694e018a15ba40ecb02
            '%s/debian/data.tar.gz' % self.source_dir,
        ]
        task_id = json.loads(subprocess.check_output(command))['task']['id']
        return self.TASK_LINK_TMPL % task_id

    def pack_and_upload(self, package_name, package_version, days_to_live=TTL_INF):
        self.pack(package_name)
        link = self.upload(package_name, package_version, days_to_live)
        report_build(package_version, package_name, link)


def build_pkg(version, release):
    os.system('dch --distribution=unstable -b -v %s-%s "Common Release"' % (version, release))
    os.system('debuild --no-tgz-check --preserve-envvar=MPFS_CONFIG_PATH')


def main_build():
    version = MPFSRepo.get_current_branch_version()

    config = BuildConfig()

    last_release_on_dist = dist.get_last_disk_release_on_dist(version)
    # Если на dist загружена более новая версия, чем по нашему конфигу сборки, то
    # возьмем более актуальную версию
    release = max(int(config.get_release_number()),
                  last_release_on_dist)
    new_release = release + 1

    print '"##teamcity[buildNumber \'%s-%s\']"' % (version, new_release)
    config.update(version=version, release=new_release)
    MPFSRepo.update_mpfs_version(version, new_release)

    subprocess.check_call('cp apps/setup.py .', shell=True)

    source_dir = os.path.abspath(os.curdir)
    ya_uploader = YaUpload(source_dir)

    packages = config.get_packages_list()
    for package_type, package_name in packages:
        copy_deb_related_files(source_dir, package_type)
        build_pkg(version, release=new_release)

        package_version = "%s-%s" % (version, new_release)
        os.chdir('../')
        ya_uploader.pack_and_upload(package_name, package_version)

        for to in ('yandex-precise', 'yandex-trusty', 'yandex-disk-common'):
            cmd = "dupload --to %s %s_amd64.changes --nomail" % (to, "%s_%s" % (package_name, package_version))
            subprocess.check_call(cmd, shell=True)

        remove_deb_related_files(source_dir, package_type)
        os.chdir(source_dir)

    subprocess.check_call('rm setup.py', shell=True)

    subprocess.check_call('svn commit --with-revprop arcanum:review-skip=yes --with-revprop arcanum:check-skip=yes -m "building all: version %s release %s platform deb"' % (version, new_release), shell=True)
    return "%s-%s" % (version, new_release)


if __name__ == "__main__":
    print "\tStart buildrelease\n"
    start = time()
    main_build()
    print "\n\tBuildrelease finished in %.3f seconds.\n" % (time() - start)
