# -*- coding: utf-8 -*-
"""
Тулза для сравнивания хешей в MFPS и сторадже

Через кладун локально выкачиватеся файл и сравнивается с хешами из запроса в MPFS
"""
import os
import requests
import shutil
import hashlib

from argparse import ArgumentParser

# отключаем предупреждения
requests.packages.urllib3.disable_warnings()

MPFS_HOST = 'http://mpfs01g.dsp.yandex.net'
DOWNLOAD_DIR = './'


def download(file_name, file_url):
    r = requests.get(file_url, stream=True)
    r.raise_for_status()
    file_path = os.path.join(DOWNLOAD_DIR, file_name)
    with open(file_path, 'wb') as f:
        shutil.copyfileobj(r.raw, f)
    return file_path


def get_hashes(file_path):
    hash_md5 = hashlib.md5()
    hash_sha256 = hashlib.sha256()
    with open(file_path, "rb") as f:
        for chunk in iter(lambda: f.read(4096), b""):
            hash_md5.update(chunk)
            hash_sha256.update(chunk)
    return {
        'md5': hash_md5.hexdigest(),
        'sha256': hash_sha256.hexdigest(),
        'size': os.path.getsize(file_path)
    }


def main(uid, path):
    params = {
        'uid': uid,
        'path': path,
        'meta': 'file_url,md5,sha256,size,file_mid'
    }
    resp = requests.get("%s%s" % (MPFS_HOST, '/json/info'), params=params)
    try:
        resp.raise_for_status()
    except requests.exceptions.HTTPError:
        print resp.content
        raise
    resp_data = resp.json()
    assert resp_data['type'] == 'file'
    mpfs_hashes = {
        'md5': str(resp_data['meta']['md5']),
        'sha256': str(resp_data['meta']['sha256']),
        'size': resp_data['meta']['size'],
    }
    file_path = download(resp_data['name'], resp_data['meta']['file_url'])
    storage_hashes = get_hashes(file_path)

    print '=' * 20
    print 'Local file path: %s' % os.path.abspath(file_path)
    print 'Disk file path:  %s' % resp_data['path']
    print 'Storage file id: %s' % resp_data['meta']['file_mid']
    for key in mpfs_hashes:
        print key.upper()
        print '\tMPFS: %s' % mpfs_hashes[key]
        print '\tSTOR: %s' % storage_hashes[key]
        print '\tIsOk: %s' % (storage_hashes[key] == mpfs_hashes[key])
    print '=' * 20


if __name__ == '__main__':
    parser = ArgumentParser(description='Сверка хешей')
    parser.add_argument('address', help='Адрес ресурса `uid:path`')
    namespace = parser.parse_args()
    uid, path = namespace.address.split(':', 1)
    main(uid, path)
